//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006-2012 Mikhail Fedotov <anyremote@mail.ru>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <time.h>
#include <unistd.h>
#ifdef __FreeBSD__
#include <signal.h>
#endif
#include <glib.h>

#include "atsend.h"
#include "btio.h"
#include "conf.h"
#include "common.h"
#include "utils.h"
#include "executor.h"
#include "dispatcher.h"

#include "pr_frontend.h"
#ifdef USE_L2CAP
#include "pr_l2cap.h"
#endif
#include "pr_web.h"

#define TMPDISCONN_TIME 60

extern CONF	conf;
extern char	tmp[MAXMAXLEN];
extern GAsyncQueue *q2disp;

char callerId	    [MAXLEN];

gboolean dispIsJoinable = TRUE;

int  callTimer	  = 0;

int  useCallId    = 1;
int  callOn       = 0;

int  gotExitSignal  = 0;

ConnectInfo connectInfo;

static int  initialized  = 0;

//
// fill ConnectInfo structure, return EXIT_ABORT in unsuccessful
//
static int definePort(char * portIn)
{

    if (portIn == NULL) {
        return EXIT_ABORT;
    }
    INFO2("[DS]: definePort() %s", portIn);

    char* strPort;

    // device should be in format:

    // deprecated
    // 1. (AT mode)     /dev/something							used with AT mode commands

    // 1. (AT mode)     rfcomm:00:12:EF:32:21:1A:xx (xx is integer from 1 to 32)
    // 2. (Server mode) socket:NNNN or 							used with java client
    //                  socket:/path/to/socket						used with java client
    // 3. (Server mode) bluetooth:NN or bluetooth (port will be = DEFAULT_BT_CHANNEL) 	used with java client
    // 4. local:/some/path								used with java client, like 1, but no AT commands
    // 5. ilirc:/dev/something 								like 1, but no AT commands
    // 6. stdio								                used with java client, like 1, but no AT commands
    // 7. (Server mode) web:NNNN							built-in web server

    if ((strPort = strstr(portIn, INET_SOCKET)) != NULL) {

        strPort += strlen(INET_SOCKET);

        // check is it port or unix socket
        char * ch = strPort;
        int isPort = 1;
        while (*ch != '\0') {
            if (isdigit(*ch)) {
                ch++;
            } else {
                isPort = 0;
                break;
            }
        }

        if (isPort) {

	    //connectInfo.proto = PROTO_ANYREMOTE;
	    connectInfo.mode = SERVER_TCP;
	    connectInfo.port = atoi(strPort);
            DEBUG2("[DS]: TCP Server mode, port %d\n", connectInfo.port);
	    
        } else {

            /*sprintf(tmp, "Unix socket Server mode. Use socket %s\n", strPort);
            logger(L_DBG, tmp);

            if ((ret = openSocketPort(SERVER_UX, -1, strPort)) < 0) {
            	return -1;
            }
	    connectInfo.proto = PROTO_ANYREMOTE;
	    connectInfo.mode = SERVER_UX;
	    g_string_assign(connectInfo.portStr, strPort);
            */
            printf("ERROR: incorrect port\n");
            return EXIT_ABORT;

        }
    } else if ((strPort = strstr(portIn, ILIRC_SOCKET)) != NULL) {

        strPort += strlen(ILIRC_SOCKET);
	connectInfo.mode = CLIENT_ILIRC;
	g_string_assign(connectInfo.portStr, strPort);
        DEBUG2("[DS]: Unix socket client mode, socket %s\n", connectInfo.portStr->str);

    } else if ((strPort = strstr(portIn, BT_SOCKET)) != NULL) {

        strPort += strlen(BT_SOCKET);

	connectInfo.mode = SERVER_BT;
        if (strstr(strPort, ":") == NULL) { // just "bluetooth
	    connectInfo.port = DEFAULT_BT_CHANNEL;
        } else {
            strPort++;
            connectInfo.port = atoi(strPort);
        }
        DEBUG2("[DS]: Bluetooth Sockets Server mode, channel %d", connectInfo.port);

    #ifdef USE_L2CAP
    } else if ((strPort = strstr(portIn, L2CAP_SOCKET)) != NULL) {

        strPort += strlen(L2CAP_SOCKET);

	connectInfo.mode = SERVER_L2CAP;
        if (strstr(strPort, ":") == NULL) { // just "l2cap"
            connectInfo.port = DEFAULT_L2CAP_PORT;
        } else {
            strPort++;
            connectInfo.port = atoi(strPort);
        }
        DEBUG2("[DS]: L2CAP Server mode, port %d", connectInfo.port);
	
    #endif

    } else if ((strPort = strstr(portIn, UNIX_SOCKET)) != NULL) {

        strPort += strlen(UNIX_SOCKET);
	
	connectInfo.mode = CLIENT_NOAT;
	g_string_assign(connectInfo.portStr, strPort);
        DEBUG2("[DS]: Serial Client mode (no AT). Use device %s", connectInfo.portStr->str);

    } else if ((strPort = strstr(portIn, STDIN_STREAM)) != NULL) {

 	connectInfo.mode = SERVER_STDIN;

    } else if ((strPort = strstr(portIn, AT_DEVICE)) != NULL) {

        if (!(strlen(portIn) == strlen(AT_DEVICE) + BT_ADDR_LEN + 3    || // 00:12:EF:32:21:1A:p
                strlen(portIn) == strlen(AT_DEVICE) + BT_ADDR_LEN + 4) || // 00:12:EF:32:21:1A:pp
                portIn[strlen(AT_DEVICE)] != ':' ||
                portIn[strlen(AT_DEVICE) + BT_ADDR_LEN + 1] != ':') {
            printf("ERROR: Improper connect string !\n");
            return EXIT_ABORT;
        }

        char sBtAddr[18];
        strncpy(sBtAddr,portIn + strlen(AT_DEVICE) + 1,17);
        sBtAddr[17] = '\0';

 	connectInfo.mode    = CLIENT_RFCOMM;
	connectInfo.port    = atoi(portIn + strlen(AT_DEVICE) + BT_ADDR_LEN + 2);;
	g_string_assign(connectInfo.portStr, sBtAddr);
	
        DEBUG2("[DS]: Serial Client mode, device %s : %d", connectInfo.portStr->str, connectInfo.port);
	
    } else if ((strPort = strstr(portIn, WEB_SOCKET)) != NULL) {
	
        strPort += strlen(WEB_SOCKET);

        // check is it port
        char * ch = strPort;
        int isPort = 1;
        while (*ch != '\0') {
            if (isdigit(*ch)) {
                ch++;
            } else {
                isPort = 0;
                break;
            }
        }

        if (isPort) {

 	    connectInfo.mode = SERVER_WEB;
	    connectInfo.port = atoi(strPort);
            DEBUG2("[DS]: Web Server mode. Use port %d", connectInfo.port);
	    
        } else {	
            logger(L_ERR,"[DS]: can not determine web server port");
            return EXIT_ABORT;
        }
	
    } else {
    
 	connectInfo.mode = CLIENT_AT;
	g_string_assign(connectInfo.portStr, portIn);
        DEBUG2("[DS]: Serial Client mode. Use device %s", connectInfo.portStr->str);
    }

    return EXIT_OK;
}

static int openPort(ConnectInfo* connInfo)
{
    logger(L_INF, "[DS]: openPort()");
    
    switch (connInfo->mode) {
        
	case SERVER_BT:
	case SERVER_TCP:
	    
	    if (connInfo->mode == SERVER_BT) {
	        DEBUG2("[DS]: Bluetooth Sockets Server mode. Use channel %d", connInfo->port);
	    } else {
	        DEBUG2("[DS]: TCP Server mode. Use port %d", connInfo->port);
	    }
	    
            if (openSocketPort(connInfo->mode, connInfo->port, NULL) < 0) {
                return EXIT_NOK;
            }
	    break;
	
	case CLIENT_ILIRC:
	    
	    DEBUG2("[DS]: Unix socket client mode. Use port %s", connInfo->portStr->str);
            if (unix_open_port(connInfo->portStr->str) < 0) {
            	printf("ERROR: open inputlircd socket\n");
            	return EXIT_NOK;
            }
	    break;
	    
	#ifdef USE_L2CAP
	case SERVER_L2CAP:
	     
	    DEBUG2("[DS]: L2CAP Server mode. Use port %d", connInfo->port);
            if (openL2capPort(connInfo->port) < 0) {
                return EXIT_NOK;
            }
	    break;
	#endif
	
	case CLIENT_NOAT:
	    
	    // nothing to do
	    break;
	
	case SERVER_STDIN:
	    
	    DEBUG2("[DS]: Server mode. Use standard input");
	    openStdin();
	    break;
	    
	case CLIENT_RFCOMM:
       
            DEBUG2("[DS]: Serial Client mode. Use device %s : %d", connInfo->portStr->str, connInfo->port);
            if (rfcommConnect(connInfo->portStr->str, connInfo->port) < 0) {
                logger(L_ERR,"[DS]: can not open RFCOMM channel");
                return EXIT_NOK;
            }
	    break;
	
	case SERVER_WEB:
	
	    DEBUG2("[DS]: Web Server mode. Use port %d", connInfo->port);
	    if (openWebPort(connInfo->port) < 0) {
                return EXIT_NOK;
            }
	    break;
	
	case CLIENT_AT:
        
	    DEBUG2("[DS]: Serial Client mode. Use device %s", connInfo->portStr->str);
	    if (openSerialPort(connInfo->portStr->str) < 0) {
                logger(L_ERR,"[DS]: can not open serial port");
                return EXIT_NOK;
	    }
	    break;
        
	default:
	    
	    DEBUG2("[DS]: Improper mode %d", connInfo->mode);
	    return EXIT_NOK;
    }
    return EXIT_OK;
}

// Then port will be closed from a forked child use 0
void closePort(int final)
{
    switch (connectInfo.mode) {

        case SERVER_BT:
        case SERVER_TCP:
        case SERVER_UX:
	
            closeSocketPort(connectInfo.mode, final);
	    break;

        #ifdef USE_L2CAP
        case SERVER_L2CAP:
	
            closeL2capPort(final);
	    break;
        #endif
 
        case CLIENT_ILIRC:

            if (unix_close_port() < 0) {
                logger(L_ERR,"[DS]: Error on closing AF_UNIX socket\n");
            }
 	    break;

        case CLIENT_RFCOMM:
	    
            rfcommClose();
 	    break;
   
        case CLIENT_AT:
        case CLIENT_NOAT:
        
	    if (closeSerialPort(final) < 0) {
                logger(L_ERR,"[DS]: Error on closing port\n");
            }
 	    break;

        case SERVER_STDIN:
    
            if (final)  {
	        closeStdin();
	    }
 	    break;
	    
        case SERVER_WEB:
    
            closeWebPort(final);
 	    break;
    }
    
    if (getFrontEnd() > 0) {
        if (final)  {
            sendToFrontEnd("Exiting");
	}
        disconnectFrontEnd();
    }
}

static void handleActiveCall(void)
{
    if (callTimer >= 20) {
        logger(L_DBG, "[DS]: Get call list again");

        // Will set global callerId [MAXLEN];
        int ret = getClip(callerId);
        if (ret == EXIT_EXACT) {
            DEBUG2("[DS]: Call still active (%s)", callerId);
        } else {
            logger(L_DBG,"[DS]: Call was finished");
            callOn = 0;
            strcpy(callerId, "NO CALLER ID");

            sendEventToExecutor(ID_EVT_ENDCALL);
        }
        callTimer = 0;
    } else {
        callTimer++;
    }
}

//
// return code is used in case of exit only
//
static int proceedDisconnect(void)
{
    int canForceExit = 0;
    
    if (connectInfo.mode == SERVER_BT  ||
        connectInfo.mode == SERVER_TCP ||
        connectInfo.mode == SERVER_UX) {

        logger(L_INF, "[DS]: Got exit event: send disconnect message");
        if (writeSocketConn("Set(disconnect);", 16) != EXIT_OK) {
            canForceExit = 1;
        }

    } else if (connectInfo.mode == CLIENT_RFCOMM || 
               connectInfo.mode == CLIENT_AT) {

        sendCMER(CMER_OFF);
        canForceExit = 1;

    #ifdef USE_L2CAP
    } else if (connectInfo.mode == SERVER_L2CAP) {

        if (writeL2capConn("Set(disconnect);", 16) != EXIT_OK) {
            canForceExit = 1;
        }
    #endif
    
    } else {
        canForceExit = 1;
    }
    return canForceExit;
}

static char* readFromFile(char *cmdTag, char* file, int* size) 
{
    FILE *fp;
    struct stat buf;

    long fLen = 0;
    if(stat(file, &buf) == -1) {
        logger(L_ERR,"can't get file size!");
    } else {
        fLen = (int) buf.st_size;
	
	if (!S_ISREG (buf.st_mode)) {
            logger(L_ERR,"not regular file");
	} else {
	
	    fp=fopen(file,"r");
	    if (fp == NULL) {
        	logger(L_ERR,"can't open file!");
	    } else {
	    
                int prefixSz = strlen(cmdTag);

		uint32_t szh32 = (uint32_t) buf.st_size;
		uint32_t szi32 = htonl(szh32);

		char * fBuffer = (char*) calloc(fLen+prefixSz+7,1);	// 7 =   <size of bin data=4bytes>    ");\0"
		if (fBuffer == NULL) {
        	    logger(L_ERR,"no more memory!");
		    
		    *size = -1;
        	    return NULL;
		}

		strcat(fBuffer, cmdTag);
		memcpy((void*)fBuffer+prefixSz, (const void *) &szi32, 4);	// length on binary data

		if (fp) {
        	    #ifdef __cplusplus
        	    size_t dummy =  
        	    #endif
        	    fread(fBuffer+prefixSz+4, sizeof(char), fLen, fp);

        	    fclose(fp);
		}
		strcpy(fBuffer+prefixSz+fLen+4,");");
		
		*size = fLen+prefixSz+7;
		return fBuffer;
	    }
	}
    }
    
    if (strncmp(cmdTag, "cover", 5) == 0) {  // cleanup the cover on client
    
        char * fBuffer = (char*) malloc(12);
	strcpy(fBuffer, "Set(cover);");
	
	*size = 11;
	return fBuffer;
    }
    
    *size = -1;
    return NULL;
}

static int checkMsgQueue()
{
    //logger(L_DBG,"[DS]: checkMsgQueue");

    // Verify commands from queue (timeout about 1/2 sec)
    dMessage* dm = (dMessage*) g_async_queue_try_pop(q2disp);
    if (dm != NULL) {
        //logger(L_DBG, "[DS]: Got event");

        if (dm->type == DM_EVENT_INIT) {
            logger(L_INF, "[DS]: Got init OK event");
            free(dm);
	    
            return EXIT_INITOK;
        } else if (dm->type == DM_EVENT_EXIT) {

            logger(L_INF, "[DS]: Got exit event");

            gotExitSignal = 1;

            // in server mode wait client to close connection
            // and then exit
            if (proceedDisconnect()) {
                logger(L_INF, "[DS]: Got exit event: exit immediately");
                closePort(1);

                free(dm);
                return EXIT_ABORT;
            }

        } else if (dm->type == DM_EVENT_DISCONNECT) {

            logger(L_INF, "[DS]: Got disconnect event");
            proceedDisconnect();
            closePort(1);

            free(dm);
            return EXIT_DISCON;

        } else if (connectInfo.connected) {
	    
            char *msgIn = (char*) dm->value;

            if (dm->type == DM_SET) {
	    
	        INFO2("[DS]: DM_SET %d", connectInfo.mode);

		#ifdef USE_L2CAP
                if (connectInfo.mode == SERVER_L2CAP) {
                    if (writeL2capConn(msgIn, dm->size) != EXIT_OK) {
                        logger(L_DBG,"[DS]: Fails in writeBytesL2capConn()");
                    }
                } else 
		#endif
		if (connectInfo.mode == SERVER_WEB) {
		    writeWebConn(dm);
		} else {
		    logger(L_DBG, "[DS]: Set()");
                    writeSocketConn(msgIn, dm->size);
                }
		
            } else if (dm->type == DM_SETFILE) {
	    
	        INFO2("[DS]: DM_SETFILE %d", connectInfo.mode);

		if (connectInfo.mode == SERVER_WEB) {
		    writeWebConn(dm);
		} else {
		    
		    // read data from file
		    int size = 0;
		    char* buf = readFromFile(dm->value, dm->file, &size);
                    
		    INFO2("[DS]: got from file %d bytes", size);

                    if (buf && size > 0) {
			#ifdef USE_L2CAP
                	if (connectInfo.mode == SERVER_L2CAP) {
                            if (writeL2capConn(buf, size) != EXIT_OK) {
                        	logger(L_DBG,"[DS]: Fails in writeBytesL2capConn()");
                            }
                	} else {
			#endif

			   logger(L_DBG, "[DS]: Set() with binary data");
                	   writeSocketConn(buf, size);

			#ifdef USE_L2CAP
			}
			#endif
		    }
		    free(buf);
                }
		
            } else if (dm->type == DM_SENDS) {

                logger(L_DBG, "[DS]: Send(string)");
                if (writeSocketConn(msgIn, dm->size) != EXIT_OK) {
                    logger(L_DBG,"[DS]: Fails in writeSocketConn()");
                }

            } else if (dm->type == DM_SENDB) {

                logger(L_DBG, "[DS]: Send(bytes)");
                if (connectInfo.mode != SERVER_L2CAP) {
                    if (writeBytesSocketConn(msgIn) != EXIT_OK) {
                        logger(L_DBG,"[DS]: Fails in writeBytesSocketConn()");
                    }
		} else {
		    #ifdef USE_L2CAP
                    if (writeBytesL2capConn(msgIn) != EXIT_OK) {
                        logger(L_DBG,"[DS]: Fails in writeBytesL2capConn()");
                    }
		    #endif
		}
            } else if (dm->type == DM_CKPD) {

                logger(L_DBG, "[DS]: SendCKPD");
                sendSeq(msgIn);

            } else if (dm->type == DM_CMER) {

                logger(L_DBG, "[DS]: Send CMER");
                sendCMER(dm->size);

            } else {
                logger(L_DBG,"[DS]: Invalid message type");
            }
        } else {
            logger(L_DBG, "[DS]: No connection. Skip event");
        }

        if (dm->value != NULL) {
            free(dm->value);
        }
        if (dm->file != NULL) {
            free(dm->file);
        }
        free(dm);
    }
    return EXIT_OK;
}

void parseCommand(char* cmd)
{
    char *prev, *next;

    if (cmd == NULL) {
        return ;
    }

    //skip lines starting with \n and \r
    if (cmd[0] == '\r') {
        cmd++;
    }
    if (cmd[0] == '\n') {
        cmd++;
    }

    // most common case
    if (!cmd[0]) {
        return;
    }

    // if recieved multiline command - handle line by line and return
    prev = cmd;
    next = strchr(cmd, '\r');
    
    if (next == NULL &&
        !(connectInfo.mode == CLIENT_RFCOMM || 
	  connectInfo.mode == CLIENT_AT)) {	// Java client will send +CKEV: 1,1; +CKEV: 1,0
	  
        next = strchr(cmd, ';');
    }
    
    if (next == NULL && 
        !(connectInfo.mode == CLIENT_RFCOMM || 
	connectInfo.mode == CLIENT_AT)) {
	
        next = strchr(cmd, '\3');		// end-of-text marker in CommandFusion
    }
    
    if (next) {
        logger(L_DBG, "[DS]: parseCommand multiline");

        char copy[1024];
        int len;

        do {
            len = next-prev;

            if (len >= 2) {
                memcpy(copy, prev, len);
                copy[len] = 0;

                // use recursion
                parseCommand(copy);
            }
            prev = next+1;
            next = strchr(prev, '\r');
            if (next == NULL &&
                !(connectInfo.mode == CLIENT_RFCOMM || 
		  connectInfo.mode == CLIENT_AT)) { // Java client will send +CKEV: 1,1; +CKEV: 1,0
		  
                next = strchr(prev, ';');
            }
	    
            if (next == NULL &&
                !(connectInfo.mode == CLIENT_RFCOMM || 
		  connectInfo.mode == CLIENT_AT)) {
		  
                next = strchr(prev, '\3');		// end-of-text marker in CommandFusion
            }
        } while (next) ;

        // now return
        return;
    }

    logger(L_DBG,"[DS]: -------------------- Command read --------------------");
    DEBUG2("[DS]: parseCommand >%s<", cmd);

    if (IS_OK(cmd)) {   // OK - nothing to do
        return;
    }

    if (strncmp(cmd, "AT+CKPD=", 8) == 0) {        // This is echo of sent message in AT mode; nothing to do
        return;
    }

    if (strncmp(cmd, "Msg:_DEBUG_(", 12) == 0) {   // This is debug message from java client; nothing to do
        return;
    }

    // hack to make it work correctly with Bemused clients
    int hack = 0;
    char c;
    if (strncmp(cmd, "VOLM", 4) == 0 && strlen(cmd) == 4) {  // read only VOLM without value to set
        logger(L_DBG,"[DS]: read one more byte");
        char ret;
        if ((ret = bt_readchar(&c,500000)) < 0) {
            DEBUG2("[DS]: parseCommand: Bemused hack: read < 0");
        } else {
            sprintf(tmp, "[DS]: parseCommand: Bemused hack: read >%c<", c);
            logger(L_DBG,tmp);
            hack = 1;
        }
    }

    // This is heartbeat of iViewer; answer to it fast
    if (strncmp(cmd, "h=0", 3) == 0 && getIViewer()) {
        logger(L_DBG, "[DS]: Got heartbeat message from iViewer; auto-reply to it");

        // it is a trick, we do not want to waste a time to handle message
        // through normal "shell" cycle
        sendIViewerHeartbeat();

        // propagate event to executor anyway
        // (to be able to disconnect in case of client dying)
    }


    eMessage* em = (eMessage*) malloc(sizeof(eMessage));
    em->type  = EM_KEY;
    if (hack) {
        em->value = malloc(strlen(cmd)+2);
        strcpy(((char*)em->value),cmd);
        ((char*)em->value)[strlen(cmd)]   = c;
        ((char*)em->value)[strlen(cmd)+1] = '\0';
    } else {
        em->value = strdup(cmd);
    }
    logger(L_DBG, "[DS]: Send message to executor thread");
    sendToExecutor(em);

    return;
}

static int read_command(char* buf, int l)
{
    int ch, len = 0;
    //printf("read_command\n");

    if (connectInfo.mode == SERVER_STDIN) {
        /* this blocks any other operation
                char* r = fgets((char*)buf, l, (FILE*)stdin);
            printf("read_command SERVER_STDIN %d\n", len);
        if (r == NULL) {
        	buf[0] = 0;
        	printf("read_command SERVER_STDIN empty\n");
                	return 0;
        } else {
                len = strlen(buf);
        	printf("read_command SERVER_STDIN %d\n", len);
        }
                */

        int ret = readStdin(buf,l);
        if (ret == 0) {
            buf[0] = 0;
            return 0;
        } else {
            len = strlen(buf);
            DEBUG2("[DS]: read_command SERVER_STDIN %d %s\n", len, buf);
        }
    } else if (connectInfo.mode == SERVER_WEB) {
        
	return checkWebPort(buf, l);
	
    } else {

        char* p = buf;

        while (len < l) {
            char c;
            ch = bt_readchar(&c,100);

            if (ch == EOF) {
                buf[0] = 0;
                return EOF;
            } else if (ch == EOF-1) {
                break;
            } else if (ch >= 0 && (c == '\r' || c == ';')) {
                break;
            } else  if (ch >= 0 && c != '\r' && c != ';') {
                *p++ = c;
                len++;
            }
        }
    }
    buf[len] = '\0';
    return len;
}

int initPort()
{
    char answer[1024];
    char cmd[32];

    logger(L_DBG,"[DS]: initPort");

    if (connectInfo.mode == SERVER_BT  ||
        connectInfo.mode == SERVER_TCP ||
        connectInfo.mode == SERVER_UX) {  // Wait for incoming connection
        
	logger(L_INF, "[DS]: Server mode: Waiting connection");
        return listenAndAcceptSocketConn(connectInfo.mode);
    }

    #ifdef USE_L2CAP
    if (connectInfo.mode == SERVER_L2CAP) {
        logger(L_INF, "[DS]: Server mode/L2CAP: Waiting connection");
        return listenAndAcceptL2capConn();
    }
    #endif
    
    if (connectInfo.mode == SERVER_WEB) {
        logger(L_INF, "[DS]: Built-in web server: init OK");
	return 1;  // nothing to init
    }

    if (connectInfo.mode == CLIENT_ILIRC ||
        connectInfo.mode == CLIENT_NOAT  ||
        connectInfo.mode == SERVER_STDIN) {
	
        logger(L_INF, "[DS]: Unix client: init OK");
        return 1;	// nothing to init
    }
    
    // CLIENT_RFCOMM / CLIENT_AT
    
    answer[0] = '\0';
    sendSyncCommand("ATZ\r", answer, sizeof(answer), NULL);
    //sendSyncCommand("AT\r", answer, sizeof(answer), NULL);
    // Do not care - it can fail after reconnect
    //if(!IS_OK(answer)) {
    //    printf("ERROR: AT -> %s\n", answer);
    //    return -1;
    //}

    sendSyncCommand("ATE0\r", answer, sizeof(answer), NULL);
    if(!IS_OK(answer)) {
        printf("ERROR: ATE0 -> %s\n", answer);
        return -1;
    }
    char* ptr = getCharset();
    if (ptr) {
        sprintf(cmd,"%s\"%s\"\r", DEF_AT_CSCS, ptr);
        sendSyncCommand(cmd, answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
            printf("ERROR: Can't set charset to %s\n", ptr);
        }
        free(ptr);
    }

    // Determine model
    sprintf(cmd,"%s\r", DEF_AT_CGMI);
    sendSyncCommand(cmd, answer, sizeof(answer), NULL);

    //Set model in conf. data
    setModel(answer);

    if (getModel() == MODEL_MOTOROLA) {	// Motorola RIZR Z3 needs to set MODE=2 to allow AT+CLIP command
        // do not care about responce
        sendSyncCommand("AT+MODE=2\r", answer, sizeof(answer), NULL);
    }

    sprintf(cmd,"%s\r", DEF_AT_CLIP);
    sendSyncCommand(cmd, answer, sizeof(answer), NULL);
    if(!IS_OK(answer)) {
        printf("ERROR: Can't set CLIP ON\n");
    }


    int ret = sendCMER(CMER_ON);
    if(ret != EXIT_OK) {
        printf("ERROR: fails in set event reporting on");
        return -1;
    }

    // Siemens S55 needs additional AT+CMEC=2 to make AT+CKPD works
    // not sure about other Siemens phones
    if (getModel() == MODEL_SIEMENS) {
        sprintf(cmd,"%s\r", DEF_AT_CMEC);
        sendSyncCommand(cmd, answer, sizeof(answer), NULL);
        if(!IS_OK(answer)) {
            printf("ERROR: ON AT+CMEC\n");
        }
    }

    // Will set global callerId [MAXLEN];
    ret = getClip(callerId);
    if(ret == EXIT_NOK) {       // Got ERROR on AT+CLCC; probably phone does not supports this command
        printf("ERROR: fails in getClip\n");
        useCallId = 0;
    }
    return 1;
}

static void hookInitOnce() 
{
    if (initialized == 0) {

        // setgid
        if(conf.uid && getuid()==0) {
            DEBUG2("[DS]: setuid/setgid %d,%d",conf.uid,conf.gid);
            setgid(conf.gid);
            setuid(conf.uid);
        }

        initialized++;
    }
}

int doCheckPassword()
{
    logger(L_DBG,"[DS]: Do password verification");

    char answer[MAXCMDLEN];
    int ret = EXIT_NOK;
    
    char *pass = getStoredPass();
    if (pass) {
        int gotReply = 1;
        while (1) {

            if (gotReply == 1) {
        	sendToFrontEnd("Verifying password");
		
		if (connectInfo.mode == SERVER_WEB) {
		    ret = writeWebConnStr("Get(password);");
		} else {
        	    ret = writeSocketConn("Get(password);", 14);
                }

        	if (ret != EXIT_OK) {
        	    logger(L_DBG,"[DS]: Fails in writeSocketConn()");
        	    continue;
        	}
        	if (getHttp() || connectInfo.mode == SERVER_WEB) {
		    if (connectInfo.mode == SERVER_WEB) {
		        ret = writeWebConnStr("End();");
		    } else {
        	        ret = writeSocketConn("End();", 6);
		    }
        	    if (ret != EXIT_OK) {
        		logger(L_DBG,"[DS]: Fails in writeSocketConn()");
        		continue;
        	    }
        	}
        	gotReply = 0;
            }

            int retRead = read_command(answer, sizeof(answer));

            if (retRead == EOF) {
        	logger(L_DBG,"[DS]: Got disconnect on verifying password");
        	sendToFrontEnd("Disconnected");
        	sendDisconnect();
        	break;
            } else if (retRead > 0) {
        	DEBUG2("[DS]: Password => %s\n",answer);
        	//DEBUG2("[DS]: Password => %s\n",answer+16);
        	//DEBUG2("[DS]: Compare to => %s\n",pass);

        	gotReply = 1;

        	if (memcmp(answer,   DEF_MSG, 4) == 0 &&
                    memcmp(answer+4, "_PASSWORD_(,", 12) == 0) {	    // Got verification response from Java client

        	    int plen = strlen(pass);
        	    if (memcmp(answer+16, pass, plen) == 0) {	// _PASSWORD_(,<password here>)
        		logger(L_INF,"[DS]: Password OK");
        		free(pass);
			
			ret = EXIT_OK;
        		break;
        	    } else {
        		logger(L_INF,"[DS]: Verification failed. Incorrect password.");
        		sendToFrontEnd("Verification failed");
        	    }

        	} else {
        	    logger(L_INF,"[DS]: Verification failed. Incorrect response.");
        	    sendToFrontEnd("Verification failed");
        	}
            }

            if (checkMsgQueue() == EXIT_ABORT) {
         	return EXIT_ABORT;
            }
            usleep(50000);
        }
    } else {
        ret = EXIT_OK;
    }
    
    if (pass && ret == EXIT_OK) {
        logger(L_DBG,"[DS]: Password verification OK");
    } else if (pass && ret == EXIT_NOK) {
        logger(L_DBG,"[DS]: Connection problem.");
    } else if (!pass) {
        logger(L_DBG,"[DS]: No password specified. Skip check.");
    }
    return ret;
}

int doMessageLoop()
{	            
    char answer[MAXCMDLEN];
    int ret = EXIT_DISCON;
    
    while (1) {
        //logger(L_DBG,"[DS]: main loop");

        int retRead = read_command(answer, sizeof(answer));
        //if (retRead != 0) printf("GOT %d\n",retRead);

        if (retRead == EOF) {
            logger(L_INF, "[DS]: Got disconnected");
            //printf("Got disconnected\n");

            freeBtAddress();

            if (gotExitSignal) {
                logger(L_INF, "[DS]: Got signal, exiting");
                closePort(1);
                return EXIT_ABORT;

            }
            sendToFrontEnd("Disconnected");
            sendDisconnect();
            sendEventToExecutor(ID_EVT_DISCONNECT);

            if (connectInfo.mode == CLIENT_RFCOMM ||
                connectInfo.mode == CLIENT_AT     ||
                connectInfo.mode == CLIENT_ILIRC  ||
                connectInfo.mode == CLIENT_NOAT) {  // In server mode we will reuse already opened socket

                logger(L_INF, "[DS]: Closing the port");
                closePort(1);
            }
            ret = EXIT_NOK;

            connectInfo.connected = FALSE;

            break;

        } else if (retRead > 0) {

            if (connectInfo.mode == CLIENT_NOAT) {	// in IR communication a lot of 'empty' chars (= -1) could be added
                char * k2 = answer;
                while (*k2 != '\0') {
                    if (*k2 == -1) {
                        *k2 = '\r';
                    }
                    k2++;
                }
            }

            parseCommand(answer);
            if (getHttp() || connectInfo.mode == SERVER_WEB) {

                eMessage* em = (eMessage*) malloc(sizeof(eMessage));
                em->type  = EM_AS_IS;
                em->value = strdup("End();");
                sendToExecutor(em);
            }
        }

        // Is call still active (timeout about 1 seconds) ?
        if (callOn) {
            handleActiveCall();
        }

        ret = checkMsgQueue();
        if (ret == EXIT_ABORT) {
            return EXIT_ABORT;
        } else if (ret == EXIT_DISCON) {
            break;
        }

        // Main loop timer (1/50 of second)
        usleep(20000);
    } // while (forever)
    
    return ret;
}		

gpointer dispatcherRoutine(gpointer thread)
{
    int ret = EXIT_OK;

    logger(L_DBG,"[DS]: start dispatcher thread");

    connectInfo.portStr = g_string_new("");
    connectInfo.connected   = FALSE;

    setCmdFiles();

    // wait init ok event
    while (1) {
        ret = checkMsgQueue();
        if (ret == EXIT_ABORT) {
            return NULL;
        } else if (ret == EXIT_INITOK) {
            break;
        }
        logger(L_DBG,"[DS]: wait init OK event");
        usleep(50000);
    }

    int dmn = autoConnect();
    int rs  = getRetrySecs();
    strcpy(callerId, "NO CALLER ID");
        
    char* dv = getDevice();
    
    if (definePort(dv) == EXIT_ABORT) {
    	logger(L_DBG,"[DS]: Incorrect device specification");
	free(dv);
	return NULL;
    }
    free(dv);

    while (1) {
 
        logger(L_DBG,"[DS]: ************ outer loop **********");
	
        if(openPort(&connectInfo) == EXIT_OK) {   // Open device

            logger(L_DBG,"[DS]: Device open OK");

            hookInitOnce();

            dispIsJoinable = FALSE;

            if (initPort() == 1 ) {  // Init modem: AT, ATE0, AT+CMER, in server mode waits for connection

                logger(L_DBG,"[DS]: Init connection OK");

                dispIsJoinable = TRUE;

                connectInfo.connected = TRUE;
                
		int pass_ok = (usePassword() ? doCheckPassword() : EXIT_OK);

                if (pass_ok == EXIT_ABORT) {
                     return NULL;
                } 

                if (pass_ok == EXIT_OK) {  // not EOF
		
                    sendToFrontEnd("Connected");
                    sendEventToExecutor(ID_EVT_CONNECT);

                    logger(L_INF,"[DS]: Start event forwarding");

                    int retLoop = doMessageLoop();
                    if (retLoop == EXIT_ABORT) {
                        return NULL;
                    } 
		    // EXIT_DISCON / EXIT_NOK is OK here

                } else {  	// not EOF after password verification
                    logger(L_DBG,"[DS]: Connection EOF");
                }
            } else {		// init port
                logger(L_DBG,"[DS]: Init connection error");
            }
        } else {		// open port
            logger(L_DBG,"[DS]: Device open error");
        }

        //printf("Connection closed or lost\n");
        logger(L_INF, "[DS]: Connection closed or lost");

        // Can't open port or it closed again
        if (!gotExitSignal &&
                (dmn ||
                 connectInfo.mode == SERVER_BT   ||
                 #ifdef USE_L2CAP
                 connectInfo.mode == SERVER_L2CAP ||
                 #endif
                 connectInfo.mode == SERVER_TCP  ||
                 connectInfo.mode == SERVER_WEB  ||
                 connectInfo.mode == SERVER_UX   ||
                 connectInfo.mode == CLIENT_NOAT ||
                 ret == EXIT_DISCON)) {

            int timeout;

            if (connectInfo.mode == SERVER_BT  ||
                #ifdef USE_L2CAP
                connectInfo.mode == SERVER_L2CAP ||
                #endif
                connectInfo.mode == SERVER_TCP ||
                connectInfo.mode == SERVER_WEB ||
                connectInfo.mode == SERVER_UX  ||
                connectInfo.mode == CLIENT_NOAT) {

                timeout = 2;	// wait only 2 seconds

            } else if (ret == EXIT_DISCON) {

                timeout = TMPDISCONN_TIME;
                ret = EXIT_OK;

            } else {
                timeout = rs;
            }

            INFO2("[DS]: Wait %d seconds to connect/open server socket ...", timeout);
            //printf("Wait %d seconds to connect/open server socket ...\n", timeout);

            dispIsJoinable = FALSE;
            sleep(timeout);
            dispIsJoinable = TRUE;

        } else {	// Proceed to exit
            printf("Exiting ...\n");
            sendAbort();
            break;
        }
    }

    // Finish all
    logger(L_INF, "[DS]: Stop dispatcher ...");
    closePort(1); 
    
    g_string_free(connectInfo.portStr,TRUE);

    return NULL;
}

void sendToDispatcher(dMessage *buf)
{
    if (q2disp == NULL || buf == NULL) {
        return;
    }

    DEBUG2("send to dispatcher %d", buf->type );

    g_async_queue_push(q2disp,buf);

    //free(buf);
}
