/**********************************************************************
 ** Skill - contains the attributes needed to store info on a skill
 **  
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef SKILL_C
#define SKILL_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "skill.h"
#include "objtype.h"
#include "newfuncts.h"
#include "flags.h"
#include "player.h"
#include "skillflags.h"

extern char *depend_name[];

/***********************************************************************
 ** Skill (constructor) - loads skill name and initializes attributes
 **
 ** Parameters: spell_name - this spell name 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Skill::Skill(char *skill_name)
{
   set_name(skill_name);
   set_area("ability");

   obj_type = OBJ_TYPE_SKILL;

   min_str = min_dex = 0;
   skill_flags = new_Flags(1);
}


/***********************************************************************
 ** ~Skill (destructor) - cleans up for destruction
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Skill::~Skill(void)
{
   delete_Flags(skill_flags);
}

/***********************************************************************
 ** set_min_str - sets the minimum strength the user must have
 **
 ** Parameters: new_min_str - the new value
 **
 ***********************************************************************/

void Skill::set_min_str(int new_min_str)
{
   if (new_min_str >= 0)
      min_str = new_min_str;
}


/***********************************************************************
 ** get_min_str - returns the minimum strength the user must have
 **
 ***********************************************************************/
   
int Skill::get_min_str()
{
   return min_str;
}


/***********************************************************************
 ** set_min_dex - sets the minimum dexterity the user must have
 **
 ** Parameters: new_min_dex - the new value
 **
 ***********************************************************************/

void Skill::set_min_dex(int new_min_dex)
{
   if (new_min_dex >= 0)
      min_dex = new_min_dex;
}


/***********************************************************************
 ** get_min_dex - returns the minimum dexterity the user must have
 **
 ***********************************************************************/
   
int Skill::get_min_dex()
{
   return min_dex;
}


/***********************************************************************
 ** get_skillflags - returns a pointer to the skill flags
 **
 ***********************************************************************/

Flags *Skill::get_skillflags()
{
   return skill_flags;
}


/***********************************************************************
 ** read_skill_attrib - reads in skill attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Skill::read_skill_attrib(FILE *read_file, ErrLog *error_log)
{
   Strings      holder;

// This is now defunct
#if 0
   /* Set min_str value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute MinStr for skill %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "read_skill_attrib");
      return -1;
   }
   set_min_str(atoi(the_token->the_string));

   /* Set min_dex value */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute MinDex for skill %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "read_skill_attrib");
      return -1;
   }
   set_min_dex(atoi(the_token->the_string));
#endif

   /* read in the skill flags */
   if (skill_flags->read_flags(read_file, error_log) <= 0)
   {
      holder.sprintf("Error reading skillflags for skill '%s@%s'", get_name(),
                                                                 get_area()); 
      error_log->log_err(holder.str_show(), "read_skill_attrib");
      return -1;
   }   

   return 1;
}


/***********************************************************************
 ** describe - describes the skill to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Skill::describe(Builder *the_builder)
{
   dependency *tmp_dep;
   the_builder->send_bldr("\n&+GSkill: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GActingRoom:&*\n%s\n", 
        (get_acting_str(Room) == NULL) ? "" : get_acting_str(Room));
   the_builder->send_bldr("&+GActingActor:&*\n%s\n",
        (get_acting_str(Actor) == NULL) ? "" : get_acting_str(Actor));
   the_builder->send_bldr("&+GActingTarget:&*\n%s\n",
        (get_acting_str(Target) == NULL) ? "" : get_acting_str(Target));
   the_builder->send_bldr("&+GSuccessRoom:&*\n%s\n",
        (get_success_str(Room) == NULL) ? "" : get_success_str(Room));
   the_builder->send_bldr("&+GSuccessActor:&*\n%s\n",
        (get_success_str(Actor) == NULL) ? "" : get_success_str(Actor));
   the_builder->send_bldr("&+GSuccessTarget:&*\n%s\n", 
        (get_success_str(Target) == NULL) ? "" : get_success_str(Target));
   the_builder->send_bldr("&+GFailRoom:&*\n%s\n",
        (get_failure_str(Room) == NULL) ? "" : get_failure_str(Room));
   the_builder->send_bldr("&+GFailActor:&*\n%s\n",
        (get_failure_str(Actor) == NULL) ? "" : get_failure_str(Actor));
   the_builder->send_bldr("&+GFailTarget:&*\n%s\n",
        (get_failure_str(Target) == NULL) ? "" : get_failure_str(Target));
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n",
                  (get_special_name() == NULL) ? "" : get_special_name());
   the_builder->send_bldr("&+GSuccTrig: \t&+g%s&*\n",
                  (get_succ_trig() == NULL) ? "" : get_succ_trig());
   the_builder->send_bldr("&+GFailTrig: \t&+g%s&*\n",
                  (get_fail_trig() == NULL) ? "" : get_fail_trig());
   the_builder->send_bldr("&+GAttemptTrig: \t&+g%s&*\n",
                  (get_attempt_trig() == NULL) ? "" : get_attempt_trig());
   the_builder->send_bldr("&+GDrain: \t\t&+w%d&*\n", get_drain());
   the_builder->send_bldr("&+GMinStr: \t&+w%d&*\n", get_min_str());
   the_builder->send_bldr("&+GMinDex: \t&+w%d&*\n\n", get_min_dex());

   the_builder->send_bldr("&+CDependencies:&*\n");
   tmp_dep = depend_list;
   while (tmp_dep != NULL)
   {
     the_builder->send_bldr("&+GType: \t\t&+g%s&*\n", depend_name[tmp_dep->the_type]);
     if (tmp_dep->the_type == MustHaveItem)
       the_builder->send_bldr("&+GDepString: \t&+w%s&*\n", 
			      tmp_dep->str_value.str_show());
     else
       the_builder->send_bldr("&+GDepNumber: \t&+w%d&*\n", tmp_dep->num_value);
     tmp_dep = tmp_dep->next_depend;
   }
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the skill to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 **
 ***********************************************************************/

void Skill::describe(Player *the_player)
{
   dependency *tmp_dep;
   the_player->send_plr("\n&+GSkill: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GActingRoom:&*\n%s\n", 
        (get_acting_str(Room) == NULL) ? "" : get_acting_str(Room));
   the_player->send_plr("&+GActingActor:&*\n%s\n",
        (get_acting_str(Actor) == NULL) ? "" : get_acting_str(Actor));
   the_player->send_plr("&+GActingTarget:&*\n%s\n",
        (get_acting_str(Target) == NULL) ? "" : get_acting_str(Target));
   the_player->send_plr("&+GSuccessRoom:&*\n%s\n",
        (get_success_str(Room) == NULL) ? "" : get_success_str(Room));
   the_player->send_plr("&+GSuccessActor:&*\n%s\n",
        (get_success_str(Actor) == NULL) ? "" : get_success_str(Actor));
   the_player->send_plr("&+GSuccessTarget:&*\n%s\n", 
        (get_success_str(Target) == NULL) ? "" : get_success_str(Target));
   the_player->send_plr("&+GFailRoom:&*\n%s\n",
        (get_failure_str(Room) == NULL) ? "" : get_failure_str(Room));
   the_player->send_plr("&+GFailActor:&*\n%s\n",
        (get_failure_str(Actor) == NULL) ? "" : get_failure_str(Actor));
   the_player->send_plr("&+GFailTarget:&*\n%s\n",
        (get_failure_str(Target) == NULL) ? "" : get_failure_str(Target));
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n",
                  (get_special_name() == NULL) ? "" : get_special_name());
   the_player->send_plr("&+GSuccTrig: \t&+g%s&*\n",
                  (get_succ_trig() == NULL) ? "" : get_succ_trig());
   the_player->send_plr("&+GFailTrig: \t&+g%s&*\n",
                  (get_fail_trig() == NULL) ? "" : get_fail_trig());
   the_player->send_plr("&+GAttemptTrig: \t&+g%s&*\n",
                  (get_attempt_trig() == NULL) ? "" : get_attempt_trig());
   the_player->send_plr("&+GDrain: \t\t&+w%d&*\n", get_drain());
   the_player->send_plr("&+GMinStr: \t&+w%d&*\n", get_min_str());
   the_player->send_plr("&+GMinDex: \t&+w%d&*\n\n", get_min_dex());

   the_player->send_plr("&+CDependencies:&*\n");
   tmp_dep = depend_list;
   while (tmp_dep != NULL)
   {
     the_player->send_plr("&+GType: \t\t&+g%s&*\n", depend_name[tmp_dep->the_type]);
     if (tmp_dep->the_type == MustHaveItem)
       the_player->send_plr("&+GDepString: \t&+w%s&*\n", 
			      tmp_dep->str_value.str_show());
     else
       the_player->send_plr("&+GDepNumber: \t&+w%d&*\n", tmp_dep->num_value);
     tmp_dep = tmp_dep->next_depend;
   }
   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib_skillflags - for set attribute, sets the spellflags for any
 **                         spell
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Skill::set_attrib_skillflags(Parse *the_parsed, Builder *the_builder)
{
   Flags *tmp_skillflags;
   int flagnum;
   Strings holder;

   tmp_skillflags = get_skillflags();
      
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("Set which skill flag?\n");
      return -1;
   }
   holder.assign_word(the_parsed->get_speech(), 1);
     
   if ((flagnum = 
       tmp_skillflags->get_by_name(holder.str_show(), skillflagnames)) == -1)
   {
      the_builder->send_bldr("That is not an skill flag.\n");
      return -1;
   }

   holder.assign_word(the_parsed->get_speech(), 2);
   if (holder.str_show() == NULL)
   {
      the_builder->send_bldr("Set that flag to what?\n"
                                "Valid choices are: On or Off\n");
      return -1;
   }

   if (holder.str_n_cmp("On", holder.str_len()))
      tmp_skillflags->set_flag(flagnum);
   else if (holder.str_n_cmp("Off", holder.str_len()))
      tmp_skillflags->clr_flag(flagnum);
   else
   {
      the_builder->send_bldr("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n");
      return -1;
   }
   the_builder->send_bldr("Flag &+M%s&* has been set to: %s\n",
       skillflagnames[flagnum], (tmp_skillflags->get_flag(flagnum)) ? 
			     "&+GOn&*" : "&+ROff&*");
   return 1;
 
}

/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Skill::set_attrib(Builder *the_builder, Parse *the_parsed){
  int results = 0;

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
            send_bldr("You can set the following attributes on a skill.\n"
               "   actingroom, actingactor, actingtarget, successroom,\n"
               "   successactor, successtarget, failroom, failactor,\n"
               "   failtarget, specials, drain, minstr,\n"
               "   depstring, depnumber, and mindex\n");
       return -1;
   }

   if ((results = set_attrib_abil(the_builder, the_parsed)) == 1)
     return results;
     
   if (results != 0)
     return results;

   if (!STRNCASECMP(the_parsed->get_target1(), "skillflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_skillflags(the_parsed, the_builder);
   }

   the_builder->send_bldr("The attribute '%s' is not a skill attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the skill to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Skill::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nskill %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   write_ability_attrib(the_file);

   skill_flags->write_flag(the_file);
}


/***********************************************************************
 ** copy_object - copies the skill to a skill of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Skill::copy_object(Entity *copy_obj)
{
   Skill *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_SKILL)
      return 0;

   copy_from = (Skill *) copy_obj;

   copy_ability_attrib((Ability *) copy_from);

   set_min_str(copy_from->get_min_dex());  
   set_min_dex(copy_from->get_min_dex());  
   skill_flags->copy_flags(copy_from->get_skillflags());

   return 1;
}

/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Skill::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Skill::get_mem_size_dynamic()
{
   int  size = 0;
   dependency *tmp_dep = depend_list;

   size += skill_flags->get_mem_size();   
   size += get_mem_size_ability();
   size += get_mem_size_entity();

   while (tmp_dep != NULL)
   {
     size += sizeof(tmp_dep);
     size += tmp_dep->str_value.get_mem_size_dynamic();
     tmp_dep = tmp_dep->next_depend;
   }
   return size;
}



#endif







