/**********************************************************************
 ** individual class: This contains methods that are shared between players
 **                   and mobiles
 **
 **   
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef INDIVIDUAL_C
#define INDIVIDUAL_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "strings.h"
#include "global.h"
#include "individual.h"
#include "objtype.h"
#include "merger.h"
#include "indflags.h"
#include "player.h"
#include "wearable.h"
#include "wearflags.h"
#include "newfuncts.h"
#include "specials.h"
#include "code.h"
#include "player.h"
#include "utils.h"
#include "adminflags.h"
#include "boat.h"
#include "gameflags.h"
#include "mobile.h"
#include "race.h"

/***********************************************************************
 ** Individual (constructor) - creates the individual object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Individual::Individual()
{
   int i;

   left_hand = right_hand = NULL;
   indflags = new_Flags(2);

   if (get_type() == OBJ_TYPE_MOBILE)
      indflags->set_flag(INDFLAG_ATTACKPLAYER);

   for (i=0; i <= WEARFLAG_FEET; i++)
      worn_on[i] = NULL;

   strength = the_config.default_strength;
   dexterity = the_config.default_dexterity;
   constitution = the_config.default_constitution;
   health = get_maxhealth();
   intelligence = the_config.default_intel;
   wisdom = the_config.default_wisdom;
   charisma = the_config.default_charisma;
   cur_loc = NULL;
   target = NULL;
}


/***********************************************************************
 ** Individual (destructor) - destroys this individual object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Individual::~Individual()
{
   int i;
   worn_item *tmp_worn;
   fighting_ind *tmp_fighter;   

   /* delete all the worn_item objects */
   for (i=0; i<WEARFLAG_FEET+1; i++)
   {
      tmp_worn = worn_on[i];
      while (tmp_worn != NULL)
      {
         worn_on[i] = tmp_worn->next_worn;
         delete_worn_item(tmp_worn);
         tmp_worn = worn_on[i];
      }
   }

   delete_Flags(indflags);

   /* now make double sure that nobody is fighting us before we delete */
   if (mainstruct != NULL)
   {
      mainstruct->reset_fight_list();
      tmp_fighter = mainstruct->get_next_fighter();

      while (tmp_fighter != NULL)
      {
         if (((Individual *) tmp_fighter->the_fighter) == this)
         {
            mainstruct->remove_fighter(this);
            mainstruct->clear_fighter(this);
         }

         tmp_fighter = mainstruct->get_next_fighter();
      }
   }
}



/***********************************************************************
 ** tell_ind - does a tell to an individual on the game
 **
 ** Parameters: target - the individual who is being told
 **             the_string - the string we are telling to them
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Individual::tell_ind(Individual *target, char *the_string)
{
   Player *target_plr = NULL;
   Player *teller_plr = NULL;
   Strings holder;
   Strings formatted;
   Flags *tmp_indflags;

   if ((target == NULL) || (the_string == NULL))
      return -1;

   /* first send all info to the target */
   if (target->get_type() == OBJ_TYPE_PLAYER)
   {
      target_plr = (Player *) target;

      formatted.format_for_comm(the_string, (strlen(get_title()) + 12), 
                                             the_config.max_line_len);
      target_plr->send_plr("&+B%s&+w tells &+Byou &+W'&+G%s&+W'&*\n", 
                                       get_title(), formatted.str_show());
   }

   tmp_indflags = target->get_indflags();

   /* next send all info to the teller */
   if (get_type() == OBJ_TYPE_PLAYER)
   {
      teller_plr = (Player *) this;
      if (tmp_indflags->get_flag(INDFLAG_CORPSE))
      {
         teller_plr->send_plr("You converse with the corpse.  "
                      "Strange, it does not respond.\n");
      }
      else
      {
         formatted.format_for_comm(the_string, (strlen(get_title()) + 12), 
                                             the_config.max_line_len);
         teller_plr->send_plr("&+BYou&+w tell &+B%s &+W'&+G%s&+W'&*\n", 
                            target->get_title(), formatted.str_show());
      }
   }

   if (teller_plr != NULL)
   {
      if (tmp_indflags->get_flag(INDFLAG_CORPSE))
      {
	 holder.sprintf("%s tries to talk with a corpse.\n", 
                                       teller_plr->get_title());
      }
      else
      {
         holder.sprintf("%s holds a conversation with %s.\n", 
                teller_plr->get_title(), target->get_title());
      }
      (teller_plr->get_loc())->send_location(holder.str_show(), 
                                           teller_plr, target_plr, CAN_SEE);
      (teller_plr->get_loc())->send_location(
        "You hear the sounds of a conversation in the darkness.\n", teller_plr, 
                                                     target_plr, CANT_SEE);

   }
   return 1;
}


/***********************************************************************
 ** move_merger - moves a merger from one individual to another
 **
 ** Parameters: target - the individual who is being told
 **             the_string - the string we are telling to them
 **
 ** Returns:  1 for success, 
 *           -1 for failure,
 **          -2 for not enough mergers in the item to move
 **
 ***********************************************************************/

int Individual::move_merger(Individual *target, Merger **the_merger, 
                                                            int num_of)
{
   Merger *new_merger;
   MudObject *parent_merger;
   int    new_num;

   if (num_of > (*the_merger)->get_number_of())
      return -2;

   if (num_of == (*the_merger)->get_number_of())
   {
      remove_contained((MudObject *) *the_merger);
   }
   else
   {         
      if ((*the_merger)->get_parent() != NULL)
      {
         parent_merger = mainstruct->get_object((*the_merger)->get_parent());
         new_merger = (Merger *)  
                   (mainstruct->get_dbase())->clone_object(parent_merger);
      }
      else
         new_merger = (Merger *) 
          (mainstruct->get_dbase())->clone_object((MudObject *) *the_merger);
      new_merger->set_number_of(num_of);
      new_num = (*the_merger)->get_number_of() - num_of;
      (*the_merger)->set_number_of(new_num);
      *the_merger = new_merger;
   }

   if (target->add_contained((MudObject *) *the_merger) == 2)
   {
      return 2;
   }
   return 1;
}


/***********************************************************************
 ** wield_moveable - wields a moveable object, accounting for how many
 **                  hands it takes to wield it
 **
 ** Parameters: the_item - the item to wield
 **
 ** Returns: 1 for success, -1 for failure, -2 for not enough hands
 **
 ***********************************************************************/
   
int Individual::wield_moveable(Moveable *the_item)
{
   Weapon *the_weapon;

   if (the_item->get_type() == OBJ_TYPE_WEAPON)
   {
      the_weapon = (Weapon *) the_item;
    
      if (the_weapon->get_wield_type() == One_Handed)
      {
         if (right_hand == NULL)
            right_hand = the_item;
         else if (left_hand == NULL)
            left_hand = the_item;
         else
	 {
            unwield_moveable(Right);
            right_hand = the_item;
         }
         return 1;
      }
      else
      {
         if ((left_hand != NULL) || (right_hand != NULL))
	 {
            unwield_moveable(Left);
            unwield_moveable(Right);
         }

         left_hand = right_hand = the_item;
         return 1;
      }
   }
   return -1;   
}


/***********************************************************************
 ** unwield_moveable - unwields a moveable object from a particular hand.
 **                    if the object is two-handed, it unwields both
 **
 ** Parameters: the_hand - the hand we are going to unwield
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Individual::unwield_moveable(playerhand the_hand)
{
   Weapon *the_weapon;
   Moveable *the_moveable;
   Moveable **hand_ptr;

   if (the_hand == Left)
   {
      if ((the_moveable = left_hand) == NULL)
         return -1;

      hand_ptr = &left_hand;
   }
   else
   {
      if ((the_moveable = right_hand) == NULL)
         return -1;

      hand_ptr = &right_hand;

   }
   if (the_moveable->get_type() == OBJ_TYPE_WEAPON)
   {
      the_weapon = (Weapon *) the_moveable;
      if (the_weapon->get_wield_type() == Two_Handed)
      {
         left_hand = right_hand = NULL;
         return 1;
      }
      *hand_ptr = NULL;
      return 1; 
   }

   return -1;
}


/***********************************************************************
 ** get_wielded - gets the object that is wielded in a particular hand
 **
 ** Parameters: the_hand - the hand we want to get the object from
 **
 ** Returns: a pointer to the object if found
 **
 ***********************************************************************/

Moveable *Individual::get_wielded(playerhand the_hand)
{
   if (the_hand == Left)
      return left_hand;
   else
      return right_hand;

}


/***********************************************************************
 ** get_indflags - gets the individual flags object
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the individual flags structure
 **
 ***********************************************************************/

Flags *Individual::get_indflags()
{
   return indflags;
}


/***********************************************************************
 ** is_male - is this individual male
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 for male, 0 for not
 **
 ***********************************************************************/

int Individual::is_male()
{
   return indflags->get_flag(INDFLAG_MALE);
}


/***********************************************************************
 ** wear_item - places an item in the player's linked list of body parts
 **             depending on where it should go
 **
 ** Parameters: the_item - the item to place on this player
 **
 ** Returns: 1 for success, -1 for failure, -2 if it can't be worn, 
 **                         -3 for already worn
 **
 ***********************************************************************/

int Individual::wear_item(Wearable *the_item)
{
   int i;
   Flags *tmp_wearflags;
   Flags *tmp2_wearflags;
   int is_toplayer;
   worn_item *new_worn;

   if (the_item == NULL)
      return -1;

   tmp_wearflags = the_item->get_wearflags();

   /* check for shield, if so, we need to wield */
   if (tmp_wearflags->get_flag(WEARFLAG_WIELDABLE))
   {
      if (right_hand == NULL)
         right_hand = the_item;
      else if (left_hand == NULL)
         left_hand = the_item;
      else
         return -4;         
   }

   /* first make sure we can add it everywhere we need to */
   is_toplayer = tmp_wearflags->get_flag(WEARFLAG_TOPLAYER);
   for (i=0; i<=WEARFLAG_FEET; i++)
   {
      /* see if the needed spots are free */
      if (is_toplayer)
      {
         if ((tmp_wearflags->get_flag(i)) && (worn_on[i] != NULL))
	 {
            tmp2_wearflags = (worn_on[i]->the_obj)->get_wearflags();
            if (tmp2_wearflags->get_flag(WEARFLAG_TOPLAYER))
               return -2;

         }
      }

      /* make sure we're not already wearing it */
      new_worn = worn_on[i];
      while (new_worn != NULL)
      {
         if (the_item == new_worn->the_obj)
           return -3;
         new_worn = new_worn->next_worn;
      }
   }

   /* now we add it to all places that it should be worn */
   for (i=0; i<=WEARFLAG_FEET; i++)
   {
      if (tmp_wearflags->get_flag(i))
      {
         new_worn = new_worn_item();
         new_worn->next_worn = NULL;
         new_worn->the_obj = the_item;
         if (worn_on[i] == NULL)
	 {
            worn_on[i] = new_worn;
         }
         else if (is_toplayer)
	 {
            new_worn->next_worn = worn_on[i];
            worn_on[i] = new_worn;
         }
         else
	 {
            tmp2_wearflags = (worn_on[i]->the_obj)->get_wearflags();
            if (tmp2_wearflags->get_flag(WEARFLAG_TOPLAYER))
	    {
               new_worn->next_worn = worn_on[i]->next_worn;
               worn_on[i]->next_worn = new_worn;
            }
            else
	    {
               new_worn->next_worn = worn_on[i];
               worn_on[i] = new_worn;
            }
         }
      }
   }
   (the_item->get_itemflags())->set_flag(ITEMFLAG_WORN);

   return 1;

}


/***********************************************************************
 ** remove_item - removes the item from the worn lists
 **
 ** Parameters: the_item - the item to remove
 **
 ** Returns: 1 for removed, 0 for not worn
 **
 ***********************************************************************/

int Individual::remove_item(Wearable *the_item)
{
   int       i;
   worn_item *tmp_worn;
   worn_item *prev_worn = NULL;
   int       found = 0;
   Flags     *tmp_wearflags;

   tmp_wearflags = the_item->get_wearflags();

   /* first if it is a shield unwield it */
   if (tmp_wearflags->get_flag(WEARFLAG_WIELDABLE))
   {
      if (left_hand == the_item)
         left_hand = NULL;
      else if (right_hand == the_item)
         right_hand = NULL;
      else
         return 0;
      return 1;
   }


   for (i=0; i<=WEARFLAG_FEET; i++)
   {
      tmp_worn = worn_on[i];
      if ((worn_on[i] != NULL) && (worn_on[i]->the_obj == the_item))
      {
         worn_on[i] = worn_on[i]->next_worn;
         found = 1;
         delete_worn_item(tmp_worn);
      }
      else
      {
         while ((tmp_worn != NULL) && (tmp_worn->the_obj != the_item))
         {
            prev_worn = tmp_worn;
            tmp_worn = tmp_worn->next_worn;
         }
         if (tmp_worn != NULL)
	 {
            prev_worn->next_worn = tmp_worn->next_worn;
            found = 1;
            delete_worn_item(tmp_worn);
         }
      }
   }
   if (found)
      (the_item->get_itemflags())->clr_flag(ITEMFLAG_WORN);

   return found;
}


/***********************************************************************
 ** is_worn - is this item worn by the player
 **
 ** Parameters: the_item - the item to look for
 **
 ** Returns: 1 for worn, 0 for not worn
 **
 ***********************************************************************/

int Individual::is_worn(Wearable *the_item)
{
   int       i;
   worn_item *tmp_worn;

   for (i=0; i<=WEARFLAG_FEET; i++)
   {
      tmp_worn = worn_on[i];
      while (tmp_worn != NULL)
      {
         if (tmp_worn->the_obj == the_item)
            return 1;
         tmp_worn = tmp_worn->next_worn;
      }
   }
   return 0;
}


/***********************************************************************
 ** display_top_worn - displays only those items that are at the top
 **                    of the stacks since they are the only ones that
 **                    we will make visible
 **
 ** Parameters: the_item - the item to remove
 **
 ** Returns: num objects displayed for success, -1 for failure
 **
 ***********************************************************************/

int Individual::display_top_worn(Player *the_player)
{
   int       i, j;
   worn_item *tmp_worn;
   Strings   holder;
   Strings   send_str;
   int       line_len = 0;
   int       count = 0;
   int       already_shown = 0;

   send_str = "They are wearing: ";
   for (i=0; i<=WEARFLAG_FEET; i++)
   {
      tmp_worn = worn_on[i];
      if (tmp_worn != NULL)
      {
         already_shown = 0;
         for (j=0; j<i; j++)
	 {
            if ((worn_on[j] != NULL) && 
                (worn_on[j]->the_obj == worn_on[i]->the_obj))
	    {
               already_shown = 1; 
            }
         }
         if (!already_shown)
	 {
            holder.sprintf("%s%s", ((line_len == 0) ? "" : ", "), 
                                       (tmp_worn->the_obj)->get_title());

            send_str.str_cat(holder.str_show());
            count++;
         }

         if ((line_len = send_str.str_len()) >= the_config.max_line_len)
         {
            the_player->send_plr(send_str.str_show());
            the_player->send_plr("\n");
            send_str = "                  ";
            line_len = 0;
         } 
      }
   }
   if (line_len != 0)
      the_player->send_plr(send_str.str_show());
   the_player->send_plr("\n");
   return count;
}


/***********************************************************************
 ** display_all_worn - displays all items worn and where worn
 **
 ** Parameters: Nothing
 **
 ** Returns: num objects displayed for success, -1 for failure
 **
 ***********************************************************************/

int Individual::display_all_worn()
{
   int       i;
   worn_item *tmp_worn;
   Strings   holder;
   Strings   send_str;
   int       line_len = 0;
   int       count = 0;
   int       first_display;
   Player    *the_player;

   char *body_parts[] = {"Head",  "Face",  "Neck", "Chest", "Back", "Arms", 
                         "Hands", "Waist", "Legs", "Feet"};

   if (get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) this;

   the_player->send_plr("Worn on body:\n");
   for (i=0; i<=WEARFLAG_FEET; i++)
   {
      tmp_worn = worn_on[i];

      first_display = 0;
      while (tmp_worn != NULL)
      {
         holder.sprintf("%s%s", ((line_len == 0) ? "" : ", "), 
                                       (tmp_worn->the_obj)->get_title());

         send_str.str_cat(holder.str_show());
         count++;

         
         if ((line_len = send_str.str_len()) >= (the_config.max_line_len - 10))
         {
            holder.sprintf("\t%s: \t%s\n", body_parts[i], 
                                              send_str.str_show());
            the_player->send_plr(holder.str_show());
            send_str.truncate(0);
            line_len = 0;
            first_display = 1;
         }
         tmp_worn = tmp_worn->next_worn;
      }
      if (line_len != 0)
      {
         if (first_display)
            holder.sprintf("\t\t%s\n", send_str.str_show());
         else
            holder.sprintf("\t%s: \t%s\n", body_parts[i], 
                                                  send_str.str_show());
         the_player->send_plr(holder.str_show());
         send_str.truncate(0);
         line_len = 0;
      }
      else if (worn_on[i] == NULL)
         the_player->send_plr("\t%s: \t -\n", body_parts[i]);
   }

   return count;
}



/***********************************************************************
 ** has_lit - individual contains an object that is lit
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 for has one, 0 for not
 **
 ***********************************************************************/

int Individual::has_lit(void)
{
   LinkedList *ind_list;
   MudObject  *next_obj;
   Item       *an_item;
   Flags      *itemflags;

   ind_list = get_inventory();
   ind_list->reset_current();
   
   next_obj = ind_list->get_next();
   while (next_obj != NULL)
   {
      if (next_obj->is_an_item())
      {
         an_item = (Item *) next_obj;
         itemflags = an_item->get_itemflags();
         if ((itemflags->get_flag(ITEMFLAG_LIT)) || 
             (itemflags->get_flag(ITEMFLAG_GLOWING)))
            return 1;
    
      }
      next_obj = ind_list->get_next();
   }
   return 0;
}


/***********************************************************************
 ** get_strength - returns the strength value
 **
 **
 ***********************************************************************/

int Individual::get_strength(void)
{
   return strength;
}


/***********************************************************************
 ** set_strength - sets the strength value
 **
 **
 ***********************************************************************/

void Individual::set_strength(int the_value)
{
   if (the_value < 0)
      return;

   if (the_value > 100)
      strength = 100;
   else
      strength = the_value;

}


/***********************************************************************
 ** set_dexterity - sets the dexterity value
 **
 **
 ***********************************************************************/

void Individual::set_dexterity(int the_value)
{
   if (the_value < 0)
      return;

   if (the_value > 100)
      dexterity = 100;
   else
      dexterity = the_value;
}


/***********************************************************************
 ** get_dexterity - returns the dexterity value
 **
 **
 ***********************************************************************/

int Individual::get_dexterity(void)
{
   return dexterity;
}


/***********************************************************************
 ** set_health - sets the health value
 **
 **
 ***********************************************************************/

void Individual::set_health(int the_value)
{
   health = the_value;
}


/***********************************************************************
 ** get_health - returns the health value
 **
 **
 ***********************************************************************/

int Individual::get_health(void)
{
   return health;
}


/***********************************************************************
 ** set_constitution - sets the constitution value
 **
 **
 ***********************************************************************/

void Individual::set_constitution(int the_value)
{
   if (the_value < 0)
      return;

   if (the_value > 100)
      constitution = 100;
   else
      constitution = the_value;
}

/***********************************************************************
 ** get_constitution - returns the constitution value
 **
 **
 ***********************************************************************/

int Individual::get_constitution(void)
{
   return constitution;
}


/***********************************************************************
 ** get_maxhealth - returns the maxhealth value
 **
 **
 ***********************************************************************/

int Individual::get_maxhealth(void)
{
   return constitution * 5;
}


/***********************************************************************
 ** set_intel - sets the intelligence value
 **
 **
 ***********************************************************************/

void Individual::set_intel(int the_value)
{
   if (the_value < 0)
      return;

   if (the_value > 100)
      intelligence = 100;
   else
      intelligence = the_value;
}


/***********************************************************************
 ** get_intel - returns the intelligence value
 **
 **
 ***********************************************************************/

int Individual::get_intel(void)
{
   return intelligence;
}

/***********************************************************************
 ** set_wisdom - sets the wisdom value
 **
 **
 ***********************************************************************/

void Individual::set_wisdom(int the_value)
{
   if (the_value < 0)
      return;

   if (the_value > 100)
      wisdom = 100;
   else
      wisdom = the_value;
}


/***********************************************************************
 ** get_wisdom - returns the wisdom value
 **
 **
 ***********************************************************************/

int Individual::get_wisdom(void)
{
   return wisdom;
}


/***********************************************************************
 ** set_charisma - sets the charisma value
 **
 **
 ***********************************************************************/

void Individual::set_charisma(int the_value)
{
   if (the_value < 0)
      return;

   if (the_value > 100)
      charisma = 100;
   else
      charisma = the_value;
}


/***********************************************************************
 ** get_charisma - returns the charisma value
 **
 **
 ***********************************************************************/

int Individual::get_charisma(void)
{
   return charisma;
}


/***********************************************************************
 ** request_fight - requests to fight a mobile (politely of course), and
 **                 if its ok, the mobile sets appropriate attributes. If
 **                 not, the return value states why
 **
 ** Parameters: the_aggressor - who is requesting the attack
 **
 ** Returns: 1 for the fight is on!
 **          -1 for error
 **          -2 for too many people attacking already
 **          -3 for im already fighting you
 **          -4 for I'm already dead you idiot!
 **          -5 for you are dead so can't fight
 **
 ***********************************************************************/

int Individual::request_fight(Individual *the_target)
{
   int num_attack;

   if (the_target == NULL)
      return -1;

   if ((indflags->get_flag(INDFLAG_CORPSE)) ||
       (indflags->get_flag(INDFLAG_GHOST)))
   {
      return -5;
   }

   if (((the_target->get_indflags())->get_flag(INDFLAG_CORPSE)) ||
       ((the_target->get_indflags())->get_flag(INDFLAG_GHOST)))
      return -4;

   if ((num_attack = mainstruct->get_num_attacking(the_target)) >=
                                                  the_config.max_num_attacking)
      return -2;

   if (target == the_target)
      return -3;

   the_target->start_fight(this);

   return 1;
}


/***********************************************************************
 ** start_fight - starts a fight with the person indicated
 **
 ** Parameters: the_target - who we are attacking
 **
 ** Returns: 1 for the fight is on!
 **          -1 for error
 **
 ***********************************************************************/

int Individual::start_fight(Individual *the_target)
{
   Player  *the_player = NULL;
   int     stand = 0;
   Strings holder;

   if (the_target == NULL)
      return -1;

   if (target == NULL)
      target = the_target;

   /* they shouldn't get here, but in case they do, don't let them fight
      a corpse */
   if ((target->get_indflags())->get_flag(INDFLAG_CORPSE))
      return 0;

   if (is_asleep())
   {
      indflags->clr_flag(INDFLAG_SLEEPING);
      if (get_type() == OBJ_TYPE_PLAYER)
      {
         the_player = (Player *) this;
         the_player->send_plr("You awaken and clamber to your feet.\n");
      }
      stand = 1;
   }

   if (is_sitting())
   {
      indflags->clr_flag(INDFLAG_SITTING);
      if (get_type() == OBJ_TYPE_PLAYER)
      {
         the_player = (Player *) this;
         the_player->send_plr("You clamber to your feet.\n");
      }
      stand = 1;
   }

   if (stand)
   {
      holder.sprintf("%s clambers to %s feet.\n", get_title(), 
                indflags->get_flag(INDFLAG_MALE) ? "his" : "her");
      (get_loc())->send_location(holder.str_show(), the_player);
   }

   indflags->set_flag(INDFLAG_FIGHTING);
   mainstruct->add_fighter(this, 0);
   return 1;
}

/***********************************************************************
 ** stop_fighting - stops fighting with all
 **
 ** Returns: 1 for the fight is done
 **          -1 for failed
 **
 ***********************************************************************/

int Individual::stop_fighting(void)
{
   target = NULL;
   indflags->clr_flag(INDFLAG_FIGHTING);
   mainstruct->remove_fighter(this);
   return 1;
}

/***********************************************************************
 ** switch_focus - change who we are directing our blows at
 **
 ** Parameters: the_target - who we are attacking
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Individual::switch_focus(Individual *the_target)
{
   if (target == NULL)
      return;

   target = the_target;
}


/***********************************************************************
 ** get_target - gets the target of this player's attack
 **
 **
 ***********************************************************************/

Individual *Individual::get_target(void)
{
   return target;
}


/***********************************************************************
 ** develop_difference - develops a difference index that is used to
 **                      determine if you hit or block, or if the shot 
 **                      misses or
 **
 ** Parameters: the_wielder - who we are calculating for
 **             the_weapon - used to return a pointer to the weapon that
 **                          they will use to do the shot or block
 **
 ** Returns: the difference index
 **
 ***********************************************************************/

int Individual::develop_difference(Individual *the_wielder, 
                                                Moveable **the_wielded)
{
   int      rand_num;
   float    str_index;
   float    dex_index;
   Weapon   *the_weapon;
   int      chance_index;
   int      difference;
   float    disadvantage = 1.0;

   /* if they can't see, they can't hit squat */
   if ((the_wielder->get_type() == OBJ_TYPE_PLAYER) &&
       !((the_wielder->get_loc())->is_lit(the_wielder)))
      disadvantage = (float)0.3;

   *the_wielded = NULL;
   rand_num = 1+ (int) (2.0*(rand()/(RAND_MAX+1.0)));
   switch(rand_num)
   {
      case 1:
         *the_wielded = the_wielder->get_wielded(Left);
         break;
      case 2:  
         *the_wielded = the_wielder->get_wielded(Right);
         break;
      default:
         printf("Invalid hand chosen!\n");
         return -1;
   }

   /* is it bare hands? If so, try the other hand */
   if (*the_wielded == NULL)
   {
      switch(rand_num)
      {
         case 2:
            *the_wielded = the_wielder->get_wielded(Left);
            break;
         case 1:  
            *the_wielded = the_wielder->get_wielded(Right);
            break;
         default:
            printf("Invalid hand chosen!\n");
            return -1;
      }
   }

   rand_num = 1+ (int) (100.0*(rand()/(RAND_MAX+1.0)));

   /* if it still equals NULL, we just return */
   if (*the_wielded == NULL)
   {
      return (int) ((float) (rand_num - (50 - (the_wielder->get_dexterity()/2)))
                                                * disadvantage);
   }

   /* is it a weapon or a shield? */
   if ((*the_wielded)->get_type() != OBJ_TYPE_WEAPON)
   {
      return (int) (1000.0 * disadvantage);
   }

   the_weapon = (Weapon *) *the_wielded;

   /* calculate the chance of hitting, and get the difference */   
   dex_index = ((((float) (the_wielder->get_dexterity() - 25)) * 
                ((((float) the_weapon->get_dep_dex())) / 100)) / 50);
   str_index = ((((float) (the_wielder->get_strength() - 25)) *
                (((float) the_weapon->get_dep_strength()) / 100)) / 50);

   chance_index = (int) (50 + (dex_index * 67) + (str_index * 67));

   difference = ((int) ((float) chance_index * disadvantage)) - rand_num;
   return difference;
}

/***********************************************************************
 ** take_a_swing - strikes at the set target, trying to do a little damage
 **
 ** Parameters: None
 **
 ** Returns:  0 if no longer fighting this ind
 **           1 for successfully executed
 **          -1 for failure
 **
 ***********************************************************************/

int Individual::take_a_swing(void)
{
   Location       *target_loc;
   int            rand_num;
   Moveable       *tmp_moveable = NULL;
   Weapon         *attack_wpn = NULL;
   Weapon         *defend_wpn = NULL;
   int            attack_hands = 0;
   int            defend_hands = 0;
   int            diff_attack;
   int            diff_defend;
   char           *defend_name;
   char           *wpn_name;
   int            body_part = 0;
   Player         *tmp_player = NULL;
   Strings        holder;
   int            i;
   int            target_range = 0;
   int            attacker_range = 0;
   int            damage_done;
   int            max_damage_poss = 1;
   int            prev_health;
   MudObject      *projectile = NULL;
   special_holder *proj_special = NULL;
   int            can_see;
   int            hit = 0;

   int chance_of_attack[10] = {
                      95, /* head 5 percent */
                      90, /* face 5 percent*/
                      80, /* neck 10 percent*/
                      50, /* chest 30 percent*/
                      30, /* back 20 percent */
		      20, /* arms 10 percent */
                      15, /* hands 5 percent */
                      10, /* waist 5 percent */
                       5, /* legs 5 percent */
		      0}; /* feet 5 percent */
                              
   char *body_part_name[10] = {"head", "face", "neck", "chest", "back", 
                               "arms", "hands", "waist", "legs", "feet"};

   /* if the person they were fighting is no longer available */
   if (target == NULL)
   {
      return 0;
   }

   /* if they are a corpse, we don't fight them */
   if ((target->get_indflags())->get_flag(INDFLAG_CORPSE))
      return 0;

   target_loc = target->get_loc();
   /* if theyre not here anymore, stop fighting */
   if (target_loc != get_loc())
   {
      target = NULL;
      if (mainstruct->get_num_attacking(this) == 0)
         indflags->clr_flag(INDFLAG_FIGHTING);
      return 0;
   }

   can_see = target_loc->is_lit(this);

   /* calculate the chance of hitting and assign weapon type */
   diff_attack = develop_difference(this, &tmp_moveable);

   /* does the attacker wield a long range weapon (bow, slingshot, etc) */
   if ((tmp_moveable != NULL) && 
       (tmp_moveable->get_type() == OBJ_TYPE_WEAPON) && 
       (!STRCASECMP(((Weapon *) tmp_moveable)->get_weapon_class(), "bow")))
      attacker_range = 1;

   if (tmp_moveable == NULL)
   {
      if (get_bare_weapon() == NULL)
         wpn_name = "bare fist";
      else
         wpn_name = get_bare_weapon();
      attack_hands = 1;
   }
   /* its a shield, we can't attack with this */
   else if (diff_attack == 1000)
      return 0;
   else
   { 
      attack_wpn = (Weapon *) tmp_moveable;
      wpn_name = attack_wpn->get_title();
   }

   diff_defend = develop_difference(target, &tmp_moveable);
   if (tmp_moveable == NULL)
   {
      if (target->get_bare_weapon() == NULL)
         defend_name = "bare fist";
      else
         defend_name = target->get_bare_weapon();

      defend_hands = 1;
   }      
      /* its a shield, we try to deflect */
   else if (diff_defend == 1000)
   {
      diff_defend = 50;
      defend_name = tmp_moveable->get_title();
   }
   else
   {
      defend_wpn = (Weapon *) tmp_moveable;
      defend_name = defend_wpn->get_title();
   }

   /* does the target it a long range weapon (bow, slingshot, etc) */
   if ((tmp_moveable != NULL) && 
       (tmp_moveable->get_type() == OBJ_TYPE_WEAPON) && 
       (!STRCASECMP(((Weapon *) tmp_moveable)->get_weapon_class(), "bow")))
      target_range = 1;
   
   /* if we are not fighting, start fight.  If it is a long range weapon,
      50 percent chance we won't start a fight on our own to signify they
      are out of range for an attack */
   if ((target->get_target() == NULL) &&
       ((!attacker_range) || (!target_range && 
                         (((int) (100.0*(rand()/(RAND_MAX+1.0)))) < 50))))
   {
      target->start_fight(this);
   }

   /* if it is a range weapon, we need to get the ammo being used, so
      lets grab the first object we find in the range weapon */
   if (attacker_range)
   {
      LinkedList *weapon_inv;

      weapon_inv = attack_wpn->get_inventory();
      weapon_inv->reset_current();
      projectile = weapon_inv->get_next();

      while (projectile != NULL)
      {
         if ((proj_special = projectile->find_special("on_launch_hit")) != NULL)
         {
            break;
         }
         projectile = weapon_inv->get_next();
      }
   }

   if ((attacker_range) && (projectile == NULL))
   {
      if (get_type() == OBJ_TYPE_PLAYER)
      { 
         tmp_player = (Player *) this;
 
         tmp_player->send_plr("&+CYou have no ammunition in your %s!&*\n", 
                                                     attack_wpn->get_title());
      }
      attacker_range = 0;
   }
   else if (attacker_range)
      wpn_name = projectile->get_title();


   /* if the difference is less than 0, we miss */
   if (diff_attack < 0)
   {
      int       results = 0;

      /* if they have a special set up to run if the arrow misses, we
         do the special and let it decrement the number of projectiles,
         otherwise we decrement */
      if ((attacker_range) && 
       ((proj_special = projectile->find_special("on_launch_miss")) != NULL))
      {
         in_params fill_param;

         if (get_type() == OBJ_TYPE_PLAYER)
            tmp_player = (Player *) this;
         else
            tmp_player = NULL;

         fill_param.primary_obj = (MudObject *) target;
         fill_param.secondary_obj = (MudObject *) projectile;
         fill_param.this_obj = (MudObject *) this;

         results = (proj_special->the_special)->
             run_special(tmp_player, &fill_param, &proj_special->environment);
         if (results == 0)
            return 1;
         if (results == 2)
	 {
            if (get_type() == OBJ_TYPE_PLAYER)
              ((Player *) this)->send_plr("&+RYou have run out of %s!\n", 
                                            projectile->get_title());
            attack_wpn->remove_contained(projectile);
            mainstruct->delete_object(projectile);
            return 0;
         }
      }
      else if (attacker_range)
      {
         results = 1;
      }
      if (attacker_range)
         holder.sprintf("%s's %s flies wide, &+cmissing&* %s.\n", 
	        get_title(), projectile->get_title(), target->get_title());
      else
         holder.sprintf("\n%s swings %s %s wide, &+cmissing&* %s.\n", 
		     get_title(), (is_male() ? "his" : "her"), 
                     wpn_name, target->get_title());

      if (get_type() == OBJ_TYPE_PLAYER)
      {
         tmp_player = (Player *) this;

         if (attacker_range)
	 {
            if (can_see)
               tmp_player->send_plr("Your %s flies wide, &+cmissing&* %s.\n", 
	                        projectile->get_title(), target->get_title());
            else
               tmp_player->send_plr(
                           "Your %s flies into the darkness and vanishes.\n", 
                                projectile->get_title());
	 }
         else
	 {
            if (can_see)
               tmp_player->
                  send_plr("\n&+B-&*Your %s swings wide, &+cmissing&* %s.\n", 
                            wpn_name, target->get_title());
            else
               tmp_player->send_plr("\n&+B-&*Your %s swings into the darkness,"
                      " catching &+conly air&*.\n", wpn_name); 
	 }
      }
      if (target->get_type() ==  OBJ_TYPE_PLAYER)
      {
        tmp_player = (Player *) target;

        if (attacker_range)
           if (target_loc->is_lit(tmp_player))
              tmp_player->send_plr("%s's %s flies wide, missing you.\n", 
	                        get_title(), projectile->get_title(), 
                                                        target->get_title());
           else
              tmp_player->send_plr("\nYou hear a swoosh as something flies by.\n");
        else
           if (target_loc->is_lit(tmp_player))
              tmp_player->send_plr("\n%s swings %s %s wide, missing you.\n", 
     	                get_title(), (is_male() ? "his" : "her"), wpn_name);
           else
              tmp_player->send_plr(
                     "\nYou hear something swing through the air nearby.\n");
      }

      if ((get_type() == OBJ_TYPE_PLAYER) &&
          (target->get_type() == OBJ_TYPE_PLAYER))
      {
         (get_loc())->send_location(holder.str_show(), (Player *) this, 
                                        (Player *) target, CAN_SEE);
         (get_loc())->send_location("The sound of battle fills the darkness\n",
                           (Player *) this, (Player *) target, CANT_SEE);
      }
      else
      {
         (get_loc())->send_location(holder.str_show(), tmp_player, CAN_SEE);
         (get_loc())->send_location("The sound of battle fills the darkness\n",
                                         tmp_player, CANT_SEE);
      }

      if ((results) && (projectile->is_merger()))
      {
         Merger *the_merger;

         the_merger = (Merger *) projectile;
         the_merger->set_number_of(the_merger->get_number_of() - 1);
         if (the_merger->get_number_of() == 0)
	 {
            if (get_type() == OBJ_TYPE_PLAYER)
              ((Player *) this)->send_plr("&+RYou have run out of %s!\n", 
                                            the_merger->get_title());
            attack_wpn->remove_contained(the_merger);
            mainstruct->delete_object(the_merger);
         }
      }
      if ((attack_wpn != NULL) && (get_type() == OBJ_TYPE_PLAYER))
         ((Player *) this)->increase_rank_exp(2, attack_wpn, (2 - (1 * hit)));
         
      return 1;
   }


   rand_num = (int) (100.0*(rand()/(RAND_MAX+1.0)));
   for (i=0; i<10; i++)
   {
      if (rand_num >= chance_of_attack[i])
      {
         body_part = i;
         break;
      }
   }

   tmp_player = NULL;

   if (attacker_range)
      holder.sprintf("\n%s's %s streaks towards %s's %s.\n", 
		  get_title(), wpn_name, 
                  target->get_title(), body_part_name[body_part]);
   else
      holder.sprintf("\n%s swings %s %s at %s's %s.\n", 
		  get_title(), (is_male() ? "his" : "her"), 
                  wpn_name, target->get_title(), body_part_name[body_part]);
   if (get_type() == OBJ_TYPE_PLAYER)
   {
      tmp_player = (Player *) this;

      if (attacker_range)
         if (can_see)
            tmp_player->send_plr("\nYour %s streaks towards %s's %s.\n", 
                                  wpn_name, target->get_title(), 
                                  body_part_name[body_part]);
         else
            tmp_player->send_plr("\nYour %s streaks into the darkness.\n",
                                 wpn_name);
      else
         if (can_see)
            tmp_player->send_plr("\nYour %s swings at %s's %s.\n", 
                            wpn_name, target->get_title(), 
                            body_part_name[body_part]);
         else
            tmp_player->send_plr("\nYour %s swings into the darkness.\n",
                                 wpn_name);

   }
   if (target->get_type() ==  OBJ_TYPE_PLAYER)
   {
      tmp_player = (Player *) target;
      if (attacker_range)
         if (target_loc->is_lit(tmp_player))
            tmp_player->send_plr("\n%s's %s streaks towards your %s.\n", 
	               get_title(), wpn_name, body_part_name[body_part]);
      else
         if (target_loc->is_lit(tmp_player))
            tmp_player->send_plr("\n%s swings %s %s at your %s.\n", 
	             get_title(), (is_male() ? "his" : "her"), wpn_name, 
                                              body_part_name[body_part]);

   }

   if ((get_type() == OBJ_TYPE_PLAYER) &&
       (target->get_type() == OBJ_TYPE_PLAYER))
   {
      (get_loc())->send_location(holder.str_show(), (Player *) this, 
                                     (Player *) target, CAN_SEE);
   }
   else
   {
      (get_loc())->send_location(holder.str_show(), tmp_player, CAN_SEE);
   }
   
   /* we hit, not a range weapon */
   if ((diff_attack > diff_defend) || (defend_hands && !attack_hands))
   {
      char *hardness[] = {"a glancing blow", "a solid blow", "hard", 
                          "very hard", "a crushing blow"};
      int  the_hardness;
      int  results;

      hit = 1;
      if (attack_wpn != NULL)
      {
         Player *tmp_player = NULL;

         if (get_type() == OBJ_TYPE_PLAYER)
            tmp_player = (Player *) this;
         if (check_specials("on_hit", target, target, this, attack_wpn, 
                                                          tmp_player) == 3)
            return -1;
      }

      if (attacker_range)
      {
         in_params fill_param;

         if (get_type() == OBJ_TYPE_PLAYER)
            tmp_player = (Player *) this;
         else
            tmp_player = NULL;

         fill_param.primary_obj = (MudObject *) target;
         fill_param.secondary_obj = (MudObject *) projectile;
         fill_param.this_obj = (MudObject *) this;

         results = (proj_special->the_special)->
             run_special(tmp_player, &fill_param, &proj_special->environment);

         hit = 1;
         if (results == 2)
	 {
            if (get_type() == OBJ_TYPE_PLAYER)
              ((Player *) this)->send_plr("&+RYou have run out of %s!&*\n", 
                                            projectile->get_title());
            attack_wpn->remove_contained(projectile);
            mainstruct->delete_object(projectile);
         }

         damage_done = proj_special->environment.counter;
      }
      else
      {
         if (attack_wpn != NULL)
         {
            damage_done = 1+ (int) (((float) attack_wpn->get_damage())*
                                                 (rand()/(RAND_MAX+1.0)));
	    max_damage_poss = attack_wpn->get_damage();
         }
         else
         {
            damage_done = 1 + (int)
                       ((get_strength() / 4)*(rand()/(RAND_MAX+1.0)));
	    max_damage_poss = (get_strength() / 8);
         }
      }

      /* prepare to award experience */
      prev_health = target->get_health();
      
//############################################################################
//                              BEGINNING OF DEATH BY BLOW

     if ((target->get_health() <= 0) || 
         (target->take_blow(damage_done, body_part) <= 0))
     {
        Individual *the_victim;

        the_victim = target;
        holder.sprintf("%s strikes a &+Rfatal blow&* to %s with %s %s!\n\n", 
	      	        get_title(), the_victim->get_title(),  
                        (is_male() ? "his" : "her"), wpn_name); 

        if (get_type() == OBJ_TYPE_PLAYER)
        {
            tmp_player = (Player *) this;

            // award points for the damage done 
            // possibly changing how this works 
	    // tmp_player->increase_exp(prev_health - target->get_health());

            tmp_player->
                  send_plr("You strike %s &+Rfatally&* with your %s.\n\n", 
                     (can_see) ? the_victim->get_title() : "someone", wpn_name);
        }

        if (the_victim->get_type() ==  OBJ_TYPE_PLAYER)
        {
            tmp_player = (Player *) the_victim;
            tmp_player->
               send_plr("%s strikes you &+rfatally&* with %s %s.\n\n", 
	            (target_loc->is_lit(tmp_player) ? get_title() : "Someone"), 
                       (is_male() ? "his" : "her"), wpn_name);
            tmp_player->send_plr("Darkness overtakes you as life slips "
                                 "away from your bleeding form.\n");
        }

        if ((get_type() == OBJ_TYPE_PLAYER) &&
            (the_victim->get_type() == OBJ_TYPE_PLAYER))
        {
            (get_loc())->send_location(holder.str_show(), (Player *) this, 
                                (Player *) the_victim, CAN_SEE);
            (get_loc())->send_location(
                "You hear the pained death cry of someone in the darkness.\n", 
                             (Player *) this, (Player *) the_victim, CANT_SEE);
        }
        else
        {
            (get_loc())->send_location(holder.str_show(), tmp_player, CAN_SEE);
            (get_loc())->send_location(
                "You hear the pained death cry of someone in the darkness.\n", 
                                                   tmp_player, CANT_SEE);

        }


        holder.sprintf("%s falls to the ground &+rdead.&*\n",
                                             the_victim->get_title());
        (get_loc())->send_location(holder.str_show(), NULL, CAN_SEE);

        // Finalize the death of this MOB by executing it's death function. 
        the_victim->death(DEATHFLAG_BLOW, this);
        if ((attack_wpn != NULL) && (get_type() == OBJ_TYPE_PLAYER))
           ((Player *) this)->increase_rank_exp(2, attack_wpn, (2 - (1 * hit)));

        return 0; 
      }

//                               ENDING OF DEATH BY BLOW
//############################################################################

      /* award experience for damage done */
      /* this will probably be changing */
      /*      if (get_type() == OBJ_TYPE_PLAYER)
         ((Player *) this)->increase_exp(prev_health - target->get_health());
      */

      /* if it is a range weapon, we already ran the special which
         should have taken care of the following section, so return */
      if (attacker_range)
      {
         if (target->get_type() == OBJ_TYPE_PLAYER)
         {
            Player *flee_player;

            flee_player  = (Player *) target;

            if (flee_player->get_health() < flee_player->get_wimpy())
            {
               flee_player->
                  send_plr("\nYou &+Rflee&* from the fight in a panic!\n");

               run_command(flee_player, "flee");
               return 0;
            }
         }

         return 1;
      }

      tmp_player = NULL;

      the_hardness = (int) 
          ((((float) damage_done) / ((float) max_damage_poss)) * 5);
      if (the_hardness >= 5)
         the_hardness = 4;

      holder.sprintf("%s strikes %s &+r%s&* with %s %s.\n", 
	   	     get_title(), target->get_title(), 
                     hardness[the_hardness], 
                     (is_male() ? "his" : "her"), wpn_name); 
 
      if (get_type() == OBJ_TYPE_PLAYER)
      {
         tmp_player = (Player *) this;
         tmp_player->send_plr("You strike %s &+B%s&* with your %s.\n", 
                               (can_see) ? target->get_title() : "someone",                                     hardness[the_hardness], wpn_name);
      }
      if (target->get_type() ==  OBJ_TYPE_PLAYER)
      {
         int percent;
         char *color;

         tmp_player = (Player *) target;

         percent = (int) (((float) tmp_player->get_health() / 
                     (float) tmp_player->get_maxhealth()) * 100.0);

         if (percent == 100)
            color = "&+C";
         else if (percent > 80)
            color = "&+c";
         else if (percent > 60)
            color = "&+Y";
         else if (percent > 40)
            color = "&+Y";
         else if (percent > 20)
             color = "&+R";
         else
            color = "&+r";

         tmp_player->send_plr("%s strikes you &+R%s&* with %s %s.\n", 
	   target_loc->is_lit(tmp_player) ? get_title() : "Someone", 
           hardness[the_hardness], 
          target_loc->is_lit(tmp_player) ? (is_male() ? "his" : "her") : "their",
           wpn_name);
      }

      if ((get_type() == OBJ_TYPE_PLAYER) &&
          (target->get_type() == OBJ_TYPE_PLAYER))
      {
         (get_loc())->send_location(holder.str_show(), (Player *) this, 
                                               (Player *) target, CAN_SEE);
         (get_loc())->send_location(
                   "You hear a crash as someone makes contact.\n", 
                              (Player *) this, (Player *) target, CANT_SEE);
      }
      else
      {
         (get_loc())->send_location(holder.str_show(), tmp_player, CAN_SEE);

         (get_loc())->send_location(
                   "You hear a crash as someone makes contact.\n", 
                                                 tmp_player, CANT_SEE);
 
      }

      if ((attack_wpn != NULL) && (get_type() == OBJ_TYPE_PLAYER))
         ((Player *) this)->increase_rank_exp(2, attack_wpn, (2 - (1 * hit)));

      if (tmp_player != NULL)
      {
         if (tmp_player->get_health() < tmp_player->get_wimpy())
         {
            tmp_player->
                    send_plr("\nYou &+Rflee&* from the fight in a panic!\n");

            run_command(tmp_player, "flee");
            return 0;
         }
      }
      
      return 1;
      
   }

   /* we were blocked */
   else
   {
      holder.sprintf("%s deflects %s's %s with %s %s.\n", 
	   	     target->get_title(), get_title(), wpn_name,  
                     (is_male() ? "his" : "her"), defend_name); 
 
      if (get_type() == OBJ_TYPE_PLAYER)
      {
         tmp_player = (Player *) this;
         tmp_player->send_plr("%s deflects your %s with his %s.\n", 
                               can_see ? target->get_title() : "Someone", 
                                                        wpn_name, defend_name);
      }
      if (target->get_type() ==  OBJ_TYPE_PLAYER)
      {
         tmp_player = (Player *) target;
           tmp_player->send_plr("You deflect %s's %s with your %s.\n", 
	     target_loc->is_lit(tmp_player) ? get_title() : "someone", 
                                                   wpn_name, defend_name);
      }

      if ((get_type() == OBJ_TYPE_PLAYER) &&
          (target->get_type() == OBJ_TYPE_PLAYER))
      {
         (get_loc())->send_location(holder.str_show(), (Player *) this, 
                                      (Player *) target, CAN_SEE);
         (get_loc())->send_location("You hear the loud clang of deflection.\n", 
                            (Player *) this, (Player *) target, CANT_SEE);
      }
      else
      {
         (get_loc())->send_location(holder.str_show(), tmp_player, CAN_SEE);
         (get_loc())->send_location("You hear the loud clang of deflection.\n", 
                                                  tmp_player, CANT_SEE);
      }
      if ((attack_wpn != NULL) && (get_type() == OBJ_TYPE_PLAYER))
         ((Player *) this)->increase_rank_exp(2, attack_wpn, (2 - (1 * hit)));

      return 1;
   }

   return 1;
}

/***********************************************************************
 ** death - execute the death of this Individual.
 **
 ** Parameters: pointer to Individual attacking (in case of BLOW)         
 **             flag of attack type [BLOW, SPECIAL, SLAY]
 **
 ** Returns:  1 death occured.
 **           0 death did not occur; a Special pre-empted death. (not impl. yet)
 **          -1 for error                                        (not impl. yet)
 **
 ** ## AMERIST 07-20-00 ##
 ***********************************************************************/

/*
    DEATH()

    Death can be caused by a variation of different stimuli in the codebase

    (1) BLOW - 
        This is how most adventuers die.  In battle.  It should be extended 
        to leave a corpse (and not just a spillage of equipment.)

    (2) SPECIAL -
        MOBs and Players run across any number of deadly things are they are
        adventuring.  Rooms that instantly kill, spells that instantly kill,
        objects that instantly kill, in every case this is caused by a Special.
        In the case of a Special activated by another player that information
        should be treated as if the person died FIGHTING.  In the case that
        a Special kills the Individual that is activated by an object that
        object (the room, the cursed rod, etc.) should put its name in the 
        SEESLAIN message/log.  There should be an extention to this to define
        if it leaves a corpse or not.

    (3) SLAY - 
        The SLAY command simply blows an Individual away be it a MOB or a 
        player without even executing a death special, it also puts a message 
        out into the LOG and should spit up a SEESLAIN message stating that
        an Individial was SLAYn; perhaps an option for not leaving a corpse
        should also be added.

The death function should mostly do cleanup and codebase mechanism operations;
the function that calls the death() function should do any messages to players
and rooms about the death.  
 */

 // Easy Interface for a SPECIAL death (dealt by an object).
int Individual::death( int attack_flag, char *name /*, int corpse */ )
{
    return death( attack_flag, NULL, name );
}

// Easy Interface for a BLOW or SLAY command.
int Individual::death( int attack_flag, Individual *the_attacker /*, 
                                                                int corpse */ )
{
    return death( attack_flag, the_attacker, NULL );
}

int Individual::death( int attack_flag, Individual *the_attacker, 
                                                 char *name /*, int corpse */ )
{
    Player  *tmp_player = NULL;
    Strings holder;

    name = NULL;  // to avoid compile warnings, I assume we may use this later?

    // Adjust the message sent out to anyone with SEESLAIN set to On appropriately.
    switch(attack_flag)
    {
        case DEATHFLAG_BLOW:
            holder.sprintf("&+Y[&+M%s&+W has been slain by &+M%s&+Y]&*\n",
                            get_title(), the_attacker->get_title());
            break;

        case DEATHFLAG_SPECIAL:
            holder.sprintf("&+Y[&+M%s&+W has been slain by &+Ma special&+Y]&*\n",
                            get_title());
            break;

        case DEATHFLAG_SLAY:
            holder.sprintf("&+Y[&+M%s&+W has been smote by &+M%s&+Y]&*\n",
                            get_title(), the_attacker->get_title());
            break;

        default:
            holder.sprintf("&+Y[&+M%s&+W has been slain...&+Y]&* (unknown flag error)\n",
                            get_title());
    }

    mainstruct->send_all_players(holder.str_show(), NULL, ADMINFLAG_SEESLAIN);

    // set the mobile as a corpse
    if (get_type() == OBJ_TYPE_MOBILE)
    {
       indflags->set_flag(INDFLAG_CORPSE);
       holder = get_altname();
       holder.str_cat(" corpse body");
       set_altname(holder.str_show());
    }

    drop_unworn();
    stop_fighting();

    //Dead men don't fight.
    mainstruct->remove_fighter(this);
    mainstruct->clear_fighter(this);

    // This is because it was done this way in the take_a_swing algorithm
    if(get_type() == OBJ_TYPE_PLAYER)
        tmp_player = ( Player *) this;
    else if((the_attacker != NULL) && 
            (the_attacker->get_type() == OBJ_TYPE_PLAYER))
        tmp_player = ( Player *) the_attacker;

    // When the slay command is used it shouldn't set off any death specials.
    if( attack_flag != DEATHFLAG_SLAY )
        check_specials("on_death", this, the_attacker, NULL, this, tmp_player);

    // Remove the mobile from the database.
    if (get_type() == OBJ_TYPE_PLAYER)
    {
       Mobile      *corpse_mob;
       Mobile      *ghost_mob;
       Mobile      *ghost_clone;
       Object_List *the_dbase;
       MudObject   *tmp_obj = NULL;
       Location    *new_loc;
       Race        *the_race;
       Connection  *the_connection;
 
       tmp_player = (Player *) this;
       /* give them 10 minutes to go get their stuff before the zone
          resets */
       (mainstruct->get_dbase())->extend_reset(this, 600);

       /* now make a corpse to put in their place */
       the_dbase = mainstruct->get_dbase();
       corpse_mob = the_dbase->player_to_mobile(tmp_player);
       (corpse_mob->get_indflags())->set_flag(INDFLAG_CORPSE);

       /* swap their stuff */
       swap_worn(corpse_mob);

       /* initialize the corpse mobile for being a corpse */
       (corpse_mob->get_indflags())->set_flag(INDFLAG_CORPSE);
       holder = corpse_mob->get_altname();
       holder.str_cat(" corpse body");
       corpse_mob->set_altname(holder.str_show());
       corpse_mob->set_location(tmp_player->get_loc(), NULL);
       corpse_mob->set_rot_timer(the_config.secs_to_rot);

       /* now create a ghost and place it with the corpse */
       if ((ghost_mob = (Mobile *) the_dbase->get_mudobject(
                               the_config.hardcoded_areaname.str_show(), 
                               the_config.ghost_name.str_show())) == NULL)
       {
          mainstruct->
             log_error("Could not get ghost template, death sequence failed.\n",
                                                                        "death");
          return -1;
       } 

       ghost_clone = (Mobile *) the_dbase->clone_object(ghost_mob);
       ghost_clone->set_location(tmp_player->get_loc(), NULL);

       /* Not exactly what clones is for, but better than creating another
          variable */
       ghost_clone->set_clones(tmp_player->get_name());

       corpse_mob->set_clones(ghost_clone->get_name());

       if ((tmp_player->get_indflags())->get_flag(INDFLAG_MALE))
          (ghost_clone->get_indflags())->set_flag(INDFLAG_MALE);
       else
          (ghost_clone->get_indflags())->clr_flag(INDFLAG_MALE);

       /* now log their death if appropriate */
       if (the_config.conf_flags->set_flag(CF_LOGDEATH))
       {
          holder.sprintf("%s has been slain.", get_title());
          mainstruct->log_event(holder.str_show());
       }

       /* now we move the player to the death start location for their race
          and display the death text */
       if (((the_race = mainstruct->get_race(tmp_player->get_race())) == NULL) ||
           ((tmp_obj = mainstruct->get_object(the_race->get_death_location())) 
                                                               == NULL) ||
           (tmp_obj->get_type() != OBJ_TYPE_LOCATION))
       {
          holder.sprintf("Could not get death loc player's race '%s'.", 
                                                     tmp_player->get_race());
          mainstruct->log_error(holder.str_show(), "death");
          holder.sprintf("%s@%s", the_config.backup_locname.str_show(), 
                                  the_config.hardcoded_areaname.str_show());
          new_loc = (Location *) mainstruct->get_object(holder.str_show());
       }
       else
          new_loc = (Location *) tmp_obj;

       the_connection = tmp_player->get_connection();
       the_connection->start_paging();
       
       tmp_player->set_location(new_loc, tmp_player->get_loc());

       tmp_player->send_plr("%s\n", the_race->get_death_text());

       new_loc->show_location(tmp_player, 
                       (tmp_player->get_gameflags())->get_flag(GAMEFLAG_BRIEF));
       the_connection->end_paging(tmp_player);
       tmp_player->set_health(1);
    }

    if (get_type() == OBJ_TYPE_MOBILE)
       ((Mobile *) this)->set_rot_timer(the_config.secs_to_rot);

    // If this was a BLOW then this MOB was in a fight, the attacking MOB should
    //      be checked to see if its still in a fight.
    if ((attack_flag == DEATHFLAG_BLOW) && 
        (mainstruct->get_num_attacking(the_attacker) == 0))
            the_attacker->indflags->clr_flag(INDFLAG_FIGHTING);         

    return 1;
}


/***********************************************************************
 ** try_step_away - if we have a range weapon, try to step out of the
 **                 enemy's range
 **
 ** Parameters: None
 **
 ** Returns: 1 for stepped away, 
 **          0 for no,
 **         -1 for error
 **
 ***********************************************************************/

int Individual::try_step_away(void)
{
   Moveable   *tmp_moveable = NULL;
   int        attacker_range = 0;
   int        target_range = 0;
   Player     *tmp_player = NULL;
   Individual *target;
   Strings    holder;

   if (get_target() == NULL)
      return -1;

   target = get_target();

   if ((tmp_moveable = get_wielded(Left)) == NULL)
         tmp_moveable = get_wielded(Right);

   /* does the attacker wield a long range weapon (bow, slingshot, etc) */
   if ((tmp_moveable != NULL) && 
       (tmp_moveable->get_type() == OBJ_TYPE_WEAPON) && 
       (!STRCASECMP(((Weapon *) tmp_moveable)->get_weapon_class(), "bow")))
      attacker_range = 1;


   if ((tmp_moveable = target->get_wielded(Left)) == NULL)
      tmp_moveable = target->get_wielded(Right);

   /* does the target it a long range weapon (bow, slingshot, etc) */
   if ((tmp_moveable != NULL) && 
       (tmp_moveable->get_type() == OBJ_TYPE_WEAPON) && 
       (!STRCASECMP(((Weapon *) tmp_moveable)->get_weapon_class(), "bow")))
      target_range = 1;
   
   /* 50 percent chance we step away, unless the target has a range
      weapon as well, so then we do nothing */
   if ((target->get_target() == this) &&
       attacker_range && !target_range && 
       (tmp_player->get_loc())->is_lit(tmp_player) &&
       (((int) (100.0*(rand()/(RAND_MAX+1.0)))) < 50))
   {
      target->clr_target();

      if (get_type() == OBJ_TYPE_PLAYER)
      {
         tmp_player = (Player *) this;
         tmp_player->send_plr("You step out of range of %s.\n\n", 
                                                       target->get_title());
      }

      if (target->get_type() ==  OBJ_TYPE_PLAYER)
      {
         tmp_player = (Player *) target;
         tmp_player->send_plr("%s steps out of range of you.\n\n", get_title());
      }

      holder.sprintf("%s steps out of range of %s!\n\n", 
	      	         get_title(), target->get_title());  

      if ((get_type() == OBJ_TYPE_PLAYER) &&
          (target->get_type() == OBJ_TYPE_PLAYER))
      {
         (get_loc())->send_location(holder.str_show(), (Player *) this, 
                                                      (Player *) target);
      }
      else
      {
         (get_loc())->send_location(holder.str_show(), tmp_player);
      }
      return 1;
   }
   return 0;
}


/***********************************************************************
 ** get_loc - gets the location of the individual
 **
 **
 ***********************************************************************/
   
Location *Individual::get_loc()
{
   return get_loc(0);
}

/***********************************************************************
 ** get_loc - gets the location of the individual
 **
 ** Parameters: override_checks - shall we override the checks for
 **                               screwed up locations?
 **
 ***********************************************************************/
   
Location *Individual::get_loc(int override_checks)
{
   override_checks = 0;

   return cur_loc;
}

/***********************************************************************
 ** get_loc_name - gets the name of the player location
 **
 ** Parameters: the_name - the name to set the player to
 **
 ** Returns:  name character string for success
 **           NULL for failure
 **
 ***********************************************************************/
   
char *Individual::get_loc_name()
{
   if (cur_loc == NULL)
      return NULL;
   return cur_loc->get_name();
}


/***********************************************************************
 ** clr_curr_loc - clears the pointer to the current location
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Individual::clr_curr_loc(void)
{
   cur_loc = NULL;
   return 1;
}


/***********************************************************************
 ** reset_curr_loc - resets the pointer to the location before reload
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Individual::reset_curr_loc(void)
{
   MudObject *new_loc;
   Strings   holder;

   if (get_location() == NULL)
   {
      mainstruct->log_error("locname set to null, couldn't reset loc", 
                                                          "reset_curr_loc");
      return -1;
   }

   new_loc = mainstruct->get_object(get_location());
   if ((new_loc == NULL) || (new_loc->get_type() != OBJ_TYPE_LOCATION))
   {
      holder.sprintf("Location '%s' for object '%s' is not valid!\n", 
                           get_location(), get_name());
      mainstruct->log_error(holder.str_show(), "reset_curr_loc");
      return -1;
   }
   return set_location((Location *) new_loc, get_loc());
}


/***********************************************************************
 ** set_curr_loc - sets the pointer to the location before reload
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Individual::set_curr_loc(Location *the_loc)
{
   Strings holder;

   if (the_loc == NULL)
   {
      printf("tried to assign player to null loc\n");
      return -1;
   }

   if (get_location() == NULL)
   { 
      holder.sprintf("%s@%s", the_loc->get_name(), the_loc->get_area());
      set_location(holder.str_show());
   }

   cur_loc = the_loc;

   holder.sprintf("%s@%s", cur_loc->get_name(), cur_loc->get_area());


   if (get_contained_by() == NULL)
   {
      fault("Player not contained by anything!\n");
   }

   if (get_type() == OBJ_TYPE_PLAYER)
      ((Player *) this)->set_pronouns(cur_loc->get_inventory());

   return 1;
}

/***********************************************************************
 ** take_blow - calculates damage and subtracts from health
 **
 ** Parameters: None
 **
 ** Returns: damage done for success, -1 for failure
 **
 ***********************************************************************/

int Individual::take_blow(int the_damage, int body_part)
{
   worn_item *tmp_items;
   int       armor_value = 0;
   int       real_damage;
   int       bonus_armor = 0;

   if ((body_part < 0) || (body_part > WEARFLAG_FEET))
      return -1;

   /* if they hold a shield, 50 percent chance the shield absorbs the
      blow dependant on the armor value */
   if (((left_hand != NULL) && 
                             (left_hand->get_type() == OBJ_TYPE_WEARABLE)) ||
     ((right_hand != NULL) && (right_hand->get_type() == OBJ_TYPE_WEARABLE)))
   {
      Wearable *the_shield;
      int      will_absorb;

      if ((left_hand != NULL) && (left_hand->get_type() == OBJ_TYPE_WEARABLE))
         the_shield = (Wearable *) left_hand;
      else
         the_shield = (Wearable *) right_hand;

      will_absorb = (int) (2.0 * (rand()/(RAND_MAX+1.0)));

      if (will_absorb)
         bonus_armor += the_shield->get_armor();
   }
   tmp_items = worn_on[body_part];
   while (tmp_items != NULL)
   {
      armor_value = armor_value + (tmp_items->the_obj)->get_armor();
      tmp_items = tmp_items->next_worn;
   }

   armor_value += bonus_armor;

   real_damage = the_damage - armor_value;

   /* if they have lots of shielding, we do a min damage which ends up
      being 1/10 of their attempted damage done.  If that still is 0, 
      we just give em 1 damage */
   if (real_damage <= 0)
      real_damage = (int) the_damage / 10;
   if (real_damage == 0)
      real_damage = 1;

   health = health - real_damage;
   return health;
}


/***********************************************************************
 ** do_damage - calculates damage and subtracts from health, accounting
 **             for possible death
 **
 ** Parameters: None
 **
 ** Returns: health of player after damage is done
 **
 ***********************************************************************/

int Individual::do_damage(int the_damage, int body_part)
{
   int results;
   Strings holder;

   /* if the player dies on this blow */
   if ((results = take_blow(the_damage, body_part)) <= 0)
   {
      holder.sprintf("%s falls over, dead!\n\n", get_title()); 

      if (get_type() == OBJ_TYPE_PLAYER)
         (get_loc())->send_location(holder.str_show(), (Player *) this);
      else
         (get_loc())->send_location(holder.str_show(), NULL);

      // We use the death function now

      death(DEATHFLAG_SPECIAL, NULL, get_name());

      if (get_type() == OBJ_TYPE_PLAYER)
      {
         ((Player *) this)->send_plr(
                               "Darkness envelops you...you are dead.\n");
      }
      else
      {
	//         (get_loc())->remove_contained((MudObject *) this);
      }
      return results;
   }

   if (get_type() == OBJ_TYPE_PLAYER)
   {
      Player *the_player;
      Flags  *tmp_indflags;

      the_player  = (Player *) this;

      tmp_indflags = the_player->get_indflags();
      if ((tmp_indflags->get_flag(INDFLAG_FIGHTING)) && 
          (results < the_player->get_wimpy()))
      {
         run_command(the_player, "flee");
      }
   }
   return results;
}



/***********************************************************************
 ** drop_all - drops all objects in the room
 **
 ** Parameters: None
 **
 ** Returns: returns the number of objects dropped
 **
 ***********************************************************************/
   
int Individual::drop_all(void)
{
   int count = 0;
   Location  *the_loc;
   MudObject *the_obj;

   /* Lets drop all the inventory of the ind in its current location. */
   the_loc = get_loc();
   inventory.reset_current();
   the_obj = inventory.get_next();

   unwield_moveable(Left);
   unwield_moveable(Right);

   while (the_obj != NULL)
   {
       if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
          remove_item((Wearable *) the_obj);

       remove_contained(the_obj);
       if (the_obj->is_a_moveable())
          ((Moveable *)the_obj)->set_moved();
       the_loc->add_contained(the_obj);
       inventory.reset_current();
       the_obj = inventory.get_next();
       count++;
   }
   return count;

}


/***********************************************************************
 ** drop_unworn - drops all objects not worn in the room
 **
 ** Parameters: None
 **
 ** Returns: returns the number of objects dropped
 **
 ***********************************************************************/
   
int Individual::drop_unworn(void)
{
   int count = 0;
   Location  *the_loc;
   MudObject *the_obj;

   /* Lets drop all the inventory of the ind in its current location. */
   the_loc = get_loc();
   inventory.reset_current();
   the_obj = inventory.get_next();

   unwield_moveable(Left);
   unwield_moveable(Right);

   while (the_obj != NULL)
   {
       if ((the_obj->get_type() != OBJ_TYPE_WEARABLE) ||
           (!is_worn((Wearable *) the_obj)))
       {
          remove_contained(the_obj);
          if (the_obj->is_a_moveable())
             ((Moveable *)the_obj)->set_moved();
          the_loc->add_contained(the_obj);
          inventory.reset_current();
       }
       the_obj = inventory.get_next();
       count++;
   }
   return count;
}


/***********************************************************************
 ** swap_worn - takes all worn items from this individual and removes them,
 **             moving them to the inventory of the individual passed in and
 **             making them wear it
 **
 ** Parameters: the_ind - the individual to wear the items
 **
 ** Returns: returns the number of objects swapped
 **
 ***********************************************************************/
   
int Individual::swap_worn(Individual *the_ind)
{
   int count = 0;
   MudObject *the_obj;

   /* Lets drop all the inventory of the ind in its current location. */
   inventory.reset_current();
   the_obj = inventory.get_next();

   while (the_obj != NULL)
   {
       if ((the_obj->get_type() == OBJ_TYPE_WEARABLE) &&
           (is_worn((Wearable *) the_obj)))
       {
          remove_item((Wearable *) the_obj);
          remove_contained(the_obj);
          the_ind->add_contained(the_obj);
          the_ind->wear_item((Wearable *) the_obj);
          inventory.reset_current();
          count++;
       }
       the_obj = inventory.get_next();
   }
   return count;
}


/***********************************************************************
 ** clr_target - clears the target pointer
 **
 **
 ***********************************************************************/

void Individual::clr_target(void)
{
   target = NULL;
}


/***********************************************************************
 ** get_bare_weapon - gets the bare weapon string for this individual
 **
 **
 ***********************************************************************/


char *Individual::get_bare_weapon()
{
   return bare_weapon.str_show();
}


/***********************************************************************
 ** set_bare_weapon - sets the bare weapon string for this weapon
 **
 **
 ***********************************************************************/

void Individual::set_bare_weapon(char *the_name)
{
   bare_weapon = the_name;
}


/***********************************************************************
 ** copy_individual_attrib - copies all the individual attributes over from 
 **                          another individual
 **
 ** Parameters: copy_from - the object we copy attributes from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Individual::copy_individual_attrib(Individual *copy_from)
{
   /******* set the individual attributes *****/
   set_constitution(copy_from->get_constitution());
   set_health(copy_from->get_maxhealth());
   set_strength(copy_from->get_strength());
   set_dexterity(copy_from->get_dexterity());
   indflags->copy_flags(copy_from->get_indflags());

   return 1;
}


/***********************************************************************
 ** display_health - displays the health of the individual in a string
 **
 ** Parameters: the_player - the player to display to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Individual::display_health(Player *the_player)
{
   char *health_level[] = {"is near death", "is severely injured",
             "is badly hurt", "is bleeding profusely", 
             "has several deep wounds", "has minor cuts and scrapes", 
	     "has superficial wounds", "is in excellent health"};
   int  health_num;

   /* print the health of hte mobile */
   health_num = (int) ((((float) get_health()) / 
                             ((float) get_maxhealth())) * 7.0);
   
   if (health_num > 7)
      the_player->send_plr("%s is almost immortal!\n", get_title());
   else
      the_player->send_plr("%s %s.\n", get_title(), 
                                                health_level[health_num]);

}


/***********************************************************************
 ** is_asleep - returns 1 for asleep and 0 for awake
 **
 **
 ***********************************************************************/

int Individual::is_asleep()
{
   return indflags->get_flag(INDFLAG_SLEEPING);
}


/***********************************************************************
 ** is_sitting - returns 1 for sitting and 0 for standing
 **
 **
 ***********************************************************************/

int Individual::is_sitting()
{
   return indflags->get_flag(INDFLAG_SITTING);
}


/***********************************************************************
 ** game_get_obj - gets an object in the game and puts it in the individual's
 **                inventory, passing the appropriate messages around
 **
 ** Parameters: the_obj - the object we are getting
 **             from_obj - where we are getting the object from
 **             num_of - for mergers, the num of objects we are getting 
 **             is_held - is the player holding the from object
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
   
int Individual::game_get_obj(MudObject *the_obj, MudObject *from_obj, 
                                                 int num_of, int is_held)
{
   Strings holder;
   char *sex;
   int  is_player = 0;
   Player *the_player = NULL;
   int  results;

   if (!the_obj->is_a_moveable())
      return -1;

   if (get_type() == OBJ_TYPE_PLAYER)
   {
      the_player = (Player *) this;
      is_player = 1;
   }

   if (the_obj->is_an_item())
   {
      Flags  *tmp_itemflags;

      tmp_itemflags = ((Item *)the_obj)->get_itemflags();
      if (the_obj->get_type() == OBJ_TYPE_ROPE)
      {
         if (tmp_itemflags->get_flag(ITEMFLAG_ISTIED))
	 {
            if (is_player)
	    {
               the_player->send_plr("You need to untie it first.\n");
            }
            return -1;
         }
      }
      else if (the_obj->get_type() == OBJ_TYPE_BOAT)
      {
         if (((Boat *) the_obj)->contains_players())
	 {
            the_player->send_plr("It has someone in it!\n");
            return -1;
         }
      }
   }

   if (is_player)
   {
      if (from_obj->is_an_individual())
      {
         if (the_obj->get_type() == OBJ_TYPE_WEARABLE)
         {
            Wearable *the_wearable;

            the_wearable = (Wearable *) the_obj;
            if (((Individual *) from_obj)->is_worn(the_wearable))
	    {
               the_player->send_plr("The %s is worn and must be removed.\n",
                                         the_wearable->get_title());
               return 0;
	    }
         }
      }

      if ((results = check_specials("pre_get", the_obj, NULL, NULL, 
                                               the_obj, the_player)) == 3)
                return 0;

   }

   /* Check if can carry */
   /* move to players inventory */
   if (num_of > 0)
   {
      Merger *the_merger;
      Merger *new_merger;
      MudObject *parent_merger;
      int    new_num;
      Strings parentname;

      if (is_player)
      {
         if (get_capacity() < (get_size_held() + 
                               (((Moveable *) the_obj)->get_size() * num_of)))
         {
            the_player->send_plr("You can't carry that many.\n");
            return -2;
         }

         if (the_player->get_weight_capacity() < (get_weight_held() + 
                            (((Moveable *) the_obj)->get_weight() * num_of)))
         {
            the_player->send_plr("You can't carry that many, "
                                                  "they are too heavy.\n");
            return -2;
         }
      }

      if (!the_obj->is_merger())
      {
         from_obj->remove_contained(the_obj);
      }
      else
      {
         the_merger = (Merger *) the_obj;
       
         if (num_of > the_merger->get_number_of())
	 {
            if (is_player)
               the_player->send_plr("There are not that many here.\n");

            return -1;
         }
         if (num_of == the_merger->get_number_of())
	 {
            from_obj->remove_contained(the_obj);
         }
         else
	 {         
            if (the_merger->get_parent() != NULL)
	    {
               parent_merger = 
                           mainstruct->get_object(the_merger->get_parent());
               new_merger = (Merger *)  
                   (mainstruct->get_dbase())->clone_object(parent_merger);
            }
            else
               new_merger = (Merger *) 
                          (mainstruct->get_dbase())->clone_object(the_obj);
            new_merger->set_number_of(num_of);
            the_obj = (MudObject *) new_merger;
            new_num = the_merger->get_number_of() - num_of;
            the_merger->set_number_of(new_num);
         }
      }
   }
   else
   {
      if (is_player)
      {
         if (get_capacity() < (get_size_held() + 
                                     ((Moveable *) the_obj)->get_size()))
         {
            the_player->send_plr("You can't carry that, "
                                      "you're carrying too much bulk.\n");
            return -2;
         }

         if (the_player->get_weight_capacity() < (get_weight_held() + 
               (((Moveable *) the_obj)->get_weight_held())) + 
                ((Moveable *) the_obj)->get_weight())
         {
            the_player->send_plr("You can't carry that, "
                                      "you're carrying too much weight.\n");
            return -2;
         }
      }
      from_obj->remove_contained(the_obj);
   }


   if (from_obj->get_type() == OBJ_TYPE_LOCATION)
   {
      if (is_player)
         the_player->send_plr("You get the %s.\n", the_obj->get_title());
      holder.sprintf("%s gets the %s.\n", 
                             get_title(), the_obj->get_title());
   }
   else
   {
      if (is_player)
         the_player->send_plr("You get the %s from the %s.\n", 
                            the_obj->get_title(), from_obj->get_title());
   
      if (is_held)
      {
         if (is_male())
            sex = "his";
         else
            sex = "her";
      }
      else
         sex = "the";
      
      holder.sprintf("%s gets the %s from %s %s.\n", 
           get_title(), the_obj->get_title(), sex, 
                                            from_obj->get_title());
   }
   
   (get_loc())->send_location(holder.str_show(), the_player, CAN_SEE);
 
   if (add_contained(the_obj) == 2)
      mainstruct->delete_object(the_obj);

   ((Moveable *) the_obj)->set_moved();

   if (is_player)
   {
      if ((results = check_specials("post_get", the_obj, NULL, NULL, 
                                                the_obj, the_player)) == 3)
                return 0;

   }
   
   return 1;
}


/***********************************************************************
 ** write_object - writes a individual to a file.  This write raises an
 **                error as we should not be writing just individuals ever
 **
 ** Parameters: the_file - the file we would write to
 **             build_format - is it in builder format
 **
 ***********************************************************************/

void Individual::write_object(FILE *the_file, int build_format)
{
   build_format = 0;
   the_file = NULL;
   mainstruct->log_error("Tried to write individual to file.", "write_object");
}


/***********************************************************************
 ** get_brief - this should never be run
 **
 ***********************************************************************/

char *Individual::get_brief()
{
   return NULL;
}



/***********************************************************************
 ** get_mem_size_ind - gets how much memory is being taken up by the
 **                    individual part of this object
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Individual::get_mem_size_ind()
{
   int size = 0;
   int i;
   worn_item *tmp_worn;

   size += bare_weapon.get_mem_size_dynamic();

   for (i=0; i<WEARFLAG_FEET + 1; i++)
   {
      tmp_worn = worn_on[i];
      while (tmp_worn != NULL)
      {
         size += sizeof(*tmp_worn);
         tmp_worn = tmp_worn->next_worn;
      }
   }

   return size;
}


/***********************************************************************
 ** reset_individual: resets the individual's variables to default
 **
 ***********************************************************************/

void Individual::reset_individual( void )
{
   strength = the_config.default_strength;
   dexterity = the_config.default_dexterity;
   constitution = the_config.default_constitution;
   health = get_maxhealth();
   intelligence = the_config.default_intel;
   wisdom = the_config.default_wisdom;
   charisma = the_config.default_charisma;
}


#endif

