/*******************************************************************************
 *  PROJECT: Agave
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <iostream>
#include <sstream>
#include <fstream>

#include <glibmm/error.h>
#include <gdkmm/cursor.h>

#include "gcs-paletteview.h"
#include "gcs-colorswatch.h"
#include "core/log-stream.h"
#include "gcs-i18n.h"
#include "gcs-conf.h"
#include "palettetreemodel.h"

#define NO_SIZE_REQUEST (-1)

namespace gcs
{
    namespace Widgets
    {
        const gint PaletteView::m_minCellSize = 20;
        const gint PaletteView::m_minPaletteWidth = 200;
        const gint PaletteView::m_maxVisibleRows = 6;
        const gint PaletteView::m_maxPaletteHeight = m_minCellSize * m_maxVisibleRows;

        PaletteView::PaletteView(GtkExpander* cobject, Glib::RefPtr<Glade::Xml>& glade) :
            Gtk::Expander(cobject), m_table(1, 1, true),
            m_palette(shared_ptr<pp::Palette>(new pp::Palette))
        {
            assert(m_palette);

            vector<string> palette_dirs;
            palette_dirs.push_back(AGAVE_PALETTEDIR);
            palette_dirs.push_back(Conf::get_user_palette_dir());
            m_paletteSelector = shared_ptr<PaletteSelector>(new PaletteSelector(palette_dirs));
            assert(m_paletteSelector);

            m_refPaletteModel = PaletteTreeModel::create(*m_palette);
            m_scrolledWindow.set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);
            m_table.set_spacings(0);

            // we need to put the table into an event box so that we can
            // respond to GDK enter and leave signals which are used to change
            // the cursor to a hand and back to a regular pointer
            m_eventBox.add(m_table);
            m_scrolledWindow.add(m_eventBox);
            m_vbox.set_spacing(Conf::UI_SPACING_SMALL);
            m_vbox.pack_start(m_scrolledWindow, Gtk::PACK_EXPAND_WIDGET);
            m_vbox.pack_start(*m_paletteSelector, Gtk::PACK_SHRINK);

            add(m_vbox);
            set_label(_("Palette"));

            // set spacing between table cells
            m_table.set_spacings(0);
            
            m_paletteSelector->signal_changed().connect(
                    sigc::mem_fun(*this, &PaletteView::on_palette_file_changed));
            //m_eventBox.signal_expose_event().connect(sigc::mem_fun(*this,
                        //&PaletteView::on_eventbox_expose));

            // display the currently selected palette at startup
            display_palette(m_paletteSelector->get_palette());
        }

        // The goal of this function was to draw some hash marks on the
        // background of the palette view so that it is more obvious when table
        // cells don't contain color swatches.  Disabled for now because of some
        // weird corruption
        bool PaletteView::on_eventbox_expose(GdkEventExpose* e)
        {
            m_eventBox.propagate_expose(m_table, e);
            LOG("PaletteView::on_eventbox_expose");
            LOG("area: " << e->area.x << "," << e->area.y << " @ " <<
                    e->area.width << "x" << e->area.height);
            cairo_t* cr = gdk_cairo_create(m_eventBox.get_window()->gobj());
            cairo_rectangle(cr, e->area.x, e->area.y, e->area.width, e->area.height);
            cairo_clip(cr);
            int w, h;
            w = m_eventBox.get_width();
            h = m_eventBox.get_height();
            // draw hatch marks in medium gray on the background of the eventbox
            // to make it obvious when there is no swatch at the end of a row
            LOG("drawing hatch paths");
            for (int i = 0; i < (w + h); i += Conf::UI_SPACING_SMALL)
            {
                int current_x = e->area.x + i;
                cairo_move_to(cr, current_x, e->area.y);
                cairo_line_to(cr, current_x - h, h);
                cairo_move_to(cr, w - current_x, e->area.y);
                cairo_line_to(cr, w - current_x + h, h);
            }
            cairo_set_line_width(cr, 1.0);
            cairo_set_source_rgb(cr, 0.6, 0.6, 0.6);
            LOG("stroking hatches");
            cairo_stroke(cr);
            cairo_destroy(cr);
            return true;
        }

        bool PaletteView::set_from_file(const Glib::ustring& fname)
        {
            LOG("Setting PaletteView from file " << fname);
            // try to set the new palette file
            if (m_paletteSelector->set_palette_file(fname))
            {
                // yep, the palette file was found, update the view's palette
                // object to point to the newly selected one
                display_palette(m_paletteSelector->get_palette());
                return true;
            }
            return false;
        }

        void PaletteView::display_palette(const shared_ptr<pp::Palette>& palette)
        {
            if (palette)
            {
                m_palette = palette;
                // hide the swatches while we remove old ones and add new ones
                hide_all();

                LOG("resetting PaletteView");
                // delete the ColorSwatches from the widget
                std::list<Gtk::Widget*> swatches = m_table.get_children();
                for (std::list<Gtk::Widget*>::iterator sw = swatches.begin();
                        sw != swatches.end(); sw++)
                {
                    m_table.remove(**sw);
                    //delete (dynamic_cast<ColorSwatch*>(*sw));
                    delete (*sw);
                }
                m_rows = 0;

                LOG("Populating PaletteView");
                guint cols = m_palette->columns();
                if (!cols) // columns wasn't defined by the palette file
                {
                    cols = m_minPaletteWidth / m_minCellSize;
                    // if there's only enough colors for one row, just set the
                    // number of columns to that value
                    if (cols > m_palette->size())
                        cols = m_palette->size();
                }
                m_rows = m_palette->size() / cols;

                if (m_palette->size() % cols)
                {
                    m_rows += 1;
                }

                m_table.resize(m_rows, cols);
                gint min_height = m_rows * m_minCellSize;
                m_table.set_size_request(NO_SIZE_REQUEST, min_height);
                if (min_height > m_maxPaletteHeight)
                {
                    m_scrolledWindow.set_size_request(NO_SIZE_REQUEST, m_maxPaletteHeight);
                }
                else
                {
                    m_scrolledWindow.set_size_request(NO_SIZE_REQUEST, min_height + 2);
                }

                for (guint r = 0; r < m_rows; r++)
                {
                    for (guint c = 0; c < cols; c++)
                    {
                        guint index = c + r * cols;
                        if (index < m_palette->size())
                        {
                            ColorSwatch *pSwatch = (new
                                    ColorSwatch((*m_palette)[index]));
                            pSwatch->signal_selected().connect(
                                    sigc::bind(sigc::mem_fun(*this,
                                            &PaletteView::on_color_selected), index));
                            pSwatch->signal_enter_notify_event().connect(
                                    sigc::mem_fun(*this, &PaletteView::on_enter_swatch));
                            pSwatch->signal_leave_notify_event().connect(
                                    sigc::mem_fun(*this, &PaletteView::on_leave_swatch));
                            m_table.attach(*pSwatch, c, c + 1, r, r + 1);
                        }
                    }
                }

                // re-show them -- we're done.
                show_all();
            }
        }


        bool PaletteView::on_color_selected(gint index)
        {
            signal_color_selected().emit((*m_palette)[index]);
            LOG("PaletteView::on_color_selected");
            return true;
        }


        bool PaletteView::on_enter_swatch(GdkEventCrossing* event)
        {
            Glib::RefPtr<Gdk::Window> win = m_eventBox.get_window();
            if (win)
            {
                Glib::RefPtr<Gdk::Display> dpy = Gdk::Display::get_default();
                Gdk::Cursor cursor(dpy, Gdk::HAND2);
                win->set_cursor(cursor);
            }
            return true;
        }


        bool PaletteView::on_leave_swatch(GdkEventCrossing* event)
        {
            Glib::RefPtr<Gdk::Window> win = m_eventBox.get_window();
            if (win)
            {
                win->set_cursor();
            }
            return true;
        }


        void PaletteView::on_palette_file_changed(void)
        {
            display_palette(m_paletteSelector->get_palette());
        }

    } // namespace Widgets
} // namespace gcs
