/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *  Portions (c) the Coaster Development Team
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtkmm/main.h>

#ifdef HAVE_GNOME
#include <libgnome/gnome-init.h>
#endif // HAVE_GNOME

#ifdef HAVE_GCONFMM
#include <gconfmm.h>
#endif // HAVE_GCONFMM

#include "gcs-mainwindow.h"
#include "gcs-i18n.h"
#include "core/log-stream.h"

namespace gcs
{

    class OptionGroup : public Glib::OptionGroup
    {
        public:
            OptionGroup();
    };

    OptionGroup::OptionGroup(void) :
        Glib::OptionGroup ("Agave", "A Colorscheme Designer",
                "A tool for generating pleasing colorschemes from an initial color")
    {
        Glib::OptionEntry entry1;
        entry1.set_long_name("debug");
        entry1.set_short_name('d');
        entry1.set_description("Enable Debugging output");
        add_entry(entry1, agave_debug);
    }

} // namespace gcs

int main(int argc, char **argv)
{
    // wrap everything in a try..catch block and display a message telling who
    // to contact about any uncaught exceptions
    try
    {
        bindtextdomain(GETTEXT_PACKAGE, AGAVE_LOCALEDIR);
        bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
        textdomain(GETTEXT_PACKAGE);

        Glib::OptionContext context;

        gcs::OptionGroup option_group;
        context.set_main_group(option_group);
        option_group.set_translation_domain(GETTEXT_PACKAGE);

        // this Gtk::Main constructor requires gtkmm 2.6
        Gtk::Main app(argc, argv, context);

        LOG("PREFIX: " << PREFIX);
        LOG("UIDIR: " << AGAVE_UIDIR);
        LOG("ICONDIR: " << AGAVE_ICONDIR);
        LOG("PALETTEDIR: " << AGAVE_PALETTEDIR);

#ifdef HAVE_GNOME
        gnome_program_init(PACKAGE, PACKAGE_VERSION, LIBGNOME_MODULE,
                argc, argv, GNOME_PROGRAM_STANDARD_PROPERTIES, NULL);
#endif // HAVE_GNOME

#ifdef HAVE_GCONFMM
        Gnome::Conf::init();
#endif // HAVE_GCONFMM
        gcs::MainWindow& w = gcs::MainWindow::Instance();
        app.run(w);
    }
    catch (Glib::Error& ex)
    {
        std::cerr << std::endl << "Ouch, that hurt." << std::endl <<
            "Please report this error to " << PACKAGE_BUGREPORT << std::endl
            << "Include the following information:" << std::endl
            << ex.what() << std::endl;
    }

    return 0;
}

/** \mainpage Agave Developer Documentation
 *
 * \section Introduction
 * This document provides an overview of the code of the Agave application.
 * Obviously this information will be most likely of interest to those who want
 * to hack on Agave or contribute to its development.  Obviously you can get all
 * of this information simply from looking at the code, but this presents the
 * information in a much easier-to-reference way.  Additions and corrections are
 * welcome to this documentation, as well as to the code itself.
 *
 * \section getting-started Getting Started
 * Most things in the application should be inside the 'gcs' namespace, which is
 * further divided into functional groups like widgets or dialog windows.  A
 * good place to start is to to get an idea of what the application is about is
 * the MainWindow class.  This is the main window of the application.  The
 * Color and Scheme classes are also interesting since they do a majority of
 * the work.  Initially the plan was to split core things like this out into
 * its own library, but I'm not sure how useful that would be.  It could still
 * be done in the future, but I don't really have any plans to do so at this
 * point.  
 */
