/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#ifndef __GCS_COLOR_H
#define __GCS_COLOR_H

#include "gcs-types.h"
#include <gdkmm/color.h>
#include <iostream>     // for std::ostream, etc.
#include <memory>       // for std::auto_ptr

namespace gcs
{

    /** The Color class keeps track of a color by a couple different property
     * sets that define a color.
     *
     * The first property set is a specification of the color in terms of Red,
     * Green, and Blue values (RGB).  It also specifies the color in terms of
     * Hue, Saturation, and Value (HSV).  The RGB value can also be specified
     * in a hex string, similar to how colors are usually specified in HTML and
     * CSS.
     */
    class Color
    {
        public:
            // Create a color object and return it as a shared pointer
            static ColorPtr create(void);
            static ColorPtr create(tHexString str);
            static ColorPtr create(const gint r, const gint g, const gint b, const Glib::ustring name = Glib::ustring());
            static ColorPtr create(const Gdk::Color&);
            static ColorPtr create(const Color&);

            /** Destroy a Color object */
            ~Color(void);		// destructor

            Gdk::Color& gdk(void) const;

            /** Sets the r, g, and b values of the Color object, and updates
             * other values as necessary
             * \param r The Red component
             * \param g The Green component
             * \param b The Blue component */
            void set(const gint r, const gint g, const gint b, const Glib::ustring name = Glib::ustring());

            /** Sets the Hex string representation of the Color object, and
             * updates other values as necessary
             * \param hex The hex string representing RGB */
            bool set(const tHexString hex);

            /** Gets the RGB hex string */
            tHexString get_hexstring() const;
            /** Gets the name of the color */
            Glib::ustring get_name() const;
            /** Sets the name of the color */
            void set_name(Glib::ustring name);
            /** Gets the red component */
            gint get_red(void) const;
            /** Sets the red component.  Must also update h, s, and v */
            void set_red(const gint r);	// must update h, s, v
            /** Gets the green component */
            gint get_green(void) const;
            /** Sets the green component.  Must also update h, s, and v */
            void set_green(const gint g);	// must update h, s, v
            /** Gets the blue component */
            gint get_blue(void) const;
            /** Sets the blue component.  Must also update h, s, and v */
            void set_blue(const gint b);	// must update h, s, v
            /** Gets the hue component */
            gint get_hue(void) const;
            /** Sets the hue component.  Must also update r, g, and b */
            void set_hue(const gint h);	// must update r, g, b
            /** Gets the saturation component */
            gint get_saturation(void) const;
            /** Sets the saturation component.  Must also update r, g, and b */
            void set_saturation(const gint s);	// must update r, g, b
            /** Gets the value component */
            gint get_value(void) const;
            /** Sets the value component.  Must also update r, g, and b */
            void set_value(const gint v);	// must update r, g, b

            gint get_luminance(void) const;


            // Conversion functions
            /** Converts an RGB color definition to a color definition in hex
             * string format
             * \param color A color definition in RGB format
             * \return A hex string representation of RGB
             * \sa hex_to_rgb */
            static tHexString rgb_to_hex(const tColorRgb color);

            /** Converts an RGB value to an HSV value */
            static tColorHsv rgb_to_hsv(const tColorRgb rgb);

            static gdouble rgb_as_p(gint rgbVal);

            static tHexString normalize_hex(tHexString hex);

            /** compares two Color objects to see if they're equal */
            bool operator==(const Color& c);
            bool operator!=(const Color& c);
            bool operator<(const Color& c);

            Color& operator=(const Color& c);

            /** Prints a representation of the Color to an output stream */
            friend std::ostream& operator<<(std::ostream& out, const Color c);
            friend std::ostream& operator<<(std::ostream& out, const ColorPtr c);


            /** Gets the values of red, green, and blue in a struct */
            tColorRgb get_rgb(void) const;
            /** Gets the values of Hue, Saturation, and Value in a struct */
            tColorHsv get_hsv(void) const;
            void set_hsv(tColorHsv);

#ifdef UNIT_TEST
            friend class ColorConstructionSuite;
#endif

        protected:
            Color(void);

            /** Create a Color object by specifying RGB in hex string
             * \param hex the hex string that dfines the color */
            Color(const tHexString hex);

            /** Create a Color by specifying separate red, green, and blue values
             * \param r The value of Red, between 0 and 255
             * \param g The value of Green, between 0 and 255
             * \param b The value of Blue, between 0 and 255 */
            Color(const gint r, const gint g, const gint b, const Glib::ustring name = Glib::ustring());

            /** Construct a Color object from a Gdk::Color object */
            Color(const Gdk::Color& c);

            /** Copy Constructor */
            Color(const Color& c);

        private:
            std::auto_ptr<Gdk::Color> pColor;
            Glib::ustring m_name;

            static const double m_redLuminance;
            static const double m_greenLuminance;
            static const double m_blueLuminance;
    };

} // namespace gcs

#endif // __GCS_COLOR_H
