/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * $Id: cardtilewidget.cpp,v 1.3 2001/05/18 10:09:48 alex Exp $
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <contactobj.h>
#include <miscfunctions.h>
#include <cardtilewidget.h>
#include <qcolor.h>
#include <qevent.h>
#include <qsize.h>
#include <qfont.h>
#include <qmessagebox.h>
#include <kconfig.h>
#include <ctype.h>
#include <carddialog.h>
#include <videotoolbox.h>
#include <tilewidget.h>

#define CARD_TITLE_COLOR 165, 188, 209
#define CARD_ACTIVE_TITLE_COLOR 165, 188, 100
#define CARD_FIELD_SPACING 3
#define CARD_INNER_MARGIN 2

#define IDSTRING "CardTileWidget: "

//#define DEBUG_CARD_TILEWIDGET

extern KConfig *GlobalConfig;

QDict<QWidget> CardTileWidget::contactEditDialogs;

/**
 *		The default constructor...
 */
CardTileWidget::CardTileWidget(CardTileView *parent, ContactObject *c, const char *name)
  : TileWidget(parent, name), TemplateView(c)
{
	contactEditDialogs.setAutoDelete(false);
	setCard(c);
	title=0;
	fCount=0;
	localFields=&CardTileView::getFields();
	_localParent=parent;
	c->setView(this);
	setBackgroundColor(QColor("white"));
	setFrameStyle(QFrame::Box | QFrame::Plain);
	spacing=CARD_FIELD_SPACING;
	fieldList.setAutoDelete(true);
	valueList.setAutoDelete(true);
	if(card() && localFields)
	{
		allocateNewFields(localFields);
		allocateTitle();
		update();
	}
}

CardTileWidget::~CardTileWidget()
{
	// Destructor
	if(title)
		delete title;
}

ContactObject *CardTileWidget::card()
{
	return _card;
}

void CardTileWidget::setCard(ContactObject *c)
{
	_card=c;
}

void CardTileWidget::allocateTitle(QString titleName)
{
	if (!title)
		title=new QLabel(titleName, this);
	if (localParent())
		title->installEventFilter( localParent() );
	title->setAutoResize(true);
	title->setBackgroundColor(QColor(CARD_TITLE_COLOR));
	title->setAlignment(AlignCenter);
	title->move(CARD_INNER_MARGIN,CARD_INNER_MARGIN);
	title->setFixedWidth(width()-CARD_INNER_MARGIN*2);
	QFont f=title->font();
	f.setBold(true);
	title->setFont(f);
}

void CardTileWidget::allocateNewFields(QStringList *fields)
{
	unsigned int count = fields->count();
	if (count < valueList.count())
	{
#ifdef DEBUG_CARD_TILEWIDGET
		printf(IDSTRING"Removing %d*2 QLabels.\n",valueList.count()-count);
#endif
		while ((valueList.count()-count) != 0)
		{
			valueList.remove(valueList.at(0));
			fieldList.remove(fieldList.at(0));
		}
	}
	else if (count > valueList.count())
	{
#ifdef DEBUG_CARD_TILEWIDGET
		printf(IDSTRING"Adding %d*2 QLabels.\n",count-valueList.count());
#endif
		for(;(count-valueList.count())!=0;)
		{
			valueList.append(new QLabel(this));
			if( localParent() ) valueList.at(valueList.count()-1)->installEventFilter(localParent());
			valueList.at(valueList.count()-1)->hide();
			valueList.at(valueList.count()-1)->setFont(QPxFont("Helvetica",12,QFont::Normal));
			valueList.at(valueList.count()-1)->setBackgroundColor(QColor("white"));
			valueList.at(valueList.count()-1)->setAutoResize(true);
			valueList.at(valueList.count()-1)->setAlignment(QLabel::WordBreak);
			fieldList.append(new QLabel(this));
			if( localParent() ) fieldList.at(fieldList.count()-1)->installEventFilter(localParent());
			fieldList.at(fieldList.count()-1)->hide();
			fieldList.at(fieldList.count()-1)->setFont(QPxFont("Helvetica",12,QFont::Bold));
			fieldList.at(fieldList.count()-1)->setBackgroundColor(QColor("white"));
			fieldList.at(fieldList.count()-1)->setAutoResize(true);
		}
	}
}

int CardTileWidget::keyCount()
{
	return 1;
}

QString CardTileWidget::key(int number)
{
	if (number!=1)
		return QString::null;
	if(title)
		return title->text();
	return QString::null;
}

QString CardTileWidget::keyName(int number)
{
	if (number!=1)
		return QString::null;
	return QString::fromLatin1("Card name");
}

void CardTileWidget::resize(int w, int h)
{
#ifdef DEBUG_CARD_TILEWIDGET
	printf(IDSTRING"1:\t\tw():%d, h():%d w:%d, h:%d\n", width(), height(), w, h);
#endif
	if(height()==h && width()==w)
		return;
	QWidget::resize(w,h);
#ifdef DEBUG_CARD_TILEWIDGET
	printf(IDSTRING"2:\t\tw():%d, h():%d w:%d, h:%d\n", width(), height(), w, h);
#endif
//	update();
}

void CardTileWidget::update()
{
	setFields(0, card(), localFields, false, localParent());

	unsigned int count=fieldList.count();
	int lastX=CARD_INNER_MARGIN, lastY=CARD_INNER_MARGIN;
  
	if(title)
	{
		title->move(lastX,lastY);
		title->setFixedWidth(width()-CARD_INNER_MARGIN*2);
		lastY=title->height()+spacing;
		title->repaint();
	}

	for(unsigned int i=0;i<count;i++)
	{
		valueList.at(i)->hide();
		fieldList.at(i)->hide();
	}

	for(unsigned int i=0;i<count;i++)
	{
#ifdef DEBUG_CARDTILE
		printf(IDSTRING"Field %d = <%s>\n",i, (const char *)valueList.at(i)->text());
#endif
		if(!valueList.at(i)->text().isEmpty())
		{
			QFontMetrics fm(valueList.at(i)->font());
			QLabel *l=fieldList.at(i);
			l->move(lastX, lastY);
			l->repaint();
			l->show();
			lastX+=l->width();
			l=valueList.at(i);
			l->setFixedWidth(width()-CARD_INNER_MARGIN-lastX);
			l->move(lastX, lastY);
			l->repaint();
			l->show();
			// Resizing ...
			int w=l->width();
			QStringList lines=split(l->text(),'\n');
			unsigned int len=0;
			for (unsigned int i=0;i<lines.count();i++)
			{
				len+=(fm.width(lines[i])+w-1)/w;
			}
			//int h=l->height();
			l->resize(w,fm.height()*len);

			lastX=CARD_INNER_MARGIN;
			lastY+=l->height()+spacing;
		}
	}
  
	resize(width(), lastY-spacing+CARD_INNER_MARGIN*2);

	static_cast<SortingWidget *>(localParent())->screenUpdate();
}

void CardTileWidget::setState(ViewState state)
{
	
	if (!title)
		return;
 	if (state==Selected) {
		title->setBackgroundColor(QColor(CARD_ACTIVE_TITLE_COLOR));
	} else {
		title->setBackgroundColor(QColor(CARD_TITLE_COLOR));
	}

	DataWidget::setState(state);
	title->repaint();
}

void CardTileWidget::setTitle(QString tn, QWidget */*parent*/)
{
	if (title)
		title->setText(tn);
}

void CardTileWidget::addField(int fieldNo, QString fn, QString fv, bool multiline, QWidget */*parent*/)
{
#ifdef DEBUG_CARDTILE
	printf(IDSTRING"Adding @ position: %d.\n",fieldNo);
#endif
	QString label;
	if(multiline)
		label= QString::null;
	else
		label=fn+QString::fromLatin1(": ");
	fieldList.at(fieldNo)->setText(label);
	valueList.at(fieldNo)->setText(fv);
}

void CardTileWidget::setFields(int offset, ContactObject *card, QStringList *f, bool isRecursed, QWidget *parent)
{
	if(!f || f->count()==0)
	{
#ifdef DEBUG_CARD_TILEWIDGET
		printf(IDSTRING" Error: No fields!\n");
#endif
		QMessageBox::information(this,"No control fields", "There are no control fields for viewing the "
					 "contacts in the tile view.\nPlease run the Configuration Browser (Address Book/Contacts).", "&Ok");
		return;
	}

	QStringList tempList=split((*f)[0],'=');
	QString list_top = tempList[0].lower();
	int which;

	if (list_top=="name")
		which=NAME_FIELD;
	else if (list_top =="work address")
		which=WORK_ADDRESS_FIELD;
	else if (list_top =="home address")
		which=HOME_ADDRESS_FIELD;
	else if (list_top =="phone")
		which=PHONE_FIELD;
	else if(list_top =="email")
		which=DEFAULT_EMAIL_FIELD;
	else if (list_top =="url")
		which=URL_FIELD;
	else if (list_top =="title")
		which=TITLE_FIELD;
	else if (list_top =="role")
		which=ROLE_FIELD;
	else if (list_top =="org")
		which=ORGANIZATION_FIELD;
	else if (list_top =="class")
		which=CLASS_FIELD;
	else if (list_top =="fn")
		which=FN_FIELD;

	QString temp=getField(which,card,(tempList.count()==2)?tempList[1]:QString::null);

	if (!isRecursed)
		setTitle((temp.isEmpty()) ? QString("(Empty)") : temp, parent);

	bool genericAdded=false;

	for (unsigned int i=(isRecursed) ? 0 : 1; i < f->count(); i++)
	{
		tempList=split((*f)[i],'=');
		QString content;
		list_top = tempList[0].lower();

		if (list_top == "name")
		{
			content=getField(NAME_FIELD,card,(tempList.count()==2)?tempList[1]:QString::null);
			addField(i+offset, "Name", content, false, parent);
		}
		else if (list_top == "work address")
		{
			content=getField(WORK_ADDRESS_FIELD,card,(tempList.count()==2)?tempList[1]:QString::null);
			addField(i+offset, "Work addr.",content, false, parent);
		}
		else if (list_top == "home address")
		{
			content=getField(HOME_ADDRESS_FIELD,card,(tempList.count()==2)?tempList[1]:QString::null);
			addField(i+offset, "Home addr.",content, false, parent);
	
		}
		else if (list_top == "phone")
		{
			content=getField(PHONE_FIELD,card,(tempList.count()==2)?tempList[1]:QString::null);
			if(!content.isEmpty())
				addField(i+offset, "Phone",content, false, parent);
			else if (!genericAdded)
			{
				QStringList s("generic");
				setFields(i, card, &s, true, parent);
				genericAdded=true;
			}
		}
		else if (list_top =="mobile")
		{
			content=getField(PHONE_FIELD,card,"mobile");
			if (!content.isEmpty())
				addField(i+offset, "Mobile",content, false, parent);
			else if (!genericAdded)
			{
				QStringList s("generic");
				setFields(i, card, &s, true, parent);
				genericAdded=true;
			}
		}
		else if (list_top =="generic")
		{
			content=getField(PHONE_FIELD,card,"generic");
			addField(i+offset, "Phone",content, false, parent);
		}
		else if (list_top =="fax")
		{
			content=getField(PHONE_FIELD,card,"fax");
			if (!content.isEmpty())
				addField(i+offset, "Fax",content, false, parent);
			else if(!genericAdded)
			{
				QStringList s("generic");
				setFields(i, card, &s, true, parent);
				genericAdded=true;
			}
		}
		else if (list_top=="voice")
		{
			content=getField(PHONE_FIELD,card,"voice");
			if (!content.isEmpty())
				addField(i+offset, "Voice",content, false, parent);
			else if(!genericAdded)
			{
				QStringList s("generic");
				setFields(i, card, &s, true, parent);
				genericAdded=true;
			}
		}
		else if(list_top=="email")
		{
			content=getField(DEFAULT_EMAIL_FIELD,card);
			addField(i+offset, "email",content, false, parent);
		}
		else if (list_top=="url")
		{
			content=getField(URL_FIELD,card);
			addField(i+offset, "URL",content, false, parent);
		}
		else if (list_top=="title")
		{
			content=getField(TITLE_FIELD,card);
			addField(i+offset, "Title",content, false, parent);
		}
		else if (list_top=="role")
		{
			content=getField(ROLE_FIELD,card);
			addField(i+offset, "Role",content, false, parent);
		}
		else if (list_top=="org")
		{
			content=getField(ORGANIZATION_FIELD,card);
			addField(i+offset, "Org",content, false, parent);
		}
		else if (list_top=="class")
		{
			content=getField(CLASS_FIELD,card);
			addField(i+offset, "Class",content, false, parent);
		}
	}
}

QString CardTileWidget::getField(int which, ContactObject *card, QString codes)
{
#ifdef DEBUG_CARD_TILEWIDGET
	printf(IDSTRING"getField %d...\n",which);
#endif
	bool flag;
	switch(which)
	{
		case NAME_FIELD:
			if(card->n().count())
				return processName(card->n(), codes);
		case HOME_ADDRESS_FIELD:
		{
			flag=false;
			for (unsigned int i=0; i < card->homeAdr().count(); i++) {
				if (!card->homeAdr()[i].isEmpty()) {
					flag=true;
				}
			}
			if (flag) {
				return processAddress(card->homeAdr(), codes);
			}
			break;
		}
		case PHONE_FIELD:
			if(card->phone().count())
				return processPhone(card, codes);
			break;
		case WORK_ADDRESS_FIELD:
		{
			flag=false;
			for (unsigned int i=0;i<card->workAdr().count();i++) {
				if (!card->workAdr()[i].isEmpty()) {
					flag=true;
				}
			}
			if (flag)
				return processAddress(card->workAdr(), codes);
			break;
		}
		case DEFAULT_EMAIL_FIELD:
			if(card->emails().count())
				return card->emails()[card->defaultEmail()];
			break;
		case URL_FIELD:
			if(card->URLs().count())
				return card->URLs()[0];
			break;
		case TITLE_FIELD:
			if(!card->title().isEmpty())
				return card->title();
			break;
		case ROLE_FIELD:
			if(!card->role().isEmpty())
				return card->role();
			break;
		case ORGANIZATION_FIELD:
			if(!card->org().isEmpty())
				return card->org();
			break;
		case CLASS_FIELD:
			if(!card->classCard().isEmpty())
				return card->classCard();
			break;
		case FN_FIELD:
			if(!card->fn().isEmpty())
				return card->fn();
			break;
	}
	return QString::null;
}

QString CardTileWidget::processName(QStringList n, QString codes)
{
#ifdef DEBUG_CARD_TILEWIDGET
	printf(IDSTRING"Processing name...\n");
#endif
	QString result=QString(" ")+codes+QString::fromLatin1("  ");
	if(codes.isEmpty())
		result=" name surname, other  ";
	int pos, t;
	pos=result.find("surname",0,false);
	if(pos!=-1) result.replace(pos,7,(n[1].length()) ? n[1] : QString::fromLatin1(" "));
	pos=result.find("name",0,false);
	if(pos!=-1) result.replace(pos,4,(n[0].length()) ? n[0] : QString::fromLatin1(" "));
	pos=result.find("suffix",0,false);
	if(pos!=-1) result.replace(pos,6,(n[4].length()) ? n[4] : QString::fromLatin1(" "));
	pos=result.find("prefix",0,false);
	if(pos!=-1) result.replace(pos,6,(n[3].length()) ? n[3] : QString::fromLatin1(" "));
	pos=result.find("other",0,false);
	if(pos!=-1) result.replace(pos,5,(n[2].length()) ? n[2] : QString::fromLatin1(" "));
	bool flag=true;
	// i have @ least 3 spaces!
	pos=0;
	while(flag)
	{
		QString tmp;
		t=pos;
		if(result.find(' ',pos+2)==t+2)
		{
			tmp.sprintf("%c",((const char *)result)[pos+2]);
			result.replace(pos,3,tmp);
		}
		if(result.find("\\n",pos+2)==t+2)
		{
			result.replace(pos,2,"");
		}
		pos=result.find(' ',pos+1);
		if(pos==-1) flag=false;
	}
	flag=true;
	// replacing \\n with \n
	pos=0;
	result=result.simplifyWhiteSpace().stripWhiteSpace();
#ifdef DEBUG_CARD_TILEWIDGET
	printf(IDSTRING"Final name: %s\n", (const char *)result);
#endif
	while(!isalnum(((const char *)result)[result.length()-1]) && result.length()!=0)
		result.remove(result.length()-1,1);
	return result;
}

QString CardTileWidget::processAddress(QStringList adr, QString codes)
{
	QString result;
	if(!codes)
		result=" po-box, street\\ncity, region\\ncountry  ";
	else
		result=QString(" ")+codes+QString(" ");
	int pos, t;
	pos=result.find("po-box",0,false);
	if(pos!=-1) result.replace(pos,6,(adr[0].length()) ? adr[0] : QString::fromLatin1(" "));
	pos=result.find("extended",0,false);
	if(pos!=-1) result.replace(pos,8,(adr[1].length()) ? adr[1] : QString::fromLatin1(" "));
	pos=result.find("street",0,false);
	if(pos!=-1) result.replace(pos,6,(adr[2].length()) ? adr[2] : QString::fromLatin1(" "));
	pos=result.find("city",0,false);
	if(pos!=-1) result.replace(pos,4,(adr[3].length()) ? adr[3] : QString::fromLatin1(" "));
	pos=result.find("region",0,false);
	if(pos!=-1) result.replace(pos,6,(adr[4].length()) ? adr[4] : QString::fromLatin1(" "));
	pos=result.find("code",0,false);
	if(pos!=-1) result.replace(pos,4,(adr[5].length()) ? adr[5] : QString::fromLatin1(" "));
	pos=result.find("country",0,false);
	if(pos!=-1) result.replace(pos,7,(adr[6].length()) ? adr[6] : QString::fromLatin1(" "));
	bool flag=true;
	// i have @ least 3 spaces!
	pos=0;
	while(flag)
	{
		QString tmp;
		t=pos;
		if(result.find(' ',pos+2)==t+2)
		{
			tmp.sprintf("%c",((const char *)result)[pos+2]);
			result.replace(pos,3,tmp);
		}
		if(result.find("\\n",pos+2)==t+2)
		{
			result.replace(pos,2,"");
		}
		pos=result.find(' ',pos+1);
		if(pos==-1) flag=false;
	}
	flag=true;
	// replacing \\n with \n
	pos=0;
	while(flag)
	{
		pos=result.find("\\n",pos);
		if(pos!=-1)
			result.replace(pos,2,"\n");
		else
			flag=false;
	}
	result=result.stripWhiteSpace();
	while(!isalnum(((const char *)result)[result.length()-1]) && result.length()!=0)
		result.remove(result.length()-1,1);
	return result;
}

QString CardTileWidget::processPhone(ContactObject *card, QString codes)
{
	QString list_cur, result=QString::null;
	QStringList cList=split(codes, QChar(','));
	QStringList p=card->phone();
	QStringList pt=card->phoneType();

	for (unsigned int i = 0; i< cList.count(); i++)
	{
		list_cur = cList[i];
		if (list_cur == "work+voice")
		{
			for (unsigned int j = 0; j < pt.count(); j++) {
				if (pt[i].find("work",0,false)!=-1 && pt[i].find("voice",0,false)!=-1) {
					result.append(p[i].stripWhiteSpace()+QString::fromLatin1(" (work, voice)\n"));
				}
			}
		}
		else if (list_cur == "work+fax")
		{
			for (unsigned int j=0; j < pt.count(); j++)
				if(pt[i].find("work",0,false)!=-1 && pt[i].find("fax",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+" (work,fax)\n");
		}
		else if (list_cur=="work+mobile")
		{
			for (unsigned int j=0; j < pt.count(); j++)
				if(pt[i].find("work",0,false)!=-1 && pt[i].find("mobile",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+" (work,mobile)\n");
		}
		else if(list_cur=="work+other")
		{
			for (unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("work",0,false)!=-1 && pt[i].find("voice",0,false)==-1 && pt[i].find("fax",0,false)==-1 &&
						pt[i].find("mobile",0,false)==-1)
					result.append(p[i].stripWhiteSpace()+" (work,mobile)\n");
		}
		if(list_cur=="home+voice")
		{
			for(unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("home",0,false)!=-1 && pt[i].find("voice",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+" (home, voice)\n");
		}
		else if(list_cur=="home+fax")
		{
			for(unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("home",0,false)!=-1 && pt[i].find("fax",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+" (home,fax)\n");
		}
		else if(list_cur=="home+mobile")
		{
			for(unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("home",0,false)!=-1 && pt[i].find("mobile",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+" (home,mobile)\n");
		}
		else if(list_cur=="home+other")
		{
			for(unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("home",0,false)!=-1 && pt[i].find("voice",0,false)==-1 && pt[i].find("fax",0,false)==-1 &&
						pt[i].find("mobile",0,false)==-1)
					result.append(p[i].stripWhiteSpace()+" (home,mobile)\n");
		}
		else if(list_cur=="mobile")
		{
			for(unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("mobile",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+"\n");
		}
		else if (list_cur=="fax")
		{
			for(unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("fax",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+"\n");
		}
		else if (list_cur=="voice")
		{
			for(unsigned int j=0;j<pt.count();j++)
				if(pt[i].find("voice",0,false)!=-1)
					result.append(p[i].stripWhiteSpace()+"\n");
		}
		else if (list_cur=="generic")
		{
			result.append(p[0].stripWhiteSpace()+"\n");
		}
	}
	// removing the last \n thing
	if(!result.isEmpty())
		result.remove(result.length()-1,1);
	return result;
}

void CardTileWidget::open()
{
	CardDialog *vcd=static_cast<CardDialog *>(dialogContactQuery(_card->path()));

	if (vcd) {
		vcd->raise();
	} else {
		vcd = new CardDialog(_card, this);
		addContactDialog(_card->path(), vcd);
		vcd->show();
		connect(vcd, SIGNAL(dialogClosed(QString)), this, SLOT(removeDialog(QString)));
	}
}


QWidget *CardTileWidget::dialogContactQuery(QString path)
{
#ifdef DEBUG_CTW
	printf(IDSTRING"Querying: %s\n", (const char *)path);
#endif
	return contactEditDialogs[path];
}

void CardTileWidget::addContactDialog(QString path, QWidget *widget)
{
#ifdef DEBUG_CTW
	printf(IDSTRING"Adding: %s\n", (const char *)path);
#endif
	if(!contactEditDialogs[path])
		contactEditDialogs.insert(path, widget);
}

void CardTileWidget::removeContactDialog(QString path)
{
#ifdef DEBUG_CTW
	printf(IDSTRING"Removing: %s\n", (const char *)path);
#endif
	contactEditDialogs.remove(path);
}


void CardTileWidget::removeDialog(QString path)
{
	removeContactDialog(path);
}

QWidget* CardTileWidget::localParent()
{
  return _localParent;
}

