/***************************************************************************
                localmailfolder.cpp  -  the local mail folder class
                             -------------------
    begin                : Mon Mar  5 16:16:00 EET 2001
    copyright            : (C) 2001 by theKompany (www.thekompany.com>
    author               : Eugen Constantinescu
    email                : eug@thekompany.com
 ***************************************************************************/

#include <config.h>
#include <localmailfolder.h>
#include <mailfolder.h>
#include <qdir.h>
#include <qfile.h>
#include <qdatetime.h>
#include <indexclass.h>
#include <qtextstream.h>
#include <servernotifier.h>
#include <kconfig.h>
#include <accounts.h>
#include <accountmanager.h>
#include <messagedevice.h>
#include <messagedescriptor.h>
#include <messagefactory.h>
#include <mimecodec.h>
#include <dateclass.h>

extern KConfig *GlobalConfig;

LocalMailFolder::LocalMailFolder(const QString &_storageDevice) : MailFolder(_storageDevice)
{
  setupFiles();
	loadIndex();
};

LocalMailFolder::~LocalMailFolder()
{
}

void LocalMailFolder::setupFiles()
{
	messagesFileName=getStorageDevice()+"/messages";
};

void LocalMailFolder::loadIndex()
{
	// debug
	printf("mailfolder: [%s] is reading index...\n", (const char *)name());
	
	IndexClass *tindex;
  setUnread(0);
  setPruneIndexCount(0);

	QFile f(getIndexFileName());
	QDataStream stream(&f);

  // check index version number
	if(f.open(IO_ReadOnly))
	{
		// debug
		printf("mailfolder: index file opened.\n");
		
  	QString idxExpectedVersion=QString::number(INDEX_VERSION);
		QString idxVersion;
		stream>>idxVersion;

  	if(idxExpectedVersion!=idxVersion)
  	{
    	printf("Folder %s attempted to read an older version or invalid index\
        	cache, discarding index...\n", (const char *)name());
  	}
  	else
  	{
	  	while(!stream.atEnd())
	  	{
		  	tindex=new IndexClass(this);

				// read index
      	stream>>(*tindex);

				// check to see if the index was already included
				IndexClass *pruneIndex;
				if((pruneIndex=indexCollection[tindex->getID()]))
				{
					// ...it was, increment the prune list
					incrementPruneIndexCount();

					// update folder stats
					if(pruneIndex->getUnreadMark()) decrementUnread();
				}
				else
				{
					// update the last index id
					unsigned long tlast=tindex->getID().mid(3).toULong();
					if(tlast>getLastIndexID())
						setLastIndexID(tlast+1);
				}
				// add index to collection
				indexCollection.replace(tindex->getID(), tindex);
				
				// debug
				// printf("localmailfolder: [loadIndex] loaded %s...\n", (const char *)tindex->getID());

				// update folder stats
		  	if(tindex->getUnreadMark()) incrementUnread();
	  	}
  	}
		
		f.close();
	}
}
			
IndexClass *LocalMailFolder::createMessage(const QCString &text, const QCString &uid, const QDateTime &rcvtime, const QString &account, MessageClass *parsedMessage, bool bSync, const unsigned flags)
{
  if( !uid || uid.isNull() || uid.isEmpty() )
    return (IndexClass*)0;

	// rfc-parse message
	MessageClass *rfcMessage;
	
	if ( parsedMessage )
	  rfcMessage = parsedMessage;
	else
  	rfcMessage = new MessageClass(text);
  	
	// create index, set id and add to collection
	IndexClass *index=new IndexClass(this);
	syncIndexID(index);
	indexCollection.insert(index->getID(), index);
	
	// create message device
	MessageDevice *dev=new MessageDevice(index);
	
	// populate descriptor
	MessageDescriptor &descriptor=dev->getDescriptor();
	descriptor.load(*rfcMessage);
	
	// add other data (account, uid, received time, index id etc)
	descriptor.account=account;
	descriptor.indexID=index->getID();
	descriptor.receivedDate=(QCString)DateClass(rcvtime);
	descriptor.uid=uid;
	
	if( -1!=descriptor.uid.find("_copy", -6) )
	{
	  // we need a new UID and message Id.
  	descriptor.messageID=MIMECodec::encodeMessageId(descriptor.indexID.right(5), "aethera");
  	descriptor.messageID+="@localhost";
  	descriptor.uid=descriptor.messageID;
	}
	
	// debug
//	printf("localmailfolder: created message with:\n");
//	printf("\taccount %s\n", (const char *)descriptor.account);
//	printf("\tuid %s\n", (const char *)descriptor.uid);
//	printf("\treceived time %s\n", (const char *)descriptor.receivedDate);
//	printf("\tindex id %s\n", (const char *)descriptor.indexID);
	
	index->setMessageID(descriptor.messageID);
	if( flags&MailFolder::Seen )
	{
	  index->setUnreadMark(false);
  	descriptor.status="Read";
	}
	else
	{
	  index->setUnreadMark(true);
  	descriptor.status="New";
	}
	
	// update folder stats
	if(index->getUnreadMark())
	  incrementUnread();
	
	// save message
	QFile dataFile(getMessagesFileName());
	dataFile.open(IO_WriteOnly|IO_Append);

	// write MBOX information
	QCString mboxInfo;
	mboxInfo="From - " + descriptor.receivedDate + "\r\n";
	dataFile.writeBlock((const char *)mboxInfo, mboxInfo.length());
	
	// write uniblock data (index is uniblock by default)
	index->setUniblockOffset(dataFile.at());
	index->setUniblockLength(text.length());
	index->setMultipartOnly(false);
	
	dataFile.writeBlock((const char *)text, index->getUniblockLength());
	dataFile.writeBlock("\r\n", 2);
	
	dataFile.close();
	
	// create index part list and update offsets
	for(unsigned int i=0;i<rfcMessage->partList.count();i++)
		index->addPart(new MimePart(*rfcMessage->partList.at(i)), true);
		
	// save descriptor (that will save the index as well)
	dev->saveDescriptor();
	
	// cleanup
	delete dev;
	
	// add index and descriptor to hierarchy
	GlobalConfig->setGroup("Threading");
	if(GlobalConfig->readEntry("Enable")=="Yes")
	{
		reparentIndex(index);
		crossReferenceIndexSet(index);
	}
	
	// notify client or wait for a sync signal
	if( bSync )
    syncMessages.append(name()+"/"+index->getID());
	else
	{
  	ServerNotifier::thisInstance()->objectCreated(name()+"/"+index->getID());
  	ServerNotifier::thisInstance()->objectChanged(name());
	}
	
	if ( !parsedMessage )
	  delete parsedMessage;
	
	return index;
}

// Notify the client when we have a timeout
void LocalMailFolder::sync()
{
  if( !syncMessages.count() )
  {
    printf("\nLocalMailFolder::sync() : Nothing to sync!\n");
  }
  else
  {
    ServerNotifier::thisInstance()->objectCreated(syncMessages);
    syncMessages.clear();
  	ServerNotifier::thisInstance()->objectChanged(name());
  }
}
	
IndexClass *LocalMailFolder::copyMessage(IndexClass *idx)
{
	if(!idx) return 0;
	// load descriptor
	MessageDevice *dev=new MessageDevice(idx);
	dev->loadDescriptor();
	MessageDescriptor &descriptor=dev->getDescriptor();
	
	QCString message=dev->rfc822Message();
	DateClass rcvTime((QCString)descriptor.receivedDate.latin1());
	QString newUid=descriptor.uid+"_copy";
	// Create the new message
  IndexClass *newIndex=createMessage( message, newUid.latin1(), rcvTime.toQDateTime(),
                                      descriptor.account.latin1(), 0, false,
                                      idx->getUnreadMark() ? MailFolder::NO_FLAGS : MailFolder::Seen);
  // Delete the allocated device
  delete dev;
  return newIndex;
}

IndexClass *LocalMailFolder::moveMessage(IndexClass *idx)
{
	if(!idx) return 0;
  IndexClass *newIndex=copyMessage(idx);
  idx->getParentFolder()->deleteMessage(idx);

	return newIndex;
}

bool LocalMailFolder::deleteMessage(IndexClass *idx)
{
	// debug
	printf("\nlocalmailfolder: message deleted\n");
	fflush(stdout);
	
	if(!idx || idx->getParentFolder()!=this) return false;
	
	QString path=name()+"/"+idx->getID();
	
	if(idx->getUnreadMark()) decrementUnread();	
	
	indexCollection.remove(idx->getID());

	delete idx;
	saveIndex();
		
	// notify the client
	ServerNotifier::thisInstance()->objectDeleted(path);
	ServerNotifier::thisInstance()->objectChanged(name());
	
//	return getAutoExpunge()?expunge():true;
  return true;
}

bool LocalMailFolder::shouldExpunge()
{
	// get waste configuration
	GlobalConfig->setGroup("Folder Management");
	unsigned wasteSize=GlobalConfig->readNumEntry("Waste");
	
	if(wasteSize<51200) wasteSize=102400; // default 100k, at least 50k allowed per folder
	
	// debug
	printf("localmailfolder: computing waste...\n");
	
	// compute waste
	unsigned int folderSize=QFileInfo(getMessagesFileName()).size(), usedSize=0;

	QDictIterator<IndexClass> it(indexCollection);
	IndexClass *index;
	while((index=it.current()))
	{
		// debug
		// printf("localmailfolder: adding waste for index %s...\n", (const char *)index->getID());
		
		if(index->isMultipartOnly())
		{
			// debug
			// printf("\tindex is multipart, adding part lengths\n");
			for(int i=0;i<index->getPartCount();i++)
			{
				// debug
				// printf("\t\tpart %d: %d\n", i, index->getPartAt(i)->length);
				
				usedSize+=index->getPartAt(i)->length;
			}
		}
		else
		{
			// debug
			// printf("\tindex is uniblock, adding length %d\n", index->getUniblockLength());
			
			usedSize+=index->getUniblockLength();
		}

		// debug
		// printf("\tadding index descriptor length %d\n", index->getDescriptorLength());
		
		usedSize+=index->getDescriptorLength();
		++it;
	}

	// debug
	// printf("localmailfolder: computed waste; results: \n\tfolder size [%d]\n\tused [%d]\n\twaste [%d]\n\ttolerated waste [%d]\n", folderSize, usedSize, folderSize-usedSize, wasteSize);
	
	if((folderSize-usedSize)>wasteSize) return true;

	return false;
}

bool LocalMailFolder::expunge(bool force)
{
	// debug
	// printf("mailfolder: entering expunge...\n");

	// if(force) printf("mailfolder: this is a forced expunge, proceeding\n");
		
	if(force || shouldExpunge())
	{
		// debug
		// printf("mailfolder: expunging folder %s...\n", (const char *)name());
		
		QFile msgFile(getMessagesFileName());
		QFile tempFile(getMessagesFileName()+".compact");
		QDir msgDir(getStorageDevice());
		
		if(tempFile.open(IO_WriteOnly) && msgFile.open(IO_ReadOnly))
		{
			QDictIterator<IndexClass> it(indexCollection);
			IndexClass *idx;
			while((idx=it.current()))
			{
				// debug
				// printf(" - index %s; writing descriptor...\n", (const char *)idx->getID());
				
				// write descriptor
				int descriptorLength=idx->getDescriptorLength();
				char *buf=new char[descriptorLength];
				
				msgFile.at(idx->getDescriptorOffset());
				msgFile.readBlock(buf, descriptorLength);
				
				idx->setDescriptorOffset(tempFile.at());
				tempFile.writeBlock(buf, descriptorLength);
				
				// debug
				// printf(" - wrote %d bytes\n", descriptorLength);
				
				delete []buf;
				
				// write data
				if(idx->isMultipartOnly())
				{
					for(int i=0;i<idx->getPartCount();i++)
					{
						MimePart *part=idx->getPartAt(i);
						
						// debug
						// printf(" - moving part %d:%d...\n", part->offset, part->length);
						
						char *buf=new char[part->length];
						
						msgFile.at(part->offset);
						msgFile.readBlock(buf, part->length);
						
						part->offset=tempFile.at();
						tempFile.writeBlock(buf, part->length);
						
						// debug
						// printf(" - wrote part %d:%d...\n", part->offset, part->length);
						
						delete []buf;
					}
				}
				else
				{
					// debug
					// printf(" - moving uniblock %d:%d...\n", idx->getUniblockOffset(), idx->getUniblockLength());
					
					int uniblockLength=idx->getUniblockLength();
					char *buf=new char[uniblockLength];
					
					msgFile.at(idx->getUniblockOffset());
					msgFile.readBlock(buf, uniblockLength);
					
					int oldUniblockOffset=idx->getUniblockOffset();
					idx->setUniblockOffset(tempFile.at());
					tempFile.writeBlock(buf, uniblockLength);
					
					// debug
					// printf(" - wrote uniblock %d:%d\n", idx->getUniblockOffset(), idx->getUniblockLength());
					
					for(int i=0;i<idx->getPartCount();i++)
						idx->getPartAt(i)->offset+=idx->getUniblockOffset()-oldUniblockOffset;
					
					delete []buf;
				}
				
				++it;
			}
			
			msgFile.close();
			tempFile.close();

			// debug
			// printf(" - renaming files...\n");
						
			msgDir.remove("messages");
			msgDir.rename("messages.compact", "messages");		
			
			saveIndex();
		}
		else return false;
	}
	
	// debug
	// printf("expunge done.\n");
	
	return true;
}

QString LocalMailFolder::getMessagesFileName() const
{
  return messagesFileName;
}





