/* ------------------------------------------------------------------------
 * $Id: Group.cc,v 1.8 2001/08/28 13:18:51 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-04-24 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <algorithm>

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Polhem/TransformImpl.hh"
#include "Polhem/VisitorImpl.hh"
#include "Polhem/VolumeImpl.hh"
#include "Polhem/Group.hh"

using namespace Nobel;

Group::Group()
    : _transform(new TransformImpl),
      _bounds(new VolumeImpl)
{
    // empty
}

Group::~Group()
{    
    // @@@Remove the children? Probably.
    deactivate(_transform);
    deactivate(_bounds);
    delete _transform;
    delete _bounds;
}

Transform_ptr Group::transformation()
{
    return _transform->_this();
}

Tag Group::insert(Node_ptr n)
{
    Guard<Mutex> guard(_mutex);
    static Tag tag_counter = 0;

    // Sanity check
    if (CORBA::is_nil(n) == true) return 0;

    // Add the pointer to the geometry vector
    node_t node(Nobel::Node::_duplicate(n), tag_counter++);
    _childlist.push_back(node);

    // Start listening to the node, too
    n->addListener(Node_var(_this()));

    return node.tag;
}

void Group::remove(Tag t)
{
    Guard<Mutex> guard(_mutex);

    // Look for the geometry in the shape
    std::list<node_t>::iterator r = std::find(_childlist.begin(),
					      _childlist.end(), t);
    if (r != _childlist.end()) {

	// Remove the node from the children list and lose interest in it
	_childlist.erase(r);    
	r->item->removeListener(Node_var(_this()));
    }
}
  
void Group::traverse(Visitor_ptr v)
{
    // Traverse all the children, one by one
    for (std::list<node_t>::iterator i = _childlist.begin();
	 i != _childlist.end(); i++) {
	v->traverse(i->item, i->item->transformation());
    }
}

void Group::render(Renderer_ptr r)
{
    // empty
}

void Group::pick(Nobel::Picker_ptr p)
{
    // empty
}

Volume_ptr Group::getBoundingVolume()
{
    Guard<Mutex> guard(_mutex);
    return _bounds->_this();
}

void Group::recomputeBoundingVolume() 
{
    VolumeImpl volume;

    // Step through all the children nodes and merge them together
    for (std::list<node_t>::iterator i = _childlist.begin();
	 i != _childlist.end(); i++) {
	volume.merge(i->item->getBoundingVolume());
    }

    // Did the bounding volume change at all? 
    if (volume != *_bounds) {
	
	// Update the bounding volume 
	*_bounds = volume;
	
	// Emit the proper event
	Nobel::Event e;
	e.type = BoundingVolumeChanged;
	//emit(e);
    }
}
