/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "usrp0.h"
#include "usrp_prims.h"
#include "usrp_interfaces.h"
#include "fpga_regs0.h"
#include "fusb.h"
#include <usb.h>
#include <stdexcept>
#include <assert.h>
#include <math.h>


struct rate_to_regval {
  unsigned short	rate;
  unsigned short	regval;
};

static struct rate_to_regval rate_table[] = {
#include "rate_to_regval.h"  
};

#define NELEM(x) (sizeof (x) / sizeof (x[0]))

// These set the buffer size used for each end point using the fast
// usb interface.  The kernel ends up locking down this much memory.

static const int FUSB_BUFFER_SIZE = 8 * (1L << 20);	// 8 MB
static const int FUSB_BLOCK_SIZE = 16 * (1L << 10);	// 16KB is hard limit
static const int FUSB_NBLOCKS    = FUSB_BUFFER_SIZE / FUSB_BLOCK_SIZE;


static const double POLLING_INTERVAL = 0.1;	// seconds

#if 0
/*
 * conservatively map the given rate to a regval suitable
 * for programming into the interpolator or decimator register.
 *
 * At this time we restrict rate to be a power of two between
 * 1 and 256.  We could relax this, but we'd have to document
 * a bunch of weird corner cases that don't work well.
 */
static bool
map_rate_to_regval (unsigned int rate, unsigned int *regval)
{
  if (rate < 1 || rate > 256)
    return false;

  switch (rate){
  case   1:	*regval = 0x00;	break;
  case   2:	*regval = 0x01;	break;
  case   4:	*regval = 0x11;	break;
  case   8:	*regval = 0x13;	break;
  case  16:	*regval = 0x33;	break;
  case  32:	*regval = 0x37;	break;
  case  64:	*regval = 0x77;	break;
  case 128:	*regval = 0x7f;	break;
  case 256:	*regval = 0xff;	break;
  default:
    return false;
  }
  return true;
}

#else

/*
 * Map the given rate to a regval suitable for programming into the
 * interpolator or decimator register.
 *
 * Of the values between 1 and 256, there are 97 values that
 * work.  See rate_to_regval.h for values
 */
static bool
map_rate_to_regval (unsigned int rate, unsigned int *regval)
{
  for (unsigned i = 0; i < NELEM (rate_table); i++)
    if (rate_table[i].rate == rate){
      *regval = rate_table[i].regval;
      return true;
    }

  return false;
}

#endif


/*
 * the inverse is always defined
 */
static bool
map_regval_to_rate (unsigned int regval, unsigned int *rate)
{
  *rate = (((regval >> 4) & 0xf) + 1) * ((regval & 0xf) + 1);
  return true;
}

static unsigned int
compute_freq_control_word (double master_freq, double target_freq, double *actual_freq)
{
  static const int	NBITS = 14;

  int	v = (int) rint (target_freq / master_freq * pow (2.0, 32.0));

  // v += 1 << (32 - NBITS - 1);			// add 1/2

  v = (v >> (32 - NBITS)) << (32 - NBITS);	// keep only top NBITS

  *actual_freq = v * master_freq / pow (2.0, 32.0);

  fprintf (stderr,
	   "compute_freq_control_word: target = %g  actual = %g  delta = %g\n",
	   target_freq, *actual_freq, *actual_freq - target_freq);

  return (unsigned int) v;
}

//////////////////////////////////////////////////////////////////
//
//			usrp0_common
//
////////////////////////////////////////////////////////////////

usrp0_common::usrp0_common (int which_board)
  : d_udh (0), d_ext_clk_div (0)
{
  usrp_one_time_init ();
  if (!usrp_load_standard_bits (which_board, false))
    throw std::runtime_error ("usrp0_common/usrp_load_standard_bits");
}

usrp0_common::~usrp0_common ()
{
  if (d_udh)
    usb_close (d_udh);
}

bool
usrp0_common::set_ext_clk_div (unsigned int div)
{
  d_ext_clk_div = div & 0xff;
  return _write_fpga_reg (FR_EXT_CLK_DIV, d_ext_clk_div);
}

double
usrp0_common::get_oscillator_freq () const
{
  return 125e6;			// FIXME should be 120 MHz on new boards
}

unsigned int
usrp0_common::get_ext_clk_div () const
{
  return d_ext_clk_div;
}

bool
usrp0_common::_write_fpga_reg (int regno, int value)
{
  // printf ("_write_fpga_reg (%3d, 0x%08x)\n", regno, value);
  return usrp_write_fpga_reg (d_udh, regno, value);
}

bool
usrp0_common::_set_led (int which, bool on)
{
  return usrp_set_led (d_udh, which, on);
}

bool
usrp0_common::_set_fpga_reset (bool on)
{
  return usrp_set_fpga_reset (d_udh, on);
}

bool
usrp0_common::_set_fpga_tx_enable (bool on)
{
  return usrp_set_fpga_tx_enable (d_udh, on);
}

bool
usrp0_common::_set_fpga_rx_enable (bool on)
{
  return usrp_set_fpga_rx_enable (d_udh, on);
}

bool
usrp0_common::_set_sleep_bits (int bits, int mask)
{
  return usrp_set_sleep_bits (d_udh, bits, mask);
}

////////////////////////////////////////////////////////////////
//
//			   usrp0_rx
//
////////////////////////////////////////////////////////////////

usrp0_rx::usrp0_rx (int which_board)
  : usrp0_common (which_board), d_devhandle (0), d_ephandle (0),
    d_bytes_per_poll (0), d_bytes_seen (0), d_first_read (true)
{
  struct usb_device *udev = usrp_find_device (which_board);
  if (udev == 0){
    fprintf (stderr, "usrp0_rx: can't find usrp[%d]\n", which_board);
    goto puke;
  }

  if (!usrp_usrp0_p (udev)){
    fprintf (stderr, "usrp0: sorry, this code only works with Rev 0 USRPs\n");
    goto puke;
  }

  d_udh = usrp_open_rx_interface (udev);
  if (d_udh == 0){
    fprintf (stderr, "usrp0_rx: can't open rx interface\n");
    usb_strerror ();
    goto puke;
  }

  set_adc_clk_div (2);				// 60 MHz
  set_decim_rate (256);

  for (int i = 0; i < MAX_CHAN; i++)
    set_rx_freq (i, 0);

  d_devhandle = fusb_sysconfig::make_devhandle (d_udh);
  d_ephandle = d_devhandle->make_ephandle (USRP_RX_ENDPOINT, true,
					   FUSB_BLOCK_SIZE, FUSB_NBLOCKS);

  if (!usrp_set_fpga_rx_enable (d_udh, 1)){
    fprintf (stderr, "usrp0_rx: set_fpga_rx_enable failed\n");
    usb_strerror ();
    goto puke;
  }
  
  if (!d_ephandle->start ()){
    fprintf (stderr, "usrp0_rx: failed to start end point streaming");
    usb_strerror ();
    goto puke;
  }
  return;

 puke:
  delete d_ephandle;
  delete d_devhandle;
  if (d_udh){
    usb_close (d_udh);
    d_udh = 0;
  }

  throw std::runtime_error ("usrp0_rx");
}

usrp0_rx::~usrp0_rx ()
{
  d_ephandle->stop ();
  delete d_ephandle;
  delete d_devhandle;
}


usrp0_rx *
usrp0_rx::make (int which_board)
{
  try {
    return new usrp0_rx (which_board);
  }
  catch (...){
    return 0;
  }
}

void
usrp0_rx::setup_status_polling ()
{
  int	 nchannels_active = 1;
  double sample_rate = get_oscillator_freq () / get_adc_clk_div () / get_decim_rate ();
  double byte_rate = sample_rate * 2 * sizeof (short) * nchannels_active;
  d_bytes_per_poll = (int) (byte_rate * POLLING_INTERVAL);
}

/*
 * \brief read data from the D/A's via the FPGA.
 * \p len must be a multiple of 512 bytes.
 *
 * \returns the number of bytes read, or -1 on error.
 *
 * If overrun is non-NULL it will be set true iff an RX overrun is detected.
 */
int
usrp0_rx::read (void *buf, int len, bool *overrun)
{
  int	r;
  
  if (overrun)
    *overrun = false;
  
  if (len < 0 || (len % 512) != 0){
    fprintf (stderr, "usrp0_rx::read: invalid length = %d\n", len);
    return -1;
  }

  r = d_ephandle->read (buf, len);
  if (r > 0)
    d_bytes_seen += r;

  /*
   * In many cases, the FPGA reports an rx overrun right after we
   * enable the Rx path.  If this is our first read, check for the
   * overrun to clear the condition, then ignore the result.
   */
  if (d_first_read){
    d_first_read = false;
    bool bogus_overrun;
    usrp_check_rx_overrun (d_udh, &bogus_overrun);
  }

  if (overrun != 0 && d_bytes_seen >= d_bytes_per_poll){
    d_bytes_seen = 0;
    if (!usrp_check_rx_overrun (d_udh, overrun)){
      fprintf (stderr, "usrp0_rx: usrp_check_rx_overrun failed\n");
      usb_strerror ();
    }
  }
    
  return r;
}

bool
usrp0_rx::set_adc_clk_div (unsigned int div)	// div: [2,6]
{
  if (div < 2 || div >  6)
    return false;
  
  d_adc_clk_div = div;
  return _write_fpga_reg (FR_ADC_CLK_DIV, d_adc_clk_div);
}

bool
usrp0_rx::set_decim_rate  (unsigned int rate)
{
  unsigned int	regval;

  if (!map_rate_to_regval (rate, &regval))
    return false;

  d_decim_rate = rate;
  d_decim_regval = regval;
  setup_status_polling ();
  return _write_fpga_reg (FR_DECIM, d_decim_regval);
}

bool
usrp0_rx::_set_decim_reg (unsigned int regval)
{
  d_decim_regval = regval;
  map_regval_to_rate (regval, &d_decim_rate);
  setup_status_polling ();
  return _write_fpga_reg (FR_DECIM, d_decim_regval);
}

bool
usrp0_rx::set_rx_freq (int channel, double freq)
{
  if (channel < 0 || channel > MAX_CHAN)
    return false;

  unsigned int v =
    compute_freq_control_word (get_oscillator_freq () / d_adc_clk_div,
			       freq, &d_rx_freq[channel]);

  return _write_fpga_reg (FR_RX_FREQ_0 + channel, v);
}

unsigned int
usrp0_rx::get_decim_rate () const { return d_decim_rate; }

unsigned int
usrp0_rx::get_adc_clk_div () const { return d_adc_clk_div; }


double 
usrp0_rx::get_rx_freq (int channel) const
{
  if (channel < 0 || channel >= MAX_CHAN)
    return 0;

  return d_rx_freq[channel];
}

////////////////////////////////////////////////////////////////
//
//			   usrp0_tx
//
////////////////////////////////////////////////////////////////

usrp0_tx::usrp0_tx (int which_board)
  : usrp0_common (which_board), d_devhandle (0), d_ephandle (0),
    d_bytes_per_poll (0), d_bytes_seen (0), d_first_write (true)
{
  struct usb_device *udev = usrp_find_device (which_board);
  if (udev == 0){
    fprintf (stderr, "usrp0_tx: can't find usrp[%d]\n", which_board);
    goto puke;
  }

  if (!usrp_usrp0_p (udev)){
    fprintf (stderr, "usrp0: sorry, this code only works with Rev 0 USRPs\n");
    goto puke;
  }

  d_udh = usrp_open_tx_interface (udev);
  if (d_udh == 0){
    fprintf (stderr, "usrp0_tx: can't open rx interface\n");
    usb_strerror ();
    goto puke;
  }

  set_interp_rate (256);

  for (int i = 0; i < MAX_CHAN; i++)
    set_tx_freq (i, 0);


  d_devhandle = fusb_sysconfig::make_devhandle (d_udh);
  d_ephandle = d_devhandle->make_ephandle (USRP_TX_ENDPOINT, false,
					   FUSB_BLOCK_SIZE, FUSB_NBLOCKS);

  if (!usrp_set_fpga_tx_enable (d_udh, 1)){
    fprintf (stderr, "usrp0_tx: set_fpga_tx_enable failed\n");
    usb_strerror ();
    goto puke;
  }
  
  if (!d_ephandle->start ()){
    fprintf (stderr, "usrp0_tx: failed to start end point streaming");
    usb_strerror ();
    goto puke;
  }
  return;

 puke:
  delete d_ephandle;
  delete d_devhandle;
  if (d_udh){
    usb_close (d_udh);
    d_udh = 0;
  }

  throw std::runtime_error ("usrp0_tx");
}

usrp0_tx::~usrp0_tx ()
{
  d_ephandle->stop ();
  delete d_ephandle;
  delete d_devhandle;
}


usrp0_tx *
usrp0_tx::make (int which_board)
{
  try {
    return new usrp0_tx (which_board);
  }
  catch (...){
    return 0;
  }
}

void
usrp0_tx::setup_status_polling ()
{
  int	 nchannels_active = 1;
  double sample_rate = get_oscillator_freq () / get_interp_rate ();
  double byte_rate = sample_rate * 2 * sizeof (short) * nchannels_active;
  d_bytes_per_poll = (int) (byte_rate * POLLING_INTERVAL);
}

/*!
 * \brief Write data to the A/D's via the FPGA.
 *
 * \p len must be a multiple of 512 bytes.
 * \returns number of bytes written or -1 on error.
 *
 * if \p underrun is non-NULL, it will be set to true iff
 * a transmit underrun condition is detected.
 */
int
usrp0_tx::write (const void *buf, int len, bool *underrun)
{
  int	r;
  
  if (underrun)
    *underrun = false;
  
  if (len < 0 || (len % 512) != 0){
    fprintf (stderr, "usrp0_tx::write: invalid length = %d\n", len);
    return -1;
  }

  r = d_ephandle->write (buf, len);
  if (r > 0)
    d_bytes_seen += r;
    
  /*
   * In many cases, the FPGA reports an tx underrun right after we
   * enable the Tx path.  If this is our first write, check for the
   * underrun to clear the condition, then ignore the result.
   */
  if (d_first_write && d_bytes_seen >= 4 * FUSB_BLOCK_SIZE){
    d_first_write = false;
    bool bogus_underrun;
    usrp_check_tx_underrun (d_udh, &bogus_underrun);
  }

  if (underrun != 0 && d_bytes_seen >= d_bytes_per_poll){
    d_bytes_seen = 0;
    if (!usrp_check_tx_underrun (d_udh, underrun)){
      fprintf (stderr, "usrp0_tx: usrp_check_tx_underrun failed\n");
      usb_strerror ();
    }
  }

  return r;
}

bool
usrp0_tx::set_interp_rate (unsigned int rate)
{
  unsigned int	regval;

  if (!map_rate_to_regval (rate, &regval))
    return false;

  d_interp_rate = rate;
  d_interp_regval = regval;
  setup_status_polling ();
  return _write_fpga_reg (FR_INTERP, d_interp_regval);
}

bool
usrp0_tx::set_tx_freq (int channel, double freq)
{
  if (channel < 0 || channel >= MAX_CHAN)
    return false;

  unsigned int v =
    compute_freq_control_word (get_oscillator_freq (), freq, &d_tx_freq[channel]);

  return _write_fpga_reg (FR_TX_FREQ_0 + channel, v);
}

void
usrp0_tx::wait_for_completion ()
{
  d_ephandle->wait_for_completion ();
}


unsigned int
usrp0_tx::get_interp_rate () const { return d_interp_rate; }

double
usrp0_tx::get_tx_freq (int channel) const 
{
  if (channel < 0 || channel >= MAX_CHAN)
    return 0;

  return d_tx_freq[channel];
}

bool
usrp0_tx::_set_interp_reg (unsigned int regval)
{
  d_interp_regval = regval;
  map_regval_to_rate (regval, &d_interp_rate);
  return _write_fpga_reg (FR_INTERP, regval);
}
