/*
 * Wrappers of open/close/read (used by the hfs/hfsp libraries)
 *
 * libhfs - library for reading and writing Macintosh HFS volumes
 * Copyright (C) 1996-1998, 2001 Robert Leslie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: os.c,v 1.5 2000/10/25 05:43:04 hasi Exp $
 */

#include "mol_config.h"

#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#include <sys/stat.h>

#include "os.h"
#include "llseek.h"

typedef struct {
	ullong	offs;		/* offset to start of file system */
	int	fd;
} os_fd_t;


/*
 * NAME:	os->open()
 * DESCRIPTION:	open and lock a new descriptor from the given path and mode
 */
int
os_open( void **priv, const char *path, ullong offs )
{
	os_fd_t *p;
	int fd;

	if( (fd=open( path, O_RDONLY )) < 0 )
		return -1;
	p = malloc( sizeof(os_fd_t) );
	p->fd = fd;
	p->offs = offs;
	*priv = (void *)p;
	return 0;
}

/*
 * NAME:	os->close()
 * DESCRIPTION:	close an open descriptor
 */
int
os_close(void **priv)
{
	os_fd_t p = *(os_fd_t*)*priv;

	free( *priv );
	*priv = NULL;

	if (close(p.fd) == -1)
		return -1;
	return 0;
}


/*
 * NAME:	os->seek()
 * DESCRIPTION:	set a descriptor's seek pointer (offset in blocks)
 */
unsigned long 
os_seek(void **priv, unsigned long offset, int blksize_bits)
{
	os_fd_t *p = (os_fd_t*)*priv;
	loff_t result, where = offset;
	
	/* offset == -1 special; seek to last block of device */

	if (offset == (unsigned long) -1)
		result = lseek(p->fd, 0, SEEK_END);
	else {
		where <<= blksize_bits;
		where += p->offs;
		if( _llseek( p->fd, (where>>32), (ulong)where, &result, SEEK_SET) < 0 )
			result=-1;
		else
			result -= p->offs;
	}

	result >>= blksize_bits; // -1 is no problem :)
	return (unsigned long) result;
}

/*
 * NAME:	os->read()
 * DESCRIPTION:	read blocks from an open descriptor
 */
unsigned long 
os_read(void **priv, void *buf, unsigned long len, int blksize_bits)
{
	os_fd_t *p = (os_fd_t*)*priv;
	ssize_t result;

	result = read(p->fd, buf, len << blksize_bits);

	if (result == -1)
		return -1;

	return (unsigned long) result >> blksize_bits;
}


/*
 * NAME:        os->same()
 * DESCRIPTION: return 1 iff path is same as the open descriptor
 */
int
os_same(void **priv, const char *path)
{
	os_fd_t *p = (os_fd_t*)*priv;
	struct stat fdev, dev;
	
	if (fstat(p->fd, &fdev) == -1 || stat(path, &dev) == -1)
		return -1;

	return fdev.st_dev == dev.st_dev &&
		fdev.st_ino == dev.st_ino;
}
