/**************************************************************
*   
*   Creation Date: <1998-11-11 11:56:45 samuel>
*   Time-stamp: <2001/06/28 02:14:56 samuel>
*   
*	<mmu.c>
*	
*	Handles page mappings and the mac MMU
*   
*   Copyright (C) 1998, 1999, 2000, 2001 Samuel Rydh
*  
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*   
**************************************************************/

#include "compat.h"
#include <linux/sys.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <asm/mmu_context.h>
#include <asm/mmu.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/unistd.h>

#include "kernel_vars.h"
#include "mmu.h"
#include "mmu_contexts.h"
#include "asmfuncs.h"
#include "emu.h"
#include "misc.h"
#include "rvec.h"

// #define DEBUG_FORCE_601_SPLITMODE
#define PERFORMANCE_INFO
#include "performance.h"

struct fault_param;

static int 	page_fault( kernel_vars_t *kv, struct fault_param *pb );
static int 	lookup_mphys( kernel_vars_t *kv, struct fault_param *pb );
static int	insert_pte( kernel_vars_t *kv, struct fault_param *pb );

static PTE *	lookup_mac_pte( kernel_vars_t *kv, ulong vsid, ulong ea );
static ulong 	get_phys_page( ulong va, int request_rw );
static PTE *	find_pte_slot( ulong ea, PTE *pte, int pte_present, int *pte_replaced );

extern int	dsi_exception( kernel_vars_t *kv, ulong dar, ulong dsisr );
extern int	isi_exception( kernel_vars_t *kv, ulong nip, ulong srr1 );

// privileges violation table, indexed by  {is_write | key | PP}
static char priv_table[16] = {	// 1 = violation
	0,0,0,0,1,0,0,0,   	// read
	0,0,0,1,1,1,0,1    	// write
};

// From the SDR1 register
static PTE	*hw_hash_base;
static ulong	hw_hash_mask;		// _SDR1 = Hash_mask >> 10
static char	*hash_allocation = NULL;

#define MREGS	(kv->mregs)
#define MMU	(kv->mmu)

/**************************************************************
*  init_mmu / cleanup_mmu
*    
**************************************************************/

int 
init_mmu( kernel_vars_t *kv )
{
	int success;
	ulong sdr1 = _get_sdr1();

	if( !sdr1 ) {
		/* Initialze SDR1 if it is unused by the kernel (PPC 603).
		 * We assume the kernel will not touch a zero SDR1 (nor depend upon it).
		 */
		ulong size = 1024*128;		/* 128K is the kmalloc limit */ 
		if( !(hash_allocation = kmalloc( size, GFP_KERNEL )) )
			return 1;

		hw_hash_base = (PTE*)hash_allocation;
		if( (ulong)hash_allocation & (size-1) ) {
			printk("Badly aligned SDR1 allocation - 64K wasted\n");
			size /= 2;
			hw_hash_base = (PTE*)(((ulong)hash_allocation + size) & ~(size-1));
		}
		hw_hash_mask = (size-1) >> 6;
		sdr1 = hw_hash_mask >> 10;
		sdr1 |= virt_to_phys( hw_hash_base );
		printk("SDR1 = %08lX\n", sdr1 );
		_set_sdr1( sdr1 );

		flush_all_PTEs();
	} else {
		hw_hash_base = phys_to_virt( sdr1 & ~0xffff );
		hw_hash_mask = ((sdr1 & 0x1ff) << 10) | 0x3ff;
	}

	success = 
		!setup_mol_contexts() &&
		!init_lvhash( kv ) &&
		!init_tlbie( kv ) &&
		!init_vsid_table( kv ) &&
		!init_mmu_io( kv ) &&
		!init_mmu_fb( kv ) &&
		!init_mmu_tracker( kv );

	if( !success ) {
		cleanup_mmu( kv );
		return 1;
	}

	MMU.emulator_context = CURRENT_MM->context;

#ifdef DEBUG_FORCE_601_SPLITMODE
	printk("MOL: Testing 601 split mode algorithm\n");
	MMU.splitmode_algorithm = kSplitAlgorithm601;
#else
	MMU.splitmode_algorithm = cpu_is_601() ? kSplitAlgorithm601 : kSplitAlgorithmFast;
#endif
	clear_vsids( kv, 1 /* is initialization */ );

	// SDR1 is set from fStartEmulation
	return 0;
}

void 
cleanup_mmu( kernel_vars_t *kv )
{
	cleanup_mmu_tracker( kv );
	cleanup_mmu_fb( kv );
	cleanup_mmu_io( kv );
	cleanup_vsid_table( kv );
	cleanup_tlbie( kv );
	cleanup_lvhash( kv );

	// all sessions share our allocated vsid space...
	if( !g_num_sessions )
		release_mol_contexts();

	// free the PTE hash table (we allocated it on 603 system)
	if( !g_num_sessions && hash_allocation ) {
		_set_sdr1(0);
		kfree( hash_allocation );
		hash_allocation = NULL;
	}

	memset( &MMU, 0, sizeof(mmu_vars_t) );
}

void
clear_tlb_table( kernel_vars_t *kv )
{
	clear_vsids( kv, 0 );
}


/* This function is called whenever the mac MMU-registers
 * have been manipulated externally.
 */
void
mmu_altered( kernel_vars_t *kv )
{
	int i;

	for(i=0; i<16; i++ )
		do_mtsr( kv, i, MREGS.segr[i] );

	do_mtsdr1( kv, MREGS.spr[S_SDR1] );

	for(i=0; i<16; i++ )
		do_mtbat( kv, S_IBAT0U+i, MREGS.spr[ S_IBAT0U+i ], 1 );
}


/************************************************************************/
/*	MMU Exceptions							*/
/************************************************************************/

typedef struct fault_param
{
	int	is_dsi;
	int	is_write;
	int	use_mmu;
	ulong	ea;
	ulong	mmubits;
	ulong	*sr_base;

	// returned by lookup_mphys
	PTE	*mpte;			// lvptr to mac-pte (if != NULL)
	ulong	mphys_page;		// ea of mphys page
	int	pte1;			// RPN | 000 | R | C | WIMG | 00 | PP
	int	key;			// pp key bit
} fault_param_t;

#define PTE1_R		BIT(23)
#define PTE1_C		BIT(24)
#define PTE1_WIMG	0x78
#define PTE1_M		BIT(27)
#define PTE1_PP		0x3

asmlinkage int 
dsi_exception( kernel_vars_t *kv, ulong dar, ulong dsisr )
{
	fault_param_t pb;
	
	// printk("<DSI: EA %08lX, DSISR %08lX>\n", dar, dsisr );

	if( dsisr & 0x84500000 )	/* 0,5,9,11 */
		RVEC_RETURN_2( &MREGS, RVEC_UNUSUAL_DSISR_BITS, dar, dsisr );

	pb.is_dsi = 1;
	pb.is_write = (dsisr & BIT(6));
	pb.use_mmu = (MREGS.msr & MSR_DR);
	pb.ea = dar;
	pb.mmubits = dsisr;
	pb.sr_base = phys_to_virt(MMU.sr_data);
	
	return page_fault( kv, &pb );
}

asmlinkage int 
isi_exception( kernel_vars_t *kv, ulong nip, ulong srr1 )
{
	// printk("ISI: NIP %08lX, SRR1 %08lX\n", nip, srr1 );

	// Zero out unused bits
	srr1 &= BIT(1) | BIT(3) | BIT(4);

	if( srr1 & BIT(3) ){
		printk("Guarded memory access at %08lX\n", nip );
		return RVEC_DEBUGGER;
	}
	if( srr1 & BIT(1) ) {
		fault_param_t pb;

		pb.is_dsi = 0;
		pb.is_write = 0;
		pb.use_mmu = (MREGS.msr & MSR_IR);
		pb.ea = nip;
		pb.mmubits = srr1;
		pb.sr_base = phys_to_virt(MMU.sr_inst);

		return page_fault( kv, &pb );
	}
	// must be privileges violation
	RVEC_RETURN_2( &MREGS, RVEC_ISI_TRAP, nip, BIT(4) );
}

static int
page_fault( kernel_vars_t *kv, fault_param_t *pb )
{
	int ind;

	// Context overflow checking (old linux-MM implementation)
	DETECT_LCTX_OVERFLOW();

	if( lookup_mphys( kv, pb )) {
		// r/w bit + page fault
		int sbits = BIT(1) | (pb->mmubits & BIT(6));
		RVEC_RETURN_2( &MREGS, pb->is_dsi ? RVEC_DSI_TRAP : RVEC_ISI_TRAP, pb->ea, sbits );
	}

#if 0
	printk("MPHYS_PAGE: %08lX, pp %d, key %d, wimg %d, mpte %p\n", 
	       pb->mphys_page, (pb->pte1 & 3), pb->key, ((pb->pte1 >> 3) & 0xf), pb->mpte );
#endif

	// check privileges
	ind = (pb->is_write? 8:0) | (pb->pte1 & PTE1_PP) | (pb->key?4:0);
	if( priv_table[ind] ){
		// r/w bit + priv. violation
		int sbits = BIT(4) | (pb->mmubits & BIT(6));	
		RVEC_RETURN_2( &MREGS, pb->is_dsi ? RVEC_DSI_TRAP : RVEC_ISI_TRAP, pb->ea, sbits );
	}

	// stamp R/C bits (mpte is NULL if this is not a page translation).
	if( pb->mpte ) {
		pb->mpte->r = 1;
		if( pb->is_write )
			pb->mpte->c = 1;
	}
	return insert_pte( kv, pb );
}

#define NO_MMU_PTE1	(PTE1_R | PTE1_C /*| PTE1_M*/ | 0x2 /*pp*/ )

static int 
lookup_mphys( kernel_vars_t *kv, fault_param_t *pb )
{
	ulong		ea = (pb->ea & ~0xfff);
	SEGREG_MOL	segr;
	mac_bat_t	*bp;
       	int		sv_mode, i;
	PTE		*mpte;

	BUMP( access_exception_ctr );
	pb->mpte = NULL;

	if( !pb->use_mmu ) {
		pb->mphys_page = ea;
		pb->pte1 = NO_MMU_PTE1;
		pb->key = 0;
		return 0;
	}

	segr = *(SEGREG_MOL*)&MREGS.segr[ea>>28];
	sv_mode = !(MREGS.msr & MSR_PR);

	// I/O segment?
	if( segr.t ){
		// Memory forced (601/604)? Note that the 601 uses I/O segments
		// even if translation is off (!). We don't implement this though.
		ulong sr = *(ulong*)&segr;
		BUMP( memory_forced_segment );

		if( ((sr >> 20) & 0x1ff) != 0x7f )
			return RVEC_MMU_IO_SEG_ACCESS;
		pb->mphys_page = (ea & 0x0ffff000) | ((sr & 0xf)<<28);
		pb->pte1 = NO_MMU_PTE1;
		pb->key = 0;
		return 0;
	}
	
	// BAT translation? 0-3 = IBATs, 4-7 = DBATs. Separated I/D BATS, hace 3/8/99
	bp = MMU.bats;
	if( pb->is_dsi && MREGS.processor != 1 )
		bp += 4;
	for(i=0; i<4; i++, bp++ ){
		if( !bp->valid )
			continue;
		if( (sv_mode && !bp->vs) && (!sv_mode && !bp->vp) )
			continue;
		if( ea < bp->base || ea > bp->base+bp->size-1 )
			continue;

		pb->mphys_page = ea - bp->base + bp->mbase;
		pb->pte1 = bp->pp | (bp->wimg << 3) | PTE1_R | PTE1_C;
		pb->key = sv_mode ? bp->ks : bp->ku;
		return 0;
	} 

	// Mac page table lookup
	if( (mpte = lookup_mac_pte( kv, segr.vsid, ea )) ) {
		pb->mpte = mpte;
		pb->mphys_page = (mpte->rpn << 12);
		pb->pte1 = ((ulong*)mpte)[1] & (PTE1_PP | PTE1_WIMG | PTE1_R | PTE1_C);
		pb->key = sv_mode ? segr.ks : segr.kp;
		return 0;
	}
	return 1;
}


static int 
insert_pte( kernel_vars_t *kv, fault_param_t *pb )
{
	PTE	pte, *slot;
	ulong	ea = pb->ea;
	ulong	mphys = pb->mphys_page;
	char 	*lvptr=0;
	int 	status, pte_replaced;
	ulong	sr = pb->sr_base[ea>>28];
	ulong	flags;

	// XXX: In certain situations we can insert a writeable page.

	((ulong*)&pte)[0] = 0x80000000 | (sr << 7) | ((ea>>22) & 0x3f);
	((ulong*)&pte)[1] = PTE1_M | PTE1_R | (pb->pte1 & (PTE1_R | PTE1_C | PTE1_WIMG)) | (pb->is_write ? 2:3);

	// PP and WIMG bits must set before the call to mphys_to_pte
	status = mphys_to_pte( kv, mphys, &pte, pb->is_write );

	if( !status || (pb->is_write && (status & TRANSL_RO)) ) {
		ulong addr = (mphys | (ea & 0xfff));
		if( pb->is_dsi ) {
			int rvec = pb->is_write ? RVEC_IO_WRITE : RVEC_IO_READ;
			RVEC_RETURN_2( &MREGS, rvec, addr, NULL );
		} else {
			RVEC_RETURN_1( &MREGS, RVEC_BAD_NIP, addr );
		}
	}

	if( !(status & TRANSL_PHYSICAL) ){
		lvptr = (char*)(pte.rpn<<12);
		if( pb->is_write )
			lvpage_dirty( kv, lvptr );
#if 0
		pte.rpn = get_phys_page( pte.rpn<<12, pb->is_write ) >> 12;
#else
		/* XXX: DEBUG DEBUG */
		/* We have trouble with zero pages (known bug). */
		/* One way to avoid them is requiring R/W */
		pte.rpn = get_phys_page( pte.rpn<<12, !(status & TRANSL_RO) ) >> 12;
#endif
		// printk("ea %08lX, mphys %08lX lvptr %08lX phys %08lX\n", ea, mphys, lvptr, (pte.rpn<<12) );
	}
#if 1
	// tlbhash table hit?
	if( (ulong)(pb->mphys_page - MMU.hash_mbase) < (ulong)MMU.hash_mask ) {
		// printk("hash_table_hit at %08lX\n", pb->ea );
		MMU.pthash_sr = sr;
		MMU.pthash_ea_base = ea & ~MMU.hash_mask;

		// user read (always), superuser r/w
		pte.pp = pb->is_write ? 1 : 3;
		// write accesses of the page table are handled in ptintercept.S
	}
#endif
	slot = find_pte_slot( ea, &pte, !(pb->mmubits & BIT(1)), &pte_replaced );

	// Hash the PTE in appropriate tables. We assume framebuffers don't move around.
	if( pb->use_mmu && !pte_replaced && !(status & TRANSL_FB) )
		hash_ea_to_pte( kv, ea, (ulong*)slot );

	// ...possibly we should make check !pte_replaced here too...
	if( !(status & TRANSL_PHYSICAL) ) {
		hash_lv_to_pte( kv, (ulong*)lvptr, (ulong*)slot );

		// X framebuffer acceleration
		if( status & TRANSL_FB_ACCEL )
			video_pte_inserted( kv, lvptr, (ulong*)slot, *(ulong*)&pte, ea );
	}

	/* RC bits of PTE should be set corresponding to the is_write 
	 * flag. This prevents the processor from stamping the RC bits
	 * (unnecessary overhead). Also, the C bits MUST be
	 * set on the 603 since the DataTLBMiss handler (0x1200) currently
	 * doesn't do this for us...
	 *
	 * XXX: The 601 will often fail booting MacOS if the C bit is set 
	 * below - I don't know why.
	 */
	if( !cpu_is_601() )
		pte.c = pb->is_write ? 1:0;
	
	/* We need to enforce coherence on SMP only */ 
#ifdef CONFIG_SMP
	pte.m=1;
#else
	pte.m=0;
#endif

#if 0
	printk("[%p] ",slot );
	printk("RPN %08X  API %08X  EA %08lX  ", pte.rpn << 12, pte.api<<12, ea );
	printk("%c%c %c%c; PP %d\n", 
	       pte.h ? 'H' : 'h', 
	       pte.v ? 'V' : 'v', 
	       pte.r ? 'R' : 'r', 
	       pte.c ? 'C' : 'c', pte.pp );
#endif

	save_flags(flags);
	__cli();
	*slot = pte;
	restore_flags(flags);

	_tlbie( ea );

	BUMP( page_fault_ctr );
 	return RVEC_NOP;
}


static PTE *
lookup_mac_pte( kernel_vars_t *kv, ulong vsid, ulong ea )
{
	ulong	phash, cmp, pteg, *p;
	ulong	mask;
	int	i;

	// we are only interested in the page index
	ea &= 0x0ffff000;
	mask = MMU.hash_mask>>6;

	// calculate primary hash function
	phash = (ea >> 12) ^ (vsid & 0x7ffff);
	pteg = ((phash & mask) << 6);

	// construct compare word
	cmp = BIT(0) | (vsid <<7) | ((ea&0x0fffffff)>>22);

	// look in primary PTEG
	p=(ulong*)((ulong)MMU.hash_base + pteg);
	for(i=0; i<8; i++, p+=2 )
		if( cmp == *p )
			return (PTE*)p;
				
	// look in secondary PTEG
	p = (ulong*)( (ulong)MMU.hash_base + (pteg ^ (mask << 6)) );
	cmp |= BIT(25);

	for(i=0; i<8; i++,p+=2 )
		if( cmp == *p )
			return (PTE*)p;
	return NULL;
}

/* stolen from include/asm-ppc/pgtable.h */
static inline 
ulong update_pte(ulong *p, ulong set)
{
	unsigned long old, tmp;
	
	__asm__ __volatile__("\
1:	lwarx	%0,0,%3\n\
	or	%1,%0,%4\n\
	stwcx.	%1,0,%3\n\
	bne-	1b"
	: "=&r" (old), "=&r" (tmp), "=m" (*p)
	: "r" (p), "r" (set), "m" (*p)
	: "cc" );
	return old;
}

/*
 * Get physical page corresponding to linux virtual address.
 * Invokes linux page fault handler if the page is missing.
 * This function never fails since we know there is a
 * valid mapping for va.
 */

#ifdef NEW_LINUX_MM
 #define PAGE_BITS_WRITE	(_PAGE_ACCESSED | _PAGE_DIRTY | _PAGE_HASHPTE )
 #define PAGE_BITS_READ		(_PAGE_ACCESSED | _PAGE_HASHPTE )
#else
 #define PAGE_BITS_WRITE	(_PAGE_ACCESSED | _PAGE_DIRTY | _PAGE_HWWRITE )
 #define PAGE_BITS_READ		(_PAGE_ACCESSED )
#endif

static ulong 
get_phys_page( ulong va, int request_rw )
{
	ulong	val, uptr, *ptr;
	ulong	flags;
	struct	mm_struct *mm;
	struct	vm_area_struct *vma;
	flags = request_rw ? 
		(_PAGE_USER | _PAGE_RW | _PAGE_PRESENT)
		: (_PAGE_USER | _PAGE_PRESENT );
	
#ifdef LINUX_24
	uptr = ((ulong*)current->thread.pgdir)[va>>22];	/* top 10 bits */
#else
	uptr = current->tss.pg_tables[va>>22];	/* top 10 bits */
#endif
	ptr = (ulong*)( uptr & 0xfffff000 );
	if( !ptr )
		goto no_page;
	
	ptr = ptr + ((va>>12) & 0x3ff);	      	/* next 10 bits */
	val = *ptr;
	if( flags & ~val )			/* check access priv */
		goto no_page;

	// we set _PAGE_HASHPTE in order to ensure that we get to
	// know about TLB invalidations for this page via our
	// flush_hash_page hook.
	update_pte(ptr, (request_rw? PAGE_BITS_WRITE : PAGE_BITS_READ ));
	// printk("pte at %p was %lx now %lx\n", ptr, val, *ptr);

	return val & ~0xfff;

no_page:
	BUMP( page_missing );

	// no mac page found...
	mm = CURRENT_MM;
	MMAP_SEM_DOWN( &mm->mmap_sem );

	vma = find_vma( mm,va );
	if( !vma )
		goto bad_area;
	if( vma->vm_start <= va )
		goto good_area;
	if( !(vma->vm_flags & VM_GROWSDOWN ) )
		goto bad_area;
	if( expand_stack(vma,va))
		goto bad_area;
good_area:
	// helthy protection checking
	if( request_rw ) {
		if( !(vma->vm_flags & VM_WRITE ) )
			goto bad_area;		
	} else {
		if( !(vma->vm_flags & (VM_READ | VM_EXEC)) )
			goto bad_area;
	}

#ifdef LINUX_24
	mc_handle_mm_fault( current->mm, vma, va, request_rw );
#else
	mc_handle_mm_fault( current, vma, va, request_rw );
#endif
	MMAP_SEM_UP( &mm->mmap_sem );
	return get_phys_page(va, request_rw);

bad_area:
	MMAP_SEM_UP( &mm->mmap_sem );
	printk("get_phys_page: BAD AREA, lvptr = %08lX\n",va );
	force_sig(SIGSEGV, current);
	return 0;
}


/* pte must have valid bit set (but H can be anything)
 * rpn should be linux_virtual / physical depending on flag
 *
 * pte_present is ~bit1 of SRR1/DSISR. It indicates that
 * there *might* be a valid PTE in the hash table.
 * However the translation might be in UTLB/ITLB caches
 * only.
 */
static PTE*
find_pte_slot( ulong ea, PTE *pte, int pte_present, int *pte_replaced )
{
	static int grab_add=0;
	ulong phash, cmp, pteg, *p;
	int i;
	
	// we are only interested in the page index
	ea &= 0x0ffff000;

	// calculate primary hash function
	phash = (ea >> 12) ^ (pte->vsid & 0x7ffff);
	pteg = (ulong)hw_hash_base | ((phash & hw_hash_mask) << 6);

	pte->h=0;

	if( pte_present ) {
		*pte_replaced = 1;

		// construct compare word,
		// cmp = BIT(0) | (pte->vsid <<7) | ((ea&0x0fffffff)>>22);
		cmp = *(ulong*)pte;
		
		// look in primary PTEG
		p=(ulong*)pteg;
		for(i=0; i<8; i++, p+=2 )
			if( cmp == *p ) {
				/* printk("Primary PTE found EA %08X\n",ea ); */
				return (PTE*)p;
			}
     
		// look in secondary PTEG
		p = (ulong*)(pteg ^ (hw_hash_mask << 6));
		cmp |= BIT(25);

		for(i=0; i<8; i++,p+=2 )
			if( cmp == *p ){
				pte->h=1;
				// printk("Secondary PTE found EA %08X\n",ea );
				return (PTE*)p;
			}
		// If the previous mapping was present only in the UTLB/ITLB 
		// on-chip copy, we will actually come here.
	}
	// a new PTE entry is inserted
	*pte_replaced = 0;
	
	// look in primary PTEG for a free slot
	p=(ulong*)pteg;
	for(i=0; i<8; i++, p+=2 )
		if( !(*p & BIT(0)) ) {
			// printk("Primary SLOT found EA %08X\n",ea );
			return (PTE*)p;
		}

	// look in secondary PTEG for a free slot
	p = (ulong*)(pteg ^ (hw_hash_mask << 6));

	for(i=0; i<8; i++,p+=2 )
		if( !(*p & BIT(0)) ) {
			pte->h=1;
			// printk("Secondary SLOT found EA %08X\n",ea );
			return (PTE*)p;
		}
	
	// just grab a slot
	grab_add = (grab_add+1) & 0x7;

	// printk("Grabbing slot %d, EA %08X\n",grab_add, ea );

	return (PTE*)pteg + grab_add;
}


/**************************************************************
*  do_tlbia
*    Translation Lookaside Buffer Invalidate All
**************************************************************/

void 
do_tlbia( kernel_vars_t *kv ) 
{
	table_tlbia( kv );
}


/**************************************************************
*  do_flush
*    Linux has destroyed a memory-page, we must flush any
*    PTEs corresponding to it.
*
*    Context is vsid >> 4.
**************************************************************/

void 
do_flush( ulong context, ulong va, ulong *ptep ) 
{
	int i;
	kernel_vars_t **kvp;
	BUMP( do_flush );

	for( kvp=&g_sesstab->kvars[0], i=0; i<MAX_NUM_SESSIONS; i++, kvp++ ) {
		if( !*kvp || context != (**kvp).mmu.emulator_context )
			continue;
		
		BUMP( block_destroyed_ctr );
		unmap_lv_range( *kvp, va, 0x1000 );
	}
}


/**************************************************************
*  mmu_add_map / mmu_remove_map
*    
**************************************************************/

void 
mmu_add_map( kernel_vars_t *kv, struct mmu_mapping *m )
{
	ulong flags=0;

	if( verify_area(VERIFY_WRITE, m, sizeof(struct mmu_mapping)) ) {
		printk("Bad mmu_add_map access\n");
		return;
	}
	flags = (m->flags & MAPPING_RW)? 0 : TRANSL_RO;
	flags |= (m->flags & MAPPING_PHYSICAL) ? TRANSL_PHYSICAL : 0;
	flags |= (m->flags & MAPPING_SCRATCH) ? TRANSL_SCRATCH : 0;
	flags |= (m->flags & MAPPING_FORCE_CACHE) ? TRANSL_FORCE_CACHE : 0;
	flags |= (m->flags & MAPPING_MACOS_CONTROLS_CACHE) ? TRANSL_MACOS_CONTROLS_CACHE : 0;
	flags |= (m->flags & MAPPING_FB_ACCEL) ? TRANSL_FB_ACCEL : 0;
	flags |= (m->flags & MAPPING_FB) ? TRANSL_FB : 0;
	flags |= (m->flags & MAPPING_DBAT) ? TRANSL_DBAT : 0;

	m->id = add_block_trans( kv, m->mbase, m->lvbase, m->size, flags );	
}

void 
mmu_remove_map( kernel_vars_t *kv, struct mmu_mapping *m )
{
	if( verify_area(VERIFY_READ, m, sizeof(struct mmu_mapping)) ) {
		printk("Bad mmu_remove_map access\n");
		return;
	}
	remove_block_trans( kv, m->id );
}


/**************************************************************
*  dbg_get_linux_page [DEBUGGER FUNCTION]
*    returns the linux-pte corresponding to va
**************************************************************/

int 
dbg_get_linux_page( ulong va, linux_page_t *r )
{
	ulong	val, uptr, *ptr;
	
	if( verify_area(VERIFY_WRITE, r, sizeof(*r)) )
		return -EFAULT;

#ifdef LINUX_24
	uptr = ((ulong*)current->thread.pgdir)[va>>22];	/* top 10 bits */
#else
	uptr = current->tss.pg_tables[va>>22];	/* top 10 bits */
#endif
	ptr = (ulong*)( uptr & 0xfffff000 );
	if( !ptr )
		return 1;
	val = ptr[ (va>>12)&0x3ff ];		/* next 10 bits */

	r->phys = val & ~0xfff;
	r->mflags = 
		  DBG_TRANSL_PAGE_FLAG( val, _PAGE_PRESENT )
		| DBG_TRANSL_PAGE_FLAG( val, _PAGE_USER )
		| DBG_TRANSL_PAGE_FLAG(	val, _PAGE_GUARDED )
		| DBG_TRANSL_PAGE_FLAG( val, _PAGE_COHERENT )
		| DBG_TRANSL_PAGE_FLAG( val, _PAGE_NO_CACHE )
		| DBG_TRANSL_PAGE_FLAG( val, _PAGE_WRITETHRU )
		| DBG_TRANSL_PAGE_FLAG( val, _PAGE_DIRTY )
		| DBG_TRANSL_PAGE_FLAG( val, _PAGE_ACCESSED )
		| DBG_TRANSL_PAGE_FLAG( val, _PAGE_RW )
		| DBG_TRANSL_PAGE_FLAG_N( val, _PAGE_HASHPTE )
		| DBG_TRANSL_PAGE_FLAG_N( val, _PAGE_EXEC );
	return 0;
}


/**************************************************************
*  dbg_translate_ea  [DEBUGGER_FUNCTION]
*    Used by the debugger to translate ea -> mphys
*    
**************************************************************/

int 
dbg_translate_ea( kernel_vars_t *kv, int context, ulong va, PTE *pte, int data_access )
{
	fault_param_t pb;

	if( verify_area(VERIFY_WRITE, pte, 8) )
		return 1;

	memset( &pb, 0, sizeof(pb));
	pb.ea = va;
	pb.is_dsi = data_access;

	switch(context){
	case kContextUnmapped:
		pb.sr_base = MMU.unmapped_sr;
		break;
	case kContextMapped_S:
		pb.sr_base = MMU.sv_sr;
		pb.use_mmu = 1;
		break;
	case kContextMapped_U:
		pb.sr_base = MMU.user_sr;
		pb.use_mmu = 1;
		break;
	default:
		return 1;
	}

	if( lookup_mphys( kv, &pb ) )
		return 1;

	// Only the RPN, R, C, WIMG and PP fields make sense
	((ulong*)pte)[0] = 0x80000000 | ((va>>22) & 0x3f);
	((ulong*)pte)[1] = pb.pte1;
	pte->rpn = (pb.mphys_page >> 12);
	return 0;
}



/**************************************************************
*  get_PTE   [DEBUGGER_FUNCTION]
*
**************************************************************/

int 
get_PTE( kernel_vars_t *kv, int seg_ident, ulong va, PTE *retptr ) 
{
	ulong	base, mask;
	ulong	vsid, ptmp, stmp, *pteg, *steg;
	ulong	cmp;
        ulong   *uret = (ulong*)retptr;
	int	i, num_match=0;

	switch( seg_ident ) {
	case kContextUnmapped:
		vsid = MMU.unmapped_sr[va>>28];
		break;
	case kContextMapped_S:
		vsid = MMU.sv_sr[va>>28];
		break;
	case kContextMapped_U:
		vsid = MMU.user_sr[va>>28];
		break;
	case kContextEmulator:
		vsid = (MUNGE_CONTEXT(MMU.emulator_context) + ((va>>28) * MUNGE_ESID_ADD)) & 0xffffff;
		break;
        case kContextKernel:
                vsid = 0;
                break;
	default:
		printk("get_PTE: no such context: %d\n",seg_ident );
		return 0;
	}
	
	// mask vsid and va
	vsid &= 0xffffff;
	va &= 0x0ffff000;

	// get hash base and hash mask
	base = (ulong)hw_hash_base;
	mask = hw_hash_mask | 0x3ff;

	// hash function
	ptmp = (vsid ^ (va>>12)) & mask;
	stmp = mask & ~ptmp;
	pteg = (ulong*)((ptmp << 6) + base);
	steg = (ulong*)((stmp << 6) + base);

	// construct compare word
	cmp = 0x80000000 | (vsid <<7) | (va>>22);

	// look in primary PTEG
	for(i=0; i<8; i++ ) {
		if( cmp == pteg[i*2] ) {
			if( !num_match++ && uret ) {
				uret[0] = pteg[i*2];
				uret[1] = pteg[i*2+1];
			}
			if( num_match == 2 ) {
				printk("Internal ERROR: duplicate PTEs!\n");
				printk("p-hash: low_pte: %08lX  high_pte: %08lX\n",
				      uret ? uret[0]:0, retptr? uret[1]:0 );
			}
			if( num_match >= 2 ) {
				printk("p-hash: low_pte: %08lX  high_pte: %08lX\n",
				       pteg[i*2], pteg[i*2+1] );
			}
		}
	}

	// look in secondary PTEG
	cmp |= 0x40;
	for(i=0; i<8; i++ ) {
		if( cmp == steg[i*2] ) {
			if( !num_match++ && uret ) {
				uret[0] = steg[i*2];
				uret[1] = steg[i*2+1];
			}
			if( num_match == 2 ) {
				printk("Internal ERROR: duplicate PTEs!\n");
				printk("?-hash: low_pte: %08lX  high_pte: %08lX\n",
				      uret? uret[0]:0, uret? uret[1]:0 );
			}
			if( num_match >= 2 ) {
				printk("s-hash: low_pte: %08lX  high_pte: %08lX\n",
				       steg[i*2], steg[i*2+1] );
			}
		}
	}
	return num_match;
}
