/**************************************************************
*   
*   Creation Date: <97/05/26 02:10:43 samuel>
*   Time-stamp: <2001/09/22 14:32:17 samuel>
*   
*	<misc.c>
*	
*	Kernel interface
*   
*   Copyright (C) 1997, 1999, 2000, 2001 Samuel Rydh
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "compat.h"
#include <linux/sys.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
/* #include <sys/mman.h> */
#include <asm/prom.h>
#include <asm/mmu_context.h>
#include <asm/mmu.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/unistd.h>
#include <asm/io.h>

#include "multiplexer.h"
#include "mac_registers.h"
#include "kernel_vars.h"
#include "misc.h"
#include "mmu.h"
#include "asmfuncs.h"
#include "emu.h"
#include "prom.h"
#include "version.h"

//#define PERFORMANCE_INFO
#include "performance.h"

static int 	prom_copy_node( char *dest, struct device_node *src, int bufsize );

static void	clear_perf_info( void );
static void	print_perf_info( void );
static int	get_performance_info( int index, char *retbuf, int size, int *retval );

/* Globals */
int 		g_num_sessions=0;


/**************************************************************
*  initialize/destroy session
*
* possible errors:	
*	errSessionOutOfBounds
*	errSessionInUse
*	errGeneralError
**************************************************************/

static void
free_kvar_pages( kernel_vars_t *kv )
{
	char *ptr = (char*)kv;
	int i;

	for(i=0; i<NUM_KVARS_PAGES; i++, ptr+=0x1000 )
		compat_clear_page_reserved( ptr );
	
	free_pages( (ulong)kv, NUM_KVARS_PAGES - 1 );
}


static int 
initialize_session( int index, ulong session_magic )
{
	kernel_vars_t *kv;
	char *ptr;
	int i;

	if( index < 0 || index > MAX_NUM_SESSIONS )
		return errSessionOutOfBounds;
	if( g_sesstab->kvars[index] )
		return errSessionInUse;

	if( !g_num_sessions && write_hooks() )
		return errGeneralError;

	if( !(kv = (kernel_vars_t*) __get_free_pages( GFP_KERNEL, NUM_KVARS_PAGES-1 )) ) {
		printk("MOL: memory allocation failure\n");
		return errGeneralError;
	}
	memset( kv, 0, NUM_KVARS_PAGES * 0x1000 );

	kv->kernel_vars = kv;

	/* To be able to export the kernel variables to user space, we
	 * must set the PG_reserved bit. This is due to a check
	 * in remap_pte_range() in kernel/memory.c (is this bug or a feature?).
	 */
	for(ptr=(char*)kv, i=0; i<NUM_KVARS_PAGES; i++, ptr+=0x1000 )
		compat_set_page_reserved( ptr );

	clear_perf_info();

	if( init_mmu( kv ) ) {
		printk("init_mmu failed\n");
		free_kvar_pages( kv );
		return errGeneralError;
	}

	inc_mod_count();
	g_num_sessions++;

	g_sesstab->kvars_ph[index] = virt_to_phys(kv);
	g_sesstab->kvars[index] = kv;
	g_sesstab->entry_magic[index] = session_magic;
	return 0;
}

void
destroy_session( int index )
{
	kernel_vars_t *kv;

	if( index < 0 || index > MAX_NUM_SESSIONS || !(kv=g_sesstab->kvars[index]) )
		return;

	print_perf_info();

	g_sesstab->kvars[index] = NULL;
	g_sesstab->kvars_ph[index] = 0;
	g_sesstab->entry_magic[index] = 0;

	/* Decrease before freeing anything (simplifies deallocation of shared resources) */
	g_num_sessions--;

	cleanup_mmu( kv );
	memset( kv, 0, NUM_KVARS_PAGES * 0x1000 );

	free_kvar_pages( kv );

	if( !g_num_sessions )
		remove_hooks();

	dec_mod_count();
}


/**************************************************************
*  multiplexer
*    system call multiplexer
**************************************************************/

ulong 
sys_macos_multiplexer( int what, int sess_id, ulong arg1, ulong arg2, ulong arg3, ulong arg4 )
{
	kernel_vars_t *kv;
	ulong ret=0, r;

	if (!suser())
		return -EPERM;	

	// This selector is used by the kernel-assembly code: r4-r6=parameters, r7=function, r8=kvars
	if( what == fCallKernelFunc ){
		typedef int (*kernelfunc_t)( kernel_vars_t *, ulong, ulong, ulong );
		return (*(kernelfunc_t)arg3)( (kernel_vars_t*)arg4, (ulong)sess_id, arg1, arg2 );
	}

	if( (sess_id < 0 || sess_id >= MAX_NUM_SESSIONS) && what != fInit ) {
		printk("Bad session number %d\n", sess_id );
		return 1;
	}
	if( !(kv=g_sesstab->kvars[sess_id] ) ) {
		if( what != fInit && what != fGetMolModVersion && what != fGetPVR ) {
			printk("Session %d not initialized (selector %d)\n", sess_id, what );
			return 1;
		}
	}

	switch( what ) {
	case fInit: /* arg1 = (new) session_magic */
		ret = initialize_session( sess_id, arg1 );
		break;
	case fCleanup:
		destroy_session( sess_id );
		break;

	case fBreakpointFlags:
		kv->break_flags = arg1;
		msr_altered(kv);	/* updates _msr (sets MSR_SE if needed) */
		break;

	case fGetMregs:
		/* ret = kernel virtual address of mregs */
		ret = (ulong)&kv->mregs;
		break;
	case fGetMregsPhys:
		ret = virt_to_phys( &kv->mregs );
		break;

	case fGetMolModVersion:
		ret = MOL_VERSION;
		break;

	case fSprChanged:
		msr_altered(kv);
		mmu_altered(kv);
		break;

	case fMsrChanged:
		msr_altered(kv);
		break;

	case fSetMacRam:
		/* void ( char *lvbase, size_t size, ulong mbase ) */
		kv->mmu.linux_ram_base = (char*)arg1;
		kv->mmu.ram_size = arg2;
		kv->mmu.mac_ram_base = arg3;
		break;
	
	case fAddIORange:
       		/* void ( ulong mbase, int size, void *usr_data )*/
		add_io_trans( kv, arg1, arg2, (void*)arg3 );
		break;
	case fRemoveIORange:
		/* void ( ulong mbase, int size ) */
		remove_io_trans( kv, arg1, arg2 );
		break;

	case fPromGetNodeSize: /* arg1 = node_ptr, 0=root */
		ret = prom_copy_node( NULL, (struct device_node*)arg1, 0 );
		break;

	case fPromCopyNode: /* arg1 = dest, arg2 = kernel_node, arg3 = bufsize */
		ret = prom_copy_node( (char*)arg1, (struct device_node*)arg2, arg3 );
		break;

	/* this selector is going to die peacefully */
	case fKernelMemmove: /* arg1 = dest, arg2 = src, arg3 = size */
		memmove( (char*)arg1, (char*)arg2, arg3 );		
		break;

	case fEmulateTLBIE:  /* arg1 = pageindex (bits 4-19) */
		do_tlbie( kv, arg1 );
		break;

	case fEmulateTLBIA:
		do_tlbia( kv );
		break;
	
	case fAddMMUMapping: /* arg1 = struct mmu_mapping *m */
		mmu_add_map( kv, (struct mmu_mapping*) arg1 );
		break;
	case fRemoveMMUMapping: /* arg1 = struct mmu_mapping *m */
		mmu_remove_map( kv, (struct mmu_mapping*) arg1 );
		break;

	case fGetKernelVersion:
		ret = linux_vers;
		break;

	/* THE FOLLOWING SELECTORS SHOULD NOT BE USED IN THE EMULATION */
	case fGetKvarBase: /* for debugging only */
		ret = (ulong)kv;
		break;		

	case fGetPTE: /* arg1 = segment_ident, arg2 = address, arg3 = PTE*  */
		ret = get_PTE( kv, arg1, arg2, (PTE*)arg3 );
		break;
		
	case fGetPhysicalAddr: /* arg1 = lvptr, arg2 = ret_pageinfo */
		ret = dbg_get_linux_page( arg1, (struct linux_page*)arg2 );
		break;

	case fTranslateIEa: /* context, lvptr, PTE* */
		ret = dbg_translate_ea( kv, arg1, arg2, (PTE*)arg3, 0 );
		break;

	case fTranslateDEa: /* context, lvptr, PTE* */
		ret = dbg_translate_ea( kv, arg1, arg2, (PTE*)arg3, 1 );
		break;
	
	case fSetupFBAccel:
		setup_fb_acceleration( kv, (char*)arg1, arg2, arg3 );
		break;
	case fGetDirtyFBLines:
		ret = get_dirty_fb_lines( kv, (short*)arg1, arg2 );
		break;
	case fGetPVR: /* retpvr */
		if( verify_area( VERIFY_WRITE, (char*)arg1, sizeof(ulong) ))
			return -EFAULT;
		asm volatile("mfpvr %0" : "=r" (r) : );
		*(ulong*)arg1 = r;
		break;

	case fTrackDirtyRAM:
		ret = track_lvrange( kv, (char*)arg1, (size_t)arg2 );
		break;
	case fGetDirtyRAM:
		ret = get_track_buffer( kv, (char*)arg1 );
		break;
	case fSetDirtyRAM:
		set_track_buffer( kv, (char*)arg1 );
		break;
	case fGetPerformanceInfo:
		ret = get_performance_info( arg1, (char*)arg2, arg3, (int*)arg4 );
		break;
	case fGetTBFrequency:
		ret = get_tb_frequency();	/* returns 0 on 2.2 */
		break;
	default:
		printk("MAC: multiplexer: index not implemented. Kernel up to date?\n");
		ret = ENOSYS;
	}
	return ret;
}


/**************************************************************
*  Copy the OpenFirmware device tree to user space
*    
**************************************************************/

static struct device_node *
get_prom_root( void ) 
{
	struct device_node *root;
	
	root = find_path_device("/");
	return root;
}

static void *
prom_copy_( char *buf, void *src, int size, int *total_size ) 
{
	char *ret = NULL;

	/* Pad to word boundary */
	if( (*total_size & 3) )
		*total_size += 4 - (*total_size & 3);

	if( buf ) {
		if( !src )
			memset( buf + *total_size, 0, size );
		else
			memcpy( buf + *total_size, src, size );
		ret = buf + *total_size;
	}
	*total_size += size;
	return ret;
}

static int 
do_prom_copy_node( char *ptr, struct device_node *src )
{
	mol_device_node_t *dest, dummy_dest;
	p_property_t *prev;
	struct property *pr;
	struct device_node *dn;
	int tsize=0;
	
	dn = get_prom_root();
	if(!src) {
		src = dn;
	} else {
		for( ; dn && dn!=src ; dn=dn->allnext )
			;
		if( !dn )
			return -EFAULT;
	}
	if( !(dest=prom_copy_( ptr, NULL, sizeof(mol_device_node_t), &tsize)))
		dest = &dummy_dest;
	dest->node = src->node;
	dest->n_addrs = src->n_addrs;
	dest->n_intrs = src->n_intrs;

	/* copy kernel node references */
	dest->parent = (mol_device_node_t*) src->parent;
	dest->child = (mol_device_node_t*) src->child;
	dest->sibling = (mol_device_node_t*) src->sibling;
	dest->next = (mol_device_node_t*) src->next;
	dest->allnext = (mol_device_node_t*) src->allnext;

	/* arrays */
	dest->addrs = prom_copy_( ptr, src->addrs, sizeof(struct address_range)*src->n_addrs, &tsize );
	dest->intrs = prom_copy_( ptr, src->intrs, sizeof(struct interrupt_info)*src->n_intrs, &tsize );

	/* strings */
	if( src->name )
		dest->name = prom_copy_( ptr, src->name, strlen(src->name)+1, &tsize );
	if( src->type )
		dest->type = prom_copy_( ptr, src->type, strlen(src->type)+1, &tsize );
	if( src->full_name )
		dest->full_name = prom_copy_( ptr, src->full_name, strlen(src->full_name)+1, &tsize );
	
	/* properties */
	prev = 0;
	for( pr=src->properties; pr; pr=pr->next ) {
		p_property_t dummy, *t;
		if( !(t = prom_copy_( ptr, NULL, sizeof(p_property_t), &tsize )) )
			t = &dummy;
		t->length = pr->length;

		if( pr->name )
			t->name = prom_copy_( ptr, pr->name, strlen(pr->name)+1, &tsize );
		t->value = prom_copy_( ptr, pr->value, pr->length, &tsize );
		t->next = prev;
		prev = t;
	}
	if( prev )
		dest->properties = prev;

	if( tsize & 4 )
		tsize += 4 - (tsize & 3);

	return tsize;
}

static int
prom_copy_node( char *dest, struct device_node *dn, int bufsize )
{
	int ret;
	
	if( !dest )
		return do_prom_copy_node( NULL, dn );

	if( verify_area(VERIFY_WRITE, dest, bufsize) )
		return -EFAULT;

	ret = do_prom_copy_node(NULL, dn);
	if( ret < 0 || ret > bufsize )
		return -EFAULT;

	do_prom_copy_node( dest, dn );
	return 0;
}


/************************************************************************/
/*	Performance Info						*/
/************************************************************************/

static void
clear_perf_info( void )
{
	perf_info_t *p = g_perf_info_table;
	for( ; p->name ; p++ )
		*p->ctrptr = 0;
}

static void
print_perf_info( void )
{
#ifdef PERFORMANCE_INFO
	perf_info_t *p = g_perf_info_table;
	for( ; p->name ; p++ )
		printk("%-30s: %ld\n", p->name, *p->ctrptr );
#endif
}

static int
get_performance_info( int ind, char *buf, int size, int *retval )
{
	int i, len;
	perf_info_t *p = g_perf_info_table;
	for( i=0; p->name && i!=ind; i++, p++ )
		;
	if( !p->name )
		return 1;
	len = strlen(p->name)+1;
	if( len > size )
		len = size;
	if( verify_area( VERIFY_WRITE, buf, len ) )
		return -EFAULT;
	if( verify_area( VERIFY_WRITE, retval, sizeof(int)) )
		return -EFAULT;
	strncpy( buf, p->name, len );
	*retval = *p->ctrptr;
	return 0;
}

