/* 
 *   Creation Date: <2000/07/25 00:22:18 samuel>
 *   Time-stamp: <2001/10/08 22:25:53 samuel>
 *   
 *	<async.c>
 *	
 *	Asynchronous I/O support
 *   
 *   Copyright (C) 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"
#include <signal.h>
#include <linux/version.h>
#include <sys/poll.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <pthread.h>
#include "async.h"
#include "debugger.h"
#include "thread.h"
#include "timer.h"
#include "molcpu.h"

static void signal_sigio( int sig_num );
static void poll_thread_entry( void *dummy );
static void aevent_rx( int fd, int events );

#define MAX_NUM_HANDLERS	16
#define MAX_NUM_AEVENTS		8

typedef struct 
{
	int		handler_id;
	struct pollfd	pollfd;
	async_handler_t	proc;
	int		sigio_capable;
} handler_t;

typedef struct
{
	int		 token;
	aevent_handler_t proc;
} aevent_t;

static handler_t 	handlers[ MAX_NUM_HANDLERS ];

#define LOCK 		pthread_mutex_lock(&poll_mutex);
#define UNLOCK 		pthread_mutex_unlock(&poll_mutex);

#define LOCK_HANDLERS	pthread_mutex_lock(&handlers_mutex);
#define UNLOCK_HANDLERS	pthread_mutex_unlock(&handlers_mutex);

static pthread_mutex_t 	poll_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t 	handlers_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t 	signal_ack_cond = PTHREAD_COND_INITIALIZER;
static int volatile 	signal_pending;	/* for polled I/O */
int volatile 		__io_is_pending=0;

static int 		num_handlers;
static int 		next_handler_id;
static int 		main_th_pid;
static pthread_t 	poll_th = 0;
static volatile int	cancel_poll_thread=0;

static int		num_aevents;
static aevent_t		aevents[ MAX_NUM_AEVENTS ];
static int		next_aevent_token;
static int	 	pipefds[2];
static int		initialized = 0;

void
async_init( void )
{
	sigset_t set;

	num_handlers = 0;
	next_handler_id = 1;

	main_th_pid = getpid();

	signal( SIGIO, (void*)signal_sigio );
	sigemptyset( &set );
	sigaddset( &set, SIGIO );
	pthread_sigmask( SIG_UNBLOCK, &set, NULL );

	if( pipe(pipefds) < 0) {
		perrorm("pipe");
		exit(1);
	}

	num_aevents = 0;
	next_aevent_token = 1;	/* 0 is used by us */
	add_async_handler( pipefds[0], POLLIN, aevent_rx, 0 );

	poll_th = create_thread( poll_thread_entry, NULL, "async-io thread");

	initialized = 1;
}

void
async_cleanup( void )
{
	int i;
	if( !initialized )
		return;
	
	/* Make sure thread is not spinning in the poll loop */
	cancel_poll_thread = 1;
	send_aevent(0);

	num_handlers = 0;
	num_aevents = 0;
	
	/* Wait for thread to exit */
	for(i=0; i<20 && (volatile int)poll_th; i++ )
		usleep(1000);
	if( poll_th ) {
		kill_thread( poll_th );
		poll_th = 0;
	}
}


extern int
add_async_handler( int fd, int events, async_handler_t proc, int sigio_capable )
{
	handler_t *h;
	int flags;
#if 1
	// The SIGIO signals causes problems on 2.2 kernels -
	// this seems to be a kernel bug which occurs when multiple
	// signals are delivered almost simultaneously.
	if( _get_kernel_version() <= KERNEL_VERSION(2,3,0) )
		sigio_capable = 0;
#endif
	if( num_handlers >= MAX_NUM_HANDLERS ) {
		printm("MAX_NUM_HANDLERS exceeded\n");
		return -1;
	}
	if( !proc )
		return -1;

	/* Make sure a poll() with the old handler set returns immediately */
	send_aevent(0);

	/* setup O_ASYNC, O_NONBLOCK and signal info */
	if( (flags = fcntl( fd, F_GETFL)) == -1 )
		return -1;
	flags |= O_NONBLOCK | (sigio_capable? O_ASYNC : 0);
	if( fcntl( fd, F_SETFL, flags ) == -1 )
		return -1;
	if( sigio_capable ) {
		if( fcntl( fd, F_SETSIG, SIGIO ) < 0 )
			return -1;
		if( fcntl( fd, F_SETOWN, main_th_pid ) < 0 )
			return -1;
	}

	/* add to table */
	LOCK_HANDLERS;
	h = &handlers[num_handlers];
	h->pollfd.fd = fd;
	h->pollfd.events = events;
	h->pollfd.revents = 0;

	h->handler_id = next_handler_id++;
	h->proc = proc;
	h->sigio_capable = sigio_capable;

	num_handlers++;
	UNLOCK_HANDLERS;

	return h->handler_id;
}


extern void
delete_async_handler( int handler_id )
{
	int i;
	handler_t *h;

	send_aevent(0);

#if 1
	/* await safe state */
	for(i=0; !signal_pending && i<20; i++ )
		usleep(1);
	if( !signal_pending )
		printm("async: safe_state never reached!\n");
#endif

	LOCK_HANDLERS;
	for(h=handlers, i=0; i<num_handlers; h++, i++ ){
		if( h->handler_id == handler_id ){
			memmove( h, h+1, (num_handlers-i-1)*sizeof(handler_t) );
			break;
		}
	}
	num_handlers--;
	UNLOCK_HANDLERS;
}

extern void
set_async_handler_events( int handler_id, int new_events )
{
	int i;
	handler_t *h;

	LOCK_HANDLERS;
	for(h=handlers, i=0; i<num_handlers; h++, i++ ){ 
		if( h->handler_id == handler_id ){
			h->pollfd.events = new_events;
			break;
		}
	}
	UNLOCK_HANDLERS;

	if( i >= num_handlers ) {
		printm("set_async_handler_events: Handler not found\n");
	}
}


static void
signal_sigio( int signum )
{
	__io_is_pending = 1;
}


void
__do_async_io( void )
{
	struct pollfd ufds[ MAX_NUM_HANDLERS ];
	handler_t *h;
	int i, n, do_ack;

	__io_is_pending = 0;	
	for(h=handlers, i=0; i<num_handlers; i++, h++ )
		ufds[i] = h->pollfd;

	if( (n = TEMP_FAILURE_RETRY(poll( ufds, num_handlers, 0 ))) < 0 ) {
		perrorm("poll");
		goto out;
	}
	for( h=handlers, i=0; i<num_handlers && n>0; i++, h++ )
		if( ufds[i].revents )
			(*h->proc)( h->pollfd.fd, ufds[i].revents );

out:
	/* restart polling thread */
	LOCK;
	do_ack = signal_pending;
	signal_pending = 0;
	UNLOCK;
	if( do_ack )
		pthread_cond_signal(&signal_ack_cond);
}


/*
 * Most devices are not capable of generating SIGIO - 
 * we poll these devices instead.
 */
static void
poll_thread_entry( void *dummy )
{
	struct pollfd ufds[ MAX_NUM_HANDLERS ];
	handler_t *h = handlers;
	int i, num;

	/* Increase priority to reduce latency */
	setpriority( PRIO_PROCESS, getpid(), -17 );

	while( !cancel_poll_thread ){
		num = 0;

		LOCK_HANDLERS;
		for(h=handlers, i=0; i<num_handlers; i++, h++ )
			if( /* !h->sigio_capable && */h->pollfd.events )
				ufds[num++] = h->pollfd;
		UNLOCK_HANDLERS;

		if( poll( ufds, num, -1 ) < 0 ) {
			if( errno != EINTR )
				perrorm("poll");
			continue;
		}

		signal_pending = 1;
#if 1
		__io_is_pending = 1;
		interrupt_emulation();
#else
		pthread_kill( get_main_th(), SIGIO );
		abort_doze();
#endif
		/* Wait for signal delivery */
		LOCK;
		while( signal_pending && !cancel_poll_thread )
			pthread_cond_wait( &signal_ack_cond, &poll_mutex );
		UNLOCK;
	}
	poll_th = 0;
}


/************************************************************************/
/*	Asynchronous event mechanism					*/
/************************************************************************/

static void 
aevent_rx( int fd, int dummy_events )
{
	char ev;
	int i;

	if( read( pipefds[0], &ev, 1 ) != 1 )
		return;
	if( !ev )
		return;
	
	for( i=0; i<num_aevents; i++ ){
		if( ev == aevents[i].token ) {
			(*aevents[i].proc)( (int)ev );
			return;
		}
	}
	printm("Aevent %d unhandled!\n", ev );
}


void 
send_aevent( int token )
{
	char ev = token;

	if( TEMP_FAILURE_RETRY( write( pipefds[1], &ev, 1 ) ) != 1 )
		perrorm("send_event");
}

int
add_aevent_handler( aevent_handler_t proc )
{
	aevent_t *aev = &aevents[num_aevents];

	if( num_aevents >= MAX_NUM_AEVENTS ){
		printm("MAX_NUM_AEVENTS exceeded\n");
		return -1;
	}
	aev->token = next_aevent_token++;
	aev->proc = proc;

	if( aev->token > 255 ){
		printm("run out of aevent tokens!\n");
		return -1;
	}
	num_aevents++;
	return aev->token;
}

void
delete_aevent_handler( int token )
{
	int i;
	aevent_t *aev;

	for(aev=aevents, i=0; i<num_aevents; aev++, i++ ){
		if( aev->token == token ){
			memmove( aev, aev+1, (num_aevents-i-1)*sizeof(aevent_t) );
			break;
		}
	}
}
