/**************************************************************
*   
*   Creation Date: <97/07/02 19:51:35 samuel>
*   Time-stamp: <2001/03/02 20:12:44 samuel>
*   
*	<pic.c>
*	
*	Interrupt controller
*	(The one found in the Grand Central (GC) chip)
*   
*   Copyright (C) 1997, 1999, 2000 Samuel Rydh (samuel@ibrium.se)
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "mol_config.h"
#include <pthread.h>

#include "pic.h"
#include "molcpu.h"
#include "ioports.h"
#include "promif.h"
#include "debugger.h"
#include "driver_mgr.h"
#include "gc.h"
#include "session.h"

#define FALLBACK_BASE 	0xf3000000
#define PIC_GC_OFFSET	0x10

/* This pmac-interrupt controller generates an interrupt on
 * the positive *AND* negative edge of the signal from the device.
 */

/* XXX: How should writes to LEVEL register be handled? 
 * Is LEVEL latched if ENABLE is cleared? Then a bit written to LEVEL
 * should probably release the latch.
 */

/* G3-machines have 64 interrupts, other 32 */

#define NUM_REGS	8

/* all registers are little-endian */
enum{
	r_flag_2=0,		/* IRQ 32-63 (for G3) */
	r_enable_2,
	r_ack_2,
	r_level_2,
	r_flag,		       	/* IRQ 0-31 */
	r_enable,
	r_ack,
	r_level
};

static char *reg_names[] = {
	"FLAG_2", "ENABLE_2", "ACK_2", "LEVEL_2",
	"FLAG", "ENABLE", "ACK", "LEVEL"
};

struct pic_info {
	pthread_mutex_t lock_mutex;
	gc_range_t	*pic_range;

	ulong		reg[NUM_REGS+1];  	/* +1 to avoid alignment problems */
};

static struct pic_info pic;

static void 	pic_io_print( int isread, ulong addr, ulong data, int len, void *usr );
static ulong 	pic_io_read( ulong addr, int len, void *usr );
static void 	pic_io_write( ulong addr, ulong data, int len, void *usr );
static void 	intline_upd( void );
static void 	pic_cleanup( void *usr );
static void    	_irq_interrupt( int irq );
static int	save_pic_regs( void );
static int 	cmd_pic_regs( int, char**);
static int	cmd_nmi( int argc, char **argv );

#define LOCK	pthread_mutex_lock( &pic.lock_mutex );
#define UNLOCK	pthread_mutex_unlock( &pic.lock_mutex );

static io_ops_t ops = {
	pic_cleanup,
	pic_io_read,
	pic_io_write,
	pic_io_print
};

driver_interface_t pic_driver = {
    "pic", pic_init, NULL
};


/************************************************************************/
/*	FUNCTIONS							*/
/************************************************************************/

int
pic_init( void )
{
	memset( (char*)&pic, 0, sizeof(struct pic_info) );
	pthread_mutex_init( &pic.lock_mutex, NULL );

	pic.pic_range = add_gc_range( PIC_GC_OFFSET, NUM_REGS * 4, "PIC", 0, &ops, NULL );

	add_cmd( "pic_regs", "pic_regs \ndisplay PIC regs\n", -1, cmd_pic_regs );
	add_cmd( "nmi", "dtc\nRaise NMI (Non Maskerable Interrupt)\n", -1, cmd_nmi );

	session_save_proc( save_pic_regs, NULL, kDynamicChunk );
	if( loading_session() ) {
		if( read_session_data( "PIC", 0, (char*)pic.reg, sizeof(pic.reg) ) )
			session_failure("Could not read PIC registers\n");
		intline_upd();
	}
	return 1;
}

void 
pic_cleanup( void *usr )
{
	pthread_mutex_destroy( &pic.lock_mutex );
}

static int
save_pic_regs( void )
{
	return write_session_data( "PIC", 0, (char*)pic.reg, sizeof(pic.reg) );
}


static void
pic_io_write( ulong addr, ulong data, int len, void *usr )
{
	ulong mbase = get_mbase(pic.pic_range);
	int reg = (addr - mbase)>>2;
	ulong	old;
	
/*	printm("PIC-write: %08X\n",data ); */
	LOCK;

	/* allow unaligned access */
	old = pic.reg[reg];
	write_mem( (char*)pic.reg + addr - mbase, data, len );
  
	switch( reg ){
	case r_enable:
	case r_enable_2:
		/* We should NOT clear flag bits */
		intline_upd();
		break;
	case r_ack:
		/* Experimental... It seems like ACK of the highest
		 * bit clears the flag register.
		 */
		if( pic.reg[r_ack] & 0x80 ) {
			/* printm("ACK of bit0 \n"); */
			pic.reg[r_flag]=0;
		} else {
			if( pic.reg[r_ack] != pic.reg[r_flag] ) {
				static int warned=0;
				if( warned++<3 )
					printm("**** POSSIBLE PIC IMPLEMENTATION ERROR ***\n");
				/* stop_emulation(); */
			}
		}
		pic.reg[r_flag] &= ~pic.reg[r_ack];
		intline_upd();
		break;
	case r_ack_2:
		/* experimental... se above */
		if( pic.reg[r_ack_2] & 0x80 ) {
			/* printm("ACK of bit0 \n"); */
			pic.reg[r_flag_2]=0;
		}
		pic.reg[r_flag_2] &= ~pic.reg[r_ack_2];
		intline_upd();
		break;
	case r_level:
#if 0
	{

		static int has_warned=0;
		if( has_warned++ < 3 )
			printm("PIC: Write to level register %08lX (is %08lX)\n", data, old);
	}
#endif
#if 1
/*		pic.reg[r_level] &= ~pic.reg[reg]; */
#endif
		pic.reg[reg] = old; /* assume read only */
		break;
	case r_level_2:
	case r_flag:
	case r_flag_2:
		printm("PIC: Write to register assumed to be read-only\n");
		pic.reg[reg] = old;
		break;
	}
	UNLOCK;
}

static ulong
pic_io_read( ulong addr, int len, void* usr )
{
	ulong ret;
	LOCK;
	ret = read_mem( (char*)pic.reg + addr - get_mbase(pic.pic_range), len);
	UNLOCK;
	return ret;
}


static void 
intline_upd( void )
{
	int new;

	new = pic.reg[r_flag] || pic.reg[r_flag_2];

	if( mregs->irq != new ) {
		mregs->irq = new;
		 /* make sure the main thread notices the changed irq value */
		if( new && (mregs->msr & MSR_EE) )
			interrupt_emulation();
	}
}


static void 
pic_io_print( int isread, ulong addr, ulong data, int len, void *usr )
{
	int reg = (addr - get_mbase(pic.pic_range))>>2;
	int i;
	ulong val;

	val = ld_le32( &data );
	printm( "PIC %s %s %08lX IRQ-bits: [ ", isread? "read " : "write", 
		reg_names[reg],val );

	i = reg>3 ? 0 : 32;
	while( val ){
		if( val & 1 )
			printm("%d ", i );
		i++;
		val = val >> 1;
	}
	printm("]\n");
}


static void 
_irq_interrupt( int irq )
{
	ulong	tmp;
	
	if( irq >= 32 ){
		st_le32( &tmp, 1UL << (irq-32) );
		pic.reg[ r_flag_2 ] |= tmp & pic.reg[ r_enable_2 ];
	} else {
		st_le32( &tmp, 1UL << irq );
		pic.reg[ r_flag ] |= tmp & pic.reg[ r_enable ];
	}

	/* update the interrupt line */
	intline_upd();
}



void 
irq_line_hi( int irq )
{
	ulong	old, tmp;

	LOCK;
	if( irq >= 32 ) {
		st_le32( &tmp, 1UL << (irq-32) );
		old = pic.reg[ r_level_2 ];
		pic.reg[ r_level_2 ] |= tmp;
		if( pic.reg[r_level_2] ^ old )
			_irq_interrupt( irq );	/* positive edge */
	} else {
		st_le32( &tmp, 1UL << irq );
		old = pic.reg[ r_level ];
		pic.reg[ r_level ] |= tmp;
		if( pic.reg[r_level] ^ old )
			_irq_interrupt( irq );	/* positive edge */
	}
	UNLOCK;
}


void 
irq_line_low( int irq )
{
	ulong tmp, old;
	
	LOCK;
	if( irq >= 32 ) {
		st_le32( &tmp, 1UL << (irq-32) );
		old = pic.reg[r_level_2 ];
		pic.reg[ r_level_2 ] &= ~tmp;
		if( pic.reg[r_level_2] ^ old )
			_irq_interrupt( irq );	/* negative edge */
	} else {
		st_le32( &tmp, 1UL << irq );
		old = pic.reg[r_level ];
		pic.reg[ r_level ] &= ~tmp;
		if( pic.reg[r_level] ^ old )
			_irq_interrupt( irq );	/* negative edge */
	}
	UNLOCK;
}


/************************************************************************/
/*	CMDs & DEBUG							*/
/************************************************************************/

static void
print_irq_nums( ulong mask, int add )
{
	mask = ld_le32( &mask );

	while( mask ){
		if( mask & 1 )
			printm("%d ", add );
		add++;
		mask = mask >> 1;
	}
}

static int 
cmd_pic_regs( int argc, char **argv )
{
	if(argc != 1)
		return -1;

	printm( "}\nEnable:  { ");
	print_irq_nums( pic.reg[ r_enable ], 0 );
	print_irq_nums( pic.reg[ r_enable_2 ], 32 );
	printm( "}\nFlag:    { ");
	print_irq_nums( pic.reg[ r_flag ], 0 );
	print_irq_nums( pic.reg[ r_flag_2 ], 32 );
	printm( "}\nLevel:   { ");
	print_irq_nums( pic.reg[ r_level ], 0 );
	print_irq_nums( pic.reg[ r_level_2 ], 32 );
	printm( "}\n");
	return 0;
}

static int
cmd_nmi( int argc, char **argv )
{
	// This does not work... oh well
	irq_line_hi( 20 /* NMI */);
	return 0;
}


