/* 
 *   Creation Date: <1999/03/05 16:14:42 samuel>
 *   Time-stamp: <2001/03/03 20:38:34 samuel>
 *   
 *	<pci-brdiges.c>
 *	
 *	PCI bridges (bandit, chaos and grackle)
 *   
 *   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

/* #define VERBOSE */

#include "verbose.h"
#include "promif.h"
#include "ioports.h"
#include "debugger.h"
#include "pci.h"
#include "debugger.h"
#include "extralib.h"

SET_VERBOSE_NAME("pci-bridges")

struct pci_bridge;
typedef struct pci_bridge pci_bridge_t;

static ulong 	reg_io_read( ulong addr, int len, void *u_bridge );
static void 	reg_io_write( ulong addr, ulong data, int len, void *u_bridge );
static void 	reg_io_print( int isread, ulong addr, ulong data, int len, void *u_bridge );

static int	bandit_interpret_addr( pci_bridge_t *b, ulong pciaddr, int *bus, int *dev_fn, int *offs );
static int	grackle_interpret_addr( pci_bridge_t *b, ulong pciaddr, int *bus,  int *dev_fn, int *offs );

static void	init_bridges( char *device_name, int bridge_type );
static void	init_chaos_bridge( mol_device_node_t *dn, pci_bridge_t *bridge );
static void	init_bandit_bridge( mol_device_node_t *dn, pci_bridge_t *bridge );
static void	init_grackle_bridge( mol_device_node_t *dn, pci_bridge_t *bridge );

/* #define BRIDGE_DEBUG  */

/* this hold for bandit & chaos */
#define BANDIT_CFG_ADDR_OFFS	0x800000
#define BANDIT_CFG_DATA_OFFS	0xc00000

/* This is for Grackle in map B - note that the address is by 0xffffc -
 * i.e. we must make sure we add maps. Motorola suggests FEC00CF8 and
 * FEE00CFE.
 */
#define GRACKLE_CFG_ADDR	0xFEC00000
#define GRACKLE_CFG_DATA	0xFEE00000

#define BANDIT_DEV_FN	(11<<3)
#define GRACKLE_DEV_FN	0

/* Chaos bridge:
 *   It appears as if OF reads data from offset c00004 whenever 
 *   the config address ends at 0x4 or 0xc, probably due to a 64-bit
 *   bus interface or something (the linux kernel doesn't do this).
 */

#if 0
#define DEBUG_BUS		0
#define DEBUG_DEV_FN		0x68 /*(13<<3) */
#endif

struct pci_bridge
{
	char 	*bridge_name;	/* verbose name */
	ulong	mbase;		/* register base (0xF8000000) */

	ulong	addr_mbase;	/* config register port */
	ulong	data_mbase;	/* config data port */

	int	bridge_type;	/* chaos or bandit */
	void	*type_data;	/* bridge specific data goes here */
	
	int	first_bus;
	int	last_bus;

	int	(*interpret_addr)( pci_bridge_t *b, ulong pciaddr, int *bus, int *dev_fn, int *offs );

	struct pci_bridge  *next;

	/* emulated hardware registers */
	ulong	addr;		/* address register */
};

static pci_bridge_t *all_bridges;

enum {
	bandit_bridge, chaos_bridge, grackle_bridge
};


static io_ops_t ops={
	NULL,			/* cleanup */
	reg_io_read,
	reg_io_write, 
	reg_io_print
};


/************************************************************************/
/*	FUNCTIONS							*/
/************************************************************************/

void
pci_bridges_init( void )
{
	mol_device_node_t *dn;
	
	all_bridges = NULL;
	
	init_bridges( "bandit", bandit_bridge );
	init_bridges( "chaos", chaos_bridge );
	init_bridges( "pci", grackle_bridge );

	/* Verbose information about bridges */
	dn = prom_find_devices( "pci-bridge" );
	for( ; dn ; dn=dn->next ){
		int bus, dev_fn, len;
		char *str;
		pci_device_loc( dn, &bus, &dev_fn );
		str = prom_get_property( dn, "model", &len );
		VPRINT("PCI-bridge %s at %d:%d\n", (len>0)? str : "?", bus, dev_fn>>3);
	}
}

void 
pci_bridges_cleanup( void )
{
	pci_bridge_t *br, *next;
	for( br = all_bridges; br; br=next ){
		next = br->next;
		if( br->bridge_name )
			free( br->bridge_name );
		if( br->type_data )
			free( br->type_data );
		free( br );
	}
	all_bridges = NULL;
}


static void
init_bridges( char *device_name, int bridge_type )
{
	mol_device_node_t *dn;
	int flags=0;
	pci_bridge_t *bridge;
	char namebuf[40];
#ifdef BRIDGE_DEBUG
	flags = io_print;
#endif
	dn = prom_find_devices( device_name );

#ifdef BRIDGE_DEBUG
	if( dn )
		VPRINT("Adding bridge: %s...\n", device_name);
#endif
	
	for( ; dn ; dn=dn->next ) {
		int i, first_bus, last_bus;

		if( dn->n_addrs != 1 ) {
			printm("%s: Expecting 1 addrs. Got %d)\n", device_name, 
			       dn->n_addrs );
			continue;
		}

		if( pci_get_bus_number( dn, &first_bus, &last_bus ) ) {
			LOG("Couldn't get bus number\n");
			continue;
		}
		for( i = first_bus; i <= last_bus; i++ ) {
			if( add_pci_bus( i ) ){
				LOG("Couldn't add pci bus %d\n", i);
				continue;
			}
		}

		bridge = calloc( sizeof(struct pci_bridge), 1 );
		bridge->bridge_name = strdup( device_name );
		bridge->next = all_bridges;
		all_bridges = bridge;
		
		bridge->mbase = dn->addrs[0].address;
		bridge->first_bus = first_bus;
		bridge->last_bus = last_bus;
		bridge->bridge_type = bridge_type;

		/* Is this a Grackle ? */
		if (strcmp(device_name, "pci") == 0)
		{
			bridge->addr_mbase = GRACKLE_CFG_ADDR;
			bridge->data_mbase = GRACKLE_CFG_DATA;
			bridge->interpret_addr = grackle_interpret_addr;
		}
		/* This is a bandit or chaos */
		else
		{
			bridge->addr_mbase = bridge->mbase + BANDIT_CFG_ADDR_OFFS;
			bridge->data_mbase = bridge->mbase + BANDIT_CFG_DATA_OFFS;
			bridge->interpret_addr = bandit_interpret_addr;
		}
		
		/* the minimum IO-area size is 16 bytes, we need only 4 (8 for chaos) bytes though */
		snprintf( namebuf, 40, "%s-%d-%d-addr", device_name, first_bus, last_bus );
		add_io_range( bridge->addr_mbase, 0x10, namebuf, flags, &ops, (void*)bridge );
		snprintf( namebuf, 40, "%s-%d-%d-data", device_name, first_bus, last_bus );
		add_io_range( bridge->data_mbase, 0x10, namebuf, flags, &ops, (void*)bridge );

		switch( bridge_type ){
		case chaos_bridge:
			init_chaos_bridge( dn, bridge );
			break;
		case bandit_bridge:
			init_bandit_bridge( dn, bridge );
			break;
		case grackle_bridge:
			init_grackle_bridge( dn, bridge );
			break;
		default:
			LOG("Warning: Unsupported bridge type\n");
			break;
		}
		
		VPRINT("PCI-bridge '%s' (bus %d..%d) installed at 0x%08lx\n", 
		       device_name, first_bus, last_bus, bridge->mbase );
	}
}


static ulong 
reg_io_read( ulong addr, int len, void *u_bridge )
{
	pci_bridge_t *bridge = (pci_bridge_t*)u_bridge;
	int dev_fn, offs;

	/* Address port */
	if( (addr & ~0x3) == bridge->addr_mbase ) {
		ulong buf[2];
		buf[0] = bridge->addr;
		return read_mem( (char*)buf+(addr&3), len );
	}
	/* Data port */
	if( (addr & ~0x7) == bridge->data_mbase ){
		int bus;
		ulong ret = 0;

		if( !bridge->interpret_addr( bridge, bridge->addr, &bus, &dev_fn, &offs ) )
			ret = read_pci_config( bus, dev_fn, offs+(addr&3),len );
#if defined(DEBUG_BUS) && defined(DEBUG_DEV_FN)
		if( bridge->bus == DEBUG_BUS && dev_fn == DEBUG_DEV_FN ){
			printm("PCI, read [%d] (%d:%02X)+%02X: %08lX\n",
			       len, bridge->bus, dev_fn, offs, ret );
		}
#endif
		return ret;
	}
	return 0;
}

static void 
reg_io_write( ulong addr, ulong data, int len, void *u_bridge )
{
	pci_bridge_t *bridge = (pci_bridge_t*)u_bridge;
	int bus, dev_fn, offs;
	
	/* Address port */
	if( (addr & ~0x3) == bridge->addr_mbase ) {
		ulong buf[2];
		buf[0] = bridge->addr;
		write_mem( (char*)buf+(addr&3), data, len );
		bridge->addr = buf[0];
		return;
	}

	/* Data port - note that Chaos uses a 64-bit bus interface (or?),
	 * placing the data in the upper word for xxx4/xxxc accesses 
	 */
	if( (addr & ~0x7) == bridge->data_mbase ){
		if( !bridge->interpret_addr( bridge, bridge->addr, &bus, &dev_fn, &offs ) )
			write_pci_config( bus, dev_fn, offs+(addr&3), data, len );
#if defined(DEBUG_BUS) && defined(DEBUG_DEV_FN)
		if( bridge->bus == DEBUG_BUS && dev_fn == DEBUG_DEV_FN ){
			printm("PCI, write [%d] (%d:%02X)+%02X: %08lX\n",
			       len, bridge->bus, dev_fn, offs, data );
		}
#endif
	}
}

static void 
reg_io_print( int isread, ulong addr, ulong data, int len, void *u_bridge )
{
	pci_bridge_t *bridge = (pci_bridge_t*)u_bridge;
	int offs, dev_fn, bus;
	ulong d = data;
	
	if( (addr & ~0x3) == bridge->addr_mbase ) {
#if 0
		bridge->interpret_addr( bridge, bridge->addr, &bus, &dev_fn, &offs );
		printm("Bridge '%s' bus %d ADDR %s %08lX ; (%02d;%s+%02X)\n", bridge->bridge_name, bus,
		       isread ? "read: " : "write:", ld_le32(&data), dev_fn>>3, dev_fn&7, offs );
#endif
		return;
	}

	if( (addr & ~0x7) == bridge->data_mbase ){
		bridge->interpret_addr( bridge, bridge->addr, &bus, &dev_fn, &offs );
		addr &= ~0x4;

		if( len == 4 )
			d = ld_le32( &data );
		else if( len == 2 )
			d = ld_le16( (short*)&data +1 );
		printm("Bridge '%s-%d' [%d] %s (%02d;%d+%02lX): %08lX\n", bridge->bridge_name,
		       bus, len, isread? "read " : "write", dev_fn>>3, dev_fn&7, offs+(addr&3), d);
		return;
	}

	printm("bridge ???? %08lX: (len=%d) %s %08lX\n", addr, len,
	       isread ? "read: " : "write:", data );
/* 	stop_emulation(); */
}


/* calculates the inverse of 
 *    
 *      addr = (1UL << (dev_fn >> 3)) + ((dev_fn & 7) << 8) + (offset & ~3)) 
 * 
 * (for bandit & chaos)
 */
static int
bandit_interpret_addr( pci_bridge_t *b, ulong pciaddr_le, 
		       int *bus, int *dev_fn, int *offs )
{
	ulong tbit;
	int val, hit=0, fn_hi=0;
	ulong pciaddr_be;
	
	pciaddr_be = ld_le32( &pciaddr_le );
	
	/* [16+5 device num] [3 bits function] [8 bits address offset] */

	*offs = (pciaddr_be & ~3) & 0xff;
        *dev_fn = (pciaddr_be >> 8) & 7;
	*bus = b->first_bus;	/* XXX: fix me */

	for( val=11, tbit=0x800; tbit; tbit=tbit<<1, val++ ){
		if( pciaddr_be & tbit ) {
			hit++;
			*dev_fn |= val<<3;
			fn_hi = val;
		}
	}
	if( hit != 1 ) {
		*dev_fn = 0xff;
		*offs = 0xff;
		return 1;
	}
	return 0;
}

/* Grackle: LE:  offset.B&~0x3 dev_fn.B bus.B 0x80  */
static int
grackle_interpret_addr( pci_bridge_t *b, ulong pciaddr_le, 
			int *bus, int *dev_fn, int *offs)
{
	*offs = pciaddr_le >> 24;
	*dev_fn = (pciaddr_le >> 16) & 0xff;
	*bus = (pciaddr_le >> 8) & 0xff;

	if( *bus < b->first_bus || *bus > b->last_bus )
		return 1;

	/* possibly an error should be returned if the enable bit is NOT set */
	return 0;
}

/************************************************************************/
/*	Bridges								*/
/************************************************************************/

static pci_dev_info_t bandit_pci_config = {
	/* vendor, device ID, revision, class */
	0x106b, 0x0001, 0x03, 0x0600
};
static pci_dev_info_t grackle_pci_config = {
	/* vendor, device ID, revision, class */
	0x1057, 0x0002, 0x03, 0x0600
};

static void
init_bandit_bridge( mol_device_node_t *dn, struct pci_bridge *bridge )
{
	add_pci_device( bridge->first_bus, BANDIT_DEV_FN, &bandit_pci_config, NULL );
}

static void
init_chaos_bridge( mol_device_node_t *dn, struct pci_bridge *bridge )
{

}

static void
init_grackle_bridge( mol_device_node_t *dn, struct pci_bridge *bridge )
{
	add_pci_device( bridge->first_bus, GRACKLE_DEV_FN, &grackle_pci_config, NULL );
}







