/* 
 *   Creation Date: <1999/03/05 13:58:16 samuel>
 *   Time-stamp: <2001/01/31 22:59:41 samuel>
 *   
 *	<hammerhead.c>
 *	
 *	Memory controller (7500,8500,9500,...)
 *   
 *   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

/* #define VERBOSE */
#define G3_CLOCK_FIX_HACK

#include "verbose.h"
#include "promif.h"
#include "ioports.h"
#include "debugger.h"
#include "mac_registers.h"
#include "memory.h"
#include "driver_mgr.h"

#ifdef VERBOSE
SET_VERBOSE_NAME("hammerhead")
#endif

static int 	hammerhead_init( void );
static void 	hammerhead_cleanup( void *usr );
static ulong 	reg_io_read( ulong addr, int len, void *usr );
static void 	reg_io_write( ulong addr, ulong data, int len, void *usr );
static void 	reg_io_print( int isread, ulong addr, ulong data, int len, void *usr );

#ifdef G3_CLOCK_FIX_HACK
static void 	g3_clock_sync_hack( void );
#endif

#define HAMMERHEAD_SIZE		0x800

/* #define HH_DEBUG */
/* #define HH_HARDWARE_READ */

/* 
 * NOTE:
 *   It appears we don't have to mess with the hammerhead, except 
 *   making sure that the 0xf8000000 register holds the correct 
 *   identification byte. 
 *
 *   Correction: If we add a control device to the PCI bus, then
 *   we must take a closer look at hammerhead.
 */

static struct hammerhead_status
{
	ulong	mregs;		/* register base (0xF8000000) */
} hs[1];

static io_ops_t ops={
	hammerhead_cleanup,
	reg_io_read,
	reg_io_write, 
	reg_io_print
};

driver_interface_t      hammerhead_driver =
{
    "hammerhead", hammerhead_init, NULL
};


/************************************************************************/
/*	FUNCTIONS							*/
/************************************************************************/

int
hammerhead_init( void )
{
	mol_device_node_t *dn;
	int flags=0;
	
	memset( hs, 0, sizeof(struct hammerhead_status) );

	if( (dn = prom_find_devices( "hammerhead" ))== NULL )
		return 0;

	if( dn->n_addrs != 1 || dn->addrs[0].size != HAMMERHEAD_SIZE ) {
		printm("hammerhead: Expecting 1 addrs. Got %d)\n", dn->n_addrs );
		return 0;
	}
	hs->mregs = dn->addrs[0].address;

#ifdef HH_DEBUG
	flags = io_print;
#endif
	add_io_range( hs->mregs, HAMMERHEAD_SIZE, "hammerhead", flags, &ops, NULL );

	printm("Hammerhead driver installed\n");
	
	return 1;
}

static void 
hammerhead_cleanup( void *usr )
{
}

static ulong 
reg_io_read( ulong addr, int len, void *usr )
{
	int reg = (addr - hs->mregs)>>4;
	ulong buf[2], ret;

	if( addr & 0xf )
		printm("hammerhead: Unsupported access!\n");

	buf[0] = 0;
	switch( reg ){
	case 0: 
		/* Identification register. Only the MSB seems to be
		 * used. The value is from a 8500/150.
		 *
		 * QUESTION: Is this valid also on the 7500 and the 9500?
		 * Please mail me the result of "ioread f8000000" for
		 * these machines.
		 */
		buf[0] = 0x39000000;
		break;
	case 2:
		/* This register is necessary if we register the control
		 * device on the chaos pci bus. Conversely, if we define
		 * this register, we must add a control entry...
		 */
#if 0
		buf[0] = 0x90000020;
#endif
		break;
	case 0xe:
#ifdef G3_CLOCK_FIX_HACK
		/* This hack fixes the calibration of the DEC/TB
		 * (mostly needed on the G3, but useful for other machines too).
		 * This applies to the 8200/8500/7200 ROM.
		 */
		if( mregs->nip == 0xfff03bd4 ) {
			printm("[CLOCK_FIX_HACK]\n");
			g3_clock_sync_hack();
		}
#endif
		break;

	case 0x1c:	/* control/status */
	case 0x4d:	/* base  */
		/* ... */
	case 0x4e:	/* control/status */
	case 0x4f:	/* base */
		break;
	}
	ret = read_mem( (char*)buf+(addr&3), len );

#ifdef HH_HARDWARE_READ
	ret = ioread( addr, len );
#endif
	return ret;
}


static void 
reg_io_write( ulong addr, ulong data, int len, void *usr )
{
	if( addr & 0xf )
		printm("hammerhead: Unsupported access!\n");
}

static void 
reg_io_print( int isread, ulong addr, ulong data, int len, void *usr )
{	
	printm("hammerhead: %08lx (len=%d) %s %08lX\n", addr, len,
	       isread ? "read: " : "write:", data );
}



#ifdef G3_CLOCK_FIX_HACK
/* 
 * Hack of the 8500/8200/7200 ROM to fix the G3 calibration
 * of the DEC/TB.
 *
 *  0x1104 = processor frequency (we fix this too thugh not important)
 *  0x1108 = timebase-frequency   (DEC)
 *  0x110c = timebase-frequency/4 (TB)
 */
static void 
g3_clock_sync_hack( void )
{
	mol_device_node_t *cpu;
	ulong	*lvptr, *ptr;

	lvptr = (ulong*)&ram.lvbase[0x1100];

	cpu = prom_find_type_hw( "cpu" );
	if( !cpu ) {
		printm("No 'cpu' property in the device tree - fix skipped\n");
		/* here we should perhaps set some default values */
		return;
	}
	ptr = (ulong*)prom_get_property( cpu, "timebase-frequency", NULL );

	if( !ptr ) {
		printm("No 'timebase-frequency' property - fix skipped\n");
	} else {       
		lvptr[2] = *ptr;
		lvptr[3] = *ptr / 4;
	}
	ptr = (ulong*)prom_get_property( cpu, "clock-frequency", NULL );
	if( !ptr )
		printm("No 'clock-frequency' property\n");
	else
		lvptr[1] = *ptr;
}
#endif

