/* 
 *   Creation Date: <2001/05/12 17:39:24 samuel>
 *   Time-stamp: <2001/09/30 17:48:03 samuel>
 *   
 *	<blkdev.c>
 *	
 *	Find out which block devices to use
 *   
 *   Copyright (C) 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/uio.h>
#include <sys/ioctl.h>

#include "res_manager.h"
#include "partition_table.h"
#include "hfs_mdb.h"
#include "disk.h"
#include "llseek.h"
#include "booter.h"
#include "fs.h"
#include "newworld.h"

#define BLKFLSBUF  _IO(0x12,97)		/* from <linux/fs.h> */

/* Volumes smaller than this must be specified explicitely
 * (and not through for instance 'blkdev: /dev/hda -rw').
 * The purpose of this is preventing any boot-strap partitions
 * from beeing mounted (and hence deblessed) in MacOS.
 */
#define BOOTSTRAP_SIZE_MB	20

static int find_hfs_partitions( char *name, int fd, int flags, ulong blk_start, ulong len );
static void do_setup_drives( char *res_name );

static opt_entry_t blk_opts[] = {
	{"-ro",			0 },
	{"-rw",			BF_ENABLE_WRITE },
	{"-force",		BF_FORCE },
	{"-cdrom",		BF_CD_ROM },
	{"-cd",			BF_CD_ROM },
	{"-boot",		BF_BOOT },
	{NULL, 0 }
};

#define MAX_NUM_PART		32

static bdev_desc_t *s_all_disks = NULL;
static int s_last_id = 0;


/************************************************************************/
/*	Get volume							*/
/************************************************************************/

static int
is_bootable( bdev_desc_t *bdev )
{
	if( newworld_rom_found() )
		return 1;
	if( bdev->flags & BF_HFS )
		return !load_newworld_rom( FS_HFS, bdev->dev_name, bdev->soffs );
	else if( bdev->flags & BF_HFS_PLUS )
		return !load_newworld_rom( FS_HFS_PLUS, bdev->dev_name, bdev->soffs );
	return 0;
}

void
bdev_setup_drives( void )
{
	bdev_desc_t *bdev, *b2;

	if( !is_newworld_boot() && !is_oldworld_boot() )
		return;

	printm("\n");
	if( get_bool_res("mount_mol_disk") != 0 )
		do_setup_drives("blkdev_mol");
	do_setup_drives("blkdev");
	printm("\n");

	/* Determine startup disk (newworld) */
	if( is_newworld_boot() ) {
		for( bdev = s_all_disks; bdev; bdev=bdev->priv_next ) {
			if( !(bdev->flags & BF_BOOT) )
				continue;
			if( is_bootable(bdev) )
				break;
			bdev->flags |= BF_NOT_BOOTABLE;
		}
		for( b2 = s_all_disks; b2; b2=b2->priv_next )
			b2->flags &= ~BF_BOOT;
		if( !bdev ) {
			for( bdev = s_all_disks; bdev; bdev=bdev->priv_next ) {
				if( (bdev->flags & (BF_BOOT | BF_NOT_BOOTABLE)) )
					continue;
				if( is_bootable( bdev ))
					break;
				bdev->flags |= BF_NOT_BOOTABLE;
			}
		}
		if( !bdev )
			printm("--- No bootable disk found ---\n");			
		else
			bdev->flags |= BF_BOOT;
	}
}

bdev_desc_t *
bdev_get_volume( int type )
{
	bdev_desc_t *bdev = s_all_disks;
	for( ; bdev ; bdev=bdev->priv_next ){
		int flags = bdev->flags;
		if( bdev->priv_claimed )
			continue;
		if( (type & BDEV_TYPE_HFS) && !(flags & (BF_HFS | BF_HFS_PLUS)) )
			continue;
		if( (type & BDEV_TYPE_BOOT) && !(flags & BF_BOOT) )
			continue;
		if( (type & BDEV_TYPE_NOT_BOOT) && (flags & BF_BOOT) )
			continue;

		bdev->priv_claimed = 1;
		return bdev;
	}
	return NULL;
}

void
bdev_close_volume( bdev_desc_t *dev )
{
	bdev_desc_t **bd;

	for( bd=&s_all_disks; *bd; bd=&(**bd).priv_next ){
		if( *bd != dev )
			continue;
		*bd = (**bd).priv_next;
		free( dev->dev_name );
		if( dev->vol_name )
			free( dev->vol_name );
		
		ioctl( dev->fd, BLKFLSBUF );
		if( dev->fd != -1 )
			close( dev->fd );
		return;
	}
	printm("bdev_close_volume: dev is not in the list!\n");
}


/************************************************************************/
/*	Find/prepare HFS partitions					*/
/************************************************************************/

static void
add_volume( char *name, char *vol_name, int fd, long flags, ulong blk_start, ulong num_blocks )
{
	bdev_desc_t *bdev = malloc( sizeof(bdev_desc_t) );
	CLEAR( *bdev );

	bdev->dev_name = strdup(name);
	bdev->vol_name = vol_name ? strdup(vol_name) : NULL;
	bdev->priv_id = s_last_id;

	bdev->fd = fd;
	bdev->flags = flags;
	bdev->soffs = 512ULL * blk_start;
	bdev->size = 512ULL * num_blocks;

	bdev->priv_next = s_all_disks;
	s_all_disks = bdev;
}


static int
do_open( char *name, int flags, int constructed )
{
	bdev_desc_t *bd;
	int fd, ro_fallback=1;
	int id, count=0;

	if( (fd=disk_open( name, flags, &ro_fallback, constructed )) < 0 )
		return 1;

	if( ro_fallback )
		flags &= ~BF_ENABLE_WRITE;

	/* printf("Disk '%s' opened, rw=%d\n", name, flags & BF_ENABLE_WRITE ); */

	id = ++s_last_id;
	if( !find_hfs_partitions( name, fd, flags, 0, get_file_size_in_blks(fd) ) ) {
		if( !(flags & BF_SUBDEVICE) )
			printm("No HFS partitions found in '%s'\n", name );
		close( fd );
		return 1;
	}

	/* reopen devices with multiple volumes - we want independent descriptors */
	for( bd=s_all_disks; bd; bd=bd->priv_next ) {
		char buf[ 80 ];
		int reopen=0;

		if( bd->priv_id != id )
			continue;
		if( count++ >= 1 )		/* fd already in use? */
			reopen=1;

		/* Detect boot-strap partitions by the small size */
		if( constructed && (bd->size >> 20) < BOOTSTRAP_SIZE_MB && (bd->flags & BF_ENABLE_WRITE)) {
			bd->flags &= ~BF_ENABLE_WRITE;
			if( !reopen )
				close( bd->fd );
			reopen = 1;
			printm("----> %s might be a boot-strap partition.\n", bd->dev_name );
			
		}
		
		if( reopen ) {
			bd->fd = open( bd->dev_name, (bd->flags & BF_ENABLE_WRITE) ? O_RDWR : O_RDONLY );
			if( bd->fd < 0 ) {
				perrorm("reopening %s", bd->dev_name);
				continue;
			}
			/* assign mew, unique id */
			bd->priv_id = ++s_last_id;
		}
		strnpad( buf, name, 17 );
		if( bd->vol_name )
			strncat3( buf, " ", bd->vol_name, sizeof(buf));
		strnpad( buf, buf, 36 );

		printm("%s %s <read-%s %4ld MB %s", 
		       (bd->flags & BF_HFS)? "HFS " : (bd->flags & BF_HFS_PLUS )? "HFS+" : "Disk",
		       buf, (bd->flags & BF_ENABLE_WRITE)? "write>" : "only> ",
		       (long)(bd->size >> 20), 
		       (bd->flags & BF_BOOT)? "BOOT" : "" );

		if( bd->soffs )
			printm("  start: %ld, len: %ld", (long)(bd->soffs >> 9), (long)(bd->size >> 9));
		printm("\n");
	}

	/* Clean out bad entries */
	for( bd=s_all_disks; bd; bd=bd->priv_next ) {
		if( bd->fd != -1 )
			continue;
		bdev_close_volume( bd );
		bd = s_all_disks;
	}
	return !count;
}


static void
do_setup_drives( char *res_name )
{
	static char 	*dev_list[] = { "/dev/sd", "/dev/hd", NULL };
	int 		i, j, flags;
	char 		buf[32], *name, **pp;

	for( i=0; (name=get_str_res_ind( res_name, i, 0)) ; i++ ) {
		int 	next = 0;
		int	n = 0;
		
		flags = parse_res_options( res_name, i, 1, blk_opts, "Unknown disk flag");

		if( flags & BF_CD_ROM )
			flags &= ~BF_ENABLE_WRITE;

		for( pp=dev_list; *pp; pp++ ){
			if( strncmp( *pp, name, strlen(*pp) ) || strlen(name) != strlen(*pp)+1 )
				continue;

			/* /dev/hda type device, substitute with /dev/hdaX */
			flags |= BF_SUBDEVICE;
			flags &= ~BF_FORCE;
			for(j=1; j<16; j++ ){
				sprintf( buf, "%s%d", name, j );
				n += !do_open( buf, flags, 1 );
			}
			if( !n )
				printm("No volumes found in '%s'\n", name );
			next = 1;
		}
		if( next )
			continue;

		do_open( name, flags, 0 );
	}
}



/* returns the number of HFS partitions added */
static int
find_hfs_partitions( char *name, int fd, int flags, ulong first_blk, ulong num_blks )
{
	desc_map_t dmap;
	int i, count=0;

	/* Note: num_blks might be zero here ... */
	
	/* has this drive a partition map? */
	if( first_blk==0 ) {
		lseek( fd, 0, SEEK_SET );
		read( fd, &dmap, sizeof(dmap) );
	}
	if( first_blk==0 && dmap.sbSig == DESC_MAP_SIGNATURE ){
		/* This is a block 0 in a partition map */
		part_entry_t par;
		int num_par;

		/* printm("Partition map detected\n"); */
		flags &= ~BF_FORCE;
		flags |= BF_SUBDEVICE;
		for( num_par=2000, i=1 ; i<=num_par; i++ ){
			int num;
			lseek( fd, dmap.sbBlockSize*i, SEEK_SET );
			read( fd, &par, sizeof(part_entry_t) );
			if( num_par == 2000 )
				num_par = par.pmMapBlkCnt;
			if( par.pmSig != 0x504d /* 'PM' */)
				break;

			/* printm("%s, %s\n", par.pmPartName, par.pmPartType ); */
			if( strcmp("Apple_HFS", par.pmPartType ))
				continue;

			/* Safety check - force r/o if the pmPyPartStart/pmPartBlkCnt exceeds first_blk/num_blks. */
			if( (flags & BF_ENABLE_WRITE) && ( par.pmPyPartStart * (dmap.sbBlockSize/512) < first_blk 
			    || (par.pmPyPartStart+par.pmPartBlkCnt) * (dmap.sbBlockSize/512) > first_blk + num_blks )) 
			{
				printm("Warning: unexpected partition range, forcing read-only\n");
				flags &= ~BF_ENABLE_WRITE;
			}

			num = find_hfs_partitions( name, fd, flags, 
						      par.pmPyPartStart * (dmap.sbBlockSize/512),
						      par.pmPartBlkCnt * (dmap.sbBlockSize/512) );

			/* Sometimes pmPyPartStart/BlkCnt is not given in sbBlockSize units,
			 * but in standard 512 bytes blocks. For safety, we require r/o access.
			 */
			if( !num && !(flags & BF_ENABLE_WRITE) )
				num = find_hfs_partitions( name, fd, flags, par.pmPyPartStart, par.pmPartBlkCnt );

			count += num;
		}
	} else {
		hfs_mdb_t mdb;
		int err, signature;

		/* Num_blks might be zero here. Currently this happens with SCSI CD-ROMs without 
		 * a partition table.
		 */
		if( !num_blks ){
			if( !(flags & BF_CD_ROM) ) {
				printm("Deivce '%s' has size zero (?)\n", name );
				return count;
			}
			printm("Warning: uses hardcoded size for CD-ROM\n");
			num_blks = 0x200000;	/* 1 GB... more than sufficient */
		}

		blk_lseek( fd, first_blk+2, SEEK_SET );
		err = read( fd, &mdb, sizeof(mdb)) != sizeof(mdb);
		signature = hfs_get_ushort(mdb.drSigWord);
		
		if( !err && signature == HFS_PLUS_SIGNATURE ) {
			flags |= BF_HFS_PLUS;
			/* slightly tricky to obtain the volume name - we don't for now */
			add_volume( name, "Unembedded HFS+", fd, flags, first_blk, num_blks );
			count++;
		}
		if( !err && signature == HFS_SIGNATURE ) {
			char vname[256];

			flags |= ( hfs_get_ushort(mdb.drEmbedSigWord) == HFS_PLUS_SIGNATURE )?
				BF_HFS_PLUS : BF_HFS;

			memcpy( vname, &mdb.drVN[1], mdb.drVN[0] );
			vname[mdb.drVN[0]] = 0;
			add_volume( name, vname, fd, flags, first_blk, num_blks );
			count++;
		}
		if( !(flags & (BF_HFS | BF_HFS_PLUS)) ) {
			if( (flags & BF_FORCE) ) {
				add_volume( name, NULL, fd, flags, first_blk, num_blks );
				count++;
			} else if( !(flags & BF_SUBDEVICE) ) {
				printm("Device '%s' is not a HFS/HFS+ partition\n"
				       "(Use the -force flag to make it available anyway - DANGEROUS)\n",
				       name );
			}
		}
	}
	return count;
}

#if 0
/* Calculate a checksum for the HFS(+) MDB (master directory block). 
 * In particular, the modification date is included in the checksum.
 */
static int 
get_hfs_checksum( drive_t *drv, ulong *checksum )
{
	hfs_mdb_t	mdb;
	hfs_plus_mdb_t	mdb_plus;
	ulong		val=0;

	blk_lseek( drv->fd, drv->first_block+2, SEEK_SET );
	read( drv->fd, &mdb, sizeof(mdb) );
	if( hfs_get_ushort(mdb.drSigWord) != HFS_SIGNATURE )
		return -1;

	// printm("HFS volume detected\n");

	if( hfs_get_ushort(mdb.drEmbedSigWord) == HFS_PLUS_SIGNATURE ) {
		int sblock = hfs_get_ushort(mdb.drAlBlSt);
		sblock += (hfs_get_uint(mdb.drAlBlkSiz) / 512) * (hfs_get_uint(mdb.drEmbedExtent) >> 16);

		blk_lseek( drv->fd, drv->first_block + sblock + 2, SEEK_SET );
		read( drv->fd, &mdb_plus, sizeof(mdb_plus) );

		if( mdb_plus.signature != HFS_PLUS_SIGNATURE ) {
			printm("HFS_PLUS_SIGNATURE expected\n");
			return -1;
		}
		val += calc_checksum( (char*)&mdb_plus, sizeof(mdb_plus) );

		// printm("HFS+ volume detected\n");
	}
	val += calc_checksum( (char*)&mdb, sizeof(mdb_plus) );
	*checksum = val;

	// printm("HFS-MDB checksum %08lX\n", *checksum );
	return 0;
}

static int
get_mdb_checksum( drive_t *drv, ulong *checksum )
{
	char buf[2048];

	if( !get_hfs_checksum( drv, checksum ) )
		return 0;

	if( !drv->locked || !(drv->flags & bf_force) ) {
		printm("Save session does not support r/w volumes which are not HFS(+)\n");
		return 1;
	}

	/* fallback - read the first four sectors */
	blk_lseek( drv->fd, drv->first_block, SEEK_SET );
	read( drv->fd, &buf, sizeof(buf) );

	*checksum = calc_checksum( buf, sizeof(buf) );
	return 0;
}

#endif
