/**************************************************************
*   
*   Creation Date: <97/06/29 23:33:42 samuel>
*   Time-stamp: <2001/09/30 16:41:51 samuel>
*   
*	<console.c>
*	
*	Initialization of screen and keyboard
*
*  	 Copyright (C) 1997-2001 Samuel Rydh <samuel@ibrium.se>
*
*	Inspired from the ppc-code in XPmac, with the following 
*	copyrights:
*
*	  Copyright 1988-1993 by Apple Computer, Inc., Cupertino, CA
*	  Copyright (c) 1987, 1999 by the Regents of the University of California
*	  Copyright 1987 by Sun Microsystems, Inc. Mountain View, CA.
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "mol_config.h"

/* #define VERBOSE */

#include <signal.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#include <termios.h>
#include <linux/vt.h>          
#include <linux/kd.h>
#include <asm/vc_ioctl.h>
#include <pthread.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <stdarg.h>

#include "mol_semaphore.h"
#include "thread.h"
#include "verbose.h"
#include "console.h"
#include "mac_keys.h"
#include "debugger.h"
#include "timer.h"
#include "driver_mgr.h"
#include "res_manager.h"
#include "mouse_sh.h"
#include "session.h"
#include "async.h"
#include "keycodes.h"
#include "booter.h"
#include "molcpu.h"

SET_VERBOSE_NAME("console");

/* #define USE_ADB_MOUSE_INTERFACE */

#ifdef USE_ADB_MOUSE_INTERFACE
#define mouse_ev PE_mouse_event
#else
#define mouse_ev mouse_event
#endif

#define 	VT_RELEASE_SIGNAL	40
#define 	VT_ACQUIRE_SIGNAL	41

static void 	mouse_init( void );
static void 	mouse_cleanup( void );

static int 	grab_vt( void );
static void 	vt_switch_init( void );
static void 	vt_switch_signal( int sig );
static void 	vt_switch_aevent( int token );
static int 	setup_keyboard( int init_flag );

static void 	key_event( int fd, int events );

static int 	console_fd = -1;
static int 	mouse_fd = -1;
static int 	mouse_handler_id=0, console_handler_id=0;
static int 	vt_orig, vt_no;

static int 	vt_release_token;
static int 	vt_acquire_token;
static int 	has_console=0;
static int 	activate_pending=0;

typedef void (mouse_drvr_proc)( int fd, int dummy_events );
static mouse_drvr_proc		*mouse_drvr = NULL;

static mouse_drvr_proc		mdrvr_ps2;
static mouse_drvr_proc		mdrvr_adb;

typedef struct 
{
	char 		*name;
	char		*device;
	mouse_drvr_proc *drvr;
} drvr_entry_t;

static drvr_entry_t mouse_drvr_table[] = { 
	{ "console", 	NULL,			NULL },
	{ "usb", 	"/dev/input/mice",	mdrvr_ps2 },
	{ "ps2", 	"/dev/input/mice",	mdrvr_ps2 },
	{ "adb", 	"/dev/adbmouse",	mdrvr_adb },
	{ NULL, NULL }
};

enum {
	k_kbd_init=0, k_kbd_raw, k_kbd_text 
};

void (*console_switch_hook)(int vt, int activate) = NULL;


int
console_init( void )
{
	vt_no = get_numeric_res( "vt" ); 	/* -1 if missing */
	if( grab_vt()) {
		LOG("Failed opening a VT\n");
		exit(1);
	}
	printm("Running on VT %d.\n", vt_no);
	register_key_table( kConsoleKeytable, 0, 127 );
	user_kbd_customize( kConsoleKeytable );

	setup_keyboard( k_kbd_init );
	setup_keyboard( k_kbd_text );
	console_printf( "*** Mac-on-Linux full screen video console ***\n");
	
	mouse_init();
	vt_switch_init();

	VPRINT("Console inited\n");
	return 1;
}

void
console_cleanup( void )
{
	struct vt_mode 	vt;
	int err;
	
	VPRINT("console_cleanup\n");

	mouse_cleanup();
	if( console_handler_id )
		delete_async_handler( console_handler_id );

	if( console_fd < 0 )
		return;

	setup_keyboard( k_kbd_text );

	/* Switch back to the original console. First, we make sure
	 * the console has been put in "AUTO" mode so no confirmation
	 * is expected.
	 */
	if( (err=ioctl( console_fd, VT_GETMODE, &vt )) >=0 ){
		vt.mode = VT_AUTO;
		vt.relsig = 0;
		vt.acqsig = 0;
		err = ioctl( console_fd, VT_SETMODE, &vt );
	}
	if( err ) 
		LOG_ERR("VT_GETSETMODE");
	ioctl( console_fd, VT_ACTIVATE, vt_orig );
	ioctl( console_fd, VT_WAITACTIVE, vt_orig );

	VPRINT("mode restored\n");

	/* OK... the console should now be usable again */
	close( console_fd );

	console_fd = -1;
}

void
console_make_safe( void )
{
	setup_keyboard( k_kbd_text );
	console_printf("Exiting MOL\n");
}

void
console_set_gfx_mode( int flag )
{
	setup_keyboard( flag ? k_kbd_raw : k_kbd_text );
}


static void 
mouse_init( void ) 
{
	drvr_entry_t *d = &mouse_drvr_table[0];
	char *s, *mdev=NULL;

	s = get_str_res( "mouse_protocol" );
	mouse_drvr = NULL;
	for( ; s && d->name; d++ ){
		if( strcasecmp( s, d->name ) )
			continue;
		mouse_drvr = d->drvr;
		break;
	}
	if( !d->name )
		LOG("Specified mouse protocol unknown - using the console driver\n");
	else if( !s )
		LOG("No mouse protocol specified - defaults to the console driver\n");
		
	mouse_fd = -1;
	if( mouse_drvr != NULL && (mdev=get_str_res( "mouse_device" )) )
		if( (mouse_fd = open( mdev, O_RDONLY )) < 0 )
			LOG_ERR("Failed opening mouse device '%s'", s );
	if( mouse_drvr && mouse_fd == -1 ) {
		mdev = d->device;
		mouse_fd = open( mdev, O_RDONLY );
	}
	if( mouse_fd != -1 ) {
		printm("Using %s mouse on %s\n", d->name, mdev );
		mouse_handler_id = add_async_handler( mouse_fd, 0/*POLLIN*/, mouse_drvr, 0 /* no SIGIO */ );
	} else {
		printm("Using console mouse driver\n");
		mouse_drvr = NULL;
	}
}

static void
mouse_cleanup( void ) 
{
	if( mouse_handler_id )
		delete_async_handler( mouse_handler_id );

	if( mouse_fd != -1 )
		close( mouse_fd );
	mouse_fd = -1;
}


static void
vt_switch_init( void )
{
	sigset_t	set;
	int		err;
	struct vt_mode 	vt;

	/* Setup signal handler for console switching.
	 */
	has_console = 0;

	vt_release_token = add_aevent_handler( vt_switch_aevent );
	vt_acquire_token = add_aevent_handler( vt_switch_aevent );

	signal(VT_RELEASE_SIGNAL, vt_switch_signal );
	signal(VT_ACQUIRE_SIGNAL, vt_switch_signal );
	sigemptyset( &set );
	sigaddset( &set, VT_RELEASE_SIGNAL );
	sigaddset( &set, VT_ACQUIRE_SIGNAL );
	pthread_sigmask( SIG_UNBLOCK, &set, NULL );

	if( (err=ioctl( console_fd, VT_GETMODE, &vt )) >=0 ){
		vt.mode = VT_PROCESS;
		vt.relsig = VT_RELEASE_SIGNAL;
		vt.acqsig = VT_ACQUIRE_SIGNAL;
		
		err = ioctl( console_fd, VT_SETMODE, &vt );
	}
	if( err<0 ) {
		printm("VT_GETSETMODE");
		exit(1);
	}
	console_handler_id = add_async_handler( console_fd, POLLIN, key_event, 1 /* use SIGIO */ );
}

/* Temporary... perhaps to be removed later */
int
get_console_fd( void )
{
	return console_fd;
}


#define BUFSIZE	64
static int	cmd_pressed=0;
static int	ctr_pressed=0;
static int	alt_pressed=0;

static void 
key_event( int fd, int dummy_events )
{
	unsigned char	events[BUFSIZE];
	int 		i,n;

	n = read(fd, events, BUFSIZE-4 );
	if( n<=0 )
		return;
#if 0
	printm("Key/Mouse event: ");
	for(i=0; i<n; i++ )
		printm("%d %s, ", (events[i] & 0x7f), (events[i] & 0x80)? "Release":"Press");
	printm("\n");
#endif

	for(i=0; i<n; i++ ){
		int vt, tab_released = 0;
		unsigned char ch;

		if( !uses_linux_keycodes() && KEY_DETAIL(events[i]) == MOUSE_ESCAPE ){
			while( i+3 > n ){
				int nn;
				if( (nn = read(fd, events+n, i+3-n )) <=0 )
					break;
				n += nn;
			}
			/* Queue mouse event */
			if( mouse_drvr == NULL ) {
				signed char dx = events[i+2] << 1;
				signed char dy = events[i+1] << 1;
				mouse_ev( dx/2, dy/2, ((events[i+1]&0x80)? 0:1) );
			}
			i+=2;
			continue;
		}
		ch = keycode_to_adb( kConsoleKeytable, events[i]&0x7f, (events[i]&0x80)?0:1 );

		vt = 0;
		switch( KEY_DETAIL(ch) ){
		case KEY_COMMAND:
			cmd_pressed = !(ch & 0x80);
			break;
		case KEY_LEFT_CTRL:
		case KEY_RIGHT_CTRL:
			ctr_pressed = !(ch & 0x80);
			break;
		case KEY_LEFT_OPTION:
			alt_pressed = !(ch & 0x80);
			break;
		case KEY_F1: vt=1; break;
		case KEY_F2: vt=2; break;
		case KEY_F3: vt=3; break;
		case KEY_F4: vt=4; break;
		case KEY_F5: vt=5; break;
		case KEY_F6: vt=6; break;
		case KEY_F7: vt=7; break;
		case KEY_F8: vt=8; break;
		case KEY_F9: vt=9; break;
		case KEY_F10: vt=10; break;
		case KEY_F11: vt=11; break;
		case KEY_F12: vt=12; break;
		case 0x30: /* TAB */
			if ((tab_released = ch & 0x80))
				vt = vt_orig;
			break;
		}
#if 0
		if( vt==12 && !(ch & 0x80) ) {
			save_session();
			continue;
		}
#endif
		if( vt && vt == vt_no )
			vt = 0;
		if( !activate_pending && vt && (ctr_pressed || tab_released) && (cmd_pressed || alt_pressed) ) {
			if( tab_released )
				PE_adb_key_event( 0x30 | 0x80 );
			VPRINT("Switching to vt %d\n", vt);
			activate_pending=1;
			ioctl( console_fd, VT_ACTIVATE, vt );
			continue;
		}

		/* XXX: We should handle the follow events also
		 *
		 * 	0x3f  Middle button down
		 *	0xbf  Middle button up
		 *	0x40  Right bytton down
		 *	0xc0  Right button up
		 */
		PE_adb_key_event( ch );
	}
}


static int 
grab_vt( void )
{
	struct vt_stat 	vts;
	int 		fd;
	char 		vtname[32];
	
	if ((fd = open("/dev/tty0", O_WRONLY)) < 0) {
		printm("failed to open /dev/tty0: %s\n", strerror(errno));
		return 1;
	}

	/* find original vt */
	if (ioctl(fd, VT_GETSTATE, &vts) < 0)
		return 1;
	vt_orig = vts.v_active;

	/* make sure the user specified a free vt */
	if( (vt_no > 0 && (vts.v_state & (1 << vt_no ))) || vt_no > 31 ){
		LOG("vt%d is already in use\n", vt_no );
		vt_no = -1;
	}

	/* find free console? */
	if (vt_no == -1) 
		if ((ioctl(fd, VT_OPENQRY, &vt_no) < 0) || (vt_no == -1)) {
			LOG("Cannot find a free VT\n");
			return 1;
		}
	close(fd);

	/* detach old tty */
	setpgrp();
       	if ((fd = open("/dev/tty",O_RDWR)) >= 0) {
		ioctl(fd, TIOCNOTTY, 0);
		close(fd);
	}

	/* open our vt */
	sprintf(vtname,"/dev/tty%d",vt_no);
	if ((console_fd = open(vtname, O_RDWR | O_NONBLOCK )) < 0) {
		printm("Cannot open %s (%s)\n",vtname, strerror(errno));
		return 1;
	}

	/* set controlling tty */
	if( ioctl( console_fd, TIOCSCTTY, 1 ) < 0 ){
		perrorm("Error setting controlling tty");
		return 1; 
	}
	return 0;
}


static int
setup_keyboard( int how )
{
	static struct termios save_tio;
	static int save_tio_ok=1;
	struct termios tio;
	
	if( !save_tio_ok && how != k_kbd_init )
		return 1;

	switch( how ){
	case k_kbd_init:
		if( tcgetattr( console_fd, &save_tio )< 0 ) {
			perrorm("tcgetattr");
			return 1;
		}
		save_tio_ok = 1;
		break;

	case k_kbd_raw:
		ioctl( console_fd, KDSETMODE, KD_GRAPHICS );
		if( uses_linux_keycodes() )
			ioctl( console_fd, KDSKBMODE, K_MEDIUMRAW );
		else
			ioctl( console_fd, KDSKBMODE, K_RAW );

		tio = save_tio;
		tio.c_iflag = (IGNPAR | IGNBRK) & (~PARMRK) & (~ISTRIP);
		tio.c_oflag = tio.c_lflag = 0;
		tio.c_cflag = CREAD | CS8 | B9600;
		tio.c_cc[VTIME]=0;
		tio.c_cc[VMIN]=1;
		if (tcsetattr(console_fd, TCSANOW, &tio) < 0) {
			perrorm("tcsetattr");
			return 1;
		}
		break;
		
	default:
	case k_kbd_text:
		ioctl( console_fd, KDSETMODE, KD_TEXT );
		ioctl( console_fd, KDSKBMODE, K_XLATE );
		write( console_fd, "\033]R", 3); /* reset palette */
		if( save_tio_ok )
			tcsetattr(console_fd, TCSANOW, &save_tio );
	}
	return 0;
}

int
console_to_front( void )
{
	VPRINT("Console_to_front\n");
	
	if( console_fd <= 0 )
		return 1;
	if (!has_console && !activate_pending ) {
		VPRINT("Sending VT_ACTIVATE\n");
		ioctl( console_fd, VT_ACTIVATE, vt_no );
	}
	return 0;
}


/* signal handler for virtual terminal switching */
static void 
vt_switch_signal( int sig )
{
	if( sig == VT_ACQUIRE_SIGNAL )
		send_aevent( vt_acquire_token );
	else
		send_aevent( vt_release_token );
}


static void
vt_switch_aevent( int token )
{
	if( token == vt_acquire_token ){
		if( has_console ) {
			printm("vt_acquire called, but has_console is true!\n");
			return;
		}
		VPRINT("Grabbing console\n");

		/* setpriority( PRIO_PGRP, 0, 0); */
		has_console = 1;
		ioctl(console_fd, VT_RELDISP, 2);
		if( console_switch_hook )
			console_switch_hook(console_fd, 1);
		if( mouse_handler_id )
			set_async_handler_events( mouse_handler_id, POLLIN );
		
		VPRINT("We are now the owner of the console\n");
	} else {
		if( !has_console ) {
			printm("vt_release called with !has_console\n");
			return;
		}
		VPRINT("Giving up console\n");
		if( mouse_handler_id )
			set_async_handler_events( mouse_handler_id, 0 );

		if( console_switch_hook )
			console_switch_hook(console_fd, 0);
		ioctl(console_fd, VT_RELDISP, 1);
		has_console=0;
		VPRINT("We no longer own the console...\n");
	}

	PE_zap_keyboard();
	cmd_pressed=0;
	alt_pressed=0;
	ctr_pressed=0;

	activate_pending = 0;
}


/************************************************************************/
/*	Mouse drivers							*/
/************************************************************************/

static void 
mdrvr_ps2( int fd, int dummy_events )	/* 0b00yx0321, dx, -dy, [dz] ; x/y = sign of dx/dy, 1=pressed */
{
	signed char buf[3];	

	if( read( fd, buf, 3 ) == 3 )
		mouse_ev( buf[1], -buf[2], buf[0] & kButtonMask );
}

static void
mdrvr_adb( int fd, int dummy_events )	/* 0x10000123, dx, dy - 0=pressed */
{
	signed char buf[3];

	if( read( fd, buf, 3 ) != 3 )
		return;
	buf[0] = ((buf[0] & 1) << 2) | ((buf[0] & 4) >> 2) | (buf[0] & 2);
	mouse_ev( buf[1], -buf[2], ~buf[0] & kButtonMask );
}


/************************************************************************/
/*	Console printing						*/
/************************************************************************/

void
console_printf(const char *fmt,... )
{
#if 0
        va_list args;
	int len;
	char buf[512];

	if( console_fd == -1 )
		return;

	va_start( args, fmt );
	if( (len = vsnprintf( buf, 512, fmt, args )) > 0 )
		write( console_fd, buf, len );
	va_end( args );
#endif
}
