/**************************************************************
*   
*   Creation Date: <97/06/30 15:13:55 samuel>
*   Time-stamp: <2001/03/02 00:14:58 samuel>
*   
*	<symbols.c>
*	
*	Symbols <-> addresses translation interface
*	XXX: A search tree would be nice
*   
*   Copyright (C) 1997, 1999, 2000 Samuel Rydh
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*
**************************************************************/

#include "mol_config.h"

#include "symbols.h"
#include "res_manager.h"
#include "cmdline.h"
#include "wrapper.h"
#include "macasm_syms.h"

#define HASH_TABLE_INIT_SIZE		30000
/* when hash_used > hash_size/REHASH_LIMIT a rehash will occur */
#define REHASH_LIMIT			3

/* --- Data-types ---- */

typedef struct sym_rec
{
	char 		*symbol;
	ulong		addr;
	int		flag;
	int		other;	   /* index of corresponding entry in the other table */
} sym_rec;

enum{
	used=1, deleted=2
};

/* ---- STATIC functions ----- */

static ulong 	find_prime( ulong num );
static int 	str_to_key( char *p );
static int 	addr_to_key( ulong addr );
static void	rehash( void );
static void	do_hash_symbols_from_file( char *filename );

static int 	stoa_index_from_symbol( char *symbol );
static int	atos_index_from_addr( ulong addr );
static void	hash_symbol_from_str( const char *str );

/* ---- STATIC variables ---- */

static sym_rec	*hash_atos=0;		/* address to symbol */
static sym_rec	*hash_stoa=0;		/* symbol to address */

static int	hash_size=0;		/* both tables have the same size */
static int	hash_used;


/**************************************************************
*  symbols_init
*    
**************************************************************/

void 
symbols_init( void )
{
	char **p;
	
	/* init hash-table */
	hash_size = find_prime( HASH_TABLE_INIT_SIZE );
	hash_atos = (sym_rec*)calloc( hash_size, sizeof(sym_rec) );
	hash_stoa = (sym_rec*)calloc( hash_size, sizeof(sym_rec) );
	hash_used = 0;

	/* for now import symbols from default file */
	hash_symbols_from_file( NULL );

	for(p=macasm_syms; *p; p++ )
		hash_symbol_from_str(*p);
}


/**************************************************************
*  symbols_cleanup
*    
**************************************************************/

void 
symbols_cleanup( void ) 
{
	int i;

	/* free entries */
	/* (the other table reuses the same memory for the strings) */
	for(i=0; i<hash_size; i++) {
		if( hash_atos[i].flag & used )
			free( hash_atos[i].symbol );
	}
	
	/* free tables */
	free( hash_atos );
	free( hash_stoa );
}

/**************************************************************
*  delete_all_hash_symbols
*    
**************************************************************/

void 
delete_all_hash_symbols( void ) 
{
	int i;
	sym_rec empty = {0,0,0,0};
	

	for(i=0; i<hash_size; i++) {
		if( hash_atos[i].flag & used ) {
			free( hash_atos[i].symbol );
		}
		hash_atos[i] = hash_stoa[i] = empty;
	}
	hash_used = 0;
}


/**************************************************************
*  atos_index_from_addr
*    returns index or -1
**************************************************************/

static int
atos_index_from_addr( ulong addr ) 
{
	int index;

	if( !hash_size ){
		fprintf(stderr, "Symbol hash table not allocated\n");
		return -1;
	}
	
	index = addr_to_key( addr );
	for( ;; ) {
		sym_rec *p;
		p = &hash_atos[index];
		
		if( p->addr == addr && (p->flag & used) )
			return index;
		if( !(p->flag & used) && !(p->flag & deleted) )
			break;

		index = (index + 1) % hash_size;
	}
	return -1;
}


/**************************************************************
*  symbol_from_addr
*
**************************************************************/

char *
symbol_from_addr( ulong addr ) 
{
	int index;
	
	index = atos_index_from_addr( addr );

	if( index>=0 )
		return hash_atos[index].symbol;
	return 0;
}


/**************************************************************
*  stoa_index_from_symbol
*    -1 if not found
**************************************************************/

static int 
stoa_index_from_symbol( char *symbol ) 
{
	int index;
	if( !hash_size ){
		fprintf(stderr, "Symbol hash table not allocated\n");
		return -1;
	}
	
	index = str_to_key( symbol );
	for( ;; ) {
		sym_rec *p;
		p = &hash_stoa[index];
		
		if( p->symbol && (p->flag & used) && !strcmp(p->symbol,symbol))
			return index;
		if( !(p->flag & used) && !(p->flag & deleted) )
			break;

		index = (index + 1) % hash_size;
	}
	return -1;
}


/**************************************************************
*  addr_from_symbol
*	
**************************************************************/

ulong 
addr_from_symbol( char *symbol ) 
{
	int index;
	
	index = stoa_index_from_symbol( symbol );
	if( index>=0 )
		return hash_stoa[index].addr;
	return 0;
}


/**************************************************************
*  hash_symbol
*
**************************************************************/

void 
hash_symbol( char *symstr, ulong addr ) 
{
	int index, atos_index;
	char *symbol;

	/* check if a rehash is needed */
	if( hash_used > hash_size/REHASH_LIMIT )
		rehash();

	symbol = (char*) malloc( strlen(symstr)+1 );
	strcpy( symbol,symstr );
	
	index = addr_to_key( addr );
	/* hash in atos */
	for( ;; ) {
		sym_rec *p = &hash_atos[index];

		if( p->addr == addr && (p->flag & used) ) {
			/* entry alredy exists! */
			free( p->symbol );
			p->symbol = symbol;
			p->flag = used;
			/* delete symbol in the other table */
			hash_stoa[p->other].flag = deleted;
			break;
		}
		if( !(p->flag & used) ) {
			/* new entry */
			p->addr = addr;
			p->symbol = symbol;
			p->flag = used;
			break;
		}
		index = (index + 1) % hash_size;
	}
	atos_index = index;

	/* hash in stoa */
	index = str_to_key( symstr );
	for( ;; ) {
		sym_rec *p;
		p = &hash_stoa[index];
		
		if( p->addr == addr || !(p->flag & used) ) {
			p->symbol = symbol;
			p->flag = used;
			p->addr = addr;
			p->other = atos_index;
			break;
		}
		index = (index + 1) % hash_size;
	}
	hash_atos[atos_index].other = index;
	
	hash_used++;
}


/**************************************************************
*  str_to_key
*    
**************************************************************/

static int 
str_to_key( char *p ) 
{
	int base=3;
	ulong key=0;
	
	while( *p ) {
		key = key + ((int)*p) * base;
		base = (base << 2) + base;
		p++;
	}
	return( key % hash_size );
}


/**************************************************************
*  addr_to_key
*    
**************************************************************/

static int 
addr_to_key( ulong addr ) 
{
	return( (addr>>2) % hash_size );
}

/**************************************************************
*  delete_symbol
*    -1 if failure
**************************************************************/

int 
delete_hash_symbol( char *symbol ) 
{
	int index = stoa_index_from_symbol( symbol );
	if( index<0 )
		return -1;

	free( hash_stoa[index].symbol );
	hash_stoa[index].flag = deleted;
	hash_atos[ hash_stoa[index].other ].flag = deleted;
	hash_used--;
	
	return 0;
}


/**************************************************************
*  delete_hash_addr
*    -1 if failure
**************************************************************/

int 
delete_hash_addr( ulong addr ) 
{
	int index = atos_index_from_addr( addr );
	if( index<0 )
		return -1;

	free( hash_atos[index].symbol );
	hash_atos[index].flag = deleted;
	hash_stoa[ hash_atos[index].other ].flag = deleted;
	hash_used--;
	
	return 0;
}


/**************************************************************
*  rehash
*    
**************************************************************/

static void 
rehash( void )
{
	int i;
	sym_rec *p;
	int old_hash_size = hash_size;
	sym_rec *old_hash_atos = hash_atos;
	sym_rec *old_hash_stoa = hash_stoa;

	/* allocate new hashtables */
	hash_size = find_prime( old_hash_size * 2 );
	hash_used = 0;
	hash_atos = (sym_rec*)calloc( hash_size, sizeof(sym_rec) );
	hash_stoa = (sym_rec*)calloc( hash_size, sizeof(sym_rec) );

	p = old_hash_atos;
	for(i=0; i<old_hash_size; i++,p++ ) {
		if( p->flag & used ) {
			hash_symbol( p->symbol, p->addr );
		}
		free( p->symbol );
	}

	/* the two tables share the same string-memory */
	free( old_hash_atos );
	free( old_hash_stoa );
}


/**************************************************************
*  hash_symbols_from_file
*    If filename == NULL, obtain filename(s) from symfile
*    resources.
**************************************************************/

void 
hash_symbols_from_file( char *filename ) 
{
	if( filename )
		do_hash_symbols_from_file( filename );
	else {
		int i=0;
		while( (filename = get_str_res_ind("symfile",i++,0)) != NULL )
			do_hash_symbols_from_file( filename );
	}
}

static void
hash_symbol_from_str( const char *str )
{
	char buf2[102];
	char *p, *s = strdup(str);
	ulong addr;
	int valid = 1;
	
	if( s[0]=='*' )
		valid = sscanf(s+1,"%lx %100s", &addr, buf2 ) == 2;
	else {
		valid = (sscanf(s+1,"0x%lx %100s,", &addr, buf2) == 2)
			|| (sscanf(s+1, "%lx %100s", &addr, buf2) == 2 );
	}
	buf2[sizeof(buf2)-1]=0;
	if( (p=strchr( buf2, '(' )) )
		*p = 0;
	if( valid )
		hash_symbol( buf2, addr );
	free(s);
}

static void
do_hash_symbols_from_file( char *filename )
{
	char 	buf[200];
	FILE *f;
	
	if( !(f=fopen( filename, "r" )) ) {
		printm("Symbol file '%s' not found\n",filename );
		return;
	}
	printm("Hashing symbols from '%s'\n", filename);

	while( fgets( buf, sizeof(buf), f ) )
		hash_symbol_from_str(buf);
	fclose( f );
}


/**************************************************************
*  export_symbols
*    
**************************************************************/

void 
export_symbols( char *filename ) 
{
	int	i;
	FILE 	*f;
	
	if( filename == NULL ) {
		filename = get_str_res("export_symfile");
		if( !filename ) {
			printm("Please specify a filename\n");
			return;
		}
	}
	if( !(f=fopen( filename, "w" )) ) {
		printm("File '%s' could not be created\n",filename );
		return;
	}
	for( i=0; i<hash_size; i++)
		if( hash_atos[i].flag & used )
			fprintf(f,"*%08lX\t%s\n",hash_atos[i].addr, hash_atos[i].symbol );
	
	fclose( f );
}



/**************************************************************
*  find_prime
*    Find a prime bigger than num
*
*    This is _really_ a naive method, but it's all we need
**************************************************************/

static ulong 
find_prime( ulong num ) 
{
	ulong p, i;

	p = (num % 2)? num : num+1;

	for( ;; ) {
		for(i=3; i<p; i+=2 )
			if( !(p%i) )
				break;
		if( i==p )
			break;
		p+=2;
	}
	return p;
}

