/* 
 *   Creation Date: <1999/11/07 19:02:11 samuel>
 *   Time-stamp: <2001/01/31 23:07:29 samuel>
 *   
 *	<mem.c>
 *	
 *	OF Memory manager
 *   
 *   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

/* #define VERBOSE */

#include "booter.h"
#include "verbose.h"
#include "ofmem.h"
#include "res_manager.h"
#include "debugger.h"
#include "memory.h"
#include "mac_registers.h"
#include "molcpu.h"
#include "os_interface.h"
#include "promif.h"
#include "wrapper.h"
#include "session.h"

#include <asm/mmu.h>
/* #include <sys/param.h> */

SET_VERBOSE_NAME("ofmem");

extern ulong of_entry_start[1];
extern ulong of_entry_end[1];


#define MAX_NUM_TRANS		70

/* segmnet number range to use */
#define SEGR_BASE		0x400

/* #define REVERSE_ALLOC */
#define MEM_ALLOC_START		0x400000

/* translations table */
typedef struct
{
	ulong	mvirt;
	ulong	size;
	ulong	mphys;
	int	mode;		/* for protection bits etc */
} ofmem_trans_t;

typedef struct 
{
	ulong	addr;
	ulong	size;
} mem_pair_t;

typedef struct 
{
	int		nent;
	ulong		size;
	mem_pair_t	tab[ MAX_NUM_TRANS ];
} mem_tab_t;

/* translations table */
static mem_tab_t	ptab[1], vtab[1];
static ofmem_trans_t	trans[ MAX_NUM_TRANS ];
static int		num_trans;


static int 		is_free( ulong virt, ulong size, mem_tab_t *table );
static void		add_entry( ulong addr, ulong size, mem_tab_t *table );
static void		of_exceptions_init( void );
static  void		hash_translation( ulong ea, ulong mphys, ulong size, int mode );

static int 		cmd_oftrans( int, char **);

/************************************************************************/
/*	Init / cleanup							*/
/************************************************************************/

void
ofmem_init( void )
{
	mol_device_node_t *dn;
	
	memset( &vtab, 0, sizeof( mem_tab_t ) );
	memset( &ptab, 0, sizeof( mem_tab_t ) );	
	num_trans = 0;

	ptab->size = ram.size;

	if( !loading_session() )
		of_exceptions_init();

	/* Fix the /memory/reg property */
	dn = prom_find_dev_by_path( "/memory" );
	if( !dn ){
		LOG("----> Node /memory is missing!\n");
	} else {
		ulong tab[8];
		memset( tab, 0, sizeof(tab) );
		tab[0] = ram.mbase;
		tab[1] = ram.size;
		prom_add_property( dn, "reg", (char*)tab, sizeof(tab) );
	}

	add_cmd( "oftrans", "oftrans\nshow ofmem translations\n", -1, cmd_oftrans );
}

void
ofmem_cleanup( void )
{
}


/************************************************************************/
/*	Implementation							*/
/************************************************************************/

static inline int
def_memmode( ulong phys )
{
	/* XXX: Guard bit not set as it should! */
	if( phys <= ram.size || phys >= rom.mbase )
		return 0x02;	/*0xa*/	/* wim GxPp */

	return 0x6a;		/* WIm GxPp, I/O */
}


/* if align != 0, phys is ignored. Returns -1 on error */
ulong
ofmem_claim_phys( ulong phys, ulong size, ulong align )
{
	if( !align ){
		if( !is_free( phys, size, ptab ) ) {
			printm("Non-free physical memory claimed!\n");
			return -1;
		}
		add_entry( phys, size, ptab );
		return phys;
	}

	/* for simplicity, everything is at least page aligned */
	if( align < 0x1000 )
		align = 0x1000;

#ifdef REVERSE_ALLOC
	phys = ram.size - size;
	phys &= ~(align-1);

	for( ; (int)phys >= 0 ; phys -= align )
		if( is_free( phys, size, ptab ) )
			break;
#else
	phys = MEM_ALLOC_START+align-1;
	phys &= ~(align-1);

	for( ; (int)phys < ram.size; phys += align )
		if( is_free( phys, size, ptab ) )
			break;
#endif

	if( (int)phys <= 0 ) {
		printm("No free memory area could be found...\n");
		return -1;
	}
	add_entry( phys, size, ptab );
	return phys;
}

ulong
ofmem_claim_virt( ulong virt, ulong size, ulong align )
{
/*	printm("ofmem_claim_virt @%08lX (size %08lX)\n", virt, size );  */
	if( !align ){
		if( !is_free( virt, size, vtab ) ) {
			printm("Non-free virtual space claimed!\n");
			return -1;
		}
		add_entry( virt, size, vtab );
		return virt;
	}

	/* page align for simplicity */
	if( align < 0x1000 )
		align = 0x1000;

#ifdef REVERSE_ALLOC
	/* try to keep things 1-1, thus try below ram.size */
	virt = ram.size - size;
	virt &= ~(align-1);

	for( ; (int)virt >= 0 ; virt -= align )
		if( is_free( virt, size, vtab ) )
			break;
#else
	virt = MEM_ALLOC_START + align-1;
	virt &= ~(align-1);

	for( ; (int)virt < ram.size; virt += align )
		if( is_free( virt, size, vtab ) )
			break;
#endif

	if( (int)virt <= 0 ) {
		printm("No free virtual memory area could be found...\n");
		return -1;
	}
	add_entry( virt, size, vtab );
	return virt;
}

/* allocate both physical and virtual space and add a translation */
ulong
ofmem_claim( ulong addr, ulong size, ulong align )
{
	ulong	virt, phys;
	ulong offs = addr & 0xfff;
	
	/* let pages be the smallest grain (for simplicity) */
	size = size + (addr & 0xfff);
	size = (size + 0xfff) & ~0xfff;
	addr &= ~0xfff;
	if( align && align < 0x1000 )
		align = 0x1000;

	virt = phys = 0;
	if( !align ){
		if( is_free( addr, size, vtab ) && is_free( addr, size, ptab ) ){
			ofmem_claim_phys( addr, size, 0 );
			ofmem_claim_virt( addr, size, 0 );
			virt = phys = addr;
		} else {
			printm("**** ofmem_claim failure!\n");
			return -1;
		}
	} else {
		phys = ofmem_claim_phys( addr, size, align );
		virt = ofmem_claim_virt( addr, size, align );
		if( phys == -1 || virt == -1 ) {
			printm("ofmem_claim failed\n");
			return -1;
		}
		/* printm("...phys = %08lX, virt = %08lX, size = %08lX\n", phys, virt, size ); */
	}
	/* printm("...free memory found... phys: %08lX, virt: %08lX, size %lX\n", phys, virt, size ); */
	ofmem_map( phys, virt, size, def_memmode( phys ) );
	return virt + offs;
}

int
ofmem_map( ulong phys, ulong virt, ulong size, int mode )
{
	ulong start, len, cur;
	int i;

/*	printm("ofmem_map: (virt->phys) %08lX ----> %08lX (size %08lX, mode 0x%02X)\n", 
		virt, phys, size, mode ); */
       
	if( phys & 0xfff || virt & 0xfff || size & 0xfff ) {
		printm("ofmem_map: Bad parameters (%08lX %08lX %08lX)\n", phys, virt, size );
		phys &= ~0xfff;
		virt &= ~0xfff;
		size = (size + 0xfff) & ~0xfff;
	}

	/* Claim any unclaimed virtual memory in the range */
	for( cur=virt, len=0, start=virt; cur < virt + size; cur += 0x1000 ) {
		if( !is_free( cur, 0x1000, vtab )) {
			if( len ) {
				printm("extra vclaim, virt: %08lX, size %lX\n", start, len );
				ofmem_claim_virt( start, len, 0 );
			}
			len = 0;
			start = cur + 0x1000;
			continue;
		}
		len += 0x1000;
	}
	if( len ) {
		printm("extra vclaim, virt: %08lX, size %lX\n", start, len );
		ofmem_claim_virt( start, len, 0 );
	}

	for(i=0; i<num_trans; i++ ){
		if( virt+size > trans[i].mvirt && virt < trans[i].mvirt + trans[i].size ) {
			printm("ofmem_map: Conflicting mapping detected --\n");
			printm("-- mapping inserted with reduced size=0x1000...\n");
			size = 0x1000;
			/* stop_emulation();*/
			break;
		}
	}
	if( num_trans >= MAX_NUM_TRANS ) {
		printm("ofmem_map: MAX_NUM_TRANS exceeded\n");
		return -1;
	}
	trans[num_trans].mvirt = virt;
	trans[num_trans].size = size;
	trans[num_trans].mphys = phys;
	trans[num_trans].mode = (mode == -1)? def_memmode(phys) : mode;
	num_trans++;

	hash_translation( virt, phys, size, mode );
	return 0;
}


/* virtual -> physical. */
ulong
ofmem_translate( ulong virt, ulong *mode )
{
	int i;
	for(i=0; i<num_trans; i++ ){
		if( virt >= trans[i].mvirt && virt < trans[i].mvirt + trans[i].size ){
			if( mode )
				*mode = trans[i].mode;
			return trans[i].mphys + virt - trans[i].mvirt;
		}
	}
	return -1;
}

void
ofmem_release( ulong virt, ulong size )
{
	VPRINT("----> ofmem_release unmplemented\n");
}


static int
is_free( ulong addr, ulong size, mem_tab_t *t )
{
	int i;

	for( i=0; i<t->nent; i++ )
		if( addr+size > t->tab[i].addr && addr < t->tab[i].addr + t->tab[i].size )
			return 0;
	if( t->size && addr+size > t->size )
		return 0;
	return 1;
}


static void
add_entry( ulong addr, ulong size, mem_tab_t *t )
{
	if( t->nent >= MAX_NUM_TRANS ) {
		printm("add_entry: MAX_NUM_TRANS exceeded!\n");
		return;
	}
	t->tab[t->nent].addr = addr;
	t->tab[t->nent++].size = size;
}

static int 
cmd_oftrans( int argc, char **argv )
{
	int i;
	if( argc != 1 )
		return -1;

	for(i=0; i<num_trans; i++ ){
		printm(" %08lX -> %08lX [size %08lX],   mode %02x\n", 
		       trans[i].mvirt, trans[i].mphys, trans[i].size, trans[i].mode );
	}

	printm("\n\nPHYS:\n");
	for(i=0; i<ptab->nent; i++ )
		printm(" %08lX, size %08lX\n", ptab->tab[i].addr, ptab->tab[i].size );

	printm("\n\nVIRT:\n");
	for(i=0; i<vtab->nent; i++ )
		printm(" %08lX, size %08lX\n", vtab->tab[i].addr, vtab->tab[i].size );
	return 0;
}


/************************************************************************/
/*	MMU Exception handlers (used during the OF-phase)		*/
/************************************************************************/

/* exports from ofglue.S */	
extern ulong of_trap_start[1];
extern ulong of_trap_end[1];

static int osip_of_trap( int sel, int *dummy_params );
static void handle_page_fault( ulong dar, ulong dsisr );

#define HASH_SIZE	(2<<15)
static ulong 		s_hashbase;

static void
of_exceptions_init( void )
{
	ulong offs;
	int i;

	os_interface_add_proc( OSI_OF_TRAP, osip_of_trap );

	/* claim memory and copy trap handlers */
	if( ofmem_claim_phys( 0, 0x1000, 0 ) == -1 ) {
		LOG("----> Claim failure (of_exceptions_init)!\n");
		exit(1);
	}
	for( offs=0x100; offs<=0x2f00; offs+=0x100 )
		memcpy( ram.lvbase+offs, of_trap_start, (ulong)of_trap_end-(ulong)of_trap_start );

	/* calim memory for 128K page tables */
	s_hashbase = ofmem_claim_phys( 0, HASH_SIZE, HASH_SIZE );

	/* try to maintain 1-1 mappings for simplicity */
	ofmem_claim_virt( s_hashbase, HASH_SIZE, 0 );	

	if( s_hashbase == -1 ){
		LOG("----> Failed to allocate pagetables\n");
		exit(1);
	}
	/* VPRINT("Hashtable at %08lX\n", s_hashbase ); */
	memset( ram.lvbase + s_hashbase, 0, HASH_SIZE );

	for(i=0; i<16; i++)
		mregs->segr[i] = (0x20 << 24) | (SEGR_BASE+i);	/* vsid + user-state protection key */
	mregs->spr[ S_SDR1 ] = s_hashbase | ((HASH_SIZE-1) >> 16);
	mregs->spr[ S_SPRG0 ] = 0;	/* we use the first 256 bytes for register saving */
	_spr_changed();
}

static int
osip_of_trap( int sel, int *dummy_params )
{
	int trap = mregs->nip >> 8;
#if 0
	printm("Trap 0x%x at %08lX\n", trap, mregs->spr[ S_SRR0 ] );
	stop_emulation();
#endif
	switch( trap ){
	case 0x3: /* DSI */
		handle_page_fault( mregs->spr[S_DAR], mregs->spr[S_DSISR] );
		break;
	case 0x4: /* ISI */
		handle_page_fault( mregs->spr[S_SRR0], mregs->spr[S_SRR1] );
		break;
	}
	return 0;
}

static void
handle_page_fault( ulong dar, ulong dsisr )
{
	ulong 	phys, mode;

	VPRINT("Page fault: dar %08lX, dsisr %08lX\n", dar, dsisr );

	/* we only handle true page faults */
	if( !(dsisr & BIT(1)) ) {
		LOG("Strange page fault\n");
		stop_emulation();
		return;
	}
	_emulate_tlbie( dar );

	if( (phys = ofmem_translate( dar, &mode )) == -1 ) {
		VPRINT("Virtual address %08lX is unmapped!\n", dar );
		/* this is probably IO - simply map it 1-1 */
		phys = dar;
		mode = 0;
		/* stop_emulation(); */
	}

	hash_translation( dar & ~0xfff, phys & ~0xfff, 0x1000, mode );
}

static void
hash_translation( ulong ea, ulong mphys, ulong size, int mode )
{
	static int next_slot=0;
	ulong 	hash1, *upte;
	ulong	vsid;
	mol_PTE_t *pte;
	int	i;

/*	printm("hash_translation: %08lX -> %08lX, size %08lX, mode %02X\n", ea, mphys, size, mode );*/
	
	if( ea & 0xfff || mphys & 0xfff || size & 0xfff ) {
		printm("hash_translation: BAD ea (%08lX) or lphys (%08lX)\n", ea, mphys );
		return;
	}

	for( ; size>0 ; size-=0x1000, mphys+=0x1000, ea+=0x1000 ) {
		_emulate_tlbie( ea );

		vsid = (ea>>28) + SEGR_BASE;
		
		/* we always use the primary PTEG for simplicity */
		hash1 = vsid;
		hash1 ^= (ea >> 12) & 0xffff;
		hash1 &= (HASH_SIZE-1) >> 6;
	
		pte = (mol_PTE_t*)( ram.lvbase + s_hashbase + (hash1 <<6) );
		for(i=0; i<8; i++) {
			if( !pte[i].v )
				break;

			/* XXX: This check is not strict enough... */
			if( pte[i].api == ((ea>>22) & 0x3f ) ) {
				/* previous translation detected */
				printm("Replacing old translation (EA %08lX, old-phys %08X)\n", ea, (pte[i].rpn<<12) );
				break;
			}
		}

		if( i==8 ) {
			printm("---> Warning: PTE table overflow\n");
			i = (next_slot++) % 8;
		}
		pte = &pte[i];
		upte = (ulong*)pte;

		upte[0] = upte[1] = 0;
		pte->api = (ea >> 22) & 0x3f;
		pte->vsid = vsid;
		pte->v = 1;
		pte->rpn = mphys >> 12;

		upte[1] |= mode; 
	}
}

