/* 
 *   Creation Date: <1999/11/02 01:17:09 samuel>
 *   Time-stamp: <2001/01/31 23:02:14 samuel>
 *   
 *	<of1275.c>
 *	
 *	OF interface implementation (IEEE 1275)
 *   
 *   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

/* The draft 1275 standard is available from ftp://playground.sun.com/pub/p1275/coredoc/ */

#include "mol_config.h"
#include <sys/param.h>
#include "verbose.h"

#include "ofmem.h"
#include "res_manager.h"
#include "debugger.h"
#include "memory.h"
#include "mac_registers.h"
#include "os_interface.h"
#include "promif.h"
#include "phandles.h"
#include "rtas.h"
#include "nvram.h"
#include "wrapper.h"


SET_VERBOSE_NAME( "of1275" );

/* #define MEM_DEBUG(args...) printm( ">>>> " ## args) */
#define MEM_DEBUG(args...)

/* #define VGETPROP */

static int ofvirt_to_lvptr( ulong ofvirt, char **lvptr );
static char* my_strncpy( char *dest, const char *src, size_t n );

typedef struct prom_args {
        const char *service;
        int nargs;
        int nret;
        ulong args[10];			/* MAX NUM ARGS! */
} prom_args_t;

typedef int of_proc( ulong args[], ulong ret[] );

typedef struct of_cmd
{
	char 	*service;
	of_proc	*proc;
	int	nargs;
	int	nret;
	int	translate_args;
} of_cmd_t;

#define A0		0x1
#define A1		0x2
#define A2		0x4
#define A3		0x8

static of_proc		of_nop;
static of_proc 		of_finddevice, of_canon;
static of_proc 		of_getprop, of_getproplen, of_setprop;
static of_proc 		of_open, of_read, of_write, of_seek;
static of_proc 		of_claim, of_release;
static of_proc 		of_call_method;
static of_proc		of_inst_to_path, of_inst_to_pack, of_pack_to_path;
static of_proc		of_peer, of_parent, of_child, of_nextprop;
static of_proc		of_exit, of_quiesce;

static of_cmd_t cmd_table[] = {
/* 6.3.2.1 Client interface */
	{"test", 		of_nop, 	1,1, 	A0},	/* char *name -- int missing */

/* 6.3.2.2 Device tree */
	{"peer", 		of_peer, 	1,1, 	0},	/* int phandle -- int sibling_phandle */
	{"child", 		of_child, 	1,1, 	0},	/* int phandle -- int child_phandle */
	{"parent", 		of_parent, 	1,1, 	0},	/* int phandle -- int parent_phandle */
	{"getproplen", 		of_getproplen, 	2,1, 	A1},	/* int phandle, char *name -- int proplen */
	{"getprop", 		of_getprop, 	4,1, 	A1|A2},	/* int phandle, char *name, void *buf, int buflen -- int size */
	{"nextprop", 		of_nextprop, 	3,1, 	A1|A2},	/* int phandle, char *previous, void *buf -- int flag */
	{"setprop",		of_setprop,	4,1,	A1|A2},	/* int phandle, char *name, void *buf, int len -- int size */
	{"canon",		of_canon,	3,1,	A0|A1},	/* char *device_specifier, void *buf, int buflen -- int length */
	{"finddevice", 		of_finddevice, 	1,1, 	A0},	/* char *device_specifier -- int phandle */
	{"instance-to-package", of_inst_to_pack,1,1, 	0},	/* int ihandle -- int phandle */
	{"instance-to-path",	of_inst_to_path,3,1,	A1 },	/* int ihandle, void *buf, int buflen -- int length */
	{"package-to-path",	of_pack_to_path,3,1,	A1 },	/* int phandle, void *buf, int buflen -- int length */
	{"call-method",		of_call_method,	-1,-1,	A0 },	/* char *method, int ihandle -- int args[] */

/* 6.3.2.3 Device I/O */
	{"open",		of_open,	1,1,	A0 },	/* char *device_specifier -- int ihandle */
	{"close",		of_nop,		1,0,	0 },	/* int ihandle -- */
	{"read",		of_read,	3,1,	0 },	/* int ihandle, void *addr, int len -- int actual */
	{"write",		of_write,	3,1,	0 },	/* int ihandle, void *addr, int len -- int actual */
	{"seek",		of_seek,	3,1,	0 },	/* int ihandle, int pos_hi, int pos_lo -- int status */

/* 6.3.2.4 Memory */
	{"claim",		of_claim,	3,1,	0 },	/* void *virt, int size, int align -- void *baseaddr */
	{"release",		of_release,	2,0,	0 },	/* void *virt, int size -- */

/* 6.3.2.5 Control transfer */
	{"boot",		of_nop,		1,0,	A0 },	/* char *bootspec -- */
	{"enter",		of_nop,		0,0,	0 },	/* -- */
	{"exit",		of_exit,	0,0,	0 },	/* -- */
	{"chain",		of_nop,		4,0,	0/*?*/},/* void *virt, int size, void *entry, void *args, int len -- */

/* 6.3.2.6 User interface */
	{"interpret",		of_nop,		-1,-1,	A0 },	/* char *cmd args[] -- args[] */
	{"set-callback",	of_nop,		1,1,	0 },	/* void *newfunc -- void *oldfunc */
	{"set-symbol-lookup",	of_nop,		2,0,	0/*?*/},/* void *sym_to_value, void *value_to_sym, -- */

/* 6.3.2.7 Time */
	{"milliseconds",	of_nop,		0,1,	0 },	/* -- int ms */

/* Documented in one of the supplements? */
	{"quiesce",		of_quiesce,	0,0,	0 },	/* -- */
	
	{NULL, 0, 0, 0,0 }
};


/************************************************************************/
/*	NODE METHODS							*/
/************************************************************************/

static int of_call_method_( ulong args[], ulong ret[], int nargs, int nret );

typedef int met_proc( ulong ihandle, ulong args[], ulong ret[] );

static met_proc		met_mem_claim;
static met_proc		met_mmu_claim, met_mmu_translate, met_mmu_map;
static met_proc		met_rtas_instantiate;
static met_proc		met_nop;
static met_proc		met_nvram_seek, met_nvram_read, met_nvram_write, met_nvram_size;
static met_proc		met_stdout_write;

typedef struct
{
	char 		*method;
	int		molih;
	met_proc	*proc;
	int		nargs;
	int		nret;
	int		translate_args;
} of_method_t;

/* node methods (invoked by call-method) */
static of_method_t method_table[] = {
	{"claim",		molih_memory, 	met_mem_claim,		3,1,	0 },	/* ( phys, size, align --- base ) */
	{"release",		molih_memory, 	met_nop,		2,0,	0 },	/* ( phys, size --- ) */

	{"claim",		molih_mmu, 	met_mmu_claim,		3,1,	0 },	/* ( virt, size, align ---  base ) */
	{"release",		molih_mmu, 	met_nop,		2,0,	0 },	/* ( virt, size --- ) */
	{"map",			molih_mmu, 	met_mmu_map,		4,-1,	0 },	/* ( phys, virt, size, mode -- [ret] ) */
	{"unmap",		molih_mmu, 	met_nop,		2,0,	0 },	/* ( virt, size --- ) */
	{"translate",		molih_mmu,	met_mmu_translate,	1,3,	0 },	/* ( virt --- phys mode true ) */

	{"instantiate-rtas",	molih_rtas,	met_rtas_instantiate,	1,1,	0 },	/* ( physbase -- rtas_callback ) */

	{"seek",		molih_nvram,	met_nvram_seek,		2,1,	0 },	/* ( pos_hi, pos_lo -- status ) */
	{"size",		molih_nvram,	met_nvram_size,		0,1,	0 },	/* ( -- size ) */
	{"read",		molih_nvram,	met_nvram_read,		2,1,	A0 },	/* ( addr, len -- actual ) */
	{"write",		molih_nvram,	met_nvram_write,	2,1,	A0 },	/* ( addr, len -- actual ) */

	{"write",		molih_stdout,	met_stdout_write,	2,1,	A0 },	/* ( addr, len -- actual ) */	

	{NULL, 0, 0, 0,0 }
};


/************************************************************************/
/*	OS interface proc						*/
/************************************************************************/

prom_args_t *s_pargs;
char *s_service;

/* -1 error, 0=RAM, 1=ROM */
static int
ofvirt_to_lvptr( ulong ofvirt, char **lvptr )
{
	ulong mphys = ofmem_translate( ofvirt, NULL );
	if( mphys == -1 )
		return -1;

	return mphys_to_lvptr( mphys, lvptr );
}


int
osip_of_interface( int sel, int *params )
{
	prom_args_t 	*args;
	char 		*service;
	int 		i, j;
	ulong		largs[10];	/* passed arguments must be local */

	if( ofvirt_to_lvptr( (ulong)params[0], (void*)&args ) < 0 )
		goto bail;
	if( ofvirt_to_lvptr( (ulong)args->service, (void*)&service ) < 0 )
		goto bail;

	if( args->nargs <0 || args->nret < 0 ) {
		LOG("nargs or nret negative (%d,%d)\n",args->nargs, args->nret );
		return 0;
	}

	for( i=0; cmd_table[i].service; i++ ){
		if( strcmp( cmd_table[i].service, service ) )
			continue;
		if( (cmd_table[i].nargs == args->nargs || cmd_table[i].nargs == -1 ) &&
		    (cmd_table[i].nret == args->nret || cmd_table[i].nret == -1 )) 
		{
			for( j=0; j<args->nargs; j++ ){
				largs[j] = args->args[j];
				if( cmd_table[i].translate_args & (1<<j) && args->args[j] ){
					if( ofvirt_to_lvptr( (ulong)args->args[j], (char**)&largs[j] ) < 0 )
						goto bail;
				}
			}
#if 0
			stop_emulation();
#endif
			s_pargs = args;
			s_service = service;
			return (*cmd_table[i].proc)( largs, &args->args[args->nargs] );
		}
	}
	printm( "Unimplemented service '%s' [%d,%d]\n", service, args->nargs, args->nret );
	return 0;

bail:
	LOG("Address lookup failure\n");
	return 0;
}

/************************************************************************/
/*	IMPLEMENTATION							*/
/************************************************************************/

#define PHANDLE_TO_DN( ph ) \
	({ mol_device_node_t *tmp = prom_phandle_to_dn( ph ); \
	  if( !tmp ) { printm("----> '%s': Invalid phandle %08lX\n", s_service, ph ); /*stop_emulation();*/ } \
	  tmp; })

static int 
of_nop( ulong args[], ulong ret[] ) 
{
	int i;
	printm("*** nop_'%s' [%d,%d]  ", s_service, s_pargs->nargs, s_pargs->nret );

	for(i=0; i<s_pargs->nargs; i++ )
		printm(" %08lX", args[i] );
	printm("\n");

	return 0;
}

/* phandle -- int sibling_phandle */
static int 
of_peer( ulong args[], ulong ret[] ) 
{
	mol_device_node_t *dn=0, *dn2;

	ret[0] = -1;
	if( args[0] ) {
 		if( !(dn = PHANDLE_TO_DN(args[0])))
			return 0;
		dn2 = dn->sibling;
	} else
		dn = dn2 = prom_get_root();

	ret[0] = prom_dn_to_phandle( dn2 );
	/* printm("of_peer '%s' -- '%s'\n", dn->full_name, dn2 ? dn2->full_name : "last" ); */
	return 0;
}

/* int phandle -- int parent_phandle */
static int 
of_parent( ulong args[], ulong ret[] ) 
{
	mol_device_node_t *dn=PHANDLE_TO_DN(args[0]);

	ret[0] = -1;
	if( !dn )
		return 0;
	ret[0] = prom_dn_to_phandle( dn->parent );
	/* printm("of_parent '%s' -- '%s'\n", dn->full_name, dn->parent ? dn->parent->full_name : "root" ); */
	return 0;
}

/* int phandle -- int child_phandle */
static int 
of_child( ulong args[], ulong ret[] ) 
{
	mol_device_node_t *dn=PHANDLE_TO_DN(args[0]);

	ret[0] = -1;
	if( !dn )
		return 0;

	ret[0] = prom_dn_to_phandle( dn->child );
	/* printm("of_child '%s' -- '%s'\n", dn->full_name, dn->child ? dn->child->full_name : "last" );  */
	return 0;
}

/* char *device_specifier --- int phandle */
static int 
of_finddevice( ulong args[], ulong ret[] ) 
{
	ret[0] = prom_dn_to_phandle( prom_find_dev_by_path( (char*)args[0] ) );
	if( !ret[0]  )
		ret[0] = -1;

	/* printm("of_finddevice %s [ %08lX ]\n", (char*)args[0], ret[0] );  */
	return 0;
}


/* int phandle, char *previous, void *buf -- int flag */
static int 
of_nextprop( ulong args[], ulong ret[] ) 
{
	char *next;
	mol_device_node_t *dn = PHANDLE_TO_DN( args[0] );

	if( !dn ) {
		ret[0] = -1;
		return 0;
	}
	ret[0] = 0;
	next = NULL;
	if( args[1] && strlen((char*)args[1]) && !prom_get_property( dn, (char*)args[1],NULL ) ) {
		ret[0] = -1;
	} else {
		if( (next = prom_next_property( dn, (char*)args[1] )) != NULL )
			ret[0] = 1;
	}
 	/* printm("of_nextprop: %s -- %s\n", (char*)args[1], next );   */

	if( args[2] && next )
		my_strncpy( (char*)args[2], next, 32 );
	else if( args[2] )
		*(char*)args[2] = 0;
	return 0;
}

/* int phandle, char *name, void *buf, int buflen --- int size */
static int 
of_getprop( ulong args[], ulong ret[] ) 
{
	char *buf;
	int len;
	mol_device_node_t *dn;

	/* Sometimes args[0]==0. This might be due to a bug somewhere.
	 * Possibly it is correct to return the root node, but I'm not sure.
	 */
	if( args[0] ) {
		dn = PHANDLE_TO_DN( args[0] );
	} else {
		static int warned=0;
		if( !warned++ )
			printm("************ WARNING, phandle == 0 ***********\n");
		dn = prom_get_root();
	}

	ret[0] = -1;
	if( !dn ) {
		/* printm("...getprop %s\n", (char*)args[1] );*/
		return 0;
	}

	buf = prom_get_property( dn, (char*)args[1], &len );
	ret[0] = buf ? len : -1;
#ifdef VGETPROP
	printm("of_getprop %s/%s -- %p {%d/%ld}\n", dn->full_name, (char*)args[1], 
	       buf, len, args[3] );
#endif
	if( buf ) {
		memcpy( (char*)args[2], buf, MIN(ret[0],args[3] ) );
/*		printm("******** '%s' (%d,%d)\n", buf, ret[0], args[3] ); */
	}
	return 0;
}


/* int phandle, char *name --- int size */
static int 
of_getproplen( ulong args[], ulong ret[] ) 
{
	mol_device_node_t *dn = PHANDLE_TO_DN( args[0] );
	int len;
	char *buf;

#if 0
	printm("getproplen -- phandle %08lX\n", args[0] );
#endif
	ret[0] = -1;
	if( !dn )
		return 0;

	if( (buf = prom_get_property( dn, (char*)args[1], &len )) != NULL )
		ret[0] = len;

#ifdef VGETPROP
	printm("of_getproplen %s/%s  -- 0x%02x\n", dn->full_name, (char*)args[1], len );
#endif
	return 0;
}

/* int phandle, char *name, void *buf, int len -- int size */
static int 
of_setprop( ulong args[], ulong ret[] ) 
{
	mol_device_node_t *dn = PHANDLE_TO_DN( args[0] );

	ret[0] = -1;
	if( !dn )
		return 0;

	printm("of_setprop: %s/%s 0x%08lX %ld\n", dn->full_name, (char*)args[1], args[2], args[3] );
	ret[0] = args[3];

	/* XXXXXXXXXXXXX not implemented yet! XXXXXXXXXXXXXXXXXX */
	return 0;
}

/* char *device_specifier, void *buf, int buflen -- int length */
static int 
of_canon( ulong args[], ulong ret[] ) 
{
	mol_device_node_t *dn = prom_find_dev_by_path( (char*)args[0] );
	
	ret[0] = -1;
	if( dn ) {
		ret[0] = strlen( (char*)dn->full_name );
		if( args[1] )
			my_strncpy( (char*)args[1], dn->full_name, args[2] );
	}
	printm("of_canon: %s -- '%s'\n", (char*)args[0], (char*)args[1] );
	return 0;
}

/* int ihandle, void *buf, int buflen -- int length */
static int
of_inst_to_path( ulong args[], ulong ret[] ) 
{
	args[0] = ihandle_to_phandle( args[0] );

#if 0
	stop_emulation();
#endif
	if( !args[0] ) {
		ret[0] = -1;
		return 0;
	}
	/* printm("of_inst_to_path... %08lX %08lX %08lX\n", args[0], args[1], args[2] );  */
	return of_pack_to_path( args, ret );
}

/* int phandle, void *buf, int buflen -- int length */
static int
of_pack_to_path( ulong args[], ulong ret[] ) 
{
	mol_device_node_t *dn = PHANDLE_TO_DN( args[0] );

	ret[0] = -1;
	if( !dn )
		return 0;

	if( args[2] ) {
		my_strncpy( (char*)args[1], dn->full_name, args[2] );
	}
	
	/* printm("of_pack_to_path ih: %08lX -- '%s'\n", args[0], (char*)args[1] ); */

	ret[0] = strlen( dn->full_name );
	return 0;	
}


/* int ihandle -- int phandle */
static int
of_inst_to_pack( ulong args[], ulong ret[] ) 
{
/*	printm("of_inst_to_pack %08lX\n", args[0] );  */
	
	ret[0] = ihandle_to_phandle( args[0] );
	if( !ret[0] )
		ret[0] = -1;
	return 0;
}


/* char *device_specifier -- int ihandle */
static int 
of_open( ulong args[], ulong ret[] ) 
{
	ulong phandle;
	ulong ih;
	
/*	printm("of_open %s\n", (char*)args[0] ); */

	phandle = prom_dn_to_phandle( prom_find_dev_by_path( (char*)args[0] ) );
	if( !phandle ) {
		printm("of_open: node '%s' missing\n", (char*)args[0]);
		ret[0] = 0;
		return -1;
	}

	ih = phandle_to_ihandle( phandle );
	if( !ih )
		ih = add_ihandle( 0, 0, phandle );

	/* printm("ihandle %08lX\n", ih ); */
	ret[0] = ih;
	return 0;
}

/* int ihandle, void *addr, int len -- int actual */
static int 
of_read( ulong args[], ulong ret[] ) 
{
	ulong largs[4] = { (ulong)"read", args[0], args[2], args[1] };
	ulong lret[2];
	int err;
	
	err = of_call_method_( largs, lret, 4, 2 ); 	
	ret[0] = lret[1];
	return err;
}

/* int ihandle, void *addr, int len -- int actual */
static int 
of_write( ulong args[], ulong ret[] ) 
{
	ulong largs[4] = { (ulong)"write", args[0], args[2], args[1] };
	ulong lret[2];
	int err;
	
	err = of_call_method_( largs, lret, 4, 2 ); 	
	ret[0] = lret[1];
	return err;
}


/* int ihandle, int pos_hi, int pos_lo -- int status */
static int
of_seek( ulong args[], ulong ret[] ) 
{
	ulong largs[4] = { (ulong)"seek", args[0], args[2], args[1] };
	ulong lret[2];
	int err;
	
	err = of_call_method_( largs, lret, 4, 2 );
	ret[0] = lret[1];
	return err;
}

/* void *virt, int size, int align -- void *baseaddr */
static int 
of_claim( ulong args[], ulong ret[] ) 
{
	/* virt should be used if align == 0 */
	MEM_DEBUG("of_claim: %lX size: %lX align: %lX\n", args[0], args[1], args[2] );

	ret[0] = ofmem_claim( args[2] ? 0 : args[0], args[1], args[2] );
	MEM_DEBUG("claim = %08lX\n", ret[0] );
#if 0
	stop_emulation();
#endif
	return 0;
}


/* void *virt, int size --  */
static int 
of_release( ulong args[], ulong ret[] ) 
{
	ofmem_release( args[0], args[1] );
	MEM_DEBUG("of_release: %08lX %08lX\n", ret[0], ret[1] );

	return 0;
}

/* --  */
static int 
of_exit( ulong args[], ulong ret[] ) 
{
	printm("********** of_exit *********\n");
	return 0;
}

/* --  */
static int 
of_quiesce( ulong args[], ulong ret[] ) 
{
	printm("********* of_quiesce *********\n");
/*	stop_emulation();*/

	/* This seems to be the correct thing to do - but I'm not sure */
	mregs->msr &= ~(MSR_DR | MSR_IR);
	_msr_changed();
	return 0;
}


/************************************************************************/
/*	NODE METHODS							*/
/************************************************************************/

static int met_nret, met_nargs;

/* char *method, int ihandle, stack_arg_top, stack_arg2, stackarg3 
 * -- catch_result, stack-res1, stack-res2
 */
static int
of_call_method( ulong args[], ulong ret[] ) 
{
	return of_call_method_( args, ret, s_pargs->nargs, s_pargs->nret );
}


static int
of_call_method_( ulong args[], ulong ret[], int tot_nargs, int tot_nret ) 
{
	of_method_t *t;
	int i, nargs, nret;
	int molih = ihandle_to_molih( args[1] );
	ulong largs[10], lret[10];

	nargs = tot_nargs - 2;
	nret = tot_nret - 1;

	for( t=method_table; t->method; t++ ){
		if( strcmp( (char*)args[0], t->method ) || molih != t->molih )
			continue;
		if( (t->nargs != -1 && nargs != t->nargs) || (t->nret != -1 && nret != t->nret ) )
			continue;

		/* stop_emulation(); */

		/* stack_argN, ..., arg1  --  retN ... ret1 */
		for(i=0; i<nargs; i++ ) {
			largs[i] = args[1 + nargs - i];
			if( t->translate_args & (1<<i) ) {
				if( largs[i] && ofvirt_to_lvptr( largs[i], (char**)&largs[i] ) < 0 ) {
					printm("******** of_call_method: Invalid pointer *******\n");
					return -1;
				}
			}
		}

		/* catch result in r[0] --- normally 0 */
		met_nargs = nargs;
		met_nret = nret;

		ret[0] = (*t->proc)( args[1], largs, lret );

		for(i=0; i<nret; i++ )
			ret[i+1] = lret[nret-i-1];
#if 0
		if( !strcmp( t->method, "read") ) return 0;
#endif
		return 0;
	}
#if 1	
	printm("**** of_call_method: cmd '%s' ih: %08lX (%d/%d) Args: ", 
	       (char*)args[0], args[1], tot_nargs, tot_nret );
	for(i=2; i<tot_nargs; i++ )
		printm(" %08lX", args[i] );
	printm("\n");
#endif
	return 0;
}

static int 
met_nop( ulong ih, ulong args[], ulong ret[] ) 
{
	printm("*** met_nop '%s'\n", (char*)s_pargs->args[2] );
	return 0;
}

/* ( phys, size, align --- base ) */
static int 
met_mem_claim( ulong ih, ulong args[], ulong ret[] ) 
{
	ret[0] = ofmem_claim_phys( args[0], args[1], args[2] );

	MEM_DEBUG("met_mem_claim: (phys %08lX, size: %08lX, align: %08lX -- %08lX)\n", 
		   args[0], args[1], args[2], ret[0] );

	return (ret[0] == -1) ? -1 : 0;
}

/* ( virt, size, align ---  base ) */
static int 
met_mmu_claim( ulong ih, ulong args[], ulong ret[] ) 
{
#if 0
#warning XXXXXXXXX MMU_CLAIM HACK DEBUG XXXXXXXXXXXXXXXXX
	if( args[1] == 0x1000 ) {
		printm("met_mmu_claim HACK -- size altered from 0x1000 to 0x30000\n");
		args[1] = 0x30000;
	}
#endif
	ret[0] = ofmem_claim_virt( args[0], args[1], args[2] );

	MEM_DEBUG("mmu_claim: (virt %08lX, size: %08lX, align: %08lX -- %08lX)\n", 
		args[0], args[1], args[2], ret[0] );

	return (ret[0] == -1) ? -1 : 0;
}

/* ( virt --- phys mode true ) */
static int 
met_mmu_translate( ulong ih, ulong args[], ulong ret[] ) 
{
	ret[1] = 0;
	ret[0] = ofmem_translate( args[0], &ret[1] );
	ret[2] = 0;

	MEM_DEBUG("mmu_translate: %08lX --> %08lX\n", args[0], ret[0] );

	return (ret[0] == -1 ) ? -1 : 0;
}

/* ( phys, virt, size, mode -- [ret] ) */
static int 
met_mmu_map( ulong ih, ulong args[], ulong ret[] ) 
{
	ulong r;
	MEM_DEBUG("met_mmu_map: %08lX %08lX %08lX %08lX\n", args[0], args[1], args[2], args[3] ); 

	r = ofmem_map( args[0], args[1], args[2], args[3] ) ? -1 : 0;
	if( met_nret == 1 )
		ret[0] = r;	/* ? */
	return r;	/* or ? */
}


/* ( physbase -- rtas_callback ) */
static int 
met_rtas_instantiate( ulong ih, ulong args[], ulong ret[] ) 
{
	/* printm("rtas_instantiate %08lX\n", args[0] );  */

	ret[0] = rtas_instantiate( args[0] );
	return 0;
}


/* ( addr, len -- actual ) */
static int 
met_stdout_write( ulong ih, ulong args[], ulong ret[] ) 
{
	char *s = malloc( args[1]+1 );

	my_strncpy( s, (char*)args[0], args[1] );
	s[args[1]]=0;
	
	printm( "* %s", s );
	free( s );
	return 0;
}

/* This version of strncpy will not pad with 0 */
static char*
my_strncpy( char *dest, const char *src, size_t n )
{
	int len = MIN( n, strlen(src)+1 );
	return memcpy( dest, src, len );
}


/************************************************************************/
/*	NVRAM								*/
/************************************************************************/

static int s_nvram_mark=0;

/* ( pos_hi, pos_lo -- status ) */
static int 
met_nvram_seek( ulong ih, ulong args[], ulong ret[] ) 
{
/*	printm("NVRAM: seek %08lX %08lX\n", args[0], args[1] );  */
	s_nvram_mark = args[1];
	return 0;
}

/* ( addr, len -- actual ) */
static int 
met_nvram_read( ulong ih, ulong args[], ulong ret[] ) 
{
	int n;
/*	printm("NVRAM: read %08lX %08lX\n", args[0], args[1] ); */

	n = nvram_read( s_nvram_mark, (char*)args[0], args[1] );
	s_nvram_mark += n;

	ret[0] = n;
	return 0;
}

/* ( addr, len -- actual ) */
static int 
met_nvram_write( ulong ih, ulong args[], ulong ret[] ) 
{
	int n;
/*	printm("NVRAM: write %08lX %08lX\n", args[0], args[1] ); */

	n = nvram_write( s_nvram_mark, (char*)args[0], args[1] );
	s_nvram_mark += n;

	ret[0] = n;
	return 0;
}


/* ( -- size ) */
static int 
met_nvram_size( ulong ih, ulong args[], ulong ret[] ) 
{
	/* printm("NVRAM: size\n");  */
	ret[0] = nvram_size();
	return 0;
}
