//
// File:        RegexMatch.java
// Package:     gov.llnl.babel.symbols
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Use java.util.regex if available to match symbol ids
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;


/**
 * Provide regular expression matching for {@link
 * gov.llnl.babel.symbols.SymbolID} against regular expressions if the
 * {@link java.util.regex} package is available. This class uses reflection
 * to avoid a compile time or runtime dependence on {@link java.util.regex}.
 */
public class RegexMatch {
  /**
   * Hold a pointer to the static {@link java.util.regex.Pattern#compile}
   * method if it's available.
   */
  private static Method s_compile;

  /**
   * Hold the class pointer for a {@link java.lang.CharSequence}.
   */
  private static java.lang.Class s_charSequence;
  
  /**
   * The compiled {@link java.util.regex.Pattern} object.
   */
  private Object d_pattern;

  /**
   * The {@link java.util.regex.Pattern#matcher} method.
   */
  private Method d_matcher;

  static {
    try {
      java.lang.Class pattern = java.lang.Class.forName("java.util.regex.Pattern");
      s_charSequence = java.lang.Class.forName("java.lang.CharSequence");
      java.lang.Class[] args = new java.lang.Class[1];
      args[0] = "".getClass();  // java.lang.String argument
      s_compile = pattern.getMethod("compile", args);
    }
    catch (ClassNotFoundException cnfe) {
      s_compile = null;
    }
    catch (NoSuchMethodException nsme) {
      System.err.println("Class java.util.regex.Pattern exists but does not have compile method: " + nsme.getMessage());
      s_compile = null;
    }
    catch (SecurityException se) {
      System.err.println("Class java.util.regex.Pattern exists but does not have public compile method: " + se.getMessage());
      s_compile = null;
    }
  }

  /**
   * Return <code>true</code> iff the current runtime system has regular
   * expression support.
   */
  public static boolean hasRegex() 
  {
    return s_compile != null;
  }

  /**
   * Create a regular expression matching object.
   *
   * @param regex  a regular expression to be compiled.
   * @throws gov.llnl.babel.symbols.RegexUnsupportedException this exception
   * indicates that the runtime environment lacks the necessary {@link
   * java.util.regex} package.
   * @throws gov.llnl.babel.symbols.BadRegexException this exception
   * indicates that the regular expression had a syntax error.  The message
   * tries to indicate where the error occured.
   */
  public RegexMatch(String regex)
    throws RegexUnsupportedException, BadRegexException
  {
    if (hasRegex()){
      try {
        Object[] args = new Object[1];
        args[0] = regex;
        d_pattern = s_compile.invoke(null, args);
        java.lang.Class[] argTypes = new java.lang.Class[1];
        argTypes[0] = s_charSequence;
        d_matcher = d_pattern.getClass().getMethod("matcher", argTypes);
        // try a bogus match to make sure the whole call works
        try {
          match("");
        }
        catch (InvocationTargetException ite) {
          throw new RegexUnsupportedException();          
        }
      }
      catch (InvocationTargetException ite) {
        throw new BadRegexException(ite.getTargetException().getMessage());
      }
      catch (Exception e){
        throw new RegexUnsupportedException();
      }
    }
    else {
      throw new RegexUnsupportedException();
    }
  }

  private boolean match(String str)
    throws NoSuchMethodException,
           InvocationTargetException,
           IllegalAccessException
  {
    Object[] args = new Object[1];
    args[0] = str;
    Object matcher = d_matcher.invoke(d_pattern, args);
    java.lang.Class[] argTypes = new java.lang.Class[0];
    args = new Object[0];
    Boolean result = (Boolean)
      (matcher.getClass().getMethod("matches", argTypes).
       invoke(matcher, args));
    return result.booleanValue();
  }

  /**
   * Return <code>true</code> if the {@link SymbolID} matches the
   * compiled regular expression.
   *
   * @param id  the symbol that will be compared with the regular
   * expression.
   * @return <code>true</code> indicates that the {@link SymbolID} matches
   * the regular expression.
   */
  public boolean match(SymbolID id)
  {
    try {
      return match(id.getFullName());
    }
    catch (NoSuchMethodException nsme) {
      System.err.println("java.util.regex package is incomplete");
    }
    catch (InvocationTargetException ite) {
      System.err.println("Unexpected exception from matches.");
    }
    catch (IllegalAccessException iae) {
      System.err.println("Required method from java.util.regex package is unavailable");
    }
    return false;
  }
}
