//
// File:        CxxStubSource.java
// Package:     gov.llnl.babel.backend.ucxx
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.ucxx;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.ucxx.Cxx;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.LevelComparator;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;


/**
 * Create and write a header for a Cxx C extension class to wrap a 
 * BABEL extendable in a Cxx object. The header has to expose a 
 * function to create a wrapped IOR, a function to check if a 
 * <code>PyObject</code> is an instance of this extension type, and
 * an import macro.
 */
public class CxxStubSource {
  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;
  private String d_self = null;
  
  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public CxxStubSource(Extendable ext) {
    d_ext = ext;
  }
  
  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() throws CodeGenerationException {
    String filename = Cxx.generateFilename( d_ext.getSymbolID(), 
                                            Cxx.FILE_ROLE_STUB, 
                                            Cxx.FILE_TYPE_CXX_SOURCE );
    //System.out.println("Create " + filename + "..." );

    if ( d_ext.isInterface() ) { 
      d_self = "loc_self->d_object";
    } else { 
      d_self = "loc_self";
    }

    try { 
      d_writer = Cxx.createSource( d_ext, Cxx.FILE_ROLE_STUB, "STUBSRCS" );
      d_writer.println();

      writeIncludes();

      writeUserDefinedMethods();

      writeConstructors();

      writeCastingOperators();

      if ( !d_ext.isInterface()) {
        writeDynamicImplStuff();
      }
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  private void writeIncludes() throws CodeGenerationException { 
    SymbolID id = d_ext.getSymbolID();
    d_writer.generateInclude( Cxx.generateFilename( d_ext.getSymbolID(), 
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER ),
                              true);
    d_writer.generateInclude( Cxx.generateFilename( "sidl.BaseInterface",
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER
                                                    ),
                              true );
    d_writer.generateInclude(  Cxx.generateFilename( "sidl.BaseClass",
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER
                                                    ),
                               true );
    if ( d_ext.hasExceptionThrowingMethod(true) ) { 
      d_writer.generateInclude( Cxx.generateFilename( "sidl.BaseException",
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER), 
                                true );
    }
    d_writer.generateInclude( "sidl_String.h", false );
    if (!BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.generateInclude("babel_config.h", false);
      d_writer.printlnUnformatted("#ifdef SIDL_DYNAMIC_LIBRARY");
      d_writer.printlnUnformatted("#include <stdio.h>");
      d_writer.printlnUnformatted("#include <stdlib.h>");
      d_writer.generateInclude("sidl_Loader.hh", false);
      d_writer.generateInclude("sidl_DLL.hh", false);

      d_writer.printlnUnformatted("#endif");
    }
    
    //Generate method dependency includes
    Cxx.generateSourceIncludes(d_writer, d_ext);
    /*Set includes = new HashSet();
    for( Iterator i = d_ext.getMethods(true).iterator(); i.hasNext(); ) { 
      Method method = (Method) i.next();
      includes.addAll(method.getSymbolReferences());
      
    }
    if (!includes.isEmpty()){
      d_writer.writeComment("Includes for all method dependencies.",false);
      
      List entries = Utilities.sort(includes);
      
      for (Iterator i = entries.iterator(); i.hasNext(); ) { 
        String header = Cxx.generateFilename( (SymbolID) i.next(), 
                                              Cxx.FILE_ROLE_STUB, 
                                              Cxx.FILE_TYPE_CXX_HEADER );
        d_writer.generateInclude( header, true );
      }
    } 
    */
    d_writer.println();
  } 

  private void writeConstructors() { 
    SymbolID id = d_ext.getSymbolID();
    final String extName = IOR.getExternalName(id);
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = 
                                  Cxx.getSymbolNameWithoutLeadingColons(id,"");
    final String name = id.getShortName();    
    final String ior_ptr = "void*"; //"ior_t*";

    if( !d_ext.isAbstract() ) { 
      d_writer.writeCommentLine("static constructor");
      d_writer.println(fullName);
      d_writer.println(fullNameWithoutLeadingColons + "::_create() {");
      d_writer.tab();
      d_writer.println( Cxx.getObjectName(id) 
        + " self( (*_get_ext()->createObject)(), false );");
      d_writer.println("return self;");
      d_writer.backTab();

      d_writer.println("}");
      d_writer.println();
    }

//     d_writer.writeCommentLine("default destructor");
//     d_writer.println(fullNameWithoutLeadingColons + "::~" + name + " () {");

//     d_writer.tab();
//     d_writer.println("if ( d_self != 0 ) {");
//     d_writer.tab();

//     d_writer.println("deleteRef();");
//     d_writer.backTab();
//     d_writer.println("}");
//     d_writer.backTab();
//     d_writer.println("}");
//     d_writer.println();

    d_writer.writeCommentLine("copy constructor");
    d_writer.println( fullNameWithoutLeadingColons + "::" + name 
      + " ( const " + fullName + "& original ) {");
    d_writer.tab();
    //d_writer.println("d_self = " + Cxx.constCast(ior_ptr, "original.d_self") 
    //    + ";");
    d_writer.println("d_self = original._cast(\"" + id.getFullName() + "\");");
    d_writer.println("d_weak_reference = original.d_weak_reference;"); 
    d_writer.println("if (d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
    
    d_writer.writeCommentLine("assignment operator");
    d_writer.println( fullName + "&");
    d_writer.println( fullNameWithoutLeadingColons + "::operator=( const " 
      + fullName + "& rhs ) {");
    d_writer.tab();
    d_writer.println("if ( d_self != rhs.d_self ) {");
    d_writer.tab();
    d_writer.println("if ( d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("deleteRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("d_self = rhs._cast(\""+id.getFullName()+"\");"); 
    //" + Cxx.constCast( ior_ptr, "rhs.d_self") + ";");
    d_writer.println("d_weak_reference = rhs.d_weak_reference;");
    d_writer.println("if ( d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return *this;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  private void writeCastingOperators() { 
    SymbolID id = d_ext.getSymbolID();
    final String extName = IOR.getExternalName(id);
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = 
                                   Cxx.getSymbolNameWithoutLeadingColons(id,"");
    final String name = id.getShortName();   

    d_writer.writeCommentLine("conversion from ior to C++ class");
    d_writer.println( fullNameWithoutLeadingColons + "::" + name 
      + " ( " + fullName + "::ior_t* ior ) ");
    
    d_writer.println("   : StubBase("+Cxx.reinterpretCast("void*","ior")
      + ") { }");
    d_writer.println();

    d_writer.writeCommentLine("Alternate constructor: does not call addRef()");
    d_writer.writeCommentLine("(sets d_weak_reference=isWeak)");
    d_writer.writeCommentLine("For internal use by Impls (fixes bug#275)");
    d_writer.println( fullNameWithoutLeadingColons + "::" + name +" ( " 
      + fullName + "::ior_t* ior, bool isWeak ) : ");
    d_writer.println( "StubBase(" + Cxx.reinterpretCast("void*","ior") 
      + ", isWeak){ ");
    //d_writer.println("    : StubBase("+Cxx.reinterpretCast("void*","ior")
    //  + ",isWeak) { ");
    //d_writer.println("d_self = "+Cxx.reinterpretCast("void*","ior") + ";");
    //d_writer.println("d_weak_reference = isWeak;");
    d_writer.println("}");
    d_writer.println();

    // d_writer.writeCommentLine("conversion from a StubBase");
    // d_writer.println( fullNameWithoutLeadingColons + "::" + name 
    //   + " ( const ::sidl::StubBase& base )");
    // d_writer.println("{");
    // d_writer.tab();
    // d_writer.println("d_self = " + Cxx.reinterpretCast("void*",
    //                  "base._cast(\"" + id.getFullName() + "\")") + ";");
    // d_writer.println("d_weak_reference = base.d_weak_reference;");
    // d_writer.println("if (d_self != 0 && !d_weak_reference) {");
    // d_writer.tab();
    // d_writer.println("addRef();");
    // d_writer.backTab();
    // d_writer.println("}");
    // d_writer.backTab();
    // d_writer.println("}");
    // d_writer.println();

    d_writer.writeCommentLine("protected method that implements casting");
    d_writer.println("void* " + fullNameWithoutLeadingColons 
      + "::_cast(const char* type) const");
    d_writer.println("{");

    d_writer.tab();

    d_writer.println("ior_t* loc_self = " 
      + Cxx.reinterpretCast("ior_t*","this->d_self") + ";");

    d_writer.println("void* ptr = 0;");
    d_writer.println("if ( loc_self != 0 ) {");
    d_writer.tab();
    d_writer.print("ptr = ");
    if (d_ext.isInterface()) {
      d_writer.print(Cxx.reinterpretCast( "void*",
        "(*loc_self->d_epv->f__cast)(loc_self->d_object, type)"));
    } else {
      d_writer.print(Cxx.reinterpretCast( "void*",
        "(*loc_self->d_epv->f__cast)(loc_self, type)"));
    }
    d_writer.println(";");
    d_writer.backTab();
    d_writer.println("}");    
    d_writer.println("return ptr;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  private void writeDynamicImplStuff() throws CodeGenerationException { 
    SymbolID id = d_ext.getSymbolID();
    final String extName = IOR.getExternalName(id);
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = 
                                  Cxx.getSymbolNameWithoutLeadingColons(id,"");
    d_writer.writeCommentLine("Static data type");
    d_writer.println("const " + fullName + "::ext_t * " 
      + fullNameWithoutLeadingColons + "::s_ext = 0;");
    d_writer.println();

    d_writer.writeCommentLine("private static method to get static data type");
    d_writer.println("const " + fullName + "::ext_t *");
    d_writer.println( fullNameWithoutLeadingColons + "::_get_ext()");
    d_writer.println("  throw ( " + Cxx.prependGlobalUCxx() 
      + "::sidl::NullIORException)");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("if (! s_ext ) {");
    d_writer.tab();
    if (BabelConfiguration.isSIDLBaseClass(id)) { 
      d_writer.println("s_ext = " + IOR.getExternalFunc(id) + "();");
    } else {
      d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
      d_writer.println("s_ext = " + IOR.getExternalFunc(id) + "();");
      d_writer.printlnUnformatted("#else");
      d_writer.println("" + Cxx.prependGlobalUCxx() + "::sidl::DLL dll = " 
        + Cxx.prependGlobalUCxx() + "::sidl::DLL::_create();");
      d_writer.println("const ext_t *(*dll_f)(void);");
      d_writer.writeCommentLine("check global namespace for symbol first");

      d_writer.println("if (dll._not_nil() && dll.loadLibrary(\"main:\", TRUE, "
        + "FALSE)) {");
      d_writer.tab();

      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const ext_t *(*)(void)) ");
      d_writer.println("dll.lookupSymbol(");
      d_writer.tab();

      d_writer.println("\"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println("s_ext = (dll_f ? (*dll_f)() : NULL);");
      d_writer.backTab();
      d_writer.println("}");

      d_writer.println("if (!s_ext) {");
      d_writer.tab();
      d_writer.println("dll = " + Cxx.prependGlobalUCxx() 
        + "::sidl::Loader::findLibrary(\"" + id.getFullName() + "\",");
      d_writer.tab();
      d_writer.println("\"ior/impl\", " + Cxx.prependGlobalUCxx() 
        + "::sidl::Scope_SCLSCOPE,");
      d_writer.println("" + Cxx.prependGlobalUCxx() 
        + "::sidl::Resolve_SCLRESOLVE);");

      d_writer.backTab();
      d_writer.println("if (dll._not_nil()) {");
      d_writer.tab();
      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const ext_t *(*)(void)) ");

      d_writer.println("dll.lookupSymbol(");
      d_writer.tab();

      d_writer.println("\"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println("s_ext = (dll_f ? (*dll_f)() : NULL);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (!s_ext) {");
      d_writer.tab();
      d_writer.disableLineBreak();
      d_writer.println("throw " + Cxx.prependGlobalUCxx() 
        + "::sidl::NullIORException( ::std::string (");
      d_writer.tab();
      d_writer.println("\"cannot find implementation for " + id.getFullName() 
        + "; please set SIDL_DLL_PATH\"");
      d_writer.backTab();
      d_writer.println("));");
      d_writer.enableLineBreak();
      d_writer.backTab();
      d_writer.println("}");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return s_ext;");
    d_writer.backTab();
    d_writer.println("}");   
    d_writer.println();    
  }

  private void writeUserDefinedMethods() throws CodeGenerationException { 
    d_writer.beginBoldComment();
    d_writer.println("User Defined Methods");
    d_writer.endBoldComment();

    Iterator m = null;
    m = d_ext.getStaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      if(!Cxx.inlineStub(method)) {
        generateMethodDispatch( method, "user defined static method", true);
        d_writer.println();
        if(method.hasRarray()) {
          generateMethodDispatch( method, "user defined static method", false);
          d_writer.println();
        }
      }
    }

    m = d_ext.getNonstaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      if(!Cxx.inlineStub(method)) {
        generateMethodDispatch(method, "user defined non-static method.", true);
        d_writer.println();
        if(method.hasRarray()) {
          generateMethodDispatch(method, "user defined non-static method", 
                                 false);
          d_writer.println();
        }
      }
    }

    d_writer.beginBoldComment();
    d_writer.println("End User Defined Methods");
    d_writer.println("(everything else in this file is specific to");
    d_writer.println(" Babel's C++ bindings)");
    d_writer.endBoldComment();
  }

  /** 
   *  Function generates function definitions. 
   */
  private void generateMethodDispatch( Method m, String altcomment, 
                                       boolean rarrays ) 
    throws CodeGenerationException 
  { 
    if ( m == null ) { return; }
    final ArrayList vArgs = m.getArgumentList();
    final int nargs = vArgs.size();
    String extra_close_paren = "";
    String shortMethodName = m.getShortMethodName();
    String longMethodName = m.getLongMethodName();
    String className = d_ext.getSymbolID().getShortName();
    Comment comment = m.getComment();
    BabelConfiguration s_babel_config = BabelConfiguration.getInstance();
    Type return_type = m.getReturnType();
    SymbolID id = return_type.getSymbolID();

    d_writer.writeComment( comment, altcomment );
    if ( shortMethodName.equals(className) ) { 
      shortMethodName = "f_" + shortMethodName;
      System.out.println("WARNING: gov.llnl.babel.backend.UCxx.CxxStubSource: "
        + "sidl / C++ conflict!");
      System.out.println("         methodName == className is not allowed in "
        + "C++");
      System.out.println("         (this is restricted to constructors in C++"
        + ")");
      System.out.println("         changing to " + className + "::" 
        + shortMethodName + "()");
    }

    //Output function deleration
    d_writer.println(Cxx.generateFunctionDeclaration(m, d_ext, d_self, 
                                                     rarrays));

    d_writer.println();
    d_writer.println("{");

    d_writer.tab();
    if ( (! m.isStatic()) &&  s_babel_config.makeCxxCheckNullIOR()) { 

      d_writer.println("if ( d_self == 0 ) {");
      d_writer.tab();
      d_writer.println("throw " + Cxx.prependGlobalUCxx() 
        + "::sidl::NullIORException( ::std::string (");
      d_writer.tab();
      d_writer.disableLineBreak();
      d_writer.println( "\"" + "Null IOR Pointer in \\\"" 
        + Cxx.getMethodStubName(d_ext.getSymbolID(),shortMethodName) 
        + "()\\\"\"");
      d_writer.enableLineBreak();
      d_writer.backTab();
      d_writer.println("));");
      d_writer.backTab();
      d_writer.println("}");
    }

    //Output almost nothing!
    d_writer.println(Cxx.generateInitialization(m, d_ext, d_self));

    if ( ! m.isStatic() ) { 
      if (shortMethodName.equals("addRef") ||
          shortMethodName.equals("deleteRef") ) { 
        d_writer.println("if ( !d_weak_reference ) {");
        d_writer.tab();
      }
    }

    //Prepare for IOR call!
    d_writer.println(Cxx.generatePreIORCall(m, d_ext, d_self, false, rarrays));

    d_writer.println(Cxx.generateIORCall(m, d_ext, d_self, false, rarrays));
    if (m.getThrows().size()>0) { // if throws exception
      d_writer.println("if (_exception != 0 ) {");      
      d_writer.tab();
      d_writer.println("void * _p = 0;");
      Object [] exceptions = m.getThrows().toArray();
      Arrays.sort(exceptions, new LevelComparator(SymbolTable.getInstance()));
      
      for( int i=0; i<exceptions.length; ++i) {
        SymbolID exid = (SymbolID)exceptions[i];
        d_writer.println("if ( (_p=(*(_exception->d_epv->f__cast))(_exception,"
          + " \"" + exid.getFullName() + "\")) != 0 ) {");
        d_writer.tab();
        d_writer.println(IOR.getObjectName(exid) + " * _realtype = " 
          + Cxx.reinterpretCast(IOR.getObjectName(exid)+"*" , "_p") + ";");
        d_writer.writeCommentLine("Note: alternate constructor does not "
          + "increment refcount.");
        d_writer.println("throw " + Cxx.getObjectName(exid) 
          + "( _realtype, false );");
        d_writer.backTab();
        d_writer.println("}");
      }
      d_writer.backTab();
      d_writer.println("}");
    }

    //Clean up from return from IOR
    d_writer.println(Cxx.generatePostIORCall(m, d_ext, d_self, rarrays));

    //d_writer.println( post_ior.toString().trim() );
    if ( ! m.isStatic() ) { 
      if ( shortMethodName.equals("addRef") 
         || shortMethodName.equals("deleteRef") ) { 
        if ( shortMethodName.equals("deleteRef") ) { 
          d_writer.println("d_self = 0;");
        }
        d_writer.backTab();
        d_writer.println("}");
      }
    }

    if ( return_type.getType() != Type.VOID ) { 
      d_writer.println("return _result;");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }
}
