#if !defined(lint)
static char rcs_id[] = "$Id: images.c,v 1.8 1996/06/17 19:39:02 stefan Exp $";
#endif

/*
** GIF image drawing functions for http-analyze.
**
** Uses the gd gif-manipulating library by Tom Boutell, <boutell@boutell.com>
** See http://sunsite.unc.edu/boutell/index.html for more infos about gd.
**
** The functions in this source file are part of http-analyze, which is
**	Copyright 1996 by Stefan Stapelberg, <stefan@rent-a-guru.de>
**
** $Log: images.c,v $
** Revision 1.8  1996/06/17  19:39:02  stefan
** Fixed a bug which caused http-analyze to dump core
** by casually writing into other memory regions.
**
** Revision 1.7  1996/06/13  11:30:43  stefan
** Added rcs_id.
**
** Revision 1.6  1996/06/13  10:40:32  stefan
** Fixed a bug which caused the graph to always start at
** 0,0 regardless of the results from previous summary periods.
** Fixed the image drawing functions to preserve 4 pixels
** for padding between the upper border and the graphs/bars.
**
** Revision 1.5  1996/06/07  11:03:13  stefan
** Now avoids same colors for consecutive countries
** in the Total transfers by Country pie chart.
**
** Revision 1.4  1996/05/25  02:00:15  stefan
** Cleaned up program structure somewhat. This is still the worstest
** kind of spaghetti code in http-analyze. Added code for a pie chart
** of the topmost countries accessing the server.
**
** Revision 1.3  1996/05/09  09:12:15  stefan
** Fixed bug in icon code (wrong aspect ratio of icon images).
**
** Revision 1.2  1996/01/28  14:35:07  stefan
** Made it lint-clean somewhat. There is still more cleanup to do.
**
** Revision 1.1  1996/01/17  00:00:00  stefan
** Initial revision
**
*/

#include <unistd.h>	/* for access(2) */
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <math.h>

#include "defs.h"
#include "gd.h"
#include "gdfontl.h"
#include "gdfonts.h"

#define FWID(arg)	((arg)->w)

static gdPoint points[50];

/* draw beveled borders */
static void makebevel(gdImagePtr im, int mx, int my, int sz,
		      int col1, int col2, int col3, int col4) {
	int pt = 0;

	points[pt].x = 0;	points[pt++].y = 0;
	points[pt].x = sz;	points[pt++].y = sz;
	points[pt].x = mx-sz;	points[pt++].y = sz;
	points[pt].x = mx;	points[pt++].y = 0;
	gdImageFilledPolygon(im, points, pt, col1);

	pt = 0;
	points[pt].x = 0;	points[pt++].y = 0;
	points[pt].x = sz;	points[pt++].y = sz;
	points[pt].x = sz;	points[pt++].y = my-sz;
	points[pt].x = 0;	points[pt++].y = my;
	gdImageFilledPolygon(im, points, pt, col2);

	pt = 0;
	points[pt].x = 0;	points[pt++].y = my;
	points[pt].x = sz;	points[pt++].y = my-sz;
	points[pt].x = mx-sz;	points[pt++].y = my-sz;
	points[pt].x = mx;	points[pt++].y = my;
	gdImageFilledPolygon(im, points, pt, col3);

	pt = 0;
	points[pt].x = mx;	points[pt++].y = my;
	points[pt].x = mx-sz;	points[pt++].y = my-sz;
	points[pt].x = mx-sz;	points[pt++].y = sz;
	points[pt].x = mx;	points[pt++].y = 0;
	gdImageFilledPolygon(im, points, pt, col4);
	return;
}

/* add a bevel to space previously reserved */
static void addbevel(gdImagePtr im, int mx, int my, int sz,
		     int col1, int col2, int col3, int col4) {

	makebevel(im, mx, my, sz, col1, col2, col3, col4);
	/*gdImageRectangle(im, 0, 0, mx, my, black);*/

	return;
}

/* draw bars */
static void drawbars(gdImagePtr im, u_long *values, u_long maxval,
		     int bx, int by, int rx, int ry, int steps,
		     int size, int bwid, int fcol, int bcol) {
	int idx, pt, top, left;
	int hdist = rx / size;

	bx += (hdist-bwid) / 2;
	for (idx=0; idx < steps; idx++) {
		top = by+ry+4-(values[idx]*ry/maxval);
		left = bx+(idx*hdist);

		if (ry+by-top < 2)
			continue;
		pt = 0;
		points[pt].x = left;
		points[pt++].y = ry+by+4;
		points[pt].x = left;
		points[pt++].y = top;
		points[pt].x = left+bwid;
		points[pt++].y = top;
		points[pt].x = left+bwid;
		points[pt++].y = ry+by+4;

		gdImageFilledPolygon(im, points, pt, fcol);
		gdImagePolygon(im, points, pt, bcol);

	}
	return;
}

/* draw bars by day */
int mn_bars(int sizex, int sizey, int base, char *name) {
	char lbuf[SMALLSIZE];
	int idx, len;
	u_long hits_max = 0L;
	u_long file_max = 0L;
	u_long site_max = 0L;
	u_long kb_max = 0L;

	/* the image size */
	int maxx = sizex-1;
	int maxy = sizey-1;

	/* the drawable region */
	int basex = base;
	int basey = base;
	int rangex = maxx-(2*basex);
	int rangey = maxy-(2*basey);

	int bl1 = (rangey+1)/2;
	int bl2 = (rangey+1)/4;
	int hdist = rangex/31;

	/* the font baselines */
	int flx = basex-12;		/* left X axis */
	int frx = maxx-basex;		/* right X axis */
	int fty = basey-18;		/* top Y axis */
	int fby = maxy-basey+4;		/* bottom Y axis */

	/* colors */
	int paper_white, white, grey1, grey2, grey3,
		black, red, green, blue, orange;

	FILE *out;
	gdImagePtr im;

	for (idx=0; idx < (int)tend.mday; idx++) {
		if (daily_hits[idx] > hits_max)
			hits_max = daily_hits[idx];
		if (daily_files[idx] > file_max)
			file_max = daily_files[idx];
		if (daily_sites[idx] > site_max)
			site_max = daily_sites[idx];
		if (daily_kbytes[idx] > kb_max)
			kb_max = daily_kbytes[idx];
	}
	if (hits_max == 0 || site_max == 0 || kb_max == 0)	/* no hits ?!? */
		return 0;

	/* Create the image */
	if ((im = gdImageCreate(sizex, sizey)) == NULL)
		return 0;

	paper_white = gdImageColorAllocate(im, 220, 220, 220);
#if defined(lint)
	paper_white = paper_white;
#endif
	white = gdImageColorAllocate(im, 255, 255, 255);
	grey1 = gdImageColorAllocate(im, 242, 242, 242);
	grey2 = gdImageColorAllocate(im, 102, 102, 102);
	grey3 = gdImageColorAllocate(im, 153, 153, 153);
	black = gdImageColorAllocate(im, 0, 0, 0);
	red = gdImageColorAllocate(im, 220, 0, 0);
	green = gdImageColorAllocate(im, 0, 170, 0);
	blue = gdImageColorAllocate(im, 0, 0, 220);
	orange = gdImageColorAllocate(im, 255, 80, 0);

	for (idx=0; idx < 31; idx++) {
		(void) sprintf(lbuf, "%2d", idx+1);
		gdImageString(im, gdFontSmall, basex+(idx*hdist)+FWID(gdFontSmall)/2, fby, lbuf, black);
	}

	(void) sprintf(lbuf, "%lu", kb_max);
	gdImageStringUp(im, gdFontSmall, flx, maxy-basey-bl2+strlen(lbuf)*FWID(gdFontSmall), lbuf, orange);

	(void) sprintf(lbuf, "%lu", site_max);
	gdImageStringUp(im, gdFontSmall, flx, maxy-basey-(2*bl2)+strlen(lbuf)*FWID(gdFontSmall), lbuf, red);

	(void) sprintf(lbuf, "%lu", hits_max);
	gdImageStringUp(im, gdFontSmall, flx, basey+strlen(lbuf)*FWID(gdFontSmall), lbuf, green);

	(void) sprintf(lbuf, "Hits");
	len = strlen(lbuf);
	gdImageString(im, gdFontLarge, basex, fty, lbuf, green);
	gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "/", black);
	len++;
	(void) sprintf(lbuf, "Files");
	gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, lbuf, blue);
	len += strlen(lbuf);
	gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "/", black);
	len++;
	(void) sprintf(lbuf, "Sites");
	gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, lbuf, red);
	len += strlen(lbuf);
	gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "/", black);
	len++;
	(void) sprintf(lbuf, "Kilobytes");
	gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, lbuf, orange);
	len += strlen(lbuf)+1;
	gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "by day", black);

	gdImageStringUp(im, gdFontSmall, frx+1, sizey/2+gdFontSmall->w*31/2, 
		"Copyright 1996 RENT-A-GURU (TM)", black);

	gdImageLine(im, basex, bl1+basey, maxx-basex, bl1+basey, black);
	gdImageLine(im, basex, bl1+bl2+basey, maxx-basex, bl1+bl2+basey, black);

	drawbars(im, daily_hits, hits_max, basex-1, basey, rangex, bl1-4,
		tend.mday, 31, 6, green, black);

	drawbars(im, daily_files, hits_max, basex+3, basey, rangex, bl1-4,
		tend.mday, 31, 6, blue, black);

	drawbars(im, daily_sites, site_max, basex+1, basey+bl1, rangex, bl2-4,
		tend.mday, 31, 8, red, black);

	drawbars(im, daily_kbytes, kb_max, basex+1, bl1+bl2+basey, rangex, bl2-4,
		tend.mday, 31, 8, orange, black);

	/* Draw a rectangle around the central area. */
	gdImageRectangle(im, basex, basey, maxx-basex, maxy-basey, black);
	makebevel(im, maxx, maxy, 6, white, grey1, grey2, grey3);

	gdImageInterlace(im, 1);		/* make it interlaced */
	if ((out = fopen(name, "w")) != NULL) {
		gdImageGif(im, out);
		(void) fclose(out);
	}
	gdImageDestroy(im);
	return 1;
}

/* draw bars by hour */
int hr_bars(int sizex, int sizey, int base, char *name) {
	char lbuf[SMALLSIZE];
	u_long hits_max = 0L;	
	int idx;

	/* the image size */
	int maxx = sizex-1;
	int maxy = sizey-1;

	/* the drawable region */
	int basex = base;
	int basey = base;
	int rangex = maxx-(2*basex)-3;
	int rangey = maxy-(2*basey);

	int hdist = rangex/24;

	/* the font baselines */
	int flx = basex-12;		/* left X axis */
	/*int frx = maxx-basex;		/* right X axis */
	int fty = basey-18;		/* top Y axis */
	int fby = maxy-basey+4;		/* bottom Y axis */

	/* colors */
	int paper_white, white, grey1, grey2, grey3, black, orange;

	FILE *out;
	gdImagePtr im;

	for (idx=0; idx < 24; idx++) {
		if (hr_hits[idx] > hits_max)
			hits_max = hr_hits[idx];
	}
	if (hits_max == 0)	/* no hits ?!? */
		return 0;

	/* create the image */
	if ((im = gdImageCreate(sizex, sizey)) == NULL)
		return 0;

	/* first color allocated is background */
	paper_white = gdImageColorAllocate(im, 220, 220, 220);
#if defined(lint)
	paper_white = paper_white;
#endif
	white = gdImageColorAllocate(im, 255, 255, 255);
	grey1 = gdImageColorAllocate(im, 242, 242, 242);
	grey2 = gdImageColorAllocate(im, 102, 102, 102);
	grey3 = gdImageColorAllocate(im, 153, 153, 153);
	black = gdImageColorAllocate(im, 0, 0, 0);
	orange = gdImageColorAllocate(im, 255, 80, 0);

	(void) sprintf(lbuf, "Hits by hours since " TIME_FMT,
		tstart.mday, monnam[tstart.mon], EPOCH(tstart.year));

	gdImageString(im, gdFontLarge, basex, fty, lbuf, black);

	for (idx=0; idx < 24; idx++) {
		(void) sprintf(lbuf, "%2d", idx);
		gdImageString(im, gdFontSmall, basex+(idx*hdist)+FWID(gdFontSmall), fby, lbuf, black);
	}
	gdImageStringUp(im, gdFontSmall, flx, fby, "1", black);

	(void) sprintf(lbuf, "%lu", hits_max);
	gdImageStringUp(im, gdFontSmall, flx, basey+strlen(lbuf)*FWID(gdFontSmall), lbuf, orange);

	drawbars(im, hr_hits, hits_max, basex+2, basey, rangex, rangey-4,
		24, 24, 11, orange, black);

	/* Draw a rectangle around the central area. */
	gdImageRectangle(im, basex, basey, maxx-basex, maxy-basey, black);
	makebevel(im, maxx, maxy, 6, white, grey1, grey2, grey3);

	gdImageInterlace(im, 1);		/* make it interlaced */
	if ((out = fopen(name, "w")) != NULL) {
		gdImageGif(im, out);
		(void) fclose(out);
	}
	gdImageDestroy(im);
	return 1;
}

/* draw a graph */
static void drawgraph(gdImagePtr im, u_long *values, u_long maxval,
		      int bx, int by, int rx, int ry,
		      int steps, int size, int fcol, int bcol) {
	int idx, pt, top, left;
	int hdist = rx / size;

	pt = 0;
	points[pt].x = bx;	/* start point */
	points[pt++].y = by+ry+4;

	for (idx=0; idx < steps; idx++) {
		top = by+ry+4-(values[idx]*ry/maxval);
		left = bx+(idx*hdist);
		
		points[pt].x = left;
		points[pt++].y = top;
	}
	points[pt].x = left;	/* end point */
	points[pt++].y = by+ry+4;

	gdImageFilledPolygon(im, points, pt, fcol);
	gdImagePolygon(im, points, pt, bcol);
	return;
}

/* draw graph by month */
int graph(int sizex, int sizey, int base, char *name) {
	char lbuf[SMALLSIZE];
	int idx, len;
	u_long hits_max = 0L;	
	u_long file_max = 0L;
	u_long site_max = 0L;
	u_long kb_max = 0L;

	/* the image size */
	int maxx = sizex-1;
	int maxy = sizey-1;

	/* the drawable region */
	int basex = base;
	int basey = base;
	int rangex = maxx-(2*basex);
	int rangey = maxy-(2*basey);

	int bl1 = (rangey)/2;
	int bl2 = (rangey)/4;
	int hdist = rangex / 12;

	/* the font baselines */
	int flx = basex-12;		/* left X axis */
	int frx = maxx-basex;		/* right X axis */
	int fty = basey-18;		/* top Y axis */
	int fby = maxy-basey+4;		/* bottom Y axis */

	/* colors */
	int white, grey1, grey2, grey3, black, red, green, blue, orange;

	int labels = (base > 0);	/* draw labels? */

	FILE *out;
	gdImagePtr im;

	for (idx=0; idx < 13; idx++) {
		if (mn_hits[idx] > hits_max)
			hits_max = mn_hits[idx];
		if (mn_files[idx] > file_max)
			file_max = mn_files[idx];
		if (mn_sites[idx] > site_max)
			site_max = mn_sites[idx];
		if (mn_kbytes[idx] > kb_max)
			kb_max = mn_kbytes[idx];
	}

	if (!base && (hits_max == 0 || site_max == 0 || kb_max == 0))
		return 0;

	if (!base) {	/* reserve some space for bevel */
		sizex += 9; sizey += 9;
		maxx += 9;  maxy += 9;

		/* the drawable region */
		basex = 5;
		basey = 5;
	}

	/* create the image, allocate colors */
	if ((im = gdImageCreate(sizex, sizey)) == NULL)
		return 0;

	white = gdImageColorAllocate(im, 255, 255, 255);
	black = gdImageColorAllocate(im, 0, 0, 0);
	red = gdImageColorAllocate(im, 220, 0, 0);
	green = gdImageColorAllocate(im, 0, 170, 0);
	blue = gdImageColorAllocate(im, 0, 0, 220);
	orange = gdImageColorAllocate(im, 255, 80, 0);

	if (labels) {
		grey1 = gdImageColorAllocate(im, 242, 242, 242);
		grey2 = gdImageColorAllocate(im, 102, 102, 102);
		grey3 = gdImageColorAllocate(im, 153, 153, 153);
	} else {
		grey1 = gdImageColorAllocate(im, 63, 63, 63);
		grey2 = gdImageColorAllocate(im, 95, 95, 95);
		grey3 = gdImageColorAllocate(im, 215, 215, 215);
	}

	if (hits_max == 0 || site_max == 0 || kb_max == 0) {	/* no hits ?!? */
		(void) sprintf(lbuf, "not hits/sites/kbytes for current and last 12 month?!?");
		gdImageString(im, gdFontSmall,
			basex+(rangex-strlen(lbuf)*FWID(gdFontSmall))/2U, basey+(rangey/2), lbuf, black);
		gdImageInterlace(im, 1);		/* make it interlaced */
		if ((out = fopen(name, "w")) != NULL) {
			gdImageGif(im, out);
			(void) fclose(out);
		}
		gdImageDestroy(im);
		return 1;
	}

	if (labels) {
		for (idx=0; idx < 13; idx++) {
			if (!idx)
				(void) sprintf(lbuf, "%3.3s %d", mn_str[idx], EPOCH(tend.year-1));
			else if (idx == 12)
				(void) sprintf(lbuf, "%3.3s %d", mn_str[idx], EPOCH(tend.year));
			else	(void) sprintf(lbuf, "%3.3s", mn_str[idx]);
			gdImageString(im, gdFontSmall,
				basex+(idx*hdist)-(strlen(lbuf)*FWID(gdFontSmall))/2U, fby, lbuf, black);
		}

		(void) sprintf(lbuf, "%lu", kb_max);
		gdImageStringUp(im, gdFontSmall, flx, maxy-basey-bl2+strlen(lbuf)*FWID(gdFontSmall), lbuf, orange);
		(void) sprintf(lbuf, "%lu", site_max);
		gdImageStringUp(im, gdFontSmall, flx, maxy-basey-(2*bl2)+strlen(lbuf)*FWID(gdFontSmall), lbuf, red);
		(void) sprintf(lbuf, "%lu", hits_max);
		gdImageStringUp(im, gdFontSmall, flx, basey+strlen(lbuf)*FWID(gdFontSmall), lbuf, green);

		(void) sprintf(lbuf, "Hits");
		len = strlen(lbuf);
		gdImageString(im, gdFontLarge, basex, fty, lbuf, green);
		gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "/", black);
		len++;
		(void) sprintf(lbuf, "Files");
		gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, lbuf, blue);
		len += strlen(lbuf);
		gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "/", black);
		len++;
		(void) sprintf(lbuf, "Sites");
		gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty,
			lbuf, red);
		len += strlen(lbuf);
		gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "/", black);
		len++;
		(void) sprintf(lbuf, "Kilobytes");
		gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty,
			lbuf, orange);
		len += strlen(lbuf)+1;
		gdImageString(im, gdFontLarge, basex+(len*FWID(gdFontLarge)), fty, "by month", black);
		gdImageStringUp(im, gdFontSmall, frx+1, sizey/2+gdFontSmall->w*31/2, 
			"Copyright 1996 RENT-A-GURU (TM)", black);

		gdImageLine(im, basex, bl1+basey, basex+12*hdist, bl1+basey, black);
		gdImageLine(im, basex, bl1+bl2+basey, basex+12*hdist, bl1+bl2+basey, black);
	}

	drawgraph(im, mn_hits, hits_max, basex, basey, rangex, bl1-4,
		13, 12, green, black);

	drawgraph(im, mn_files, hits_max, basex, basey, rangex, bl1-4,
		13, 12, blue, black);

	drawgraph(im, mn_sites, site_max, basex, basey+bl1, rangex, bl2-4,
		13, 12, red, black);

	drawgraph(im, mn_kbytes, kb_max, basex, bl1+bl2+basey, rangex, bl2-4,
		13, 12, orange, black);

	/* Draw a rectangle around the central area. */
	gdImageRectangle(im, basex, basey, basex+12*hdist, maxy-basey, black);

	if (labels)
		makebevel(im, maxx, maxy, 6, white, grey1, grey2, grey3);
	else
		addbevel(im, maxx, maxy, 4, grey1, grey2, white, grey3);

	gdImageInterlace(im, 1);		/* make it interlaced */
	if ((out = fopen(name, "w")) != NULL) {
		gdImageGif(im, out);
		(void) fclose(out);
	}
	gdImageDestroy(im);
	return 1;
}

#define RADIUS		120
#define PI		3.14159265358979323846
#define MINPERCENT	1.4
#define MAX_CNTRY	14

#define COLOR_BG	0
#define COLOR_BLACK	1
#define COLOR_WHITE	2
#define COLOR_PWHITE	3
#define COLOR_IDX	4
#define NUM_COLORS	15

static int colors[NUM_COLORS];
static int tcolor[NUM_COLORS];

static struct {
	int x, y;
	int x2, y2;
	int fillcol;
	int textcol;
	char *name;
} pts[MAX_CNTRY+1];

int c_chart(int sizex, int sizey, int base, char *name, COUNTRY **ccp, size_t max) {
	gdImagePtr im;
	double scale, ht;
	double percent;
	char *cp, lbuf[50];
	int idx, pt, other;
	int tidx, cidx;
	int grey1, grey2, grey3;
	int leftb = (sizex-2*RADIUS)/2-4;
	int rightb = leftb+(2*RADIUS)+8;
	int oldx = 0, oldy = 0;
	long total_sum = 0L;
	long total_other = 0L;
	FILE *out;

	im = gdImageCreate(sizex, sizey);
	colors[COLOR_BG] = gdImageColorAllocate(im, 2, 2, 2);
	colors[COLOR_BLACK] = gdImageColorAllocate(im, 0, 0, 0);
	colors[COLOR_WHITE] = gdImageColorAllocate(im, 255, 255, 255);
	colors[COLOR_PWHITE] = gdImageColorAllocate(im, 220, 220, 220);
	grey1 = gdImageColorAllocate(im, 242, 242, 242);
	grey2 = gdImageColorAllocate(im, 102, 102, 102);
	grey3 = gdImageColorAllocate(im, 153, 153, 153);

	cidx = COLOR_IDX;
	colors[cidx] = gdImageColorAllocate(im,   0,   0, 154); /* navyblue */
	tcolor[cidx++] = colors[COLOR_WHITE];

	colors[cidx] = gdImageColorAllocate(im, 204,   0, 255); /* violet */
	tcolor[cidx++] = colors[COLOR_WHITE];

	colors[cidx] = gdImageColorAllocate(im,  52, 255,   0); /* green */
	tcolor[cidx++] = colors[COLOR_BLACK];

	colors[cidx] = gdImageColorAllocate(im,  52,   0,   0); /* brown */
	tcolor[cidx++] = colors[COLOR_WHITE];

	colors[cidx] = gdImageColorAllocate(im,   0, 204, 255); /* skyblue */
	tcolor[cidx++] = colors[COLOR_BLACK];

	colors[cidx] = gdImageColorAllocate(im, 255, 103,   0); /* orange */
	tcolor[cidx++] = colors[COLOR_BLACK];

	colors[cidx] = gdImageColorAllocate(im, 255, 154, 255); /* pink */
	tcolor[cidx++] = colors[COLOR_BLACK];

	colors[cidx] = gdImageColorAllocate(im,   0,  52,   0); /* darkgreen */
	tcolor[cidx++] = colors[COLOR_WHITE];

	colors[cidx] = gdImageColorAllocate(im, 255, 255,   0); /* yellow */
	tcolor[cidx++] = colors[COLOR_BLACK];

	colors[cidx] = gdImageColorAllocate(im, 255,   0,   0); /* red */
	tcolor[cidx++] = colors[COLOR_BLACK];

	colors[cidx] = gdImageColorAllocate(im, 154, 154,   0); /* olive */
	tcolor[cidx++] = colors[COLOR_BLACK];

	gdImageFilledRectangle(im, 0, 0, sizex-1, sizey-1, colors[COLOR_BG]);
	gdImageArc(im, sizex/2, sizey/2, RADIUS*2, RADIUS*2-60, 0, 360, colors[COLOR_BLACK]);   
	gdImageFill(im, sizex/2, sizey/2, colors[COLOR_BLACK]);

	scale = total_hits ? (PI*2)/total_hits : 0.0;
	total_sum = total_other = 0L;
	cidx = tidx = COLOR_IDX;
	other = 0;

	for (idx=0, pt=0; idx < max && pt < TABSIZE(pts); idx++) {
		if (idx >= MAX_CNTRY || other ||
		    (percent = (ccp[idx]->count*100.0)/total_hits) < MINPERCENT) {
			total_other += ccp[idx]->count;
			total_sum += ccp[idx]->count;
			other = 1;
			if (idx != max-1)
				continue;
		} else {
			ht = total_sum+ccp[idx]->count/2.0;
			total_sum += ccp[idx]->count;
		}

		if (other) {
			percent = (total_other*100.0)/total_hits;
			ht = total_sum-total_other/2.0;
			(void) sprintf(lbuf, "Other %d%%", (int)rint(percent));
		} else
			(void) sprintf(lbuf, "%.14s %d%%", ccp[idx]->name, (int)rint(percent));

		if (idx == max-1 && cidx == COLOR_IDX) {
			cidx += 2;	/* avoid same color for consecutive countries */
			tidx += 2;
		}
		pts[pt].name = strsave(lbuf);	/* check return code below */
		pts[pt].x = (int)(cos((double)total_sum*scale)*RADIUS+(sizex/2));
		pts[pt].y = (int)(sin((double)total_sum*scale)*RADIUS+(sizey/2));
		pts[pt].x2 = (int)(cos((double)ht*scale)*(RADIUS/2)+(sizex/2));
		pts[pt].y2 = (int)(sin((double)ht*scale)*(RADIUS/2)+(sizey/2));
		pts[pt].fillcol = colors[cidx++];
		pts[pt].textcol = tcolor[tidx++];
		pt++;

		if (cidx == TABSIZE(colors))
			cidx = COLOR_IDX;

		if (tidx == TABSIZE(tcolor))
			tidx = COLOR_IDX;
	}

	idx = 0;
	gdImageLine(im, sizex/2, sizey/2,
		pts[idx].x, pts[idx].y, colors[COLOR_BG]);
	for (++idx; idx < pt; idx++) {
		gdImageLine(im, sizex/2, sizey/2,
			pts[idx].x, pts[idx].y, colors[COLOR_BG]);
		gdImageFillToBorder(im, pts[idx].x2, pts[idx].y2,
			colors[COLOR_BG], pts[idx].fillcol);
	}
	gdImageFillToBorder(im, pts[0].x2, pts[0].y2,
		colors[COLOR_BG], pts[0].fillcol);

	idx = (sizex-2*RADIUS)/2;
	gdImageArc(im, sizex/2, sizey/2, RADIUS*2, RADIUS*2-60, 0, 360, colors[COLOR_BLACK]);   
	gdImageLine(im, idx, sizey/2, idx, sizey/2+12, colors[COLOR_BLACK]);
	gdImageLine(im, sizex-idx, sizey/2, sizex-idx, sizey/2+12, colors[COLOR_BLACK]);
	gdImageArc(im, sizex/2, sizey/2+12, RADIUS*2, RADIUS*2-60, 0, 180, colors[COLOR_BLACK]);
	gdImageFill(im, 0, 0, colors[COLOR_WHITE]);
	gdImageFill(im, sizex/2, sizey/2+RADIUS-20, colors[COLOR_BLACK]);

	for (idx=0; idx < pt; idx++) {
		int x, y, txtpos, len;

		if (!pts[idx].x)
			continue;

		x = (pts[idx].x2 <= sizex/2) ? leftb : rightb;
		if ((y=pts[idx].y2) <= sizey/2) {	/* T */
			y -= RADIUS/4;
		} else {				/* B */
			y += RADIUS/4;
		}

                if (oldx == x && oldy != 0) {
			if (x > sizex/2) {
                                if (y < oldy + 12 || (y < oldy + 48) && (y > oldy + 12))
                                        y = oldy + 12;
                        } else if (y > oldy - 12 || (y > oldy - 48) && (y < oldy - 12)) {
                                y = oldy - 12;
			}
                }
		oldx = x;
		oldy = y;
		
		txtpos = (x <= sizex/2) ? leftb-4 : rightb+4;
		gdImageLine(im, pts[idx].x2, pts[idx].y2, x, y, colors[COLOR_BLACK]);
		gdImageLine(im, x, y, txtpos, y, colors[COLOR_BLACK]);

		if (pts[idx].name != NULL) {
			len = strlen(pts[idx].name)*gdFontSmall->w;
			if (x <= sizex/2) {
				txtpos -= len;
				gdImageFilledRectangle(im, txtpos, y-5, txtpos+len, y+4, pts[idx].fillcol);
			} else
				gdImageFilledRectangle(im, txtpos, y-5, txtpos+len, y+4, pts[idx].fillcol);
			gdImageString(im, gdFontSmall,
				txtpos, y-gdFontSmall->h/2,
				pts[idx].name, pts[idx].textcol); 

			free(pts[idx].name);
		}
	}
	makebevel(im, sizex-1, sizey-1, 6, colors[COLOR_WHITE], grey1, grey2, grey3);

	cp = "Total transfers by Country";
	gdImageString(im, gdFontLarge,
		(sizex-strlen(cp)*gdFontLarge->w)/2U, sizey-base/2-gdFontLarge->h,
		cp, colors[COLOR_BLACK]);

	gdImageInterlace(im, 1);		/* make it interlaced */
	if ((out = fopen(name, "w")) != NULL) {
		gdImageGif(im, out);
		(void) fclose(out);
	}
	gdImageDestroy(im);
	return 1;
}
 
ICON_TAB icon_tab[] = {
	{ "sq_green.gif",   0,  170,   0 },
	{ "sq_blue.gif",    0,    0, 220 },
	{ "sq_red.gif",    220,   0,   0 },
	{ "sq_orange.gif", 222,  84,   0 },
	{ "sq_yellow.gif", 242, 242,   0 }
};

static void mkIcon(ICON_TAB *tp) {
	FILE *out;
	gdImagePtr im;
	int bgcol;

	if ((im=gdImageCreate(10, 8)) != NULL) {
		bgcol = gdImageColorAllocate(im, tp->color[0], tp->color[1], tp->color[2]);
#if defined(lint)		/* first color allocated is background */
		bgcol = bgcol;
#endif

		gdImageInterlace(im, 1);
		if ((out=fopen(tp->name, "w")) != NULL) {
			gdImageGif(im, out);
			(void) fclose(out);
		}
		gdImageDestroy(im);
	}
	return;
}

void checkForIcons(void) {
	int idx;

	for (idx=0; idx < TABSIZE(icon_tab); idx++)
		if (access(icon_tab[idx].name, 0))
			mkIcon(&icon_tab[idx]);
	return;
}
