/*
    Wn: A Server for the HTTP
    File: wn/chkcntrl.c
    Version 2.0.4

    Copyright (C) 1995-8  <by John Franks>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "wn.h"
#include "chkcntrl.h"
#include "access.h"


extern int	chkauth();

extern long	atol();

static int	mask_match();

static void	setdirvalue(),
		setvalue(),
		do_serveall();


/*

 * chk_cntrl( ip) checks to see that the URL sent by the client is a
 * valid one, i.e. that basename exists in a control file in the
 * designated directory.  It also enters some additional fields in the
 * Request struct pointed to by ip, namely "title", "content_type",
 * "encoding", "wrappers", "includes", etc.  It gets this information
 * from the cached control file.

 */

void
chk_cntrl( ip)
Request	*ip;
{
	struct stat stat_buf;

	FILE	*fp;

	int	i,
		access_status = ACCESS_DENIED;

	Cache_entry	*cep,
			entry;

#ifdef ACCESS_DENIED_URL
	mystrncpy( dir_p->noaccess_url, ACCESS_DENIED_URL, MIDLEN/2);
#else
	*dir_p->noaccess_url = '\0';
#endif

#ifdef NO_SUCH_FILE_URL
	mystrncpy( dir_p->cantstat_url, NO_SUCH_FILE_URL, MIDLEN/2);
#else
	*dir_p->cantstat_url = '\0';
#endif

#ifdef AUTH_DENIED_FILE
	mystrncpy( dir_p->authdenied_file, AUTH_DENIED_FILE, MIDLEN/2);
#else
	*dir_p->authdenied_file = '\0';
#endif
	if ( ip->type == RTYPE_DENIED )  /* done in parse_request */
		return;

	if ( lstat( ip->cachepath, &stat_buf) != 0 ) {
		writelog( ip,  err_m[2], ip->cachepath);
		ip->type = RTYPE_DENIED;
		ip->status |= WN_CANT_STAT;
		return;
	}


#ifndef S_ISREG
#define	S_ISREG(m)	(((m)&S_IFMT) == S_IFREG)
#endif

	if ( !(S_ISREG(stat_buf.st_mode))) {
		senderr( SERV_ERR, err_m[13], ip->cachepath);
		return;
	}
	dir_p->cache_uid = (unsigned) stat_buf.st_uid;
	dir_p->cache_gid = (unsigned) stat_buf.st_gid;
	dir_p->cmod_time =  stat_buf.st_mtime;

	if ( (!WN_OPT_U) && (WN_OPT_T) && (!IS_TRUSTED) ) {
		logerr( err_m[7], ip->cachepath);
		ip->type = RTYPE_DENIED;
		return;
	}

	if ( stat_buf.st_uid == (uid_t) user_id) {
		logerr( err_m[8], ip->cachepath);
		ip->type = RTYPE_DENIED;
		return;
	}

	if ( (fp = fopen( ip->cachepath, "r")) == (FILE *) NULL ) {
		logerr( err_m[9], ip->cachepath);
		ip->type = RTYPE_DENIED;
		return;
	}

	read_cache_dirinfo( fp, dir_p);

	access_status = chkaccess( ip->cachepath,  dir_p->accessfile);

	switch( access_status) {
	case ACCESS_GRANTED:
	case ACCESS_PRIVILEGED:
		break;
	case ACCESS_DENIED:
		ip->type = RTYPE_NOACCESS;
		fclose( fp);
		return;
	default:
		ip->type = RTYPE_DENIED;
		fclose( fp);
		return;
	}

	if ( *(dir_p->defdoc) && ( ip->filetype & WN_DEFAULT_DOC)) {
		fclose( fp);
		dolocation( dir_p->defdoc, ip, 301);
		ip->type = RTYPE_FINISHED;
		return;
	}

	if ( *(dir_p->authmod) && (access_status != ACCESS_PRIVILEGED) ) {
		if ( !(IS_TRUSTED || IS_ATRUSTED) ) {
			senderr( DENYSTATUS, err_m[90], dir_p->authmod);
			ip->type = RTYPE_FINISHED;
			fclose( fp);
			return;
		}

		exec_ok( ip);

		if ( !getfpath( dir_p->authmodule, dir_p->authmod, ip)) {
			senderr( SERV_ERR, err_m[45], dir_p->authmod);
			ip->type = RTYPE_FINISHED;
			fclose( fp);
			return;
		}
		dir_p->authmod = dir_p->authmodule;

		/* anything requiring authentication should not be cached */
		ip->attributes |= WN_NOCACHE;

		if ( !(strcasecmp( dir_p->authtype, "Digest") == 0) &&
						 !chkauth( ip)) {
			/*
			 * chkauth must be delayed for Digest until we know the
			   md5 digest.
			 */
			ip->type = RTYPE_NO_AUTH;
			fclose( fp);
			return;
		}
	}

	if ( *(dir_p->filemod)) {
		exec_ok( ip);
		if ( !getfpath( dir_p->filemodule, dir_p->filemod, ip)) {
			senderr( SERV_ERR, err_m[45], dir_p->filemod);
			return; /* to process_url */
		}
		dir_p->filemod = dir_p->filemodule;

		if ( WN_OPT_U )
			check_perm( ip, dir_p->filemodule);

		ip->attrib2 |= WN_FILEMOD;
	}

	if ( *(dir_p->cachemod)) {
		exec_ok( ip);
		if ( !getfpath( dir_p->cachemodule, dir_p->cachemod, ip)) {
			senderr( SERV_ERR, err_m[45], dir_p->cachemod);
			return; /* to process_url */
		}
		dir_p->cachemod = dir_p->cachemodule;
		if ( WN_OPT_U )
			check_perm( ip, dir_p->cachemodule);

		if ( (dir_p->defattributes) & WN_CGI)
			cgi_env( ip, FALSE);
	}

	if ( *(dir_p->indexmod)) {
		exec_ok( ip);
		if ( !getfpath( dir_p->indexmodule, dir_p->indexmod, ip)) {
			senderr( SERV_ERR, err_m[45], dir_p->indexmod);
			return;
		}
		dir_p->indexmod = dir_p->indexmodule;
		if ( WN_OPT_U )
			check_perm( ip, dir_p->indexmodule);
	}

	if ( iswndir( ip)) {
		/* It's a title, keyword, grep, or index search of this dir */
		ip->allowed |= WN_M_GET + WN_M_TRACE;
		fclose( fp);
		ip->title = ip->encoding = ip->keywords 
			= ip->filter = ip->handler = wn_empty;
		ip->content_type = "text/html";
		return;  
	}

	cep = &entry;
	cep->line = ip->cacheline;  
	/* actual cache line goes in ip struct not cep struct */

	while ( read_cache_file( cep, fp, ip->basename)) {
		if ( !*cep->basename)
			continue;
		if ( streq( ip->basename, cep->basename)) {
			if ( ip->type == RTYPE_UNCHECKED)
				ip->type = RTYPE_FILE;

			ip->attributes |= 
				(entry.attributes ? entry.attributes :
						dir_p->defattributes);

			ip->allowed |= WN_M_GET + WN_M_HEAD
						+ WN_M_TRACE + WN_M_OPTIONS;


			ip->title = entry.title;
			ip->content_type = entry.content; 
			ip->encoding = entry.encoding; 
			ip->keywords = entry.keywords; 

			ip->includes = dir_p->defincludes;
			ip->wrappers = dir_p->defwrapper;
			ip->list_incl = dir_p->deflist;

			if ( *entry.includes)
				ip->includes = entry.includes;
			if ( (*ip->includes) &&
					!streq( ip->includes, "<none>")) {
				ip->attrib2 |= WN_INCLUDE;
				ip->attrib2 &= ~(WN_LIST_INCL);
			}
			else
				ip->includes = wn_empty;

			if ( *entry.wrappers)
				ip->wrappers = entry.wrappers; 
			if ( (*ip->wrappers) &&
					!streq( ip->wrappers, "<none>")) {
				ip->attrib2 |= WN_WRAPPED;
				ip->attrib2 &= ~(WN_LIST_INCL);
			}
			else
				ip->wrappers = wn_empty;

			ip->inclptr = ip->wrappers; 

			if ( *entry.list_incl)
				ip->list_incl = entry.list_incl;

			if ( (*ip->list_incl) &&
					!streq( ip->list_incl, "<none>")) {
				ip->attrib2 |= WN_LIST_INCL;
				/* shut off any includes */
				ip->attrib2 &= ~(WN_INCLUDE + WN_WRAPPED);
				ip->includes = ip->wrappers = wn_empty;
			}
			else
				ip->list_incl = wn_empty;

			ip->swrapper = entry.swrapper; 
			ip->nomatchsub = entry.nomatchsub; 
			ip->filter = entry.filter;

			ip->expires = entry.expires;
			ip->maxage = ((*entry.maxage) 
				? entry.maxage : dir_p->default_maxage);

			ip->handler = ((*entry.handler) 
				? entry.handler : dir_p->def_handler);

			if ( *ip->handler) 
				if ( streq( ip->handler, "<none>")) {
					ip->handler = wn_empty;
					ip->attributes &= ~WN_CGI;
				}
				else {
					ip->attributes |= 
					(WN_CGI + WN_DYNAMIC +
					 WN_NOCACHE + WN_NOSEARCH);
					ip->type = RTYPE_CGI_HANDLER;
				}

			/* By default CGI is dynamic non-cachable */
			if ( (ip->type == RTYPE_CGI ) ||
						(ip->type == RTYPE_NPH_CGI ))
				ip->attributes |= 
					(WN_CGI + WN_DYNAMIC +
					 WN_NOCACHE + WN_NOSEARCH);

			/* But it can be non-dynamic  */
			if ( ip->attributes & WN_NONDYNAMIC)
				ip->attributes &= ~(WN_DYNAMIC);

			if ( ip->attributes & WN_CACHEABLE)
				ip->attributes &= ~(WN_NOCACHE);

			if ( ip->attributes & WN_CGI ) {
				ip->allowed |= (WN_M_GET + WN_M_POST);
			}

			if ( ip->attributes & WN_POST_ONLY ) {
				ip->allowed |= WN_M_POST;
				ip->allowed &= ~WN_M_GET;
			}

			if ( ip->attributes & WN_NO_POST)
				ip->allowed &= ~WN_M_POST;

			if ( ip->attributes & WN_PUT_OK)
				ip->allowed |= WN_M_PUT;

			ip->filter = ((*entry.filter) 
				? entry.filter : dir_p->def_filter);

			if ( *ip->filter) 
				if ( streq( ip->filter, "<none>")) {
					ip->filter = wn_empty;
					ip->attributes &= ~WN_FILTERED;
				}
				else {
					ip->attributes |= WN_FILTERED;
				}

			for ( i = 0; i < NUMFIELDS; i++) {
				ip->field[i] = entry.field[i];
			}

			ip->filetype |= entry.filetype;

			if ( ip->attrib2 & 
				(WN_INCLUDE + WN_WRAPPED + WN_LIST_INCL)) {
				update_mod_time( ip);
				ip->attributes |= WN_PARSE;
			}

			if ( (ip->filetype & WN_IMAGEMAP) ||
					(ip->attributes & WN_ISMAP)) {
				ip->type = RTYPE_IMAGEMAP;
			}

			if ( ip->attributes & WN_NOPARSE)
				ip->attributes &= ~(WN_PARSE);

			if ( strncasecmp( ip->content_type, "text", 4) == 0 )
				ip->filetype |= WN_TEXT;
			if ( streq( ip->content_type, "text/html"))
				ip->filetype |= WN_ISHTML;

			if ( *entry.headerlines)
				strcpy( outheadp->list, entry.headerlines);
				/* both have size BIGLEN */

			if ( *entry.status)
				mystrncpy( outheadp->status, 
						entry.status, SMALLLEN);

			if ( *entry.md5) {
				mystrncpy( outheadp->md5, "Content-MD5: ", 20);
				mystrncat( outheadp->md5, entry.md5, TINYLEN + 2);
				mystrncat( outheadp->md5, "\r\n", 3);
			}

			if ( *entry.redirect) {
				char 	*cp;

				ip->type = RTYPE_REDIRECT;
				mystrncpy( outheadp->redirect, 
						entry.redirect, SMALLLEN);

				cp = strrchr( outheadp->redirect, '?');
				if ( (cp != NULL) && (*(cp+1) == '\0') ) {
					if  ( *ip->query )
						mystrncpy( ++cp, ip->query, 
							MIDLEN - SMALLLEN);
					else
						*cp = '\0';
				}
			}
			else if ( (ip->status & WN_CANT_STAT) ) {
				if (*dir_p->filemod == '\0')
					; /* filemod handles it */
				else if ( (ip->allowed & WN_M_PUT) &&
						(inheadp->method == PUT))
					; /* put file may not exist */
				else
					ip->type = RTYPE_DENIED;
			}
			if ( strcasecmp( dir_p->authtype, "Digest") == 0 &&
							!chkauth( ip)) {
				ip->type = RTYPE_NO_AUTH;
			}
			fclose( fp);
			return;
		}
	}

#ifdef NO_SERVEALL
	ip->type = RTYPE_DENIED;
#else
	if ( dir_p->attributes & WN_SERVEALL) {
		if ( strcasecmp( dir_p->authtype, "Digest") == 0 &&
							!chkauth( ip)) {
			ip->type = RTYPE_NO_AUTH;
			fclose( fp);
			return;
		}
		do_serveall( ip);
	}
	else
		ip->type = RTYPE_DENIED;
#endif
	fclose( fp);
	return;


}



/*
 * do_serveall( ip) The file ip->filename exists but is not in the index.cache
 * file.  This function sets ip->content (and ip->encoding if needed) based
 * on the filename suffix.
 */

static void
do_serveall( ip )
Request	*ip;
{
	register char	*cp,
			*cp2;

	char		suffix[10];

	ip->title = ip->basename;

	if ( !*(dir_p->default_content))
		dir_p->default_content = DEFAULT_CONTENT_TYPE;
	ip->allowed |= WN_M_GET + WN_M_HEAD + WN_M_TRACE + WN_M_OPTIONS;
	ip->content_type = dir_p->default_content;
	ip->maxage = dir_p->default_maxage;
	ip->handler = dir_p->def_handler;
	ip->filter = dir_p->def_filter;
	if ( *ip->handler || *ip->filter)
		ip->attributes |= WN_NOSEARCH;
	ip->attributes |= dir_p->defattributes;

	if ( *(dir_p->filemod))
		/* skip check on file name */
		;
	else if ( (*(ip->title) == '.') || streq( ip->title, CACHEFNAME) ||
				(ip->status & WN_CANT_STAT) ||
				streq( ip->title, dir_p->accessfile) ||
				streq( ip->title, CONTROLFILE_NAME)) {
		ip->type = RTYPE_DENIED;
		return;
	}

	if ( ip->type == RTYPE_UNCHECKED)
		ip->type = RTYPE_FILE;

	if ( (ip->type == RTYPE_CGI ) || (ip->type == RTYPE_NPH_CGI ))
		ip->attributes |= (WN_DYNAMIC + WN_NOSEARCH);

	if ( (cp = strrchr( ip->basename, '.')) == NULL )
		return;

	mystrncpy( suffix, cp+1, 10);
	strlower( suffix);
	if ( streq( suffix, "gz") || streq( suffix, "z")) {
		ip->encoding = ( suffix[0] == 'g' ? "x-gzip" : "x-compress");
		*cp = '\0';
		if ( (cp2 = strrchr( ip->basename, '.')) == NULL ) {
			*cp = '.';
			ip->content_type = dir_p->default_content;
			return;
		}
		mystrncpy( suffix, cp2+1, 10);
		strlower( suffix);
		*cp = '.';
	}

#ifdef PARSE_EXT
	if ( streq( suffix, PARSE_EXT) && !(ip->attributes  & WN_NOPARSE))
		ip->attributes |= WN_PARSE;
#endif
	if ( streq( suffix, CGI_EXT + 1) )
		ip->attributes |= WN_NOSEARCH;

	get_mtype( suffix);
	if ( strncasecmp( ip->content_type, "text", 4) == 0 )
		ip->filetype |= WN_TEXT;
	if ( streq( ip->content_type, "text/html"))
		ip->filetype |= WN_ISHTML;

	return;
}


/*
 * chkaccess( cachepath, accessfile) checks whether the client's IP address
 * is in the allowed list in accessfile.  Returns ACCESS_PRIVILEGED if 
 * access is unconditionally allowed, ACCESS_GRANTED if further
 * authentication (through an authentication module) may be required, 
 * ACCESS_DENIED if access is denied and ACCESS_ERR on error.
 */

int
chkaccess( cachepath, accessfile)
char	*cachepath,
	*accessfile;
{
	FILE	*fp;
	int	len,
		notflag,
		priv_flag,
		match = FALSE;

	char	*cp,
		*cp2,
		buf[MIDLEN],
		linebuf[SMALLLEN];

	if ( ! *accessfile)
		return ACCESS_GRANTED;	/* No access control */

	if ( getfpath2( buf, accessfile, cachepath) == FALSE)
		return ACCESS_ERR;

	get_remote_info( );

	if ((fp = fopen( buf, "r")) == (FILE *)NULL ) {
		logerr( err_m[87], buf);
		return ACCESS_ERR;
	}

	while ( fgets( linebuf, SMALLLEN, fp)) {
		if ( !chop( linebuf)) {
			logerr( err_m[62], buf);
			return ACCESS_ERR;
		}

		cp = linebuf;

		if ( strncasecmp( cp, "access-denied-url=", 18) == 0) {
			mystrncpy( dir_p->noaccess_url, cp + 18, MIDLEN/2);
			continue;
		}

		if ( (notflag = ( *cp == '!')))
			cp++;

		if ( (priv_flag = ( *cp == '+')))
			cp++;

		len = strlen( cp);

		if ( (len == 0) || (*cp == '#') )
			continue;

		strlower( cp);

		if ( streq(cp, this_conp->remaddr)
				|| streq(cp, this_conp->remotehost)) {
			match = TRUE;
		}
		else if ( (cp2 = strchr( cp, '/')) != NULL) {
			*cp2++ = '\0';
			match = mask_match( cp, cp2);
		}
		else {
			if ( (cp[len - 1] == '.') && 
				( strncmp( cp, this_conp->remaddr, len) == 0 ))
				match = TRUE;
		}

		if ( match || wild_match( this_conp->remotehost, cp)) {
			if ( notflag)
				break;
			else {
				fclose( fp);
				if ( priv_flag)
					return  ACCESS_PRIVILEGED;
				else
					return  ACCESS_GRANTED;
			}
		}
	}
	fclose( fp);
	return ACCESS_DENIED;
}

/*
 * static int mask_match( net, mask)
 * net is "nnn.nnn.nnn.nnn", mask is "mmm.mmm.mmm.mmm" and remaddr
 * is "rrr.rrr.rrr.rrr".  Return true if nnn == mmm | rrr for each
 * of the four segments of net.  Else return false.
 */

static int
mask_match( net, mask)
char	*net,
	*mask;

{
	int	ipnet[4],
		ipmask[4],
		iprem[4];

	sscanf( net, "%d.%d.%d.%d", 
			&ipnet[0], &ipnet[1], &ipnet[2], &ipnet[3]);
	sscanf( mask, "%d.%d.%d.%d", 
			&ipmask[0], &ipmask[1], &ipmask[2], &ipmask[3]);
	sscanf( this_conp->remaddr, "%d.%d.%d.%d", 
			&iprem[0], &iprem[1], &iprem[2], &iprem[3]);

	return ( (ipnet[0] == (ipmask[0] & iprem[0]))
			&& ( ipnet[1] == (ipmask[1] & iprem[1]))
			&& ( ipnet[2] == (ipmask[2] & iprem[2]))
			&& ( ipnet[3] == (ipmask[3] & iprem[3])));
}


/*
 *  wild_match( char *l, char *p)
 *
 *  String equality routine, including matching the '*' and '?' characters.
 *  The string p contains the wildcards.  '?' matches any single character
 *  while '*' matches any string.
 *  
 */

/*
 *  Borrowed from the ANU News sources V6.1b3 newsrtl.c.  Original sources
 *  Copyright 1989 by Geoff Huston.
 *
 */

int
wild_match( l, p)
char	*l,
	*p;
{
	if (!*l) {
		if (!*p)
			return TRUE;
		else if (*p == '*') 
			return (wild_match( l, p+1));
		else
			return FALSE;
	}
	if (*p == '*') {
		while ( !wild_match( l, p+1)) {
			l++;
			if (!*l) {
				if (!*(p+1))
					return TRUE;
				else
					return FALSE;
				}
			}
		return TRUE;
	}
	if (*p == '?')
		return(wild_match( l+1, p+1));

	return ((*l == *p) && wild_match( l+1, p+1));
}


/*
 * Store the line from fp in dep->dirline.  This line consists of &
 * separated field value pairs (field=value).  Fields are access,
 * swrapper, nomatchsub, subdirs, and owner.
 * Change the &'s and ='s  to '\0' and  make dip->access, etc., point to
 * the right place in dip->line.  Return TRUE unless empty line then FALSE.
 */

void
read_cache_dirinfo( fp, dep)
FILE		*fp;
Dir_info	*dep;
{
	register char	*cp;
	char		*field,
			*value = NULL;

	cp = dep->dirline;
	if ( fgets( cp, BIGLEN, fp) == NULL) {
		*cp++ = '\n';
		*cp = '\0';
	}

	if ( strrchr( dep->dirline, '\n') == NULL) {
		senderr( SERV_ERR, err_m[63], "");
		wn_exit( 2);  /* senderr: SERV_ERR */
	}

	dep->attributes = dep->defattributes = 0;

	while ( *cp)
		cp++;

	dep->accessfile = dep->swrapper = dep->defwrapper = dep->defincludes
	= dep->deflist = dep->subdirs = dep->nomatchsub = dep->dir_owner 
	= dep->cachemod = dep->filemod = dep->indexmod
	= dep->authmod = dep->authtype = dep->authrealm = cp;

	dep->default_content = dep->defdoc = dep->default_maxage
	= dep->def_filter = dep->def_handler = cp;

	*dep->authmodule = *dep->cachemodule
	 = *dep->filemodule = *dep->indexmodule = '\0';

	cp = dep->dirline;

	if ( !*cp )
		return;

	field = cp++;
	while ( *cp) {
		switch (*cp) {
		case '=':
			*cp++ = '\0';
			value = cp;
			break;
		case '&':
			if ( *(cp-1) == '\\')	/* ignore escaped & */
				break;
			*cp = '\0';
			setdirvalue( field, value, cp, dep);
			field = ++cp;
			break;
		case '\n':
			*cp = '\0';
			setdirvalue( field, value, cp, dep);
			return;
		default:
			cp++;
		}
	}
}


static void
setdirvalue( field, value, end, dep)
char		*field,
		*value,
		*end;
Dir_info	*dep;
{
	char	buf[TINYLEN];

	if ( !value || !*value) {
		mystrncpy( buf, field, TINYLEN);
		logerr( err_m[10], buf);
		return;
	}
	if ( (end - value) > MIDLEN/2) {
		mystrncpy( buf, field, TINYLEN);
		logerr( err_m[107], buf);
		value[MIDLEN/2] = '\0';
	}

	switch (*field) {
	case 'a':
		if ( strncmp( field, "auth", 4) == 0) {
			switch( field[4]) {
			case 'd':		/* authdenied_file */
				mystrncpy( dep->authdenied_file, 
							value, MIDLEN/2);
				break;
			case 'm':
				dep->authmod = value;
				break;
			case 'r':
				dep->authrealm = value;
				break;
			case 't':
				dep->authtype = value;
				break;
			}
		}
		else {
			dep->accessfile = value;
		}
		break;

	case 'c':
		dep->cachemod = value;
		break;

	case 'd':
		if ( streq( field, "dwrapper")) {
			dep->attributes |= WN_DIRWRAPPED;
			dep->swrapper = value;
		}

		else if ( streq( field, "defwrapper")) {
			dep->defwrapper = value;
		}

		else if ( streq( field, "defincludes")) {
			dep->defincludes = value;
		}

		else if ( streq( field, "default_filter")) {
			dep->def_filter = value;
		}

		else if ( streq( field, "default_handler")) {
			dep->def_handler = value;
			dep->attributes |= WN_CGI;
		}

		else if ( streq( field, "deflist")) {
			dep->deflist = value;
		}

		else if ( streq( field, "defattributes"))
			dep->defattributes = (unsigned) atol( value);

		else if ( streq(field, "default_content"))
			dep->default_content = value;

		else if ( streq(field, "default_document"))
				dep->defdoc = value;

		else if ( streq(field, "default_maxage"))
				dep->default_maxage = value;

		else
			logerr( err_m[11], field);
		break;

	case 'f':	/* file module */
		dep->filemod = value;
		break;

	case 'i':	/* index module */
		dep->indexmod = value;
		break;

	case 'n':
		switch ( *(field + 2)) {

		case 'a':		/* noaccess_url */
			mystrncpy( dep->noaccess_url, value, MIDLEN/2);
			break;

		case 'f':		/* nofile_url */
			mystrncpy( dep->cantstat_url, value, MIDLEN/2);
			break;

		case 'm':		/* nomatchsub */
			dep->nomatchsub = value;
			break;

		case 's':
			if ( streq( value, "true")) /* nosearch=true */
			dep->attributes |= WN_DIRNOSEARCH;
			break;

		default:
			logerr( err_m[11], field);
			break;
		}
		break;

	case 'o':	/* owner */
		dep->dir_owner = value;
		break;

	case 's':
		if ( field[1] == 'u')		/* subdirs */
			dep->subdirs = value;
		else {				/* serveall */
			if ( streq( value, "true"))
				dep->attributes |= WN_SERVEALL;
		}
		break;
		
	default:
		logerr( err_m[11], field);
	}
}

/*
 * Store the line from fp in cep->line.  This line consists of &
 * separated field value pairs (field=value).  Fields are basename,
 * title, keywords, content, maxage, encoding, type, includes, and wrappers.
 * Change the &'s and ='s  to '\0' and  make cep->basename, cep->title,
 * cep->keywords, cep->content, cep->encoding and cep->type point to
 * the right place in cep->line.  Return TRUE unless no more lines then FALSE.
 */

int
read_cache_file( cep, fp, key)
Cache_entry	*cep;
FILE		*fp;
char		*key;

{
	register char	*cp;
	char		*field,
			*value,
			envkey[SMALLLEN];
	int		c,
			i;

	static FILE	*lfp;

	cp = cep->line;

	if ( *dir_p->cachemod ) {  /* invoke cache module */
		if ( key != NULL ) {
			strcpy( envkey, "WN_KEY=");
			mystrncat( envkey, key, SMALLLEN - 7);
			putenv( envkey);
		}

		if ((lfp = popen( dir_p->cachemod, "r")) == (FILE *)NULL ) {
			senderr( SERV_ERR, err_m[40], dir_p->cachemod);
			wn_exit( 2);  /* senderr: SERV_ERR */
		}

		if ( (c = getc( lfp)) == EOF ) {
			senderr( SERV_ERR, err_m[43], dir_p->cachemod);
			pclose( lfp);
			wn_exit( 2);  /* senderr: SERV_ERR */
		}
		else
			ungetc( c, lfp);

		*dir_p->cachemod = '\0';  /* don't come back here again */
	}	
	else 
		lfp = fp;

	for (;;) {		/* read until non-empty line */
		if ( fgets( cp, CACHELINE_LEN, lfp) == NULL)
			return FALSE;
		if ( *cp != '\n')
			break;
	}

	if ( lfp != fp)    /* It's a cache module and we're done */
		pclose( lfp);

	if ( strrchr( cp, '\n') == NULL) {
		senderr( SERV_ERR, err_m[63], "");
		wn_exit( 2);  /* senderr: SERV_ERR */
	}

	while ( *cp)
		cp++;
	cep->end = cep->basename = cep->title = cep->keywords 
	= cep->content = cep->encoding = cep->includes = cep->list_incl 
	= cep->wrappers = cep->swrapper = cep->nomatchsub
	= cep->filter = cep->expires = cep->maxage
	= cep->status = cep->md5 = cep->url = cep->redirect 
	= cep->handler = cp;

	for ( i = 0; i < NUMFIELDS; i++) {
		cep->field[i] = cp;
	}
	*cep->headerlines = '\0';
	cep->attributes = cep->filetype = 0;
	cp = cep->line;

	field = cp;
	if ( (value = strchr( cp, '=' )) == NULL )
		cp = value = cep->end;
	else {
		*value++ ='\0';
		cp = value;
	}
	while ( *cp) {
		switch (*cp) {
		case '&':
			if ( *(cp-1) == '\\') {	  /* handle escaped & */
				strcpy( cp-1, cp);
				break;
			}
			*cp = '\0';
			setvalue( field, value, cp, cep);
			field = ++cp;
			if ( (value = strchr( cp, '=' )) == NULL )
				cp = value = cep->end;
			else {
				*value++ ='\0';
				cp = value;
			}
			break;
		case '\n':
			*cp = '\0';
			setvalue( field, value, cp, cep);
			break;
		default:
			cp++;
		}
	}
	cp = strrchr( cep->basename, '.');
	if ( (cp && streq( cp, CGI_EXT)) || (this_rp->attrib2 & WN_ISACGIBIN)) {
		cep->attributes |= WN_NOSEARCH;
	}
	return TRUE;
}

static void
setvalue( field, value, end, cep)
char	*field,
	*value,
	*end;
Cache_entry	*cep;
{
	char	buf[SMALLLEN];
	int	i,
		errflg;

	errflg = 0;

	if ( !*value) {
		mystrncpy( buf, field, 28);
		strcat( buf, ":");
		mystrncat( buf, cep->basename, SMALLLEN - 30);
		logerr( err_m[5], buf);
		return;
	}
	if ( (end - value) > MIDLEN/2) {
		mystrncpy( buf, field,  30);
		logerr( err_m[107], buf);
		value[MIDLEN/2] = '\0';
	}

	switch (*field) {
	case 'a':		/* attributes */
		cep->attributes |= (unsigned) atol( value);
		if ( cep->attributes & WN_DYNAMIC) {
			cep->attributes |= WN_NOSEARCH;
		}
		break;
	case 'c':
		cep->content = value;
		break;

	case 'e':
		switch( field[1]) {
		case 'n':
			cep->encoding = value;
			break;
		case 'x':
			cep->expires = value;
			break;
		default:
			errflg++;
		}
		break;
	case 'f':
		switch( field[3]) {
		case 'e':		/* file */
			cep->basename = value;
			*cep->url = '\0';  /* can't have url & basename */
			break;
		case 'l':		/* field */
			i = atoi( field + 5);
			if ( (i >= 0 ) && ( i < NUMFIELDS))
				cep->field[i] = value;
			break;
		case 't':		/* filter */
			cep->filter = value;
			break;
		default:
			errflg++;
		}
		break;
	case 'h':
		if ( streq( field, "handler")) {
			cep->handler = value;
			if ( !streq( value, "<none>")) {
				cep->attributes |= WN_NOSEARCH;
			}
		}


		else if ( streq( field, "header")) {
			if ( strlen( cep->headerlines) + 
						strlen( value) > BIGLEN ) {
				senderr( SERV_ERR, err_m[70], value);
				wn_exit( 2);  /* senderr: SERV_ERR */
			}
			strcat( cep->headerlines, value);
			strcat( cep->headerlines, "\n");
		}
		else
			errflg++;
		break;
	case 'i':
		cep->includes = value;
		cep->attrib2 |= WN_INCLUDE;
		break;
	case 'k':
		cep->keywords = value;
		break;
	case 'l':			/* list */
		cep->list_incl = value;
		cep->attrib2 |= WN_LIST_INCL;
		break;
	case 'm':
		if ( streq( field, "md5"))
			cep->md5 =  value;
		else 	if ( streq( field, "maxage"))
			cep->maxage = value;
		else
			errflg++;
		break;
	case 'n':
		if ( *(field+2) == 'm')  /* nomatchsub */
			cep->nomatchsub = value;
		else if ( streq( value, "true")) /* nosearch=true */
			cep->attributes |= WN_NOSEARCH;
		else
			errflg++;
		break;
	case 'r':
		cep->redirect = value;
		break;
	case 's':
		if ( streq( field, "swrapper")) {
			cep->swrapper = value;
			cep->attrib2 |= WN_SWRAPPED;
		}
		else if ( streq( field, "status")) {
			if ( strlen( value) > SMALLLEN ) {
				senderr( SERV_ERR, err_m[70], value);
				wn_exit( 2);  /* senderr: SERV_ERR */
			}
			cep->status = value;
		}
		else
			errflg++;
		break;
	case 't':		/* title */
		cep->title = value;
		break;
	case 'u':
		cep->url = value;
		*cep->basename = '\0';  /* can't have url & basename */
		break;
	case 'w':
		cep->wrappers = value;
		cep->attrib2 |= WN_WRAPPED;
		break;
	default:
		errflg++;
	}

	if ( errflg) 
		logerr( err_m[6], field);
}


/*
 * check_perm( ip, buf)
 * This stats the file named in buf, then checks if the index.cache is
 * owned by trusted user or group (options -t & -T) or if the index.cache
 * owner must own the file ( option -u). It sends an error and quits
 * if permssion is not allowed.
 */

void
check_perm( ip, buf)
Request	*ip;
char	*buf;
{
	register char	*cp;
	char		filebuf[MIDLEN];

	struct stat stat_buf;

	if ( (!WN_OPT_T) && (!WN_OPT_U) ) {
		if ( (ip->filetype & WN_NOT_WORLD_READ)
				&& (ip->filetype & WN_NOT_REG_FILE)) {
			senderr( SERV_ERR, err_m[44], buf);
			wn_exit( 2); /* senderr: SERV_ERR */
		}
		else
			return;
	}
	else if ( IS_TRUSTED )
		return;

	if ( WN_OPT_U ) {
		mystrncpy( filebuf, buf, MIDLEN);
		cp = filebuf;
		while ( *cp && !isspace( *cp))
			cp++;
		*cp = '\0';
		if ( stat( filebuf, &stat_buf) != 0 ) {
			logerr( err_m[12], buf);
			return;
		}

		if (( dir_p->cache_uid == stat_buf.st_uid )
				 || ( cache_id == stat_buf.st_uid )
				 || ( cache_id == stat_buf.st_gid ))
			return;
	}
		
	senderr( DENYSTATUS, err_m[3], filebuf);
	wn_exit( 2); /* senderr: DENYSTATUS */
}


void
exec_ok( ip)
Request	*ip;
{
	
#ifdef FORBID_CGI
	ip->allowed &= ~(WN_M_POST + WN_M_PUT);
	senderr( DENYSTATUS, err_m[4], ip->cachepath);
	wn_exit( 2);  /* senderr: DENYSTATUS*/
#else
	if ((serv_perm & WN_FORBID_EXEC) ||
			((serv_perm & WN_RESTRICT_EXEC) && !IS_TRUSTED)) {
		senderr( DENYSTATUS, err_m[4], ip->cachepath);
		wn_exit( 2);  /* senderr: DENYSTATUS*/
	}
#endif
}

