/***

IPTraf
An IP-oriented Network Statistics Utility
   		
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998
   	
Version 1.3.0

This software is open-source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <panel.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "dirs.h"
#include "deskman.h"
#include "menurt.h"
#include "stdwinset.h"
#include "utfdefs.h"
#include "othfilter.h"
#include "utfilter.h"
#include "ifstats.h"
#include "serv.h"
#include "options.h"
#include "externs.h"
#include "log.h"
#include "attrs.h"
#include "error.h"

extern void srpromisc(int mode);
extern void about();
extern void masq_warning();

int check_ip_masquerade(void)
{
    int fd;
    
    fd = open("/proc/net/ip_masquerade", O_RDONLY);
    
    if (fd >= 0) {
        close(fd);
        return 1;
    }
        
    return 0;
}

/*
 * Get the ball rolling: The program interface routine.
 */

void program_interface(struct OPTIONS *options,
		       int opt, char *optarg, int facilitytime, 
		       int is_first_instance)
{
    struct MENU menu;
    int endloop = 0;
    int row = 1;
    int aborted;
    int faborted;
    int filtered = 0;
    struct filterlist fl;
    struct othpoptions ofilter;
    char ifname[10];
    struct porttab *ports;

    draw_desktop();
    attrset(STDATTR);
    move(0, 1);
    printw("IPTraf %s", VERSION);

    loadsavedtcpfilter(&fl, &filtered);
    loadothfilter(&ofilter);
    indicate("");
    loadaddports(&ports);

    if (check_ip_masquerade())
        masq_warning();
        
    if (opt == 0) {
	about();

	initmenu(&menu, 14, 35, 5, 20);

	additem(&menu, " IP traffic ^m^onitor", "Displays current IP traffic information");
	additem(&menu, " General interface ^s^tatistics", "Displays some statistics for attached interfaces");
	additem(&menu, " ^D^etailed interface statistics", "Displays more statistics for a selected interface");
	additem(&menu, " TCP/UDP ser^v^ice monitor", "Displays statistics on TCP/UDP traffic");
	additem(&menu, " ^L^AN station monitor", "Displays statistics on detected LAN stations");
	additem(&menu, NULL, NULL);
	additem(&menu, " ^T^CP display filters", "Manages TCP display filters");
	additem(&menu, " Other ^p^rotocol filters", "Select which non-TCP protocols to display");
	additem(&menu, NULL, NULL);
	additem(&menu, " C^o^nfigure", "Set various program options");
	additem(&menu, NULL, NULL);
	additem(&menu, " E^x^it", "Exits program");

	endloop = 0;

	do {
	    showmenu(&menu);
	    operatemenu(&menu, &row, &aborted);

	    switch (row) {
	    case 1:
		ipmon(options,
		      filtered, &fl, &ofilter, options->timeout, 0);
		break;
	    case 2:
		ifstats(options, 0);
		break;
	    case 3:
		selectiface(ifname, &aborted);
		if (!aborted)
		    detstats(ifname, options, 0);
		break;
	    case 4:
		selectiface(ifname, &aborted);
		if (!aborted)
		    servmon(ifname, ports, options, 0);
		break;
	    case 5:
		hostmon(options, 0);
		break;
	    case 7:
		tcpfilterselect(&fl, &filtered, &faborted);
		break;
	    case 8:
		othfilterselect(&ofilter);
		saveothfilter(&ofilter);
		break;
	    case 10:
		setoptions(options, &ports, is_first_instance);
		saveoptions(options);
		break;
	    case 12:
		endloop = 1;
		break;
	    }
	} while (!endloop);

	destroymenu(&menu);
    } else {
	switch (opt) {
	case 'i':
	    ipmon(options,
		  filtered, &fl, &ofilter, options->timeout, facilitytime);
	    break;
	case 'g':
	    ifstats(options, facilitytime);
	    break;
	case 'd':
	    detstats(optarg, options, facilitytime);
	    break;
	case 's':
	    servmon(optarg, ports, options, facilitytime);
	    break;
	case 'l':
	    hostmon(options, facilitytime);
	    break;
	}
    }

    destroyporttab(ports);
    erase();
    update_panels();
    doupdate();
}

void removetags()
{
    unlink(IPTIDFILE);
    unlink(IPMONIDFILE);
    unlink(GSTATIDFILE);
    unlink(DSTATIDFILE);
    unlink(TCPUDPIDFILE);
    unlink(LANMONIDFILE);
    unlink(TCPFLTIDFILE);
    unlink(UDPFLTIDFILE);
    unlink(OTHFLTIDFILE);
}

/*
 * Handler for the TERM signal.  There's nothing we can do for the KILL.
 */

void termhandler()
{
    erase();
    refresh();
    endwin();
    removetags();
    fprintf(stderr, "Received TERM signal\n");
    fprintf(stderr, "IPTraf terminated\n\n");
    exit(0);
}

/* 
 * Handler for the SIGSEGV, Segmentation Fault.  Tries to clear the screen
 * and issue a better message than "Segmentation fault".
 */

void segvhandler()
{
    erase();
    refresh();
    endwin();
    removetags();
    fprintf(stderr, "Fatal error: unable to allocate memory for a critical function\n");
    fprintf(stderr, "IPTraf terminated abnormally\n\n");
    exit(2);
}

/*
 * Command-line help facility.
 */

void commandhelp()
{
    printf("\nSyntax:\n");
    printf("    iptraf [ -f ][ { -i | -g | -d iface | -s iface | -l } [-t timeout ]]\n\n");
    printf("Issuing the iptraf command without command-line options brings the program\n");
    printf("up in interactive mode, with the various facilities accessed through a menu.\n\n");
    printf("These options can also be supplied to the command:\n\n");
    printf("	-i		- immediately start the IP traffic monitor\n");
    printf("	-g		- immediately start the general interface statistics\n");
    printf("	-d iface	- allows you to immediately start the detailed\n");
    printf("			  on the indicated interface (iface)\n");
    printf("	-s iface	- allows you to immediately monitor TCP and UDP\n");
    printf("			  traffic on the specified interface (iface)\n");
    printf("	-l		- start the LAN station monitor\n");
    printf("	-t timeout	- when used with one of the above parameters, tells\n");
    printf("			  the facility to run only for the specified number of\n");
    printf("			  minutes (timeout)\n");
    printf("	-f		- Remove all identification tag files.  This will\n");
    printf("			  allow multiple instances of facilities in different\n");
    printf("			  IPTraf processes.  Use with great caution.  May be\n");
    printf("			  used after an abnormal termination.\n\n");
    printf("IPTraf %s Copyright (c) Gerard Paul Java 1997, 1998\n", VERSION);
}

int first_instance()
{
    int fd;

    fd = open(IPTIDFILE, O_RDONLY);

    if (fd < 0)
	return !0;
    else {
	close(fd);
	return 0;
    }
}

void mark_first_instance()
{
    int fd;

    fd = open(IPTIDFILE, O_CREAT | O_WRONLY | O_TRUNC, S_IRUSR | S_IWUSR);
    if (fd < 0) {
	fprintf(stderr, "\nWarning: unable to tag this process\n");
	fprintf(stderr, "Press Enter to continue\n");
	getchar();
	return;
    }
    close(fd);
}

/*
 * Main routine
 */

int main(int argc, char **argv)
{
    struct OPTIONS options;
    extern char *optarg;
    extern int optind;
    extern int opterr;
    extern int optopt;
    int opt = 0;
    int command = 0;
    int is_first_instance;
    char keyparm[12];
    int facilitytime = 0;

#ifndef ALLOWUSERS
    if (getuid() != 0) {
	fprintf(stderr, "\nIPTraf Version %s\n", VERSION);
	fprintf(stderr, "Copyright (c) Gerard Paul Java 1997, 1998\n\n");
	fprintf(stderr, "This program can be run only by the system administrator\n\n");
	exit(1);
    }
#endif

    /*
     * Parse command line
     */

    if (argc > 1) {
	do {
	    opterr = 0;
	    opt = getopt(argc, argv, "igd:s:lhft:");

	    if (opt == 'h') {
		commandhelp();
		exit(0);
	    } else if (opt == 'f') {
		removetags();
            } else if (opt == 't') {
                facilitytime = atoi(optarg);
                if (facilitytime == 0) {
                    fprintf(stderr, "\nInvalid time value\n\n");
                    exit(1);
                }
	    } else if (opt == '?') {
		fprintf(stderr, "\nInvalid option or missing parameter\n\n");
		exit(1);
	    } else if (opt != -1) {
		if (optarg != 0) {
		    bzero(keyparm, 12);
		    strncpy(keyparm, optarg, 11);
		}
		command = opt;
	    }
	} while ((opt != '?') && (opt != -1));
    }

    is_first_instance = first_instance();
            
    signal(SIGTERM, (void *) termhandler);
    signal(SIGSEGV, (void *) segvhandler);
    signal(SIGTSTP, SIG_IGN);
    signal(SIGINT, SIG_IGN);

    if (getenv("TERM") == NULL) {
	fprintf(stderr, "Your TERM variable is not set.\n");
	fprintf(stderr, "Please set it to an appropriate value.\n");
	exit(1);
    }
    loadoptions(&options);

    mark_first_instance();

    initscr();

    if ((LINES < 24) || (COLS < 80)) {
	endwin();
	fprintf(stderr, "\nThis program requires a screen size of at least 80 columns by 24 lines\n");
	fprintf(stderr, "Please resize your window\n\n");
	exit(1);
    }
    start_color();
    standardcolors(options.color);
    noecho();
    nonl();
    cbreak();

#ifndef DEBUG
    curs_set(0);
#endif

    if (is_first_instance)
        srpromisc(options.promisc);

    program_interface(&options, command,
                      keyparm, facilitytime, is_first_instance);

    if (is_first_instance)
        srpromisc(0);

    endwin();

    if (is_first_instance)
        unlink(IPTIDFILE);

    return (0);
}
