/*
 * Program:        stopafter.c.
 * Version:        1.2.5
 * Author:         lilo@linpeople.org.
 * Date written:   27 July 1996.
 * Last modified:  4 December 1996.
 *
 * Description     Run a command, killing it with a SIGHUP after a
 *                 specified amount of time.  Forks a child process to
 *                 kill the parent process and any children if still
 *                 present.
 *
 * Bugs            Might need better parsing.  Also, stopafter has no
 *                 obvious way to know the parent has terminated if
 *                 called from inside a $() or `` construct, so it has
 *                 to time out before the construct is allowed to
 *                 terminate.  Also, the dialogue is way too
 *                 melodramatic.
 *
 * Copyright       (c) 1996 by lilo <lilo@linpeople.org>.  All rights
 * and             reserved.
 * license
 * information     This program is free software; you can redistribute
 *                 it and/or modify it under the terms of version 2 of
 *                 the GNU General Public License as published by the
 *                 Free Software Foundation.
 *
 *                 This program is distributed in the hope that it
 *                 will be useful, but WITHOUT ANY WARRANTY; without
 *                 even the implied warranty of MERCHANTABILITY or
 *                 FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *                 General Public License for more details.
 *
 *                 You should have received a copy of version 2 of the
 *                 GNU General Public License along with this program;
 *                 if not, it should be available at one of the two
 *                 following URL locations:
 *
 *                 ftp://ftp.linpeople.org/pub/LICENSE/COPYING-2.0
 *                 ftp://prep.ai.mit.edu/pub/gnu/COPYING-2.0
 *
 *                 Otherwise write to the Free Software Foundation,
 *                 Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Make sure we get POSIX function, superseded by BSD function.
 */

#define _BSD_SOURCE
#define _POSIX_SOURCE

/*
 * Header files.
 */

#include <ctype.h>

#define __KERNEL__
#include <errno.h>
#undef __KERNEL__
#include <fcntl.h>
#include <limits.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "siglist.h"

/*
 * Signal handler and lookup routine prototypes.
 */

void kill_process_group(int killsignal);
int oursignalnumber(char *oursignalstring);
char *oursignalname(int oursignalnumeric);

/*
 * Global variables:  signal number, counter, verbosity flag, previous
 * signal number, seconds count for timeout, exit code from terminated
 * process, pointer to termination status information, first child pid
 * number, terminating child pid number, signal masks, signal action
 * structures.  Array for pid filename string.
 */

int         use_signal,
            ind,
            verbose = 0,
            last_number = 0,
            seconds = 0,
            exitcode = -1,
            exitsignal = 0,
            endstatus;

pid_t       child, ending;

sigset_t    old_procmask,
            accept_no_signals;
                
struct
sigaction   ignored_signal,
            default_behavior,
            process_group_terminate;

FILE        *pidfile;
char        pid_file[_POSIX_PATH_MAX+1] = { "" };

/*
 * Main routine.
 */

void main(int argc, char *argv[])
{

/*
 * Do some initial signal-structure setup.  We don't want to wait
 * until the user feels we're running, so the earlier the better.
 */

    sigfillset(&accept_no_signals);

    ignored_signal.sa_handler = SIG_IGN;
    default_behavior.sa_handler = SIG_DFL;
    
    sigfillset(&process_group_terminate.sa_mask);
    process_group_terminate.sa_handler = kill_process_group;

/*
 * Is there a `verbose' option?  if so, set the flag and skip past it.
 */

    while (argc > 2 && *argv[1] == '-' && strlen(argv[1]) == 2)
        switch(argv[1][1])
        {
            case 'h':
            case 'H':
            case '?':
                argc = 0;
                break;
            case 'v':
                verbose = 1, argv++, argc--;
                break;
                
            case 'f':
                argv++, argc--;
                if (*pid_file)
                {
                    fprintf(stderr,
                        "stopafter: specify -f option only once\n");
                    argc = 0;
                    break;
                }
                strncpy(pid_file, argv[1], _POSIX_PATH_MAX);
                argv++, argc--;
                break;

            default:
                fprintf(stderr, "stopafter: unknown option `%s'\n", argv[1]);
                argc = 0;
        }

/*
 * If there are at least enough arguments left to include the signal
 * name, attempt to resolve it to an integer value.  (If not, the
 * default value of zero will be out of range and force a help
 * display.)
 */

    if (argc > 2)

/*
 * First look for a simple numeric value in one of the standard
 * bases.  If value is non-zero, we'll pass that down to the if
 * statement for the help display, which will check for valid range.
 */

        if (!(use_signal = (int) strtol(argv[2], NULL, 0)))

/*
 * Otherwise it's not a numeric value; treat it as the name of a
 * signal and look for it in the table.
 */

            use_signal = oursignalnumber(argv[2]);

/*
 * If we have at least one argument, evaluate it as the timeout period
 * in seconds.  Numeric values may be provided in standard C syntax in
 * the base of the user's choice.  the number of seconds in whatever
 * numeric base provided.  An invalid numeric value will be treated as
 * zero, for which no time limit is enforced.
 */

    if (argc > 1) seconds = (int) strtol(argv[1], NULL, 0);

/*
 * If there are not enough arguments to include the seconds count, the
 * signal and some sort of command, or if the signal number acquired
 * above is less than one or greater than the maximum number possible
 * on this system, produce a help display.  The help display provides
 * version, command format and a list of available signal names and
 * numbers (see header file siglist.h).  Terminate with the
 * traditional syntax exit code value of one.
 */

    if (argc < 4 || use_signal < 1 || use_signal >= NSIG)
    {
        fputs("stopafter v" VERSION ", copyright (c) 1996 "
            "by lilo@linpeople.org.\n"
            "All rights are reserved by the author.\n\n"
            "This program is offered for use under the terms of "
            "version 2 of the GNU\n"
            "Public License.  No warranty is provided, either "
            "express or implied.  See\n"
            "`ftp://ftp.linpeople.org/pub/Licenses/COPYING-2.0' "
            "for more information.\n\n"
            "Format:  stopafter [-v] [-f <pidf>] <sec> <sig> "
            "<cmd> [<arg>]...\n\n"
            "Specify a positive integer for <sec> (seconds).  "
            "Numerics should be\n"
            "standard C constants (decimal, octal or hexadecimal).  "
            "Specify signal\n"
            "type <sig> as either a name or a number.  "
            "Names may be mixed-case and\n"
            "the `SIG' prefix is not required.  Signal types are:"
            "\n\n", stderr);
        for (ind=0; ind < KNOWN_SIGNALS; ind++)
        {
            use_signal = Signal[ind].number;
            fprintf(stderr,
                "%9.9s  %2.2u  %s\n",
                Signal[ind].name,
                use_signal,
                use_signal == last_number ?
                    "   \"" :  sys_siglist[use_signal]);
            last_number = use_signal;
        }
        exit(1);
    }

/*
 * Everything seems fine.  Make this pid leader of a new process group
 * leader so that its descendants can all be sent the timeout signal.
 */

    setpgid(0, 0);

/*
 * Fork a child process to exec the requested command with no
 * provision for return.  If this is the parent process, monitor for a
 * timeout.
 */

    if ((child = fork()))    {

/*
 * A pid was provided, so this is the parent process.  If verbose mode
 * is on, display an informative message about the child process just
 * forked.  If the seconds value is zero or invalid, make sure it's
 * set to zero and, if verbose mode is on, display a message about
 * that too.
 */

        if (verbose)
            fprintf(stderr, "stopafter: forked process %i to run "
                "command\n", child);
        
        if (seconds < 1)
        {
            seconds = 0;
            if (verbose)
                fprintf(stderr, "stopafter: timeout "
                    "value set to zero, "
                    "timeout function "
                    "will not be performed\n");
        }

/*
 * Block all signals temporarily to set up our handlers.  Once we
 * restore the previous procmask value, we will either get signals
 * queued by the user, or not, and it's okay either way.
 */

        sigprocmask(SIG_SETMASK, &accept_no_signals, &old_procmask);

/*
 * Ignore everything but SIGALRM and a few intentional signals;
 * SIGABRT takes its default value, and SIGALRM, SIGHUP, SIGINT AND
 * SIGTERM are handled by the signal handler.  The other signals are
 * ignored, just as it says on the sign above my desk: Everything Not
 * Obligatory Is Strictly Prohibited.
 */

        for (ind=1; ind < NSIG; ind++) switch(ind)
        {
            case SIGABRT:
                sigaction (ind, &default_behavior, NULL);
                break;
            
            case SIGALRM:
            case SIGHUP:
            case SIGINT:
            case SIGTERM:
                sigaction (ind, &process_group_terminate, NULL);
                break;
            
            default:
                sigaction (ind, &ignored_signal, NULL);
        }

/*
 * The required signal actions have been set up.  Restore the old
 * blocking mask.  Start the timer.
 */

        sigprocmask(SIG_SETMASK, &old_procmask, NULL);
        alarm(seconds);

/*
 * We're now ready to actually write the pidfile, if one was specified,
 * indicating that we are running.  If there are errors, we'll send
 * ourselves a SIGHUP to terminate the underlying applications.
 */

        if (pid_file[0])
        {
            if ((pidfile = fopen(pid_file, "w")) == NULL)
            {
                fprintf(stderr,
                    "stopafter: %s on fopen of pid file %s\n",
                    strerror(errno),
                    pid_file);
                raise(SIGHUP);
                exit(2);
            }
            fprintf(pidfile, "%d\n", (int) getpid());
            if (fclose(pidfile) == EOF)
            {
                fprintf(stderr,
                    "stopafter: %s on fclose of pid file %s\n",
                    strerror(errno),
                    pid_file);
                raise(SIGHUP);
                exit(3);
            }
        }

/*
 * Wait for the termination of any member of our process group.  Don't
 * quit until we get an error, and it must be `no more children'; if
 * it's ERESTARTSYS, for example, we just want to keep looping.
 */

        while ((ending = waitpid(0, &endstatus, 0)) != -1 ||
            errno != ECHILD)
        {

/*
 * If waitpid() returned an error, display the error message if we are
 * in verbose mode.  We'll lowercase the message from the system error
 * list to clean up the message and to revel in the joys of copy-on-
 * write.  Loop.
 */

            if (ending == -1)
            {
                if (verbose && (errno != ERESTARTSYS))
                {
                    char *msg = strdup(sys_errlist[errno]);

                    *msg = tolower(*msg);
                    fprintf(stderr, "stopafter: process wait returned "
                        "prematurely, %s\n", msg);
                }
                continue;
            }

/*
 * Process the status information.  If this is the original child we
 * forked, capture the termination information.  If verbose mode is
 * on, provide termination messages as each process completes.
 */

            if (WIFEXITED(endstatus))
            {
                if (ending == child)
                    exitcode = WEXITSTATUS(endstatus);
                if (verbose)
                    if (exitcode)
                        fprintf(stderr,
                            "stopafter: %sprocess %i ended "
                            "with exit code %i\n",
                            ending == child ? "main child " : "",
                            ending, exitcode);
                    else
                        fprintf(stderr,
                            "stopafter: %sprocess %i ended "
                            "normally\n",
                            ending == child ? "main child " : "",
                            ending);
            }
            if (WIFSIGNALED(endstatus))
            {
                if (ending == child)
                    exitsignal = WTERMSIG(endstatus);
                if (verbose)
                    fprintf(stderr, "stopafter: %sprocess %i "
                        "terminated by uncaught %s\n",
                        ending == child ? "main child " : "",
                        ending, oursignalname(exitsignal));
            }
        }

/*
 * Everybody's dead.  Sing ricketty-ticketty-tin.  Remove the
 * pidfile if we wrote one.  Terminate in much the same fashion as
 * our original child process, providing appropriate message(s) if
 * verbose mode is set.  If we have no other information on how the
 * original child process terminated, terminate with exit code zero.
 */

        if (pid_file[0] && (unlink(pid_file) == -1))
        {
            fprintf(stderr,
                "stopafter: %s on unlink of pid file %s\n",
                strerror(errno),
                pid_file);
        }
        
        if (exitsignal)
        {
            if (verbose)
                fprintf(stderr, "stopafter: parent "
                    "process raised %s\n",
                    oursignalname(exitsignal));
            sigaction (exitsignal, &default_behavior, NULL);
            raise(exitsignal);
        }
        
        if (exitcode > -1)
        {
            if (verbose)
                if (exitcode)
                    fprintf(stderr, "stopafter: parent "
                        "process ended with exit code %i\n",
                        exitcode);
                else
                    fprintf(stderr, "stopafter: parent "
                        "process ended normally\n");
            exit(exitcode);
        }

        if (verbose)
            fprintf(stderr, "stopafter: no information is available "
                "on main child process termination\n"
                "stopafter: parent process ended normally\n");
        exit(0);
    }

/*
 * The return code from fork() is zero, so this is the child.  Start
 * the program with no provision for return.
 */

    else execvp(argv[3], &argv[3]);
}




/*
 * Signal handler function.  Ignore all signals, kill everything in
 * our process group and return to the parent process to terminate.
 */

void kill_process_group(int killsignal)
{

/*
 * Set all signal actions to ignore so we can finish up.
 */

        for (ind=1; ind < NSIG; ind++)
                sigaction (ind, &ignored_signal, NULL);

/*
 * Kill the members of the monitor process' group, using the signal
 * number provided on the command line.  Provide appropriate
 * informative message if verbose mode is on.  Return to terminate in
 * a fashion that simulates the original child process in case of
 * recalcitrant shell.
 */

    ind=kill(0, use_signal);
    if (verbose)
        if (killsignal == SIGALRM)
            fprintf(stderr, "stopafter: time limit exceeded, "
                "%s sent to process group\n",
                oursignalname(use_signal));
        else
            fprintf(stderr, "stopafter: %s received, %s sent "
                "to process group\n", oursignalname(killsignal),
                oursignalname(use_signal));
    return;
}




/*
 * Signal number retrieval function.  Takes a signal name and massages
 * it to upper case.  Compares it to each signal in the list, with and
 * without the `SIG'- prefix.
 */

int oursignalnumber(char *oursignalstring)
{


/*
 * First translate the string provided by the user to all-upper-case
 * so that it matches the signal names in our table.
 */

    for (ind=0; ind < strlen(oursignalstring); ind++)
        oursignalstring[ind] = toupper(oursignalstring[ind]);

/*
 * Loop through the signal table comparing entries to the argument
 * provided.  Compare the complete string or, if the signal table
 * string is at least three characters, just compare everything past
 * that point (skipping the `SIG-' prefix).
 */

    for (ind=0; ind < KNOWN_SIGNALS; ind++)
    {
        if (!strcmp(oursignalstring,Signal[ind].name) ||
            (strlen(Signal[ind].name) >= 3 &&
            !strcmp(oursignalstring, Signal[ind].name+3)))
        {

/*
 * If a match is found, return the integer value for the signal. 
 * Table entries must must have signal numbers greater than zero or
 * problems will happen.
 */

            return Signal[ind].number;
        }
    }

/*
 * If no match is found, return zero value to force a help display.
 */

    return 0;
}




/*
 * Signal name retrieval function.  Takes a signal number and returns
 * a canonical signal name or a string with the signal number.
 */

char *oursignalname(int oursignalnumeric)
{
    static char unknown_signal[70];

/*
 * Loop through the signal table looking for our signal.
 */

    for (ind=0; ind < KNOWN_SIGNALS; ind++)
    {
        if (Signal[ind].number == oursignalnumeric)

/*
 * If a match is found, return the pointer for the signal name.
 */

            return Signal[ind].name;
    }

/*
 * If no match is found, return the signal number in a string.
 */

    sprintf(unknown_signal, "(unknown signal %i)", oursignalnumeric);
    return unknown_signal;
}
