// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _Options_h_
#define _Options_h_

extern "C"
{
#include <stdio.h>
}

#include <qobject.h>
#include <qstring.h>

//: The Options object manages the user preferences for the entire application.
//. The Options object is responsible for reading user preferences from the
//. $HOME/.qwebrc file and the command line.  Whenever any option is changed,
//. a signal is emitted.  Upon request, the current user preferences can be
//. written back out to $HOME/.qwebrc.
class Options : public QObject {
    Q_OBJECT

    bool    _autoLoad;
    QString _bookmarksUrl;
    QString _catalogFile;
    QString _homeUrl;

    bool    _debugging;

    QString _menuFontName;
    int     _menuFontSize;
    int     _menuFontWeight;
    bool    _menuFontItalic;
    
    //. Return a single token from the preferences file.  A tokens are
    //. delimited by spaces only.
    QString getToken( FILE* fp );

    //. Read in the user preferences from $HOME/.qwebrc.
    void    load();
public:
    //. Create a new Options object.  Set some reasonable default for all the
    //. options.  Read any settings from $HOME/.qwebrc.  Override these
    //. settings with anything specified on the command line.
    Options( int argc, char* argv[] );

    //. Return the value of the auto load option.
    bool    autoLoad()     { return _autoLoad; }

    //. Return the value of the bookmarks URL option.
    QString bookmarksUrl() { return _bookmarksUrl; }

    //. Return the value of the catalog file option.
    QString catalogFile()  { return _catalogFile; }

    //. Return the value of the home URL option.
    QString homeUrl()      { return _homeUrl; }

    //. Return the value of the debugging option.
    bool    debugging()    { return _debugging; }

    QString menuFontName() { return _menuFontName; }

    int     menuFontSize() { return _menuFontSize; }

    int     menuFontWeight() { return _menuFontWeight; }

    bool    menuFontItalic() { return _menuFontItalic; }

    //. Change the auto load option.
    void setAutoLoad( bool b )          { _autoLoad = b;              emit updateAutoLoad();     }

    //. Change the bookmarks file option.
    void setBookmarksUrl( QString url ) { _bookmarksUrl = url.copy(); emit updateBookmarksUrl(); }

    //. Change the catalog file option.
    void setCatalogFile( QString url )  { _catalogFile = url.copy();  emit updateCatalogFile();  }

    //. Change the home URL option.
    void setHomeUrl( QString url )      { _homeUrl = url.copy();      emit updateHomeUrl();      }

    //. Change the debugging option.
    void setDebugging( bool b )         { _debugging = b;             emit updateDebugging();    }

    void setMenuFontName( QString name ) { _menuFontName = name; emit updateMenuFontName(); }

    void setMenuFontSize( int size ) { _menuFontSize = size; emit updateMenuFontSize(); }

    void setMenuFontWeight( int weight ) { _menuFontWeight = weight; emit updateMenuFontWeight(); }

    void setMenuFontItalic( bool italic ) { _menuFontItalic = italic; emit updateMenuFontItalic(); }
public slots:
    //. Save the current options to the file $HOME/.qwebrc.
    void save();
signals:
    //. Emitted when the auto load options is changed.
    void updateAutoLoad();

    //. Emitted when the bookmarks URL option is changed.
    void updateBookmarksUrl();

    //. Emitted when the catalog file option is changed.
    void updateCatalogFile();

    //. Emitted when the home URL options is changed.
    void updateHomeUrl();

    //. Emitted when the debugging option is changed.
    void updateDebugging();

    void updateMenuFontName();

    void updateMenuFontSize();

    void updateMenuFontWeight();

    void updateMenuFontItalic();
};

extern Options* options;

#endif
