/* Socket routine descriptions */

// thanks to Eugene Aleynikov for bug fixes

#include <sys/types.h> 
#include <sys/socket.h>
#include <string.h>
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>

#include "socket.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef HAVE_INET_ATON
#include "inet_addr.h"
#endif

#ifdef USE_SOCKS5
#define SOCKS
#include <socks.h>
#endif

//-----Socket::openSocket--------------------------------------------------------------------------
void Socket::openSocket(void)
{
   if (descriptorVal != -1) 
   {
      fcntl(descriptorVal, O_NONBLOCK);
      if (sn == NULL) sn = new QSocketNotifier(descriptorVal, QSocketNotifier::Read);
      sn->setEnabled(true);
   }
}


//-----Socket::closeSocket-------------------------------------------------------------------------
void Socket::closeSocket(void)
{
   if (sn != NULL)
   {
      delete sn;
      sn = NULL;
   }
}


//-----StdinSocket---------------------------------------------------------------------------------
StdinSocket::StdinSocket(void)
{ 
   descriptorVal = STDIN_FILENO; 
   errorVal = 0; 
   openSocket();
}


//-----INetSocket::constructor---------------------------------------------------------------------
INetSocket::INetSocket(void)
{
   localIPVal = 0;
   remoteIPVal = 0;
   remotePortVal = 0;
   descriptorVal = -1;
}



//-----INetSocket::setDestination------------------------------------------------------------------
bool INetSocket::setDestination(unsigned long rIp, unsigned short rPort)
{ 
   if (rIp == 0) return(false);  // if the rIp is invalid, exit
   /*
   hostent *localHost;
   //char hn[128];
   //gethostname(hn, 128);
   localHost = gethostbyname("127.0.0.1");
   localIPVal = *((unsigned long *)localHost->h_addr);*/

   // set up remote connection
   remoteIPVal = rIp;
   remotePortVal = rPort;
   remote.sin_port = htons(rPort);
   remote.sin_addr.s_addr = rIp;
   return(true);
}


//-----INetSocket::resetSocket---------------------------------------------------------------------
void INetSocket::resetSocket(void)
{
   if (connected()) closeConnection();
   localIPVal = 0;
   remoteIPVal = 0;
}


//-----INetSocket::lookup--------------------------------------------------------------------------
unsigned long INetSocket::lookup(char *h)
{
   struct hostent *host; 
   struct in_addr ina;

   // check if the hostname is in dot and number notation
   if (inet_aton(h, &ina)) return(ina.s_addr);
   
   // try and resolve hostname
   if ((host = gethostbyname(h)) == NULL) return (0);	// Couldn't resolve hostname/ip
   
   // return the ip
   return *((unsigned long *)host->h_addr);
}


//-----INetSocket::closeConnection-----------------------------------------------------------------
void INetSocket::closeConnection(void)
{
   closeSocket(); 
   if (descriptorVal != -1) close(descriptorVal); 
   descriptorVal = -1; 
}


//-----TCPSocket::TCPSocket------------------------------------------------------------------------
TCPSocket::TCPSocket(void)
{
}


//-----TCPSocket::openConnection-------------------------------------------------------------------
bool TCPSocket::openConnection(void)
{  
   struct sockaddr_in local;
   if(remoteIPVal == 0) return(false);  // if no destination set then someone screwed up
   
   descriptorVal = socket(AF_INET, SOCK_STREAM, 0);
   if (descriptorVal == -1) return(false);
   
   openSocket();
   socklen_t sizeofSockaddr = sizeof(struct sockaddr);
   memset(&local.sin_zero, 0, 8);
   memset(&remote.sin_zero, 0, 8);
   local.sin_family = remote.sin_family = AF_INET;
   local.sin_port = 0;
   local.sin_addr.s_addr = htonl(INADDR_ANY);
   bind(descriptorVal, (struct sockaddr*)&local, sizeof(struct sockaddr));
   getsockname(descriptorVal, (struct sockaddr*)&local, &sizeofSockaddr);
   localPortVal = ntohs(local.sin_port);
   localIPVal = ntohl(local.sin_addr.s_addr);

   // if connect fails then call closeConnection to clean up before returning
   if (connect(descriptorVal, (struct sockaddr *)&remote, sizeofSockaddr) < 0)  // connect failed
   {
      closeConnection();
      return(false);
   }
   return (true);
}


//-----TCPSocket::receiveConnection----------------------------------------------------------------
void TCPSocket::receiveConnection(TCPSocket &newSocket)
{
   socklen_t sizeofSockaddr = sizeof(struct sockaddr);
   newSocket.descriptorVal = accept(descriptorVal, (struct sockaddr*)&remote, &sizeofSockaddr);
   newSocket.openSocket();
   sizeofSockaddr = sizeof(struct sockaddr);
   getpeername(newSocket.descriptorVal, (struct sockaddr *)&newSocket.remote, &sizeofSockaddr);
   
   newSocket.localIPVal = localIPVal;
   newSocket.localPortVal = localPortVal;
   
   newSocket.remoteIPVal = newSocket.remote.sin_addr.s_addr;
   newSocket.remotePortVal = ntohs(newSocket.remote.sin_port);
}


//-----TCPSocket::closeConnection------------------------------------------------------------------
void TCPSocket::closeConnection(void)
{
   if (connected()) INetSocket::closeConnection();
}


//-----TCPSocket::startServer----------------------------------------------------------------------
bool TCPSocket::startServer(unsigned int p = 0)
{
   struct sockaddr_in local;
   socklen_t sizeofSockaddr = sizeof(struct sockaddr_in);
   descriptorVal = socket(AF_INET, SOCK_STREAM, 0);
   if (descriptorVal < 0) return (false);  // big error
   
   openSocket();
   memset(&local.sin_zero, 0, 8);
   local.sin_family = AF_INET;
   local.sin_port = htons(p);
   local.sin_addr.s_addr = htonl(INADDR_ANY);
   if (bind(descriptorVal, (struct sockaddr*)&local, sizeofSockaddr) < 0)
      return (false);
   getsockname(descriptorVal, (struct sockaddr *)&local, &sizeofSockaddr);
   localPortVal = ntohs(local.sin_port);
   localIPVal = ntohl(local.sin_addr.s_addr);
   listen(descriptorVal, 5); // Allow 5 unprocessed connections
   return(true);
}                         


//-----TCPSocket::transferConnection---------------------------------------------------------------
void TCPSocket::transferConnectionFrom(TCPSocket &from)
{
   descriptorVal = from.descriptorVal; 
   remoteIPVal = from.remoteIPVal;
   remotePortVal = from.remotePortVal;
   localIPVal = from.localIPVal;
   localPortVal = from.localPortVal;
   from.closeSocket();
   from.descriptorVal = -1;
   openSocket();
}


bool TCPSocket::sendPacket(Packet &p)
{
   return sendPacket(p, true);
}

bool TCPSocket::receivePacket(Packet &p)
{
   return receivePacket(p, true);
}


//-----TCPSocket::sendPacket-----------------------------------------------------------------------
bool TCPSocket::sendPacket(Packet &p, bool sendSize)
{
   if (descriptorVal < 0) return (false);   

   unsigned long bytesSent = 0;
   if (sendSize) 
   {
      char pSize[2];
      pSize[0] = (p.sizeVal) & 0xFF;
      pSize[1] = (p.sizeVal >> 8) & 0xFF;
   
      // send the length of the packet, close the connection and return false if unable to send
      if(send(descriptorVal, &pSize, 2, 0) <= 0) { closeConnection(); return(false); }
   }

   // send the rest of the packet
   ssize_t rc;
   while(p.sizeVal > bytesSent)
   {
      rc = send(descriptorVal, p.buff + bytesSent, p.sizeVal, 0);
      if (rc <= 0) 
      { 
         closeConnection(); 
         return(false); 
      }
      bytesSent += rc;
   }
   return (true);
}


//-----TCPSocket::receivePacket--------------------------------------------------------------------
bool TCPSocket::receivePacket(Packet &p, bool recvSize)
{
   if (descriptorVal < 0) return (false);   

   p.clearPacket();
   
   unsigned short pSize;
   if (recvSize) 
   {
      if(recv(descriptorVal, p.buff, 1, 0) <= 0)  { closeConnection(); return(false); }
      recv(descriptorVal, p.buff + 1, 1, 0);

      p.sizeVal = 2;
      p >> pSize;
   
      // make sure pSize isn't too large
      if(pSize > p.maxSize - sizeof(unsigned long)) 
         p.sizeVal = p.maxSize- sizeof(unsigned long);
      p.clearPacket();
   }
   else pSize = 1;


   ssize_t rc;
   while(p.sizeVal < pSize)
   {
      if ((rc = recv(descriptorVal, p.nextData, 1, 0)) <= 0) 
      { 
         closeConnection(); 
         return(false); 
      }
      p.sizeVal += rc;
      p.nextData = p.buff + p.sizeVal;
   }
   p.nextData = p.buff;
   return(true);
}


//-----UDPSocket::constructor----------------------------------------------------------------------
UDPSocket::UDPSocket(void)
{
}


//-----UDPSocket::destructor-----------------------------------------------------------------------
UDPSocket::~UDPSocket(void)
{
   closeConnection();
}


//-----UDPSocket::openConnection-------------------------------------------------------------------
bool UDPSocket::openConnection(void)
{  
   struct sockaddr_in local;
   if(remoteIPVal == 0) return(false);  // if destination was never set then there is a problem
   descriptorVal = socket(AF_INET, SOCK_DGRAM, 0);
   openSocket();
   memset(&local.sin_zero, 0, 8);
   memset(&remote.sin_zero, 0, 8);
   local.sin_family = remote.sin_family = AF_INET;
   local.sin_port = 0;
   local.sin_addr.s_addr = htonl(INADDR_ANY);
   socklen_t sizeofSockaddr = sizeof(struct sockaddr);
   bind(descriptorVal, (struct sockaddr*)&local, sizeofSockaddr);
   getsockname(descriptorVal, (struct sockaddr*)&local, &sizeofSockaddr);
   localPortVal = ntohs(local.sin_port);
   localIPVal = ntohl(local.sin_addr.s_addr);
   return(true);  // udp transfers are connectionless, hence always return true
}


//-----UDPSocket::receiveConnection----------------------------------------------------------------
void UDPSocket::receiveConnection(void)
{
   // Should be open socket. The socket is the same as in startServer, however.
}


//-----UDPSocket::closeConnection------------------------------------------------------------------
void UDPSocket::closeConnection(void)
{
   if (connected()) INetSocket::closeConnection();
}


//-----UDPSocket::startServer----------------------------------------------------------------------
bool UDPSocket::startServer(unsigned int p)
{
   struct sockaddr_in local;
   descriptorVal = socket(AF_INET, SOCK_DGRAM, 0); // use getprotobyname
   if (descriptorVal < 0) return (false);

   openSocket();
   memset(&local.sin_zero, 0, 8);
   local.sin_family = AF_INET;
   local.sin_port = htons(p);
   local.sin_addr.s_addr = htonl(INADDR_ANY);
   bind(descriptorVal, (struct sockaddr*)&local, sizeof(struct sockaddr));
   return(true);
}      


//-----UDPSocket::sendPacket-----------------------------------------------------------------------
bool UDPSocket::sendPacket(Packet &p)
{
   if (descriptorVal < 0) return (false);   

   int bytesSent = 0;
   bytesSent = sendto(descriptorVal, p.buff, p.sizeVal, 0, (struct sockaddr *)&remote, sizeof(struct sockaddr));

   // if unable to send anything
   if((unsigned int)bytesSent != p.sizeVal) 
   { 
      closeConnection(); 
      return(false); 
   }

   return(true);   
}


//-----UDPSocket::receivePacket--------------------------------------------------------------------
bool UDPSocket::receivePacket(Packet &p)
{
   if (descriptorVal < 0) return (false);   

   socklen_t sizeofSockaddr = sizeof(struct sockaddr);
   p.clearPacket();
   errno = 0;
   
   p.sizeVal = recvfrom(descriptorVal, p.buff, p.maxSize, 0, (struct sockaddr*)&remote, &sizeofSockaddr);
   
   // it didn't manage to receive anything, there was an error, close the socket and return false
   if(p.sizeVal <= 0) { closeConnection(); return(false); }

   // make sure the size won't overflow the buffer
   if (p.sizeVal > p.maxSize) 
      p.sizeVal = p.maxSize - sizeof(unsigned long);
   
   return (true);
}

