/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: transform.c,v 1.1 94/04/26 20:07:26 nau Exp $";

/* functions used to move or rotate pins, elements ...
 */

#include <stdlib.h>

#include "global.h"

#include "create.h"
#include "cursor.h"
#include "data.h"
#include "draw.h"
#include "error.h"
#include "search.h"
#include "transform.h"

/* ---------------------------------------------------------------------------
 * rotates a line in 90 degree steps 
 */
void RotateLine(LineTypePtr Line, Position X, Position Y, BYTE Number)
{
	Position	x1 = Line->X1,
				y1 = Line->Y1,
				x2 = Line->X2,
				y2 = Line->Y2;

	ROTATE(x1, y1, X, Y, Number);
	ROTATE(x2, y2, X, Y, Number);
	SETLINEDIRECTION(Line,x1,y1,x2,y2);
}

/* ---------------------------------------------------------------------------
 * rotates a rectangle in 90 degree steps 
 */
void RotateRect(RectTypePtr Rect, Position X, Position Y, BYTE Number)
{
	Position	x1 = Rect->X,
				y1 = Rect->Y,
				x2 = Rect->X +Rect->Width,
				y2 = Rect->Y +Rect->Height;

	ROTATE(x1, y1, X, Y, Number);
	ROTATE(x2, y2, X, Y, Number);
	Rect->X = MIN(x1, x2);
	Rect->Y = MIN(y1, y2);
	Rect->Width = MAX(x1, x2) -Rect->X;
	Rect->Height = MAX(y1, y2) -Rect->Y;
}

/* ---------------------------------------------------------------------------
 * rotates a text in 90 degree steps 
 * only the surrounding rectangle is rotated, text rotation itself
 * is done by the drawing routines
 */
void RotateText(TextTypePtr Text, Position X, Position Y, BYTE Number)
{
	RotateRect(&Text->Rect, X, Y, Number);

		/* set new direction, 0..3,  0-> to the right, 1-> straight up, 2-> to the left */
	Text->Direction = ((Text->Direction +Number) & 0x03);
}

/* ---------------------------------------------------------------------------
 * rotates an arc
 */
void RotateArc(ArcTypePtr Arc, Position X, Position Y, BYTE Number)
{
	Dimension	save;

		/* add Number*90 degrees to the startangle and check for overflow */
	Arc->StartAngle = (Arc->StartAngle +Number*90) % 360;
	ROTATE(Arc->X, Arc->Y, X, Y, Number);

		/* now change width and height */
	if (Number == 1 || Number == 3)
	{
		save = Arc->Width;
		Arc->Width = Arc->Height;
		Arc->Height = save;
	}
}

/* ---------------------------------------------------------------------------
 * rotate an element in 90 degree steps
 */
void RotateElement(ElementTypePtr Element, Position X, Position Y, BYTE Number)
{
	LineTypePtr	line;
	PinTypePtr	pin;
	ArcTypePtr	arc;
	MarkTypePtr	mark;
	Cardinal	i;

	for (line = Element->Line, i = Element->LineN; i; i--, line++)
		RotateLine(line, X, Y, Number);
	for (pin = Element->Pin, i = Element->PinN; i; i--, pin++)
		ROTATE_PIN(pin, X, Y, Number);
	for (arc = Element->Arc, i =Element->ArcN; i; i--, arc++)
		RotateArc(arc, X, Y, Number);
	for (mark = Element->Mark, i = Element->MarkN; i; i--, mark++)
		ROTATE(mark->X, mark->Y, X, Y, Number);
	ROTATE(Element->TextX, Element->TextY, X, Y, Number);
	Element->Direction = (Element->Direction +Number) & 0x03;
	RotateRect(&Element->Rect, X, Y, Number);
}

/* ---------------------------------------------------------------------------
 * moves a element
 */
void MoveElement(ElementTypePtr Element, Position X, Position Y)
{
	LineTypePtr	line;
	PinTypePtr	pin;
	ArcTypePtr	arc;
	MarkTypePtr	mark;
	Cardinal	i;

	for (line = Element->Line, i = Element->LineN; i; i--, line++)
		MOVE_LINE(line, X, Y);
	for (pin = Element->Pin, i = Element->PinN; i; i--, pin++)
		MOVE_PIN(pin, X, Y);
	MOVE(Element->TextX, Element->TextY, X, Y);
	for (arc = Element->Arc, i =Element->ArcN; i; i--, arc++)
		MOVE(arc->X, arc->Y, X, Y);
	for (mark = Element->Mark, i = Element->MarkN; i; i--, mark++)
		MOVE(mark->X, mark->Y, X, Y);
	MOVE_RECT(&Element->Rect, X, Y);
}

/* ---------------------------------------------------------------------------
 * moves vias, text, rectangles and lines located within
 * the specified block on allvisible layers (DeltaX, DeltaY) 
 * and additional operation rotates them (center is X,Y)
 */
Boolean MoveAndRotateAllInRectangle(Position X1, Position Y1, Position X2, Position Y2, Position DeltaX, Position DeltaY, Position X, Position Y, BYTE Number)
{
	Position		x1, y1,		/* passed arguments are not necessary top-left */
					x2, y2;		/* and lower-right corner */
	PinTypePtr		via;
	LineTypePtr		line;
	RectTypePtr		rect;
	LayerTypePtr	layer;
	TextTypePtr		text;
	ElementTypePtr	element;
	Cardinal		n,
					i;
	Boolean			changed = False;

	x1 = MIN(X1, X2);
	x2 = MAX(X1, X2);
	y1 = MIN(Y1, Y2);
	y2 = MAX(Y1, Y2);

		/* start with vias */
	if (PCB->ViaOn)
		for (n = PCB->ViaN, via = PCB->Via+n-1; n; n--, via--)
			if (VIA_OR_PIN_IN_RECTANGLE(via, x1, y1, x2, y2))
			{
				MOVE_VIA(via, DeltaX, DeltaY);
				ROTATE_VIA(via, X, Y, Number);
				changed = True;
			}

		/* now check all layers */
	for (i = 0; i < MAX_LAYER; i++)
		if (PCB->Layer[i].On)
		{
			layer = &PCB->Layer[i];

				/* check lines */
			for (n = layer->LineN, line = layer->Line+n-1; n; n--, line--)
				if (LINE_IN_RECTANGLE(line, x1, y1, x2, y2))
				{
					MOVE_LINE(line, DeltaX, DeltaY);
					RotateLine(line, X, Y, Number);
					changed = True;
				}

				/* check rectangles */
			for (n = layer->RectN, rect = layer->Rect+n-1; n; n--, rect--)
				if (RECT_IN_RECTANGLE(rect, x1, y1, x2, y2))
				{
					MOVE_RECT(rect, DeltaX, DeltaY);
					RotateRect(rect, X, Y, Number);
					changed = True;
				}

				/* check text */
			for (n = layer->TextN, text = layer->Text+n-1; n; n--, text--)
				if (TEXT_IN_RECTANGLE(text, x1, y1, x2, y2))
				{
					MOVE_TEXT(text, DeltaX, DeltaY);
					RotateText(text, X, Y, Number);
					changed = True;
				}
		}

		/* elements */
	for (n = PCB->ElementN, element = PCB->Element+n-1; n; n--, element--)
		if (RECT_IN_RECTANGLE(&element->Rect, x1, y1, x2, y2))
			{
				MoveElement(element, DeltaX, DeltaY);
				RotateElement(element, X, Y, Number);
				changed = True;
			}

	PCB->Changed |= changed;
	return(changed ? True : False);
}

/* ---------------------------------------------------------------------------
 * changes the size of a via
 * returns TRUE if changed
 */
Boolean ChangeViaSize(PinTypePtr Via, int Delta)
{
	Dimension	value = Via->Thickness +Delta;

	if (value <= MAX_PINORVIASIZE && value >= MIN_PINORVIASIZE)
	{
		ErasePinOrVia(Via);
		Via->Thickness = value;
		DrawNewVia(Via);
		return(True);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * changes the size of a pin
 * returns TRUE if changed
 */
Boolean ChangePinSize(PinTypePtr Pin, int Delta)
{
	Dimension	value = Pin->Thickness +Delta;

	if (value <= MAX_PINORVIASIZE && value >= MIN_PINORVIASIZE)
	{
		ErasePinOrVia(Pin);
		Pin->Thickness = value;
		DrawNewPin(Pin);
		return(True);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * changes the size of a line
 * returns TRUE if changed
 */
Boolean ChangeLineSize(LineTypePtr Line, LayerTypePtr Layer, int Delta)
{
	Dimension	value = Line->Thickness +Delta;

	if (value <= MAX_LINESIZE && value >= MIN_LINESIZE)
	{
		EraseLine(Line);
		Line->Thickness = value;
		DrawLineOnLayer(Line, Layer);
		return(True);
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * changes the size of all lines of the current layer within the rectangle
 */
Boolean ChangeLineSizeInRectangle(Position X1, Position Y1, Position X2, Position Y2, Position Delta)
{
	Position	x1, y1,		/* passed arguments are not necessary top-left */
				x2, y2;		/* and lower-right corner */
	LineTypePtr	line = CURRENT->Line;
	Cardinal	n = CURRENT->LineN;
	Boolean		changed = False;

	x1 = MIN(X1, X2);		/* get upper left and lower right corner */
	x2 = MAX(X1, X2);
	y1 = MIN(Y1, Y2);
	y2 = MAX(Y1, Y2);
	while(n)
	{
		if (LINE_IN_RECTANGLE(line, x1, y1, x2, y2))
			changed |= ChangeLineSize(line, CURRENT, Delta);
		n--;
		line++;
	}
	return(changed);
}

/* ---------------------------------------------------------------------------
 * changes the size of all via in rectangle by Delta
 * returns TRUE if something was changed
 */
Boolean ChangeViaSizeInRectangle(Position X1, Position Y1, Position X2, Position Y2, Position Delta)
{
	Position		x1, y1,		/* passed arguments are not necessary top-left */
					x2, y2;		/* and lower-right corner */
	PinTypePtr		via;
	Cardinal		n;
	Boolean			changed = False;

	if (PCB->ViaOn)
	{
			/* get upper left and lower right corner of surrounding rectangle */
		x1 = MIN(X1, X2);
		x2 = MAX(X1, X2);
		y1 = MIN(Y1, Y2);
		y2 = MAX(Y1, Y2);

		for (n = PCB->ViaN, via = PCB->Via; n; n--, via++)
			if (VIA_OR_PIN_IN_RECTANGLE(via, x1, y1, x2, y2))
			{
				ChangeViaSize(via, Delta);
				changed = True;
			}
	}
	return(changed);
}

/* ---------------------------------------------------------------------------
 * changes the size of all pins in rectangle by Delta
 * returns TRUE if something was changed
 */
Boolean ChangePinSizeInRectangle(Position X1, Position Y1, Position X2, Position Y2, Position Delta)
{
	Position		x1, y1,		/* passed arguments are not necessary top-left */
					x2, y2;		/* and lower-right corner */
	PinTypePtr		pin;
	ElementTypePtr	element;
	Cardinal		i,
					j;
	Boolean			changed = False;

	if (PCB->PinOn)
	{
			/* get upper left and lower right corner of surrounding rectangle */
		x1 = MIN(X1, X2);
		x2 = MAX(X1, X2);
		y1 = MIN(Y1, Y2);
		y2 = MAX(Y1, Y2);

		for (i = PCB->ElementN, element = PCB->Element; i; i--, element++)
			for (j = element->PinN, pin = element->Pin; j; j--, pin++)
				if (VIA_OR_PIN_IN_RECTANGLE(pin, x1, y1, x2, y2))
				{
					ChangePinSize(pin, Delta);
					changed = True;
				}
	}
	return(changed);
}

