/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: memory.c,v 1.3 94/06/19 11:22:12 nau Exp $";

/* memory management functions
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "global.h"
#include "error.h"
#include "memory.h"

#ifdef	DBMALLOC_INCLUDE
#include DBMALLOC_INCLUDE
#endif

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	void	DSRealloc(DynamicStringTypePtr, size_t);

/* ---------------------------------------------------------------------------
 * allocates memory with error handling
 */
void *MyCalloc(size_t Number, size_t Size, char *Text)
{
	void	*p;

	if ((p = calloc(Number, Size)) == NULL)
		MyFatal("out of memory during calloc() in '%s'()\n", (Text ? Text : "(unknown)"));
	return(p);
}

/* ---------------------------------------------------------------------------
 * allocates memory with error handling
 * this is a save version because BSD doesn't support the
 * handling of NULL pointers
 */
void *MyRealloc(void *Ptr, size_t Size, char *Text)
{
	void	*p;

	if (Ptr)
		p = realloc(Ptr, Size);
	else
		p = malloc(Size);
	if (!p)
		MyFatal("out of memory during realloc() in '%s'()\n", (Text ? Text : "(unknown)"));
	return(p);
}

/* ---------------------------------------------------------------------------
 * allocates memory for a new string, does some error processing
 */
char *MyStrdup(char *S, char *Text)
{
	char	*p;

	if ((p = strdup(S)) == NULL)
		MyFatal("out of memory during strdup() in '%s'\n", (Text ? Text : "(unknown)"));
	return(p);	
}
 
/* ---------------------------------------------------------------------------
 * frees memory and sets pointer to NULL
 */
void MyFree(char **Ptr)
{
	SaveFree(*Ptr);
	*Ptr = NULL;
}

/* ---------------------------------------------------------------------------
 * frees memory used by an element
 */
void FreeElementMemory(ElementTypePtr Element)
{
	PinTypePtr		pin;

	if (!Element)
		return;
	MyFree(&Element->NameOnPCB);
	MyFree(&Element->CanonicalName);
	for (pin = Element->Pin; Element->PinN; pin++, Element->PinN--)
		MyFree(&pin->Name);
	MyFree((char **) &Element->Pin);
	MyFree((char **) &Element->Line);
	MyFree((char **) &Element->Arc);
	MyFree((char **) &Element->Mark);
	memset(Element, 0, sizeof(ElementType));
}

/* ---------------------------------------------------------------------------
 * free memory used by PCB
 */
void FreePCBMemory(PCBTypePtr PCBPtr)
{
	LayerTypePtr	layer;
	TextTypePtr		text;
	ElementTypePtr	element;
	PinTypePtr		via;
	int				i;

	if (!PCBPtr)
		return;

	MyFree(&PCBPtr->Name);
	MyFree(&PCBPtr->Filename);
	for (via = PCBPtr->Via; PCBPtr->ViaN; via++, PCBPtr->ViaN--)
		MyFree(&via->Name);

	for (element = PCBPtr->Element; PCBPtr->ElementN; element++, PCBPtr->ElementN--)
		FreeElementMemory(element);

	for (layer = PCBPtr->Layer, i = 0; i < MAX_LAYER; layer++, i++)
	{
		MyFree(&layer->Name);
		MyFree((char **) &layer->Line);
		MyFree((char **) &layer->Rect);
		for (text=layer->Text; layer->TextN; text++, layer->TextN--)
			MyFree(&text->TextString);
		MyFree((char **) &layer->Text);
	}

	for (i = 0; i <= MAX_FONTPOSITION; i++)
		MyFree((char **) &PCBPtr->Font.Symbol[i].Line);

		/* clear struct */
	memset(PCBPtr, 0, sizeof(PCBType));
}

/* ---------------------------------------------------------------------------
 * a 'save' free routine which first does a quick check if the pointer
 * is zero. The routine isn't implemented as a macro to make additional
 * savety features easier to implement
 */
void SaveFree(void *Ptr)
{
	if (Ptr)
		free(Ptr);
}

/* ---------------------------------------------------------------------------
 * reallocates memory for a dynamic length string if necessary
 */
static void DSRealloc(DynamicStringTypePtr Ptr, size_t Length)
{
	if (Ptr->Data == NULL || Length >= Ptr->MaxLength)
	{
		Ptr->MaxLength = Length +512;
		Ptr->Data = MyRealloc(Ptr->Data, Ptr->MaxLength, "ReallocDS()");
	}
}

/* ---------------------------------------------------------------------------
 * adds one character to a dynamic string
 */
void DSAddCharacter(DynamicStringTypePtr Ptr, char Char)
{
	size_t	position = Ptr->Data ? strlen(Ptr->Data) : 0;

	DSRealloc(Ptr, position+1);
	Ptr->Data[position++] = Char;
	Ptr->Data[position] = '\0';
}

/* ---------------------------------------------------------------------------
 * add a string to a dynamic string
 */
void DSAddString(DynamicStringTypePtr Ptr, char *S)
{
	size_t	position = Ptr->Data ? strlen(Ptr->Data) : 0;

	DSRealloc(Ptr, position+1+strlen(S));
	strcat(&Ptr->Data[position], S);
} 
