%{
// Copyright (c) 1995  David Engberg  All rights reserved
// $Id: parser.y,v 1.11 1996/05/03 23:39:15 geppetto Exp $
#include <map>
#include <stream.h>
#include <cstdlib>
#include <deque>
#include "unicode_string.h"
#include "Compiler.h"
#include "parser_decls.h"
#include "JavaAccessFlags.h"
#include "JavaTypeSignature.h"
#include "JavaFieldSignature.h"
#include "Expression.h"
#include "VariableDeclaration.h"
#include "Statement.h"
#include "IntermediateClass.h"
#include "BinaryExpression.h"

CCompiler* gCurrentCompiler = 0;
const unicode_char kSymbolDivider = (unicode_char)'/';

CExpression* InterpretSymbolExpression(const unicode_string& symbol);

#ifndef NDEBUG
#ifndef YYDEBUG
#define YYDEBUG 1
#endif
#endif

%}

%union {
  unicode_string* text;
  deque<unicode_string>* stringList;
  unsigned long longNumber;   // also used for a single character, etc.
  unsigned long long doubleLong;
  float floatNumber;
  double doubleFloat;
  CJavaAccessFlags* modifiers;
  CJavaTypeSignature* typeSignature;
  CJavaFieldSignature* fieldSignature;
  deque<CJavaFieldSignature>* fieldList;
  CExpression* expression;
  ExpressionList* expressionList;
  deque<CVariableDeclaration*>* variableList;
  CStatement* statement;
  CCompoundStatement* compoundStatement;
  StatementList* statementList;
  deque<CCatchClause*>* catchList;
  CCatchClause* catchClause;
}

%token ERROR   /* used by the lexer. */

%token ABSTRACT BOOLEAN BREAK BYTE CASE CATCH CHAR CLASS CONTINUE DEFAULT 
%token DO DOUBLE EXTENDS FALSE_TOKEN FINAL FINALLY FLOAT FOR IF IMPLEMENTS 
%token IMPORT INSTANCEOF INT INTERFACE LONG NATIVE NULL_TOKEN PACKAGE 
%token PRIVATE PROTECTED PUBLIC RETURN SHORT STATIC SUPER SWITCH SYNCHRONIZED 
%token THIS THROW THROWS TRANSIENT VOLATILE TRUE_TOKEN TRY VOID WHILE 

%token <longNumber> INT_LITERAL CHARACTER_LITERAL
%token <doubleLong> LONG_LITERAL 
%token <floatNumber> FLOAT_LITERAL
%token <doubleFloat> DOUBLE_LITERAL
%token <text> SYMBOL STRING_LITERAL

%type <text> qualifiedSymbol simpleSymbol extends qualifiedSymbolWithLBracket
%type <stringList> qualifiedSymbolList interfaces interfaceExtends optThrows
%type <modifiers> modifierList classModifierList
%type <longNumber> optArrayBounds
%type <typeSignature> type simpleType
%type <fieldSignature> parameter
%type <fieldList> parameterList optParameterList
%type <expression> literal cast expression newExpression expressionStatement
%type <expression> nonSymbolExpression nonSymbolComplexPrimary optExpression
%type <expression> initializer optVariableInitializerList optInitializer
%type <expression> methodCall symbolArrayExpression
%type <expressionList> argumentList optArgumentList variableInitializerList
%type <expressionList> forIncr expressionStatements allocationBounds
%type <variableList> variableDeclaration partialVariable
%type <statement> statement forInit
%type <statementList> statementList optConstructorStatements
%type <compoundStatement> compoundStatement optFinally finally methodBlock
%type <catchList> catchList
%type <catchClause> catchItem

%nonassoc NOT_AN_OPERATOR
%right SHIFT_RIGHT_EQUALS FILL_SHIFT_RIGHT_EQUALS SHIFT_LEFT_EQUALS ADD_EQUALS SUB_EQUALS MUL_EQUALS DIV_EQUALS MOD_EQUALS AND_EQUALS XOR_EQUALS OR_EQUALS '='
%right '?' ':'
%left OR
%left AND
%left '|'
%left '&'
%left '^'
%left EQUAL_COMPARE NOT_EQUAL
%left LTEQ GTEQ INSTANCEOF '<' '>'
%left BITSHIFT_RIGHT FILL_SHIFT_RIGHT SHIFT_LEFT
%left '+' '-'
%left '*' '/' '%'
%nonassoc CAST
%nonassoc INCR DECR '!' '~' UMINUS UPLUS
%nonassoc POST_INCR POST_DECR
%nonassoc NEW
%nonassoc '(' ')' '[' ']'
%left '.'

/* thse two are used to fix the standard dangling 'else' problem: */
%nonassoc LOWER_THAN_ELSE
%nonassoc ELSE

/*
 * There should be three shift-reduce conflicts in the following code.  They
 * are essentially the same problem, and 'fixing' them would be extremely
 * painful, particularly given that bison's default behavior (shift) does the
 * right thing. In all three cases, expressions can end with a
 * 'qualifiedSymbol' which is defined to be   SYMBOL ('.' SYMBOL)*
 * and this conflicts with a situation where expression can itself be followed
 * by   '.' SYMBOL
 * By always shifting, we are sticking all of those extra .SYMBOL markers
 * onto the qualifiedSymbol, which is what we want to do anyway.
 * Fixing this would entail modifying every single expression to always
 * terminate the right way, which would be really difficult given that I
 * already had to do this for the beginning of expressions to disambiguate
 * statements from casts that start with a type.
 */
%expect 3

%%

compilationUnit : optPackage importList typeDeclarationList
    ;

optPackage :
    | PACKAGE qualifiedSymbol ';'
    {
      gCurrentCompiler->SetPackage(*($2));
      delete $2;
    }
    ;

importList :
    | importList import
    ;

import : IMPORT qualifiedSymbol '.' '*' ';'
    {
      if (!gCurrentCompiler->ImportPackage(*($2))) {
	string errorMessage("Invalid package: ");
	errorMessage += ::UnicodeToUTF(*($2));
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      delete $2;
    }
    | IMPORT qualifiedSymbol ';'
    {
      if (gCurrentCompiler->ImportClass(*($2)) == 0) {
	string errorMessage("Invalid class: ");
	errorMessage += ::UnicodeToUTF(*($2));
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      delete $2;
    }
    ;

typeDeclarationList :
    | typeDeclarationList typeDeclaration
    ;

typeDeclaration : ';'
    | classDeclaration
    | interfaceDeclaration
    ;

classDeclaration :
    classModifierList CLASS simpleSymbol extends interfaces
    {
      gCurrentCompiler->StartClass(*($3), $1, $4, $5);
      delete $3;
    }
    classBlock
    { gCurrentCompiler->EndClass(); }
    ;

interfaceDeclaration : classModifierList INTERFACE simpleSymbol
    interfaceExtends
    {
      CJavaAccessFlags usedFlags(*($1));
      usedFlags.fPublic = usedFlags.fAbstract = 0;
      $1->fAbstract = $1->fInterface = 1;
      if (usedFlags.Count() > 0) {
	string errorMessage("Invalid interface modifier use: ");
	errorMessage += usedFlags.FlagNames();
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      gCurrentCompiler->StartInterface(*($3), $1, $4);
    }
      classBlock
    { gCurrentCompiler->EndClass(); }
    ;

classModifierList :
    { $$ = new CJavaAccessFlags; }
    | classModifierList FINAL
    {
      $$ = $1;
      if ($$->fFinal) {
	yyerror("'final' modifier repeated");
	YYERROR;
      }
      $$->fFinal = 1;
    }
    | classModifierList PUBLIC
    {
      $$ = $1;
      if ($$->fPublic) {
	yyerror("'public' modifier repeated");
	YYERROR;
      }
      $$->fPublic = 1;
    }
    | classModifierList ABSTRACT
    {
      $$ = $1;
      if ($$->fAbstract) {
	yyerror("'abstract' modifier repeated");
	YYERROR;
      }
      $$->fAbstract = 1;
    }
    ;

extends :
    { $$ = 0; }
    | EXTENDS qualifiedSymbol
    { $$ = $2; }
    ;

interfaceExtends :
    { $$ = 0; }
    | EXTENDS qualifiedSymbolList
    { $$ = $2; }
    ;

interfaces :
    { $$ = 0; }
    | IMPLEMENTS qualifiedSymbolList
    { $$ = $2; }
    ;

classBlock : '{' '}'
    | '{' fieldList '}'
    ;

fieldList : field
    | fieldList field
    ;

field : ';'
    | methodDeclaration
    | constructorDeclaration
    | modifierList variableDeclaration
    {
      CJavaAccessFlags usedFlags(*$1);
      usedFlags.fPublic = usedFlags.fProtected = usedFlags.fPrivate =
          usedFlags.fStatic = usedFlags.fFinal = usedFlags.fTransient =
          usedFlags.fVolatile = 0;
      if (usedFlags.Count() > 0) {
	string errorMessage("Invalid modifier use in field declaration: ");
	errorMessage += usedFlags.FlagNames();
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      gCurrentCompiler->AddField(new CDeclarationStatement($2, $1));
    }
    | staticInitializer
    ;

staticInitializer : modifierList compoundStatement
    {
      if ($1->fStatic == 0) {
	string errorMessage(
		     "Static initialization block requires STATIC modifier");
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      if ($1->Count() > 1) {
	$1->fStatic = 0;
	string errorMessage("Invalid modifiers before static initializer: ");
	errorMessage += $1->FlagNames();
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      gCurrentCompiler->AddStaticCode($2);
      delete $1;
    }
    ;

methodDeclaration : modifierList type simpleSymbol '(' optParameterList ')'
         optArrayBounds optThrows
    {
      if (gCurrentCompiler->GetCurrentClass()->GetName() == *($3)) {
	string errorMessage("Invalid use of return value with constructor.");
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      if ($7 > 0 && $2->GetBaseType() == CJavaTypeSignature::Void) {
	yyerror("Invalid return value of void array.");
	YYERROR;
      }
      $2->SetArrayBounds($2->GetArrayBounds() + $7);
      CJavaAccessFlags usedFlags = *($1);
      usedFlags.fPublic = usedFlags.fProtected = usedFlags.fPrivate =
          usedFlags.fStatic = usedFlags.fFinal = usedFlags.fAbstract =
          usedFlags.fNative = usedFlags.fSynchronized = 0;
      if (usedFlags.Count() > 0) {
	string errorMessage("Invalid modifier use in method declaration: ");
	errorMessage += usedFlags.FlagNames();
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      gCurrentCompiler->StartFunction(*($2), *($3), *($1), $5, $8);
      delete $2;
    }
    methodBlock
    {
      if ($10 != 0) {
	if ($1->fAbstract != 0) {
	  string errorMessage("Non-empty block used with abstract method ");
	  errorMessage += ::UnicodeToString(*($3));
	  yyerror(errorMessage.c_str());
	  YYERROR;
	} else if ($1->fNative != 0) {
	  string errorMessage("Non-empty block used with native method ");
	  errorMessage += ::UnicodeToString(*($3));
	  yyerror(errorMessage.c_str());
	  YYERROR;
	}
      }
      gCurrentCompiler->EndFunction($10);
      delete $1;
      delete $3;
    }
    ;

constructorDeclaration : modifierList simpleSymbol
    {
      if (!(gCurrentCompiler->GetCurrentClass()->GetShortName() == *($2))) {
	string errorMessage("No type given for function: ");
	errorMessage += ::UnicodeToString(*($2));
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      delete $2;
    } 
       '(' optParameterList ')' optThrows '{'
    {
      CJavaAccessFlags usedFlags(*$1);
      usedFlags.fPublic = usedFlags.fProtected = usedFlags.fPrivate = 0;
      if (usedFlags.Count() > 0) {
	string errorMessage("Invalid modifier use in constructor: ");
	errorMessage += usedFlags.FlagNames();
	yyerror(errorMessage.c_str());
	YYERROR;
      }
      gCurrentCompiler->StartConstructor(*$1, $5, $7);
      delete $1;
    }
       optConstructorStatements '}'
    { 
      CCompoundStatement* statement = new CCompoundStatement($10);
      gCurrentCompiler->EndFunction(statement);
    }
    ;

optThrows :
    { $$ = 0; }
    | THROWS qualifiedSymbolList
    { $$ = $2; }
    ;

optConstructorStatements :
    { $$ = new StatementList; }
    | statement statementList
    {
      $$ = $2;
      if ($1 != 0) {
	$$->push_front($1);
      }
    }
    | SUPER '(' optArgumentList ')' ';' statementList
    {
      $$ = $6;
      $$->push_front(
	   new CExplicitConstructorCall(CExplicitConstructorCall::kSuper, $3));
    }
    | THIS '(' optArgumentList ')' ';' statementList
    {
      $$ = $6;
      $$->push_front(
	    new CExplicitConstructorCall(CExplicitConstructorCall::kThis, $3));
    }
    ;

methodBlock : ';'
    { $$ = 0; }
    | compoundStatement
    { $$ = $1; }
    ;

optParameterList :
    { $$ = 0; }
    | parameterList
    { $$ = $1; }
    ;

parameterList : parameter
    {
      $$ = new deque<CJavaFieldSignature>(1, *($1));
      delete $1;
    }
    | parameterList ',' parameter
    { 
      $$ = $1;
      for (deque<CJavaFieldSignature>::iterator parameterIter = $1->begin();
	   parameterIter != $1->end(); ++parameterIter) {
	if ((*parameterIter).GetFieldName() == (*($3)).GetFieldName()) {
	  string errorString("Invalid re-use of parameter name: ");
	  errorString += ::UnicodeToString((*($3)).GetFieldName());
	  yyerror(errorString.c_str());
	  YYERROR;	  
	}
      }
      $$->push_back(*($3));
      delete $3;
    }
    ;

parameter : type simpleSymbol optArrayBounds
    {
      if ($1->GetBaseType() == CJavaTypeSignature::Void) {
	yyerror("Invalid use of 'void' for a parameter type.");
	YYERROR;
      }
      $1->SetArrayBounds($1->GetArrayBounds() + $3);
      $$ = new CJavaFieldSignature(*($1), *($2));
      delete $1;
      delete $2;
    }
    ;

variableDeclaration : partialVariable ';'
    {
      $$ = $1;
      // take off the bogus entry added in partialVariable
      delete $$->front();  
      $$->pop_front();
    }
    ;

/*
 * This rule is structured a little awkwardly because each rule needs to
 * know the base type when it is being evaluated, since in a situation like
 * this:    int x=0, y=x;
 * I can't evaluate "x=0, y=x" as a normal list and then wait to add the 'int'
 * type to these variables, since 'x' has already been used once.
 */
partialVariable : type simpleSymbol optArrayBounds optInitializer
    {
      if ($1->GetBaseType() == CJavaTypeSignature::Void) {
	yyerror("Invalid use of 'void' for a variable type.");
	YYERROR;
      }
      $$ = new deque<CVariableDeclaration*>;
      // push a bogus entry at the beginning, just to hold the base type of
      // this variable declaration sequence.
      $$->push_front(new CVariableDeclaration(*($1), unicode_string()));
      $1->SetArrayBounds($1->GetArrayBounds() + $3);
      CArrayInitializer* arrayInitializer= DYNAMIC_CAST(CArrayInitializer, $4);
      if (arrayInitializer != 0) {
	if ($1->GetArrayBounds() == 0) {
	  string errorMessage("Cannot use array initializer with non-array ");
	  errorMessage += ::UnicodeToUTF(*$2);
	  yyerror(errorMessage.c_str());
	  YYERROR;
	} else {
	  arrayInitializer->SetArrayType(*($1));
	}
      }
      
      CJavaFieldSignature signature(*($1), *($2));
      if (gCurrentCompiler->InStatementBlock()) {
	if (gCurrentCompiler->LookupLocalVariable(*$2) != 0) {
	  string errorMessage = ::UnicodeToUTF(*$2);
	  errorMessage += " already declared in this scope.";
	  yyerror(errorMessage.c_str());
	  YYERROR;
	} else {
	  gCurrentCompiler->AddLocalVariable(signature);
	}
      }
      CExpression* initializer = $4;
      if (initializer != 0) {
	initializer =
	  new CBinaryExpression(::InterpretSymbolExpression(*($2)),
				'=', initializer);
      }
      $$->push_back(new CVariableDeclaration(*($1), *($2), initializer));
      delete $1;
      delete $2;
    }
    | partialVariable ',' simpleSymbol optArrayBounds optInitializer
    {
      $$ = $1;
      CJavaTypeSignature thisType = ($$)->front()->GetType();
      thisType.SetArrayBounds(thisType.GetArrayBounds() + $4);
      CArrayInitializer* arrayInitializer= DYNAMIC_CAST(CArrayInitializer, $5);
      if (arrayInitializer != 0) {
	if (thisType.GetArrayBounds() == 0) {
	  string errorMessage("Cannot use array initializer with non-array ");
	  errorMessage += ::UnicodeToUTF(*$3);
	  yyerror(errorMessage.c_str());
	  YYERROR;	  
	} else {
	  arrayInitializer->SetArrayType(thisType);
	}
      }

      CJavaFieldSignature signature(thisType, *($3));
      if (gCurrentCompiler->InStatementBlock()) {
	if (gCurrentCompiler->LookupLocalVariable(*$3) != 0) {
	  string errorMessage = ::UnicodeToUTF(*$3);
	  errorMessage += " already declared in this scope.";
	  yyerror(errorMessage.c_str());
	  YYERROR;
	} else {
	  gCurrentCompiler->AddLocalVariable(signature);
	}
      }
      CExpression* initializer = $5;
      if (initializer != 0) {
	initializer =
	  new CBinaryExpression(::InterpretSymbolExpression(*($3)),
				'=', initializer);
      }
      $$->push_back(new CVariableDeclaration(thisType, *($3), initializer));
      delete $3;
    }
    ;

optInitializer :
    { $$ = 0; }
    | '=' initializer
    { $$ = $2; }
    ;

initializer : expression
    { $$ = $1; }
    | '{' optVariableInitializerList '}'
    { $$ = $2; }
    ;

optVariableInitializerList :
    { $$ = new CArrayInitializer(); }
    | variableInitializerList optComma
    { $$ = new CArrayInitializer($1); }
    ;

variableInitializerList : initializer
    {
      $$ = new ExpressionList;
      $$->push_back($1);
    }
    | variableInitializerList ',' initializer
    {
      $$ = $1;
      $$->push_back($3);
    }
    ;

optComma :
    | ','
    ;

compoundStatement : '{'
    { gCurrentCompiler->PushLocalScope(); }
      statementList '}'
    {
      gCurrentCompiler->PopLocalScope();
      $$ = new CCompoundStatement($3);
    }
    ;

statementList :
    { $$ = new StatementList; }
    | statementList statement
    {
      $$ = $1;
      if ($2 != 0) {
	$1->push_back($2);
      }
    }
    ;

statement : ';'
    { $$ = 0; }
    | error ';'
    { $$ = 0; }  /* using bison's default error-recovery system. */
    | error '}'
    { $$ = 0; }
    | compoundStatement
    { $$ = $1; }
    | expressionStatement ';'
    { $$ = new CExpressionStatement($1); }
    | variableDeclaration
    { $$ = new CDeclarationStatement($1); }
    | IF '(' expression ')' statement     %prec LOWER_THAN_ELSE
    { $$ = new CIfStatement($3, $5); }
    | IF '(' expression ')' statement ELSE statement
    { $$ = new CIfStatement($3, $5, $7); }
    | WHILE '(' expression ')' statement
    { $$ = new CWhileStatement($5, $3); }
    | DO statement WHILE '(' expression ')' ';'
    { $$ = new CDoStatement($2, $5); }
    | BREAK ';'
    { $$ = new CBranchStatement(CBranchStatement::kBreak); }
    | BREAK simpleSymbol ';'
    { $$ = new CBranchStatement(CBranchStatement::kBreak, $2); }
    | CONTINUE ';'
    { $$ = new CBranchStatement(CBranchStatement::kContinue); }
    | CONTINUE simpleSymbol ';'
    { $$ = new CBranchStatement(CBranchStatement::kContinue, $2); }
    | RETURN ';'
    { $$ = new CReturnStatement(); }
    | RETURN expression ';'
    { $$ = new CReturnStatement($2); }
    | FOR '('   { gCurrentCompiler->PushLocalScope(); }
       forInit optExpression ';' forIncr ')' statement
    {
      $$ = new CForStatement($4, $5, $7, $9);
      gCurrentCompiler->PopLocalScope();
    }
    | THROW expression ';'
    { $$ = new CThrowStatement($2); }
    | SYNCHRONIZED '(' expression ')' statement
    { $$ = new CSynchronized($3, $5, gCurrentCompiler->AddUnnamedVariable()); }
    | simpleSymbol ':' statement    %prec NOT_AN_OPERATOR
    { $$ = new CLabelStatement($1, $3); }
    | TRY compoundStatement catchList optFinally
    { 
      unsigned short handlerVariable, subroutineVariable;
      if ($4 != 0) {
	handlerVariable = gCurrentCompiler->AddUnnamedVariable();
	subroutineVariable = gCurrentCompiler->AddUnnamedVariable();
      }
      $$ = new CTryStatement($2, $3, $4, handlerVariable, subroutineVariable);
    }
    | TRY compoundStatement finally
    {
      unsigned short handlerVariable, subroutineVariable;
      handlerVariable = gCurrentCompiler->AddUnnamedVariable();
      subroutineVariable = gCurrentCompiler->AddUnnamedVariable();
      $$ = new CTryStatement($2, 0, $3, handlerVariable, subroutineVariable);
    }
    | SWITCH '(' expression ')' compoundStatement
    { $$ = new CSwitch($3, $5); }
    | CASE expression ':' statement
    { $$ = new CLabelStatement($2, $4); }
    | DEFAULT ':' statement
    { $$ = new CLabelStatement($3); }
    ;

optExpression :
    { $$ = 0; }
    | expression
    { $$ = $1; }
    ;

forInit : ';'
    { $$ = 0; }
    | expressionStatements ';'
    {
      StatementList* statements = new StatementList;
      for (ExpressionList::iterator i = $1->begin(); i != $1->end(); ++i) {
	statements->push_back(new CExpressionStatement(*i));
      }
      delete $1;
      $$ = new CCompoundStatement(statements);
    }
    | variableDeclaration
    { $$ = new CDeclarationStatement($1); }
    ;

forIncr :
   { $$ = 0; }
   | expressionStatements
   { $$ = $1; }
   ;

expressionStatements : expressionStatement
    {
      $$ = new ExpressionList;
      $$->push_front($1);
    }
    | expressionStatements ',' expressionStatement
    {
      $$ = $1;
      $$->push_back($3);
    }
    ;

optFinally :
    { $$ = 0; }
    | finally
    { $$ = $1; }
    ;

finally : FINALLY compoundStatement
    { $$ = $2; }
    ;

catchList : catchItem
    {
      $$ = new deque<CCatchClause*>;
      $$->push_back($1);
    }
    | catchList catchItem
    {
      $$ = $1;
      $$->push_back($2);
    }
    ;

catchItem : CATCH '('
    { gCurrentCompiler->PushLocalScope(); }
      qualifiedSymbol simpleSymbol
    { 
      CJavaFieldSignature signature(*($4), *($5));
      if (gCurrentCompiler->LookupLocalVariable(*$5) != 0) {
	string errorMessage = ::UnicodeToUTF(*$5);
	errorMessage += " already declared in this scope.";
	yyerror(errorMessage.c_str());
	YYERROR;
      } else {
	$<longNumber>$ = gCurrentCompiler->AddLocalVariable(signature);
      }
    }
      ')' compoundStatement
    {
      gCurrentCompiler->PopLocalScope();
      CJavaTypeSignature catchType(*($4));
      $$ = new CCatchClause($<longNumber>6, catchType, $8);
    }
    ;

expression : qualifiedSymbol
    {
      $$ = ::InterpretSymbolExpression(*($1));
      delete $1;
    }
    | nonSymbolExpression
    { $$ = $1; }
    ;

symbolArrayExpression : qualifiedSymbolWithLBracket expression ']'
    {
      CExpression* baseExpression = ::InterpretSymbolExpression(*($1));
      delete $1;
      $$ = new CArrayIndex(baseExpression, $2);
    }

nonSymbolExpression : expressionStatement
    { $$ = $1; }
    | expression '?' expression ':' expression 
    { $$ = new CTrinaryExpression($1, $3, $5); }
    | expression OR expression
    { $$ = new CBinaryExpression($1, OR, $3); }
    | expression AND expression
    { $$ = new CBinaryExpression($1, AND, $3); }
    | expression '|' expression
    { $$ = new CBinaryExpression($1, '|', $3); }
    | expression '&' expression
    { $$ = new CBinaryExpression($1, '&', $3); }
    | expression '^' expression
    { $$ = new CBinaryExpression($1, '^', $3); }
    | expression EQUAL_COMPARE expression
    { $$ = new CBinaryExpression($1, EQUAL_COMPARE, $3); }
    | expression NOT_EQUAL expression
    { $$ = new CBinaryExpression($1, NOT_EQUAL, $3); }
    | expression LTEQ expression
    { $$ = new CBinaryExpression($1, LTEQ, $3); }
    | expression GTEQ expression
    { $$ = new CBinaryExpression($1, GTEQ, $3); }
    | expression '<' expression
    { $$ = new CBinaryExpression($1, '<', $3); }
    | expression '>' expression
    { $$ = new CBinaryExpression($1, '>', $3); }
    | expression BITSHIFT_RIGHT expression
    { $$ = new CBinaryExpression($1, BITSHIFT_RIGHT, $3); }
    | expression FILL_SHIFT_RIGHT expression
    { $$ = new CBinaryExpression($1, FILL_SHIFT_RIGHT, $3); }
    | expression SHIFT_LEFT expression
    { $$ = new CBinaryExpression($1, SHIFT_LEFT, $3); }
    | expression '+' expression
    { $$ = new CBinaryExpression($1, '+', $3); }
    | expression '-' expression
    { $$ = new CBinaryExpression($1, '-', $3); }
    | expression '*' expression
    { $$ = new CBinaryExpression($1, '*', $3); }
    | expression '/' expression
    { $$ = new CBinaryExpression($1, '/', $3); }
    | expression '%' expression
    { $$ = new CBinaryExpression($1, '%', $3); }
    | '-' expression  %prec UMINUS
    { $$ = new CUnaryExpression($2, '-', CUnaryExpression::kPrefix, false); }
    | '+' expression  %prec UPLUS
    { $$ = new CUnaryExpression($2, '+', CUnaryExpression::kPrefix, false); }
    | '!' expression
    { $$ = new CUnaryExpression($2, '!', CUnaryExpression::kPrefix, false); }
    | '~' expression
    { $$ = new CUnaryExpression($2, '~', CUnaryExpression::kPrefix, false); }
    | expression INSTANCEOF type
    {
      $$ = new CInstanceof($1, *($3));
      delete $3;
    }
    | cast
    { $$ = $1; }
    | nonSymbolComplexPrimary
    { $$ = $1; }
    | SUPER
    { $$ = new CSpecialExpression(CSpecialExpression::kSuper); }
    | THIS
    { $$ = new CSpecialExpression(CSpecialExpression::kThis); }
    | NULL_TOKEN
    { $$ = new CNull(); }
    ;

nonSymbolComplexPrimary : literal
    { $$ = $1; }
    | '(' nonSymbolExpression ')'
    { $$ = $2; }
    | '(' qualifiedSymbol ')'
    {
      $$ = ::InterpretSymbolExpression(*($2));
      delete $2;
    }
    | nonSymbolComplexPrimary '[' expression ']'
    { $$ = new CArrayIndex($1, $3); }
    | methodCall '[' expression ']'
    { $$ = new CArrayIndex($1, $3); }
    | nonSymbolExpression '.' simpleSymbol
    {
      $$ = new CClassFieldExpression($1, *($3));
      delete $3;
    }
    | symbolArrayExpression
    { $$ = $1; }
    ;

expressionStatement : expression '=' expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, '=', $3);
    }
    | expression SHIFT_RIGHT_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, SHIFT_RIGHT_EQUALS, $3);
    }
    | expression FILL_SHIFT_RIGHT_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, FILL_SHIFT_RIGHT_EQUALS, $3);
    }
    | expression SHIFT_LEFT_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, SHIFT_LEFT_EQUALS, $3);
    }
    | expression ADD_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, ADD_EQUALS, $3);
    }
    | expression SUB_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, SUB_EQUALS, $3);
    }
    | expression MUL_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, MUL_EQUALS, $3);
    }
    | expression DIV_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, DIV_EQUALS, $3);
    }
    | expression MOD_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, MOD_EQUALS, $3);
    }
    | expression AND_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, AND_EQUALS, $3);
    }
    | expression XOR_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, XOR_EQUALS, $3);
    }
    | expression OR_EQUALS expression
    {
      assert($1 != 0);
      if (! $1->IsLValue()) {
	yyerror("Invalid left hand side of assignment.");
	YYERROR;
      }
      $$ = new CBinaryExpression($1, OR_EQUALS, $3);
    }
    | INCR expression
    { $$ = new CUnaryExpression($2, INCR, CUnaryExpression::kPrefix, true); }
    | DECR expression
    { $$ = new CUnaryExpression($2, DECR, CUnaryExpression::kPrefix, true); }
    | expression INCR  %prec POST_INCR
    { $$ = new CUnaryExpression($1, INCR, CUnaryExpression::kPostfix, true); }
    | expression DECR  %prec POST_DECR
    { $$ = new CUnaryExpression($1, DECR, CUnaryExpression::kPostfix, true); }
    | methodCall
    { $$ = $1; }
    | newExpression
    { $$ = $1; }
    ;

methodCall : qualifiedSymbol '(' optArgumentList ')'
    {
      unicode_string::size_type slashPosition = $1->find(kSymbolDivider);
      if (slashPosition != unicode_string::npos) {
	unicode_string prefix(*($1), 0, slashPosition);
	const pair<CJavaFieldSignature, CCompiler::LocalVariableIndex>* match =
	  gCurrentCompiler->LookupLocalVariable(prefix);
	if (match != 0) {
	  unicode_string::size_type lastSlash =
	    $1->find_last_of(kSymbolDivider);
	  prefix.assign(*($1), 0, lastSlash);
	  CExpression* baseExpression = ::InterpretSymbolExpression(prefix);
	  unicode_string methodName(*($1), lastSlash + 1);
	  $$ = new CMethodCall(baseExpression, methodName, $3);
	} else {
	  $$ = new CMethodCall(0, *($1), $3);
	}
      } else {
	$$ = new CMethodCall(0, *($1), $3);
      }
      delete $1;
    }
    | nonSymbolExpression '.' simpleSymbol '(' optArgumentList ')'
    {
      $$ = new CMethodCall($1, *($3), $5);
      delete $3;
    }
    ;

cast : '(' simpleType optArrayBounds ')' expression  %prec CAST
    {
      if ($2->GetBaseType() == CJavaTypeSignature::Void) {
	yyerror("Invalid use of 'void' in a cast.");
	YYERROR;
      }
      $2->SetArrayBounds($3);
      $$ = new CCastExpression(*($2), $5);
      delete $2;
    }
    | '(' qualifiedSymbol ')' expression   %prec CAST
    {
      $$ = new CCastExpression(*($2), $4);
      delete $2;
    }
    | '(' qualifiedSymbolWithLBracket ']' optArrayBounds ')'
      expression   %prec CAST
    {
      CJavaTypeSignature type(*($2), 1 + $4);
      $$ = new CCastExpression(type, $6);
      delete $2;
    }
    ;

newExpression : NEW simpleType '[' allocationBounds
    {
      if ($2->GetBaseType() == CJavaTypeSignature::Void) {
	yyerror("Invalid array creation using 'void' type.");
	YYERROR;
      }
      $2->SetArrayBounds($4->size());
      while ($4->back() == 0) {
	$4->pop_back();
      }
      $$ = new CNewArray(*($2), $4);
      delete $2;
    }
    | NEW qualifiedSymbol '[' allocationBounds
    {
      CJavaTypeSignature tempSignature(*($2), $4->size());
      while ($4->back() == 0) {
	$4->pop_back();
      }
      $$ = new CNewArray(tempSignature, $4);
      delete $2;
    }
    | NEW qualifiedSymbol
    {
      $$ = new CNewObject(*($2));
      delete $2;
    }
    | NEW qualifiedSymbol '(' optArgumentList ')'
    {
      $$ = new CNewObject(*($2), $4);
      delete $2;
    }
    ;

allocationBounds : expression ']'
    {
      $$ = new ExpressionList;
      $$->push_front($1);
    }
    | expression ']' '[' allocationBounds
    {
      $$ = $4;
      $$->push_front($1);
    }
    | expression ']' '[' ']' optArrayBounds
    {
      $$ = new ExpressionList;
      $$->push_front($1);
      unsigned long arrayBounds = $5 + 1;
      while (arrayBounds-- > 0) {
	$$->push_back((CExpression*)0);
      }
    }
    ;

optArrayBounds :
    { $$ = 0; }
    | optArrayBounds '[' ']'
    { $$ = $1 + 1; }
    ;

literal : INT_LITERAL
    { $$ = new COrdinalLiteral((unsigned long)$1); }
    | STRING_LITERAL
    {
      $$ = new CStringLiteral(*($1));
      delete $1;
    }
    | CHARACTER_LITERAL
    { $$ = new COrdinalLiteral((unsigned short)$1); }
    | LONG_LITERAL
    { $$ = new COrdinalLiteral($1); }
    | FLOAT_LITERAL
    { $$ = new CFloatLiteral($1); }
    | DOUBLE_LITERAL
    { $$ = new CFloatLiteral($1); }
    | TRUE_TOKEN
    { $$ = new COrdinalLiteral(true); }
    | FALSE_TOKEN
    { $$ = new COrdinalLiteral(false); }
    ;

optArgumentList :
    { $$ = 0; }
    | argumentList
    { $$ = $1; }
    ;

argumentList : expression
    {
      $$ = new ExpressionList;
      $$->push_back($1);
    }
    | argumentList ',' expression
    {
      $$ = $1;
      $$->push_back($3);
    }
    ;

type : qualifiedSymbolWithLBracket ']' optArrayBounds
    {
      $$ = new CJavaTypeSignature(*($1), 1 + $3);
      delete $1;
    }
    | qualifiedSymbol
    {
      $$ = new CJavaTypeSignature(*($1));
      delete $1;
    }
    | simpleType optArrayBounds
    {
      $$ = $1;
      $$->SetArrayBounds($2);
      if ($2 > 0 && $$->GetBaseType() == CJavaTypeSignature::Void) {
	yyerror("Invalid use of 'void' type.");
	YYERROR;
      }
    }
    ;

qualifiedSymbolWithLBracket : qualifiedSymbol '['
    { $$ = $1; }
    ;

simpleType : BOOLEAN
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Boolean); }
    | BYTE
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Byte); }
    | CHAR
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Character); }
    | SHORT
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Short); }
    | INT
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Integer); }
    | FLOAT
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Float); }
    | LONG
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::LongInteger); }
    | DOUBLE
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Double); }
    | VOID
    { $$ = new CJavaTypeSignature(CJavaTypeSignature::Void); }
    ;

modifierList :
    { $$ = new CJavaAccessFlags; }
    | modifierList PRIVATE
    {
      $$ = $1;
      if ($$->fPrivate) {
	yyerror("'private' modifier repeated");
	YYERROR;
      }
      $$->fPrivate = 1;
    }
    | modifierList PUBLIC
    {
      $$ = $1;
      if ($$->fPublic) {
	yyerror("'public' modifier repeated");
	YYERROR;
      }
      $$->fPublic = 1;
    }
    | modifierList PROTECTED
    {
      $$ = $1;
      if ($$->fProtected) {
	yyerror("'protected' modifier repeated");
	YYERROR;
      }
      $$->fProtected = 1;
    }
    | modifierList STATIC
    {
      $$ = $1;
      if ($$->fStatic) {
	yyerror("'static' modifier repeated");
	YYERROR;
      }
      $$->fStatic = 1;
    }
    | modifierList TRANSIENT
    {
      $$ = $1;
      if ($$->fTransient) {
	yyerror("'transient' modifier repeated");
	YYERROR;
      }
      $$->fTransient = 1;
    }
    | modifierList VOLATILE
    {
      $$ = $1;
      if ($$->fVolatile) {
	yyerror("'volatile' modifier repeated");
	YYERROR;
      }
      $$->fVolatile = 1;
    }
    | modifierList FINAL
    {
      $$ = $1;
      if ($$->fFinal) {
	yyerror("'final' modifier repeated");
	YYERROR;
      }
      $$->fFinal = 1;
    }
    | modifierList ABSTRACT
    {
      $$ = $1;
      if ($$->fAbstract) {
	yyerror("'abstract' modifier repeated");
	YYERROR;
      }
      $$->fAbstract = 1;
    }
    | modifierList NATIVE
    {
      $$ = $1;
      if ($$->fNative) {
	yyerror("'native' modifier repeated");
	YYERROR;
      }
      $$->fNative = 1;
    }
    | modifierList SYNCHRONIZED
    {
      $$ = $1;
      if ($$->fSynchronized) {
	yyerror("'synchronized' modifier repeated");
	YYERROR;
      }
      $$->fSynchronized = 1;
    }
    ;

qualifiedSymbol : simpleSymbol
    { $$ = $1; }
    | qualifiedSymbol '.' simpleSymbol
    {
      $$ = $1;
      *($$) += kSymbolDivider;
      *($$) += *($3);
      delete $3;
    }
    ;

qualifiedSymbolList : qualifiedSymbol
    { 
      $$ = new deque<unicode_string>;
      $$->push_back(*($1));
      delete $1;
    }
    | qualifiedSymbolList ',' qualifiedSymbol
    {
      $$ = $1;
      $$->push_back(*($3));
      delete $3;
    }
    ;

simpleSymbol : SYMBOL
    { $$ = $1; }
    ;

%%

void
yyerror(const char* errorMessage)
{
  assert(gCurrentCompiler != 0);
  gCurrentCompiler->ParseError(yylineno, errorMessage, yytext);
}

void
InitializeParser(const string& parseString, CCompiler* compilerAlias)
{
  #ifndef NDEBUG
  if (::getenv("YYDEBUG") != 0) {
    yydebug = 1;
  }
  #endif
  gCurrentCompiler = compilerAlias;
  assert(gCurrentCompiler != 0);
  InitializeLexer(parseString);
}

void
FinishParser()
{
  FinishLexer();
  gCurrentCompiler = 0;
  #ifndef NDEBUG
  yydebug = 0;
  #endif
}

//
//  Function name : InterpretSymbolExpression
//  Description : Used to parse a qualified symbol into a base variable and
//    some sequence of field accesses.  So "x" is parsed as the expression
//    representing the variable 'x' but "x.y.z" is parsed as the variable
//    x dereferenced to get field y, dereferenced to get field z.
//
CExpression*
InterpretSymbolExpression(const unicode_string& symbol)
{
  unicode_string::size_type slashPosition = symbol.find(kSymbolDivider);
  unicode_string baseVariable = symbol;
  CExpression* baseExpression = 0; 
  if (slashPosition < symbol.size()) {
    baseVariable.assign(symbol, 0, slashPosition);
  }
  const pair<CJavaFieldSignature, CCompiler::LocalVariableIndex>* match =
    gCurrentCompiler->LookupLocalVariable(baseVariable);
  if (match != 0) {
    baseExpression =
      new CLocalVariableExpression(match->first, match->second);
    if (slashPosition < symbol.size()) {
      do {
	unicode_string::size_type previousPosition = slashPosition + 1;
	slashPosition = symbol.find(kSymbolDivider, previousPosition);
	if (slashPosition >= symbol.size()) {
	  slashPosition == symbol.size();
	}
	unicode_string nextField(symbol, previousPosition,
				 slashPosition - previousPosition);
	baseExpression = new CClassFieldExpression(baseExpression, nextField);
      } while (slashPosition < symbol.size());
    }
  } else {
    baseExpression = new CClassFieldExpression(symbol);
  }
  return baseExpression;
}
