/* The code should compile with either ANSI C or K&R compilers. */

/*
 *      Copyright (c) 1993 by California Institute of Technology.
 *      Written by William Deich.  Not derived from licensed software.

 *    You may distribute under the terms of either the GNU General Public
 *    License or the Artistic License, as specified in the README file.
 */

#include "super.h"
#include "version.h"

/*
 * Super allows users to execute other programs, particularly
 * scripts, as root (or another user/group), without unduly
 * compromising security.
 * 
 * Use:
 * 
 *     $0 [-d | -D] [-h | -? | -H | -V] commandname [args...]
 * or
 *     $0 -c [-d | -D] [superfile]
 * 
 * Options:
 *	-c -- check syntax of super file, but don't execute anything.
 *	-d -- debug mode: show what would be done, but don't actually
 *		execute a command.
 *	-D -- verbose debug mode: same as -d, plus tell more about variables.
 *	-h, -? -- help: print list of allowed commands, then exit.
 *	-H -- a long-winded variant of -h.
 *	-f -- (just the facts, ma'm) a version of -H that prints no extra
 *	      info, just a list of what you can execute, using the format:
 *		Cmd FullPath [initial args]
 *		Cmd FullPath [initial args]
 *		...
 *	      Useful for scripts that want a list of what you may execute.
 *	-V -- print version information, then exit.
 * 
 * The super.tab file names each command that super will execute, and
 * says who can use it.

 * Most lines in the super.tab file are lines like:
 * 
 *    commandname    FullPathName   <option>=... <condition>~pattern
 * or
 *    Cmd1::FullPath1 Cmd2::FullPath2[...]   <option>=... <condition>~pattern

 * The commandname is normally used as just a plain string, but it is actually
 * a pattern that is matched against the user's input command (again, brace
 * expansion is allowed).  Commands beginning with a colon (":") are
 * reserved for use by super.

 * If the commandname is matched, the FullPathName normally specifies the
 * actual command to execute.  In the special case that FullPathName contains
 * an asterisk, the asterisk is replaced by the user's "cmd" string, and
 * this forms the actual command to exec.  This wildcard form allows you to
 * give away access to MANY commands for _TRUSTED_ users -- be very careful
 * about what you give away here.

 * The first form of super.tab entries uses one one to describe one
 * command/file pair.  The second form of the command line allows you
 * to enter a list of commandname/filename pairs on one line, and the
 * ok-user-pats and options are applied to each pair.
 * That is, typing
 *    Cmd1::FullPath1 Cmd2::FullPath2 Cmd3::FullPath3 ok-user-pats | options
 * is equivalent to typing
 *    Cmd1  FullPath1 ok-user-pats | options
 *    Cmd2  FullPath2 ok-user-pats | options
 *    Cmd3  FullPath3 ok-user-pats | options
 
 * The command is accepted for execution if the conditions are met.
 * Conditions are things such as user~will@somehost  or  time~11:00-15:00/tues.
 * The user~ condition is met if the username/group/host matches one or
 * more user~ patterns.  The time~ condition is met if the current time matches
 * one or more time~ patterns.

 * Conditions can be preceded by "!" to negate them, e.g.
 *	!user~will
 * means to disallow a user named "will".  Conditions are evaluated 
 * left-to-right, so that
 *	user~*  !user~will
 * means to allow any user (user~*) except to disallow user will.  If the
 * order of conditions were reversed, 
 *	!user~will  user~*
 * the first condition would have no effect since the second condition
 * allows anybody to use the command.

 * For backwards compatibility, the "user~" can be left out, so that
 * the above could equally be written
 *	* !will

 * Similarly,
 *	time~8:00-17:00  !time~12:00-13:00
 * means to allow execution between 8am and 5pm, except not at 12h - 13h.

 * The rules for time defaults are as follows:
 *	if _ALL_ time patterns are NEGATED, super _PERMITS_ execution
 *				at any time not explicitly listed;
 *	otherwise, there is at least one non-negated pattern, and super DENIES
 *				execution at any time not explicitly listed.

 * The `ok-user-pats' are (by default) ed-style patterns, with the addition of
 * csh-style brace expansion.  The actual regular-expression code is
 * Ozan Yigit's implementation of regex; see regex.c for details.
 * (Alternatively, an option can be set so that shell-style patterns are
 * used; see global option "patterns=xxx", below.)

 * Each ok-user-pattern is in the format
 *	user[:group][@host]   |    :group[@host]   |   @host

 * By default, the "group" field can match either the group name or
 * the user's gid -- this allows the use of group numbers in the
 * /etc/passwd file that aren't in the /etc/group file.  See the
 * #define's of MATCH_DECIMAL_UID and MATCH_DECIMAL_GID for details.

 * The ok-time-patterns require that the current time and day must be in
 * the specified range.
     * dayname			(implies all hours)
     * hh:mm-hh:mm/dayname
     * hh:mm-hh:mm		(implies all days)
     * xhh:mm/dayname		(implies all days; x is one of <, >, <=, >= )
     * xhh:mm			(implies all days; x is one of <, >, <=, >= )
 * For hosts that support setlocale(), the valid daynames are those
 * used in your current locale.  They can be abbreviated to 3 or more
 * characters, or to the official abbreviation for your current locale.


 * Options can be supplied in control lines (local options) or in a
 * special ":global" command (global options).  Local options placed on
 * the control line (along with the user and time conditions), and
 * affect only the commands on that line.  Global options have scope that
 * runs from the line of definition to the end of input; typically,
 * they are set on the first non-comment line of the super.tab file.
 * Each option is of the
 *  form   key=value  (no whitespace around the equal sign is allowed).
 * Global options are set using

 *	 	:global	[globaloptions]
 *	or
 *	 	:global_options	[globaloptions]

 * ( The following is obsolescent, but is accepted for
 * backwards compatibility:
 *	 	/	/	[globaloptions]
 * i.e., commandname="/", FullPathName="/", then options. )

 *  Group 1.  Options affecting how we read the super.tab file.

 *  o	patterns=xxx	Select the pattern-matching type.  xxx must be one of:
 *	GLOBAL
 *		    "shell" -- patterns are as follows (approximately
 *				Bourne-shell style), with the addition
 *				of csh-style brace expansion:
 *				\x	force x to be a plain character;
 *				?	matches any single character;
 *				*	matches any sequence of 0 or more chars;
 *				[x..y]	matches any single character in the set;
 *				[^x..y]	matches any single char NOT in the set;
 *				^xxx	inverts the sense of the match --
 *					the string is `matched' if it doesn't
 *					match pattern xxx.  Note that this is
 *					different from the negation !pat.  For
 *					example, the pattern in `!joe' is just
 *					`joe', and thereby matches only user
 *					joe (who is then disallowed because the
 *					pattern is preceded with `!').
 *					In contrast, the pattern in
 *					`^joe' is all four characters ``^joe'',
 *					which matches any string other than joe.
 *					Example 1:
 *						j*  ^joe
 *					matches all names beginning with
 *					j (`j*'), _and_ all other names except
 *					for joe.
 *					Example 2:
 *						j*  !joe
 *					matches all names beginning with
 *					j (`j*'), and then disallows joe
 *					(`!joe').


 *		    "regex" -- patterns are ed-style, with the addition
 *				of csh-style brace expansion.

 *			DEFAULT: regex.

 *  o	relative_path=y|n	If y, allows the filenames to be relative.
 *	GLOBAL		This is in general a stupid idea.  Don't do it!
 *			Default: n

 *  o	group_slash=y|n	Allow group names to have slashes.  If you
 *	GLOBAL		allow this, you make it harder to catch certain
 *			command-line typos.  Don't do it!
 *			Default: n

 *  Group 2.  Logging options.

 *  o	logfile=xxx	Enable logging information.  Each invocation of super()
	GLOBAL		(aside from ones for help) generates an entry in
			file xxx.

 *  o	loguid=xxx	The logfile is opened for writing using uid=xxx
 *	GLOBAL		(xxx can be either a username or numeric uid).
 *			This option is useful when a network of hosts are
 *			sharing a cross-mounted logfile.  Note that networks
 *			are typically configured to not allow root on one
 *			host to have root access to files on another host,
 *			so this option allows you to have the file
 *			created/opened under another uid that does have
 *			cross-host access, such as the uid of a system manager.
 *			Default: loguid=0.

 *  o	syslog=y|n	If y, logfile messages are logged using syslog().
 *	GLOBAL		(This is done independently of the setting of
 *			logfile=xxx).

 *  o	mail="mail-command"	Notices of super failures are piped to
 *	LOCAL + GLOBAL	the shell command mail-command.  For instance,
 *			mail="mail -s Super joeblow" will cause error
 *			messages to be mailed to user joeblow (on some
 *			systems you may need to substitute "mailx" for "mail").
 *			Note: the mail-command is passed as an argument
 *			to popen(3), and the message piped in.

 *  o	mailany="mail-command"	Notices of successful uses of super as well
 *	LOCAL + GLOBAL	as failures are sent to the mail command.  You should
 *			not combine mail= and mailany=  on one line.

 *  Group 3.  Extra help information for user entering super -h or super -H.

 *  o	info=xxx	This line is printed when help is given.
 *	LOCAL

 *  Group 4.  Passwords + restrictions on environment before executing command.

 *  o	nargs=[mmm-]nnn	In addition to any command-line options supplied
 *	LOCAL		with the command string (e.g. "/Full/Path/Name arg1 "),
 *			the user must supply from mmm to nnn args (if mmm and
 *			nnn are given), else exactly nnn args (if nnn only
 *			is given), else any number of args.

 *  o	argMMM-NNN=SSS	Here MMM and NNN are numbers, and SSS is a pattern
 *	LOCAL		(which must be enclosed in quotes if it contains
 *			whitespace).  This option states that the MMM-NNN'th
 *			user-supplied argument, if given, must match
 *			pattern SSS.  Note that they don't _require_ this
 *			number of arguments.

 *  o	owner=xxx	The owner of the FullPathName must be xxx, or it
 *	LOCAL + GLOBAL	won't be executed.  N.B.  xxx is first tried as a
 *			username, then as a uid.

 *  o   password=y|n	If y, the user's password is required to execute
 *	LOCAL + GLOBAL	the command.  Default=n.

 *  o	timeout=mmm	Passwords are good for mmm minutes; then they
 *	LOCAL + GLOBAL	have to be re-entered to execute another password
 *			command.  <=0 requires password every time.
 *			Default = 5 min.

 *  o	renewtime=y|n	If y, each successfully-executed password-requiring
 *	GLOBAL		command causes the timestamp to be changed to the
 *			currrent time, allowing another timeout minutes of
 *			use before the password need be re-entered.  def=n.

 *  o	timestampbyhost=y|n	If y, timestamp files are unique for each host
 *	GLOBAL		-- they won't be shared by the same username when
 *			one timestamp directory is cross-mounted.  If n, the
 *			cross-mounted timestamp directories use the same
 *			timestamp file for all accounts with the same username.

 *  o	timestampuid=xxx	If given, timestamp files are created using
 *	GLOBAL		uid=xxx (a username or uid).  The default is to
 *			use uid=root.  The motivation is the same as for
 *			the loguid option: to allow a cross-mounted directory
 *			to be used for timestamp files, even though root may
 *			be mapped to nobody for NFS accesses.

 *  Group 5.  Modifications to environment before executing approved command.

 *  o	uid=xxx		By default, FullPathName is executed with effective
 *	LOCAL		uid=root, but no changes to the gid or real uid.
 *			This option sets the real and effective user id to xxx.
 *			N.B.  Uid's and gid's are first tried as names, then
 *			as numbers.

 *  o	gid=xxx		By default, the group id remains unchanged when
 *	LOCAL		FullPathName is executed.  This option sets the real
 *			and effective group id to xxx.

 *  o	u+g=zzz		The real and effective uid are set to zzz and
 *	LOCAL		the real and effective gid is set to zzz's login gid.
 *			(u+g conflicts with uid=xxx or gid=yyy, and can't be
 *			used w/ them).

 *  o	umask=nnn	The umask of the executed command is set to nnn, which
 *	LOCAL + GLOBAL	is either hexadecimal (0xnnn), octal (0nnn) or decimal.

 *  o	nice=nnn	The nice value of the executed command is incremented
 *	LOCAL + GLOBAL	by an amount nnn.

 *  o	env=var[,...]	Super() discards most environment variables and keeps
 *	LOCAL		only a restricted set (see below).  This option adds
 *			the variables var,...  to the kept set.  If used more
 *			than once in a single command line, the later list
 *			replaces the earlier one.  Use at your own risk.

 *  o	setenv=var=xxx	Define environment variable var to have value xxx,
 *	LOCAL		and add it to the environment variables that are
 *			not discarded.  Note that "setenv=DISPLAY" is
 *			syntactically invalid, and that "setenv=DISPLAY="
 *			sets DISPLAY to the null string, so this is probably
 *			not what you want.  Use "env=" to keep a list of
 *			environment variables w/o modifying their definitions.
 *			Can be used several times to add definitions of
 *			various variables.

 *  o	fd=m[,...]	Super() closes most file descriptors and normally keeps
 *	LOCAL		open just fd=0,1,2 before running the FullPathName.
 *			This option adds the descriptors m,...  to the open set.

 *  Group 6.  Other options

 *  o	print=msg	If the line is successfully matched, super will print
 *	LOCAL		the message before executing the command.

 *  o	die=msg		If the line is successfully matched, super will exit
 *	LOCAL		with the given message.

 *  o	user/group/host	These aren't really options at all.  But I list them
 *	GLOBAL		here just for completeness.
 
 *			Ordinary user/group/host patterns can be put in the
 *			global options list.  The list is of the form

 *			  [ before_pat before_pat ... <> ] after_pat after_pat

 *			When matching user/group/host patterns for each
 *			command, all user/group/host patterns up to `<>'
 *			are processed _before_ all of the patterns on the
 *			per-command line; and all the rest of the patterns
 *			are processed _after_ all of the patterns on the
 *			per-command line.

 * Variables:
 *	You can define variables that are substituted into command lines
 *	before _any_ other parsing is done.  This is useful, for example,
 *	in defining lists of users who should all be treated alike.

 *	The special command name / FullPathName
 *		:define VariableName xxxx
 *	defines a variable.  The VariableName must be made up only of letters,
 *	digits, and/or underscore.

 *	The text beginning at the first non-whitespace character after
 *	the VariableName and continuing up to but not including the
 *	final newline becomes the variable definition.  In a
 *	definition continued across multiple lines, the entire
 *	backslash-newline-leading_whitespace sequence is treated as
 *	whitespace if it follows a letter, digit, or underscore; otherwise,
 *	it is deleted from the definition.  This is the same as control
 *	lines.

 *	Unlike Makefiles, the variables are not scanned first and then the
 *	file re-scanned.  Instead, variables take effect at the point they
 *	are defined, and remain in effect until the end of file.

 *	Variables may contain other variables (which must have already been
 *	defined).   Variable replacement is done when a line is first read.
 *	A line is never re-scanned after variable replacement.

 *	Variables are used in a file by typing
 *		$VarName
 *	or
 *		$(VarName)
 *	The second form allows you to concatenate strings, e.g. $(xyz)abc.

 *	The special variable $$ is replaced by a single $.  Any other name $X,
 *	where X is not a letter, underscore, or left parenthesis, is a
 *	syntax error.  (It is legal, using regex expressions, to use "$"
 *	as a end-of-string anchor.  But all regular expressions in
 *	super are automatically anchored anyway, so we make the
 *	user be explicit about the $ (by using $$), or preferably get rid
 *	of it entirely.)

 *	Since variable expansion happens _first_, the text
 *		:define xyz  this and that
 *		\$xyz
 *		\$$xyz
 *	expands to
 *		\this and that
 *		\$xyz
 *	The point here is that the backslash has no special status
 *	during variable expansion.

 *	Because a line is never re-scanned after variable replacement,
 *	the following sequence:
 *		:define A $$
 *		:define B A
 *		:define C $B $$B
 *		$C
 *	defines B to be "A", then defines C to be "A $B";
 *	thus the final output is simply
 *		A $B
 *		

 * If a user is allowed to execute a given <commandname>, the <fullpathname>
 * is exec'd, with <commandname> as argv[0].
 *
 * If a symlink is made from, say, mycmd to super, then executing
 *	mycmd args...
 * or
 *	/usr/local/bin/superlinks/mycmd args...
 * is equivalent to executing
 *	super mycmd args...

 * Notes:
 *	1) when interpreting $argv[0] as the cmd, the leading directory parts
 *		are stripped off.
 *	2) Links mustn't be named "super" -- the super program won't recognize
 *		it's a link.
 * 
 * For security, the environment variables are discarded, save for TERM,
 * LINES COLUMNS, the env=... variables, and the setenv= variables.  If TERM
 * contains any characters other than [a-z][A-Z][0-9]_+.:/-, it is
 * discarded.  If LINES or COLUMNS contains any characters other than
 * [0-9], they are discarded.  To these are added reasonable values for
 * IFS, PATH, USER, HOME, and (USER and HOME are set to the username and
 * login directory, respectively, of the real uid when the command is
 * run).  LOGNAME is set to the same as USER.  ORIG_USER, ORIG_LOGNAME,
 * and ORIG_HOME are set to the username and login directory of the
 * person invoking super().  SUPERCMD is set to the <commandname>.  All
 * descriptors excepting 0,1,2 and the fd=...  set are closed.  Signals
 * are all reset to have default handling.

 * If the super.tab file isn't owned by root, or if the super.tab file
 * is group- or world-writeable, won't run setuid-root.
 * (If caller is root, won't run at all.  Otherwise, euid reverts to real uid).

 */

char *superfile = SUPERFILE;	/* The super.tab file */

/* Global info */
GlobalInfo gi = {
	"",		/* Required file owner */
	0,		/* Allow relative paths */
	0,		/* Allow group names to have slashes */
	0,		/* The nice(3) increment */
	0,		/* umask setting */

	 {		/* Global password requirements */
		0,	/* not required */
		5,	/* 5-minute expiration */
		0,	/* renew timestamp with ea. command use? */
		1,	/* create timestamp files separately for each host? */
		""},	/* value of timestampuid=xxx, if entered */


	{0,NULL,NULL},	/* userbefore: list of u/g/h pats before per-cmd pats */
    	{0,NULL,NULL},	/* userafter: list of u/g/h pats after per-cmd pats */
    	{0,NULL},	/* b_a_text: list of original text for above */
	1,		/* user_clear: clear if new val seen */

	{{0,0,0,0},	/* timebefore: permitted times before per-cmd pats */
		NULL},
	{{0,0,0,0},	/* timeafter: permitted times after per-cmd pats */
		NULL},
	1,		/* time_clear: clear if new val seen */

	0,		/* use_after: set to !0 when we see <> */

	{ NULL,		/* log: FILE *fp */
	    "",		/* filename */
	    "",		/* value of loguid=xxx */
	    0 },	/* UID under which we open logfile */

	"",		/* mailcmd */
	0,		/* mail_success */

	};

/* The list of currently open files */
FileList *currfile = NULL;

/* The struct of what things we've matched */
Conditions matches;

/* The li struct is filled in a bit at a time until it completely
 * describes the caller, the program to invoke, any options
 * set in the control line, etc.
 */
UserInfo ui;

LocalInfo li = { {NULL, -1, 0, "", 0, 0},	/* Initialize ProgMatch empty */
						/* Other elements that need
						 * start initializers are done
						 * through explict assignment.
						 */
		};

extern char *re_comp __P(( char *));	/* regular-expression compiler */
int shell_compare __P(( char * ));	/* re_comp()-style iface to wildmat() */
extern int re_exec __P(( char * ));	/* regular-expression comparison */
char *shell_compile __P(( char * ));	/* re_exec()-style iface to wildmat() */


char *prog;				/* this program */
int debug=0;				/* Set by the debug options flag */
int check_syntax=0;			/* Set by the -c options flag */

/* Routines used to compile/match user/group/host patterns */
char *(*pat_compile) __P((char *)) = re_comp;
int (*pat_compare) __P((char *)) = re_exec;
int need_re_anchor = 1;


/* For strqtokS */
unsigned char my_qm[256];		/* our quotemarks */
unsigned char my_cc[256];		/* our comment characters */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
main(argc, argv)
int argc;
char **argv;
{
    int i, status, n_builtin;
    char *s, *cmd, *path;
    char **arglist, **envp;
    extern char *error_prog;
    int iarg, givehelp=0, giveversion=0, verbosity=0;

    s = strrchr(argv[0], '/');
    prog = (s && *(s+1)) ? s+1 : argv[0];
    error_prog = ONETRUENAME;	/* same as prog, but used by Error() */
#ifdef HAVE_LOCALE
    setlocale(LC_ALL, "");
#endif
    error_srcfile = superfile;	/* error messages w/ line nums refer to this */
    error_line = -1;
    if (add_variable("$", "$") == -1)	/* Built-in variable $$ -> "$" */
	return 1;

    li.info = li.die = li.print = li.env = NULL;
    for (i=0; i<=MAXSETENV; i++)
	li.setenv[i]=NULL;
    li.fdlist = NULL;
    StrInit(&li.argpats);
    ui.ourtime.start = time(NULL);
#ifdef HAVE_LOCALTIME
    {
	struct tm *tm_p = localtime(&ui.ourtime.start);
	ui.ourtime.min = tm_p->tm_hour*60 + tm_p->tm_min;
	ui.ourtime.day = tm_p->tm_wday;
    }
#else
    ui.ourtime.min = (ui.ourtime.start/60) % (24*60);
    ui.ourtime.day = daynum(ui.ourtime.start);
#endif
    init_umask(0);		/* initialize local umask option setting */
    init_umask(1);		/* initialize global umask option setting */
    init_nice_incr(0);		/* initialize local nice_incr setting */
    init_nice_incr(1);		/* initialize global nice_incr setting */

    /*
     * We want the hostname (fully-qualified if possible), as well as a
     * lower-cased version (to try and deal with mixed case hostnames).
     */
    if (get_canonical_hostname(ui.hostname, sizeof(ui.hostname)) == -1)
	return 1;
    strcpy(ui.lc_hostname, ui.hostname);
    strtolower(ui.lc_hostname);

    /* Decide if we were invoked as "super cmd args...", or 
     * as a link: cmd args...
     */
    if (strcmp(prog, ONETRUENAME) == 0) {
	/* Invoked as super [options] cmd args... */
	iarg = do_options(argc, argv, &givehelp, &giveversion, &verbosity);
	if (iarg < 0 || (argv[0] == NULL && !(givehelp || giveversion))) {
	    /* User screwed up; give help */
	    givehelp = 1;
	    cmd = NULL;
	} else {
	    cmd = argv[iarg];
	    argv += iarg;
	}
    } else {
	/* It's been invoked via link to super.  Therefore any options
	 * go to the command, not to super.
	 */
	s = strrchr(argv[0], '/');
	cmd = (s && *(s+1)) ? s+1 : argv[0];
    }

    /* Get some info right away about user that we want right away, so that
     * we give more useful error and debugging informations.
     */
    if (basic_userinfo() == -1)
	return 1;

    if (debug)
	debug_hello();

    init_strqtokS();

    if (check_syntax)
	Error(0,0, "Checking syntax of superfile `%s'\n", superfile);

    /* Check for permission to execute, and change uid/gid as necessary */
    if ((path = approve(cmd, givehelp, giveversion, verbosity)) == NULL)
	return 1;
    else if (*path == '\0')
	return 0;

    /* Sanity check */
    if (check_syntax)
	Error(0, 2,
	    "Abort: shouldn't ever get here when check_syntax is set!\n");

    /* Check that the file to execute has proper ownership */
    if (check_owner(path) != 0)
	return 1;

    /* Get the arglist for the command, and null-terminate cmd if not
     * already done so.
     */
    arglist = newargs(path, argv, &n_builtin);

    /* Button up for security, and get a modified environment */
    envp = buttonup(cmd);
    if (!envp)
	return 1;

    /* Set uid/gid if necessary */
    status = set_u_g();

    /* Set the umask value */
    set_umask();

    if (debug) {
	debug_print(path, arglist, n_builtin);
	return 0;
    }
    if (status == -1)
	return 1;

    /* Check password requirements */
    if (check_pass(cmd) != 0)
	return 1;

    /* Log an informational message at LOG_INFO priority, not at
     * the usual error priority.
     */
    {
#ifdef USE_SYSLOG
	extern int error_priority;
	int old_pri = error_priority;
	error_priority = LOG_INFO;
#endif
	logmsg(cmd, arglist);
#ifdef USE_SYSLOG
	error_priority = old_pri;
#endif
    }
    /* Close the logfile writer, if any: we are done logging, and going
     * to exec the prog.
     */
    close_writer();

    if (set_nice_incr() == -1)
	return 1;

    /* If print option is set, write message before executing command */
    if (li.print)
	puts(li.print);

    if (execve(path, arglist, envp) == -1) {
#ifdef INTERPRETER_HACK
	if (errno == ENOEXEC) {
	    /* Open the file, check for "#!interpreter [argument]" */
	    FILE *fp;
	    char *interp, *argument, line1[1024];

	    if ((fp = fopen(path, "r")) &&		/* open the file */
		fgets(line1, sizeof(line1), fp) &&	/* read first line */
		(strlen(line1) < sizeof(line1)-1) &&	/* not too long? */
		(strncmp(line1, "#!", 2) == 0) &&	/* begins "#!"? */
		(interp = strtok(line1+2, " \t\n"))) {	/* has interpreter? */

		argument = strtok(NULL, " \t\n");	/* get opt argument */

		/* Adjust the arglist -- recall it has room for this */
		if (argument) {
		    arglist -= 2;
		    arglist[0] = arglist[2];
		    arglist[1] = argument;
		    arglist[2] = path;
		} else {
		    arglist--;
		    arglist[0] = arglist[1];
		    arglist[1] = path;
		}
		(void) execve(interp, arglist, envp);
	    }
	}
#endif
	/* If here, we failed to exec the prog.  Re-open the logfile we
	 * closed above and write a message.
	 */
	if (*gi.log.filename != '\0')
	    opensuperlog();
	(void) Error(1,1, "command `%-.500s': Couldn't exec `%s': ", cmd, path);
    }
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Set options; return index of arg that follows options, or -1 on error */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
do_options(argc, argv, givehelp, giveversion, verbosity)
int argc;
char **argv;
int *givehelp, *giveversion, *verbosity;
{
    int iarg;
    char *s;

    debug = check_syntax = *giveversion = *givehelp = 0;
    *verbosity = HELP_BASIC;	/* only matters if *givehelp != 0 */

    for (iarg = 1; iarg < argc && argv[iarg][0] == '-'; iarg++) {
	for (s=&argv[iarg][1]; *s; s++) {
	    switch (*s) {
	    case 'c': check_syntax = 1;
		    /* Check for optional superfile */
		    if (*(s+1)) {
			/* User gave -cfile */
			superfile = s+1;
			s += strlen(s) - 1;
		    } else if ((iarg+1 < argc) && argv[iarg+1][0] != '-') {
			/* User gave -c file */
			iarg++;
			superfile = argv[iarg];
		    }
		    break;
	    case 'd': debug = 1; break;
	    case 'D': debug = 2; break;
	    case 'V': *giveversion = 1; break;
	    case 'h': case '?': *givehelp = 1; break;
	    case 'f': *givehelp = 1; *verbosity = HELP_FACTS; break;
	    case 'H': *givehelp = 1;
			if (*verbosity != HELP_FACTS)
			    *verbosity = HELP_FULL;
		    break;
	    default:
		    return Error(0, 0, "Unrecognized option `%c'\n", *s);
	    }
	}
    }
    /* Default operation: give help if no args */
    if (argc == 1)
	*givehelp = 1;
    return iarg;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Print the debug startup lines */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
debug_hello()
{
    char *s = dayname(ui.ourtime.day);
    fprintf(stderr, "%s -%c:\n", prog, (debug < 2) ? 'd' : 'D');
    fprintf(stderr, "\tYou are: user=%s gid=%d hostname=%s\n\n",
			    ui.caller, ui.orig_gid, ui.hostname);

    fprintf(stderr, "\tStart time=%d:%d/%s (hr*60+min=%d daycode=%d)\n",
	ui.ourtime.min/60, ui.ourtime.min%60, s,
	ui.ourtime.min, ui.ourtime.day);
    fprintf(stderr, "\tUSE_NETGROUP is %sdefined\n",
#ifdef USE_NETGROUP
		    "");
#else
		    "not ");
#endif

    fprintf(stderr, "\tUSE_SYSLOG is %sdefined\n",
#ifdef USE_SYSLOG
		    "");
#else
		    "not ");
#endif

    fprintf(stderr, "\tUSE_GETHOSTBYNAME is %sdefined\n",
#ifdef USE_GETHOSTBYNAME
		    "");
#else
		    "not ");
#endif

    fprintf(stderr, "\tRLOG_MACHINE is %s\n",
#ifdef RLOG_MACHINE
		    RLOG_MACHINE);
#else
		    "not defined");
#endif

    fprintf(stderr, "\tHAVE_LOCALE is %sdefined\n",
#ifdef HAVE_LOCALE
		    "");
#else
		    "not ");
#endif

    fprintf(stderr, "\tHAVE_LOCALTIME is %sdefined\n",
#ifdef HAVE_LOCALTIME
		    "");
#else
		    "not ");
#endif
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Print the debug info */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
debug_print(path, arglist, n_builtin)
char *path;
char **arglist;
int n_builtin;
{
    char *s, **p, **sp;
    int isglobal, j, iarg;

    fprintf(stderr,
	"\n\tPermitted times for execution (in reverse input order):\n");
    if (gi.timebefore.next != 0 || li.time.next != 0
					    || gi.timeafter.next != 0) {
	for (j = 0; j<3; j++) {
	    TimeList *tl;
	    switch (j) {
	    case 0: isglobal=1; tl = gi.timeafter.next; break;
	    case 1: isglobal=0; tl = li.time.next; break;
	    default: isglobal=1; tl = gi.timebefore.next; break;
	    }
	    for ( ; tl ; tl=tl->next) {
		fprintf(stderr, "\t\t%stime~%d:%02d-%d:%02d/%s (%s)\n",
		    tl->te.invert ? "!" : "",
		    tl->te.begin / 60, tl->te.begin % 60,
		    tl->te.end / 60, tl->te.end % 60, dayname(tl->te.day),
		    isglobal ? "global def" : "per-cmd def");
	    }
	}
    } else {
	    fputs(" (unrestricted)\n", stderr);
    }
    fputs("\n", stderr);

    (void) fprintf(stderr, "\tCommand: <%s>\n", arglist[0]);
    (void) fprintf(stderr, "\tPath: <%s>\n", path);
    for (sp=arglist; *sp; sp++)
	;
    (void) fprintf(stderr, "\tArgc: %d\n", sp - arglist);
    for (sp=arglist; *sp; sp++) {
	iarg = sp - arglist;
	(void) fprintf(stderr, "\tArgv[%d]:  <%s>\n", iarg, *sp);
	if (iarg > n_builtin) {
	    s = StrEltGetPtr(&li.argpats, iarg-n_builtin);
	    if (s)
		fprintf(stderr, "\t\tMust match pattern: %s\n", s);
	}
    }

    if (li.usr_args[0] < 0) {
	(void) fprintf(stderr,
			    "\tAny number of user-entered args allowed.\n");
    } else if (li.usr_args[0] == li.usr_args[1] && li.usr_args[0] == 0) {
	(void) fprintf(stderr, "\tNo user-entered args are allowed.\n");
    } else if (li.usr_args[0] == li.usr_args[1]) {
	(void) fprintf(stderr,
		    "\t%d user-entered arg%s required.\n",
		    li.usr_args[0],
		    li.usr_args[0] == 1? " is" : "s are");
    } else {
	(void) fprintf(stderr,
		    "\t%d - %d user-entered args are required.\n",
		    li.usr_args[0], li.usr_args[1]);
    }

    (void) fprintf(stderr, "\tCommand executes with nice increment = %d.\n",
		rcl_nice_incr());

    (void) fprintf(stderr, "\tCommand executes with umask set to 0%o.\n",
		rcl_umask());

    (void) fprintf(stderr, "\tAdditional user's environment variables:\n");
	(void) fprintf(stderr, "\t\t%s\n",
		(li.env == NULL || *li.env == '\0') ? "(none)" : li.env);

    (void) fprintf(stderr,
			"\tEnvironment variables defined with setenv=var=:\n");
    if (li.setenv[0] == NULL) {
	(void) printf("\t\t(none)\n");
    } else {
	for (p=li.setenv; *p; p++)
	    (void) printf("\t\t%s\n", *p);
    }
    (void) fprintf(stderr,
		    "\tFile descriptors not to be closed:\n\t\t0,1,2");
    if (li.fdlist)
	(void) fprintf(stderr, ",%s", li.fdlist);
    (void) fprintf(stderr, "\n\n\tID's:\treal effective\n");
    (void) fprintf(stderr, "\tuid\t%d\t%d\n", getuid(), geteuid());
    (void) fprintf(stderr, "\tgid\t%d\t%d\n", getgid(), getegid());
    (void) fprintf(stderr, "\n\tPassword required: %s\n",
			    li.passinfo.required ? "yes" : "no");
    if (li.passinfo.required) {
	(void) fprintf(stderr, "\tPassword timeout: %d min\n",
						    li.passinfo.timeout);
	(void) fprintf(stderr,
    "\tUpdate timestamp with each use of a password-requiring command? %s\n",
				    li.passinfo.renewtime ? "Yes" : "No");
    }

}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Log a super call -- If "args" isn't a null ptr, it's printed inside
 *		parentheses, with whitespace separating the arguments.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

void
logmsg(cmd, args)
char *cmd;
char **args;
{
    char *ec, *logbuf, **ap;
    int e;
    int loglen = strlen(cmd) + 4;

    /* Determine buffer length needed to hold all arguments */
    if (args)
	for (ap = args; *ap; ap++)
	    loglen += strlen(*ap) + 1;

    if (!(logbuf = malloc(loglen)))
	(void) Error(0, 2, "failed to malloc space for logging command\n");

    if (args) {
	sprintf(logbuf, "%s (", cmd);
	for (ap = args; *ap; ) {
	    strcat(logbuf, *ap++);
	    strcat(logbuf, " ");
	}
	logbuf[loglen-3] = ')';
	logbuf[loglen-2] = '\n';
	logbuf[loglen-1] = '\0';
    } else {
	strcpy(logbuf, cmd);
    }

    /* Log the message using Error(), but
     *	- make sure msg doesn't go to stderr;
     *	- if not mail_success, don't let msg go to error_command, either.
     */
    e = error_stderr;
    ec = error_command;
    if (li.mail_success == 0 || (li.mail_success == -1 && gi.mail_success==0))
	error_command = NULL;
    error_stderr = 0;
    Error(0, 0, logbuf);
    error_stderr = e;
    error_command = ec;
} 

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Get the arglist for the command, and null-terminate cmd if nec */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char **
newargs(path_plus, argv, n_builtin)
char *path_plus;	/* string = "path [options]".  Null-terminate and put
			 * options into front of arglist. */
char **argv;		/* options to add to arglist */
int *n_builtin;		/* returned w/ number of args in path_plus */
{
    int nuser, nalloc, iarg, nargs, nxtra;
    char **arglist, **ap;
    char *s;

    /* Count user-entered args. */
    for (ap = argv; *ap; )
	ap++;

    /* Check number of user-entered args is ok. */
    nargs = ap - argv - 1;
    if (li.usr_args[0] >= 0) {
	if (nargs < li.usr_args[0] || nargs > li.usr_args[1]) {
	    if (li.usr_args[0] == li.usr_args[1] && li.usr_args[0] == 0)
		    (void) Error(0, 2,
			"you may not give any arguments to `%-.500s'\n",
			argv[0]);
	    else if (li.usr_args[0] == li.usr_args[1])
		(void) Error(0, 2,
		    "You must give %d argument%s to `%-.500s'\n",
		    li.usr_args[0], li.usr_args[0] == 1 ? "" : "s", argv[0]);
	    else
		(void) Error(0, 2,
		    "You must give %d - %d arguments to `%-.500s'\n",
			li.usr_args[0], li.usr_args[1], argv[0]);
	}
    }

    /* Check that each user-entered argument matches its pattern, if given */
    for (iarg = 1; iarg <= nargs; iarg++) {
	s = StrEltGetPtr(&li.argpats, iarg);
	if (s && match_pattern(0, argv[iarg], s) != 1)
	    (void) Error(0, 2,
		"Your argument #%d <%s> must match pattern <%s>\n",
		iarg, argv[iarg], s);
    }

    /* Start off with space for user-entered args + 100 args in super.tab.
     * We'll re-alloc if necessary.
     */
    nuser = (ap - argv) + 3;
    nalloc = nuser + 100;
    arglist = (char **) malloc(sizeof(char *) * nalloc);
    if (!arglist)
	(void) Error(1, 2, 
	    "failed to malloc space for %d ptrs\n", nalloc);

    /* Leave room for two extra args at front, in case we are handling
     * the "#! interpreter [opt]" file for OS's that don't support it.
     */
    arglist += 2;

    /* First copy the command to the new arglist */
    arglist[0] = *argv++;

    /* Then copy the extra args from super.tab to the arglist,
     * re-allocing the arglist as the number of args grows.
     */
    s=strqtokS(path_plus, SEP, QM, "", 1); /* Don't put FullPath into arglist */
    for(nxtra=0, ap=&arglist[1], s=strqtokS(NULL, SEP, NULL, NULL, 1);  s;
				s = strqtokS(NULL, SEP, NULL, NULL, 1)) {
	nxtra++;
	if (nuser + nxtra >= nalloc) {
	    char **newarglist;
	    nalloc *= 2;
	    newarglist = (char **) realloc((void *) arglist, nalloc);
	    if (!newarglist)
		(void) Error(1, 2, 
		    "failed to realloc space for %d ptrs\n", nalloc);
	    ap = newarglist + (ap - arglist);
	    arglist = newarglist;
	}
	*ap++ = s;
    }

    /* Now add the user-supplied args at the end */
    *n_builtin = ap - arglist - 1;
    while (*argv)
	*ap++ = *argv++;
    *ap = NULL;

    return arglist;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Get a safe environment for execution of the command */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
char **
buttonup(cmd)
char *cmd;		/* name of command being started */
{
    /* Closes all descriptors save 0,1,2 and the super.tab-specified fd list.
     * Resets all signal-handling to SIG_DFL.
     * Discards all env. variables save for TERM, LINES, COLUMNS, and
     * any variables listed in the super.tab file.
     * Don't allow TERM to have any but [-/:+._a-zA-Z0-9].
     * Don't allow LINES, COLUMNS to have anything but digits.
     * To these are added reasonable values for IFS, PATH, USER, HOME.
     * USER and HOME refer to the uid under which the command is executed;
     * LOGNAME is set to the same as USER, and SUPERCMD is set to cmd.
     * ORIG_USER, ORIG_LOGNAME, and ORIG_HOME refer to the user who invoked
     * super.
     * Returned:
     *  NULL on error;
     *	otherwise, a pointer to the modified environment list.
     */
    int i, fd, maxfd;
    char **p, *s;

    int fd_log;
    static char *env[200];
    static char User[100];		/* USER */
    static char Logname[100];		/* LOGNAME (alias for USER) */
    static char Home[MAXPATHLEN+5];	/* HOME */
    static char OrigUser[100];		/* ORIG_USER */
    static char OrigLogname[100];	/* ORIG_LOGNAME */
    static char OrigHome[MAXPATHLEN+9];	/* ORIG_HOME */
    static char Cmd[1200];		/* SUPERCMD */
    void (*signal())();

    fd_log = gi.log.fp ? fileno(gi.log.fp) : -1; /* don't close logfile yet */
    maxfd = MAXFD;

    for (fd=3; fd <= maxfd; fd++)
	if (li.fd[fd] == 0 && fd != fd_log)
	    (void) close(fd);
    
    for (i=0; i<NSIG; i++)
       (void) signal(i, SIG_DFL);

    s = *li.user ? li.user : *li.u_g ? li.u_g : ui.caller;
    (void) sprintf(OrigUser, "ORIG_USER=%s", ui.caller);
    (void) sprintf(User, "USER=%s", s);
    (void) sprintf(OrigLogname, "ORIG_LOGNAME=%s", ui.caller);
    (void) sprintf(Logname, "LOGNAME=%s", s);
    (void) sprintf(Cmd, "SUPERCMD=%s", cmd);
    (void) strcpy(Home, "HOME=");
    (void) getlogdir(s, Home+5);
    (void) strcpy(OrigHome, "ORIG_HOME=");
    (void) getlogdir(ui.caller, OrigHome+10);
    i = 0;
    env[i] = Getenv("TERM");
    if (env[i] && checkenv("TERM", env[i]+5, "^[-/:+._a-zA-Z0-9]*$") != -1) i++;
    env[i] = Getenv("LINES");
    if (env[i] && checkenv("LINES", env[i]+6, "^[0-9]*$") != -1) i++;
    env[i] = Getenv("COLUMNS");
    if (env[i] && checkenv("COLUMNS", env[i]+8, "^[0-9]*$") != -1) i++;
    env[i++] = SAFE_IFS;
    env[i++] = SAFE_PATH;
    env[i++] = User;
    env[i++] = Logname;
    env[i++] = Home;
    env[i++] = Cmd;
    env[i++] = OrigUser;
    env[i++] = OrigLogname;
    env[i++] = OrigHome;

    /* Now add the extra environment variables requested in the
     * super.tab file.
     */
    for (s=strtok(li.env, ","); i < NELEM(env)-1 && s; s=strtok(NULL, ",")) {
	env[i] = Getenv(s);
	if (env[i])
	    i++;
    }

    for (p = li.setenv; *p && i < NELEM(env)-1 ; )
	env[i++] = *p++;

    if (i >= NELEM(env)-1) {
	Error(0, 0, "%t\n\tAsked to save too many \
environment variables (max allowed %d).\n", NELEM(env)-1);
	return NULL;
    }

    env[i] = (char *) NULL;

    return &env[0];
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Look up some of the most basic user information. */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int
basic_userinfo()
{
    struct passwd *usrpw;
    ui.orig_uid = getuid();
    ui.orig_gid = getgid();
    usrpw = getpwuid(ui.orig_uid);
    if (!usrpw)
	return Error(0, 0, "Couldn't get your password entry: ");
    (void) strcpy(ui.caller, usrpw->pw_name);
    error_user = ui.caller;
    return 0;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Store the desired umask; set the actual umask; recall the desired umask */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
init_umask(is_global)
int is_global;
{
    if (is_global) {
	/* Get current value (but this clears it) */
	ui.orig_mask = umask(0);
	gi.mask = ui.orig_mask;

	/* ...so restore curr val */
	(void) umask(ui.orig_mask);
    } else {
	li.mask = -1;
    }
}

void
store_umask(mask, is_global)
int mask;
int is_global;
{
    if (is_global)
	gi.mask = mask;
    else
	li.mask = mask;
}

void
set_umask()
{
    umask( (li.mask >= 0) ? li.mask : gi.mask);
}

int
rcl_umask()
{
    return (li.mask >= 0) ? li.mask : gi.mask;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Init/store/set/recall niceness */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#define NICE_NOTSET 999999
void
init_nice_incr(is_global)
int is_global;
{
    if (is_global) {
	gi.nice_incr = 0;
    } else {
	li.nice_incr = NICE_NOTSET;
    }
}

void
store_nice_incr(nice_incr, is_global)
int nice_incr;
int is_global;
{
    if (is_global)
	gi.nice_incr = nice_incr;
    else
	li.nice_incr = nice_incr;
}

int
set_nice_incr()
{
    int nice_incr = li.nice_incr == NICE_NOTSET ? gi.nice_incr : li.nice_incr;
    if (nice_incr && nice(nice_incr) == -1)
	return Error(1, 0,
		"Failed to apply a ``nice'' increment = %d: ", nice_incr);
    return 0;
}

int
rcl_nice_incr()
{
    return li.nice_incr == NICE_NOTSET ? gi.nice_incr : li.nice_incr;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Frees all elements in a SimpleList, except the one it's given.
 * The "next" field of that element is set NULL.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
free_SimpleList(sl)
SimpleList *sl;
{
    SimpleList *slp;

    if (!sl || !sl->next)
	return;
    slp = sl->next;
    sl->next = NULL;
    for (sl=sl->next ; sl; sl = slp) {
	slp = sl->next;
	free(sl->pat);
	free(sl);
    }
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Frees all elements in a Simple2List, except the one it's given.
 * The "next" field of that element is set NULL.
 */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
void
free_Simple2List(sl)
Simple2List *sl;
{
    Simple2List *slp;

    if (!sl || !sl->next)
	return;
    slp = sl->next;
    sl->next = NULL;
    for (sl=sl->next ; sl; sl = slp) {
	slp = sl->next;
	free(sl->pat);
	free(sl);
    }
}
