/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPSession.cpp

	Contains:	Implemenation of RTSPSession objects
	
	$Log: RTSPSession.cpp,v $
	Revision 1.2  1999/02/19 23:08:37  ds
	Created
	

*/

#include "RTSPSession.h"
#include "RTSPRequest.h"
#include "RTSPServerInterface.h"
#include "MyAssert.h"
#include "OS.h"

#define RTSPCONNECTION_DEBUG_LOGGING 0


RTSPSession::RTSPSession()
: fStream(&fSocket),
  fTimeoutTask(this),
  fDebugFile(NULL)
{
	RTSPServerInterface::IncrementCurrentSessions();
	//Use the "real" timeout, not the timeout that gets advertised to clients
	fTimeoutInMilSecs = RTSPServerInterface::GetRTSPPrefs()->GetRealRTSPTimeoutInSecs() * 1000;
	fTimeoutTask.SetTimeoutMilSecs(fTimeoutInMilSecs);
}

RTSPSession::~RTSPSession()
{
	RTSPServerInterface::DecrementCurrentSessions();
	RTSPModuleInterface::SessionClosing(this);
}

SInt64 RTSPSession::Run()
{
	EventFlags events = this->GetEvents();
	
	if (fDebugFile != NULL)
		fprintf(fDebugFile, "Time: %qd. RTSPSession::Run: Received %d events\n",OS::Milliseconds(), events);

	//check for a timeout. If so, just consider the session dead
	if (events & Task::kTimeoutEvent)
		fLiveSession = false;
		
	while (IsLiveSession())
	{
		AssertV((events & Task::kReadEvent) || (events & Task::kWriteEvent), events);
		QTSS_ErrorCode err = QTSS_NoErr;
		
		//read some data off the socket
		if ((err = fStream.ReadRequest(fDebugFile)) == QTSS_NoErr)
			return 0;//wait for another S_DATA event
		if ((err != QTSS_RequestArrived) && (err != QTSS_NotEnoughSpace))
		{
			//Any other error implies that the client has gone away
			Assert(!this->IsLiveSession());
			break;
		}

		//ok, session is active for some reason. Make sure that the timeout mechanism knows this
		fTimeoutTask.SetTimeoutMilSecs(0);

		//Create a new request object, and parse the incoming request
		RTSPRequest* theRequest = new (fRequestMem) RTSPRequest(&fStream, this);
		Assert(theRequest != NULL);

		//Invoke filter plug-ins at this point.
		RTSPModuleInterface::FilterRequest(theRequest);
	
		RTSPProtocol::RTSPStatusCode resultCode = RTSPProtocol::kSuccessOK;
		
		//one of the filter plug-ins may have sent a response to the client. If that's
		//the case, then we don't need to even process the request. Otherwise, go ahead
		//and process it now
		if (!theRequest->HasResponseBeenSent())
		{
			//If we've run out of space in the input request buffer, just ignore the
			//data, send the proper error response, and close the connection
			if (err == QTSS_NotEnoughSpace)
				resultCode = theRequest->SendErrorResponse(RTSPProtocol::kClientBadRequest,
															RTSPMessages::kRequestTooLong);
			else
			{
				//otherwise, go ahead and parse this RTSP request and process it
				Assert(err == QTSS_RequestArrived);
				resultCode = theRequest->Parse(fStream.GetRequestBuffer());
			}
			
			//if the resultCode is kSuccessOK, there was no syntax error in the request,
			//so go on and process it normally. Only log requests that have no
			//syntax errors, so mark that with the isLoggable variable
			if (resultCode == RTSPProtocol::kSuccessOK)
				resultCode = RTSPModuleInterface::ProcessRequest(theRequest);

			//if (resultCode != RTSPProtocol::kSuccessOK)
			//{
			//	resultCode->ProcessRequest(theRequest);
			//	delete resultCode;
			//}
#if RTSPCONNECTION_DEBUG_LOGGING
			static StrPtrLen sRtspam("QTS/1.0");
			if (theRequest->GetHeaderValue(RTSPProtocol::kUserAgentHeader)->Equal(sRtspam))
			{
				int blah = ::rand();
				char yaya[50];
				sprintf(yaya, "debug.%s.%d",this->GetSocket()->GetRemoteAddrStr()->Ptr, blah);
				if (fDebugFile == NULL)
					fDebugFile = fopen(yaya, "w");
			}
#endif
			if (fDebugFile != NULL)
				fprintf(fDebugFile, "Time: %qd. RTSPSession: Finished processing %ld\n",OS::Milliseconds(), theRequest->GetMethod());
		}
		
		//If whatever processed this request hasn't done this yet, make sure it gets done.
		(void)theRequest->Close();

		//if this is not a keepalive request, we should kill the session NOW
		fLiveSession = theRequest->GetResponseKeepAlive();

		//if we are timing this session out, reset the timeout
		if (fTimeoutEnabled)
			fTimeoutTask.SetTimeoutMilSecs(fTimeoutInMilSecs);
	}
	if (fDebugFile != NULL)
	{
		fprintf(fDebugFile, "Time: %qd. RTSPSession: DELETING THIS SESSION\n", OS::Milliseconds());
		fclose(fDebugFile);
	}
	return -1;
}
