/*
 * proxy.c
 * Written by Michael Kellner
 * Copyright 1999 by Apple Computer, Inc., all rights reserved.
 *
 * $Log: /StreamingServers/proxy3/proxy.c $
		
		9     4/2/99 2:54 PM Mike Kellner
		better copyright
		
		8     3/05/99 3:27p Mike Kellner
		better recognition that socket connection has gone away
		
		7     3/5/99 12:45 PM Mike Kellner
		remove EWOULDBLOCK error case for connections - map to EAGAIN for
		non-unix platforms
		
		6     3/4/99 3:06 PM Mike Kellner
		add ASYNC connect for Mac, average for bps sent/received
		
		5     3/3/99 4:25 PM Mike Kellner
		remove newlines from ErrorStrings, make internal functions static,
		don't use hashval for add and remove session
		
		4     3/01/99 6:47p Mike Kellner
		service more often
		
		3     3/1/99 3:53 PM Mike Kellner
		cleanup, fix incomplete server response problem.
		
		2     2/26/99 7:28 PM Mike Kellner
		make sure that we deal with multiple commands in a single packet
		
		1     2/26/99 10:36 AM Mike Kellner
		Adding subproject 'proxy3' to '$/StreamingServers'
		
		7     2/17/99 10:47 AM Mike Kellner
		don't do isReadable or isWritable, just check return code from
		non-blocking reads or writes. This allows us to notice when the
		connection has died, and thus clean up nicely.
		
		6     2/16/99 5:42 PM Mike Kellner
		don't forget the "client-port" in the reconstructed Transport: line
		
		5     2/16/99 10:52 AM Mike Kellner
		change win32 to WIN32
		
		4     2/15/99 2:16 PM Mike Kellner
		use global for getopt string (so we can have different ones for
		Mac/Unix and Win) use more plat independent stuff for error checking
		use DOS LF/CR 'cause mac can deal and win can't.
		
		3     2/12/99 2:53 PM Mike Kellner
		strip out the source=xxx.xxx.xxx.xxx from the setup response (so as not
		to confuse the client)
		
		2     2/12/99 11:49 AM Mike Kellner
		make compile on Mac and Win32
		
		1     2/11/99 9:44 AM Mike Kellner
		Adding 'proxy.c' to '$/StreamingServers/proxy/'
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>

#if defined(unix)
#include <unistd.h>

#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/resource.h>
#include <signal.h>
#include <sys/signal.h>
#include <sys/socket.h>
#if sgi
#include <sys/schedctl.h>
#endif
#include <regex.h>
#elif defined(WIN32)
#include "WINSOCK.H"
#include "regex.h"
#include "getopt.h"
#elif defined(mac)
#include <Events.h>
#include <MacTypes.h>
#include <unistd.h>
#include "OpenTransport.h"
#include "regex.h"
#include "getopt.h"
#else
#error
#endif

#include "proxy_plat.h"
#include "util.h"
#include "proxy.h"

/**********************************************/
// Globals
int gQuitting = 0;
int gVerbose = 0;
int gDebug = 0;
#if defined(mac) || defined(WIN32)
int gStats = 1;
#else
int gStats = 0;
#endif

int	gNeedsService = 0;

extern char *gConfigFilePath;

rtsp_session *gSessions = NULL;

subnet_allow	*gAllowedNets = NULL;
rtsp_listener	*gListeners = NULL;
int		gUserLimit = 0;
int		gNumUsers = 0;

//int		gUDPPortMin = 4000;
//int		gUDPPortMax = 65535;

int		gProxyIP = -1;
int		gMaxPorts = 0;

unsigned long gBytesReceived = 0;
unsigned long gBytesSent = 0;
unsigned long gPacketsReceived = 0;
unsigned long gPacketsSent = 0;
unsigned long gLastPacketsReceived = 0;
unsigned long gLastPacketsSent = 0;

#define ANY_ADDRESS	-1

/**********************************************/
#if defined(unix)
void sig_catcher(int sig)
{
	gQuitting = 1;
}
#endif

/**********************************************/
static void print_usage(char *name)
{
	printf("%s: [-dvh] [-p #] [-c <file>]\n", name);
	printf("  -d        : debug\n");
	printf("  -v        : verbose\n");
	printf("  -h        : help (this message)\n");
	printf("  -p #      : listen on port # (defaults to 554)\n");
	printf("  -c <file> : configuration file (defaults to %s)\n", gConfigFilePath);
}

/**********************************************/
int main(int argc, char *argv[])
{
	int i, j;
	signed char	ch;
	int	listening_port = 554, user_listener = false;
	unsigned long time_zero, now, last;
	unsigned long	usnow, uslast;
	extern char	*optarg;
	extern int	optind;

	time_zero = time(0);
	last = time_zero;

#if defined(unix)
	//
	// increase file descriptor limit
	{
		struct rlimit rl;
		rl.rlim_cur = 4096;
		rl.rlim_max = 4096;
		setrlimit(RLIMIT_NOFILE, &rl);
	}
	//
	// deal with ctrl-c quit
	signal(SIGINT, sig_catcher);
	signal(SIGPIPE, SIG_IGN);
	//
	// boost our priority
#if sgi
	i = schedctl(NDPRI, 0, NDPHIMAX);
	if (i != 0)
		fprintf(stderr, "schedctl failed %d\n", errno);
#elif _sparc
	i = setpriority(PRIO_PROCESS, 0, -20);
	if (i != 0)
		fprintf(stderr, "schedctl failed %d\n", errno);
#endif
#endif	// defined(unix)
	//
	// initialize the network stacks
	init_network();
	init_ui();

	//
	// read command line options
	while ((ch = get_opt(argc, argv, gOptionsString)) != -1) {
		switch (ch) {
			case 'p':
				listening_port = atoi(optarg);
				user_listener = true;
				break;
			case 'd':
				gDebug = 1;
				break;
			case 'v':
				gVerbose = 1;
				break;
			case 's':
				gStats = 1;
				break;
			case 'h':
				print_usage(argv[0]);
				break;
			case 'c':
				gConfigFilePath = str_dup(optarg);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	//
	// read and deal with config
	read_config();

	//
	// set up rtsp listener (if necessary)
	//  -- if it wasn't specified in the config file, or the user had
	//     one on the command line
	if (!gListeners || user_listener)
		add_rtsp_port_listener(listening_port);

	//
	gProxyIP = get_local_ip_address();

	//
	// compile regular expressions for RTSP
	uslast = microseconds();
	usnow = uslast;
	//
	// main event loop
	i = 0, j = 0;
	while (!gQuitting) {
		if ((i++ >= 50) || !gNeedsService) {
			service_listeners();
			i = 0;
		}
		if ((j++ > 25) || !gNeedsService) {
			service_sessions();
			j = 0;
		}
		service_shoks();
		
		if (gStats) {
			now = time(0);
			if ((now - last) >= 2) {
				unsigned long msElapsed;
				static unsigned long lastBPSReceived = 0, lastBPSSent = 0;
				unsigned long bpsReceived, bpsSent;
				stats_chunk	stats;
				stats.numClients = gNumUsers;
				stats.elapsedSeconds = now - time_zero;
				usnow = microseconds();
				msElapsed = (usnow - uslast) / USEC_PER_MSEC;
				bpsReceived = ((gBytesReceived * USEC_PER_MSEC) / msElapsed) * 8;
				bpsSent = ((gBytesSent * USEC_PER_MSEC) / msElapsed) * 8;
				if (lastBPSReceived) {
					stats.bpsReceived = (bpsReceived + lastBPSReceived) / 2;
					stats.bpsSent = (bpsSent + lastBPSSent) / 2;
				}
				else {
					stats.bpsReceived = bpsReceived;
					stats.bpsSent = bpsSent;
				}
				stats.ppsReceived = ((gPacketsReceived - gLastPacketsReceived) * USEC_PER_MSEC) / msElapsed;
				stats.ppsSent = ((gPacketsSent - gLastPacketsSent) * USEC_PER_MSEC) / msElapsed;
				stats.totalPacketsReceived = gPacketsReceived;
				stats.totalPacketsSent = gPacketsSent;
				stats.numPorts = gMaxPorts;
				DoStats(&stats);
				gBytesReceived = 0;
				gBytesSent = 0;
				gLastPacketsReceived = gPacketsReceived;
				gLastPacketsSent = gPacketsSent;
				lastBPSReceived = bpsReceived;
				lastBPSSent = bpsSent;
				uslast = usnow;
				last = now;
			}
		}
		if (!gNeedsService) {
			sleep_milliseconds(1);	
		}
		else {
			service_ui(0);
			gNeedsService = 0;
		}
	}

	cleanup_sessions();
	cleanup_listeners();
	
	//
	// terminate the network stacks
	term_network();

	return 0;
}

/**********************************************/
int service_listeners()
{
	rtsp_listener *listener;

	listener = gListeners;
	while (listener) {
		answer_new_connection(listener);
		listener = listener->next;
	}
	return 0;
}

/**********************************************/
void cleanup_listeners()
{
	rtsp_listener *listener, *next;

	listener = gListeners;
	while (listener) {
		next = listener->next;
		if (gDebug) DebugString1("Closing listener socket %d", listener->skt);
		close_socket(listener->skt);
		free(listener);
		listener = next;
	}
}

/**********************************************/
void add_rtsp_port_listener(int port)
{
	rtsp_listener *listener;
	int	skt;

	if (gVerbose)
		printf("Listening for RTSP messages on port %d\n", port);

	if ((skt = new_socket_tcp(true)) == INVALID_SOCKET) {
		perror("couldn't set up RTSP listening socket");
		return;
	}
	set_socket_reuse_address(skt);
	
	if (bind_socket_to_address(skt, ANY_ADDRESS, port, true) == SOCKET_ERROR) {
		close_socket(skt);
		perror("binding RTSP listening socket");
		return;
	}

	make_socket_nonblocking(skt);

	if (listen_to_socket(skt) == SOCKET_ERROR) {
		close_socket(skt);
		perror("listening on RTSP socket");
		return;
	}
	
	listener = (rtsp_listener *)malloc(sizeof(rtsp_listener));
	if (!listener) {
		ErrorString("Couldn't allocate memory for listener.");
		exit(1);
	}
	listener->skt = skt;
	listener->port = port;
	listener->next = gListeners;
	gListeners = listener;
}

/**********************************************/
void answer_new_connection(rtsp_listener *listener) {
	int	skt;
	rtsp_session *session;

	if (! call_is_waiting(listener->skt, &skt))
		return;

	session = new_session();
	if (!session) {
		ErrorString("Couldn't create a new session");
		close_socket(skt);
		return;
	}

	session->client_skt = skt;	
	make_socket_nonblocking(session->client_skt);
	session->client_ip = get_remote_address(session->client_skt, NULL);
	session->newSession = true;

	//
	// add him to our session list
		add_session(session);
		if (gVerbose)
			printf("Added connection for client %s.\n", ip_to_string(session->client_ip));
		gNeedsService++;
	}

/**********************************************/
void add_session(rtsp_session *session)
{
	if (gSessions == NULL)
		gSessions = session;
	else {
		session->next = gSessions;
		gSessions = session;
	}

	gNumUsers++;
}

/**********************************************/
void remove_session(rtsp_session *session)
{
	if (gSessions == NULL)
		return;
	else {
		if (gSessions == session)
			gSessions = session->next;
		else {
			rtsp_session *cur, *last = gSessions;
			cur = last->next;
			while (cur) {
				if (cur == session) {
					last->next = cur->next;
					break;
				}
				last = cur;
				cur = cur->next;
			}
		}
	}
}

/**********************************************/
rtsp_session *new_session(void)
{
	rtsp_session	*s;
	int				i;

	s = (rtsp_session*)calloc(1, sizeof(rtsp_session));
	if (s) {
		s->next = NULL;
		s->die = false;
		s->client_skt = INVALID_SOCKET;
		s->client_ip = -1;
		s->server_address = NULL;
		s->server_skt = INVALID_SOCKET;
		s->server_ip = -1;
		s->server_port = 554;
		s->server_skt_pending_connection = false;
		s->state = stRecvClientCommand;
		s->transaction_type = ttNone;
		s->sessionID = NULL;

		s->cur_trk = 0;
		for (i=0; i<MAX_TRACKS; i++) {
			s->trk[i].ID = 0;
			s->trk[i].ClientRTPPort = -1;
			s->trk[i].ServerRTPPort = -1;
			s->trk[i].RTP_S2P = NULL;
			s->trk[i].RTCP_S2P = NULL;
			s->trk[i].RTP_P2C = NULL;
			s->trk[i].RTCP_P2C = NULL;
			s->trk[i].RTP_S2C_tpb.status = NULL;
			s->trk[i].RTP_S2C_tpb.send_from = NULL;
			s->trk[i].RTP_S2C_tpb.send_to_ip = -1;
			s->trk[i].RTP_S2C_tpb.send_to_port = -1;
			s->trk[i].RTCP_S2C_tpb.status = NULL;
			s->trk[i].RTCP_S2C_tpb.send_from = NULL;
			s->trk[i].RTCP_S2C_tpb.send_to_ip = -1;
			s->trk[i].RTCP_S2C_tpb.send_to_port = -1;
			s->trk[i].RTCP_C2S_tpb.status = NULL;
			s->trk[i].RTCP_C2S_tpb.send_from = NULL;
			s->trk[i].RTCP_C2S_tpb.send_to_ip = -1;
			s->trk[i].RTCP_C2S_tpb.send_to_port = -1;
		}
		s->numTracks = 0;

		s->amtInClientInBuffer = 0;
		s->amtInClientOutBuffer = 0;
		s->amtInServerInBuffer = 0;
		s->amtInServerOutBuffer = 0;
	}

	return s;
}

/**********************************************/
void cleanup_session(rtsp_session *s)
{
	int i;
	if (s->client_skt != INVALID_SOCKET) {
		if (gDebug) printf("Closing client rtsp socket %d (ip %s)\n", s->client_skt, ip_to_string(s->client_ip));
		close_socket(s->client_skt);
		s->client_skt = INVALID_SOCKET;
	}
	if (s->server_skt != INVALID_SOCKET) {
		if (gDebug) printf("Closing server rtsp socket %d (ip %s)\n", s->server_skt, ip_to_string(s->server_ip));
		close_socket(s->server_skt);
		s->server_skt = INVALID_SOCKET;
	}
	if (s->server_address)
		free(s->server_address);
	if (s->sessionID)
		free(s->sessionID);
	for (i=0; i<s->numTracks; i++) {
		if (s->trk[i].RTP_S2P)
			remove_shok_ref(s->trk[i].RTP_S2P, gProxyIP, s->server_ip, true);
		if (s->trk[i].RTP_P2C)
			remove_shok_ref(s->trk[i].RTP_P2C, gProxyIP, s->client_ip, true);
	}
}

/**********************************************/
void cleanup_sessions(void)
{
	rtsp_session	*cur, *next;

	cur = gSessions;
	while (cur) {
		next = cur->next;
		cleanup_session(cur);
		free(cur);
		cur = next;
	}
}

/**********************************************/
int service_sessions()
{
	rtsp_session	*cur, *next;

	cur = gSessions;
	while (cur) {
		next = cur->next;
		if (cur->newSession) {
			cur->newSession = false;
			//
			// check to see if this user is allowed
			if (! allow_ip(cur->client_ip)) {
				cur->die = true;
				send_rtsp_error(cur->client_skt, kPermissionDenied);
				if (gVerbose)
					printf("Refusing connection for client %s - not allowed\n", ip_to_string(cur->client_ip));
			}
			//
			// see if we're going beyond our user limit
			else if (gUserLimit && (gNumUsers > gUserLimit)) {
				cur->die = true;
				send_rtsp_error(cur->client_skt, kTooManyUsers);
				if (gVerbose)
					printf("Refusing connection for client %s - too many users\n", ip_to_string(cur->client_ip));
			}
		}
		if (cur->die) {
			gNumUsers--;
			remove_session(cur);
			cleanup_session(cur);
			free(cur);
		}
		else {
			if (cur->client_skt != INVALID_SOCKET || cur->server_skt != INVALID_SOCKET)
				service_session(cur);
		}
		cur = next;
	}
	return 0;
}

/**********************************************/
t_cmd_map cmd_map[] = {
	"DESCRIBE",	ttDescribe,
	"SETUP",	ttSetup,	
	"PLAY",		ttPlay,
	"PAUSE",	ttPause,
	"STOP",		ttStop,
	"TEARDOWN",	ttTeardown,
	NULL,		ttNone
};

static int cmd_to_transaction_type(char *cmd)
{
	t_cmd_map	*map;
	map = cmd_map;
	while (map->cmd != NULL) {
		if (str_casecmp(map->cmd, cmd) == 0)
			return map->type;
		map++;
	}
	return ttNone;
}

/**********************************************/
static int track_id_to_idx(rtsp_session *s, int id)
{
	int i;
	for (i=0; i<s->numTracks; i++) {
		if (s->trk[i].ID == id)
			return i;
	}
	return -1;
}

/**********************************************/
static int has_two_crlfs(char *s)
{
	int		l, n;
	char	*p;
	l = strlen(s);
	if (l < 4)
		return 0;
	n = 3;
	p = &s[n];
	while (n < l) {
		if (s[n] != '\n')
			n += 1;
		else if (s[n-1] != '\r' || s[n-2] != '\n' || s[n-3] != '\r')
			n += 2;
		else
			return 1;
	}
	return 0;
}


/**********************************************/
static int is_command(char *inp, char *cmd, char *server)
{
	int 	l;
	char	*p;

	l = strlen(inp);

	if (l < 17)		/* "RTSP/1.0" (8) + " rtsp:// " (9) */
		return 0;
	if (strn_casecmp(&inp[l-8], "RTSP/1.0", 8) != 0)
		return 0;

	p = inp;
	while (*p && (*p != ' '))
		*cmd++ = *p++;
	*cmd = '\0';

	if (strn_casecmp(p, " rtsp://", 8) != 0)
		return 0;

	p += 8;
	while (*p && (*p != '/'))
		*server++ = *p++;
	*server = '\0';

	return 1;
}

/**********************************************/
static int has_trackID(char *inp, int *trackID)
{
	int 	l;
	char	*p;
	l = strlen(inp);

	if (l < 18)		/* "RTSP/1.0" (8) + "trackID=n " (10) */
		return 0;
	if (strn_casecmp(&inp[l-8], "RTSP/1.0", 8) != 0)
		return 0;

	p = inp;
	while (p) {
		p = strchr(p, '=');
		if (p - 7 < inp) {
			p++;
			continue;
		}
		if (strn_casecmp(&p[-7], "trackid=", 8) != 0) {
			p++;
			continue;
		}
		*trackID = atoi(&p[1]);
		return 1;
	}
	return 0;
}

/**********************************************/
static int has_content_length(char *inp, int *len)
{
	int		l;
	char	*p;
	l = strlen(inp);

	if (l < 16)		/* "Content-Length:n" (16) */
		return 0;
	if (strn_casecmp(inp, "content-length", 14) != 0)
		return 0;
	p = strchr(inp, ':');
	p++;
	while (*p && (*p == ' '))
		p++;
	if (p) {
		*len = atoi(p);
		return 1;
	}
	else
		return 0;
}

/**********************************************/
static int has_IN_IP(char *inp, char *str)
{
	int		l;
	char	*p;
	l = strlen(inp);

	if (l < 10)		/* "c=IN IP4 n" (10) */
		return 0;
	if (strn_casecmp(inp, "c=IN IP4 ", 9) != 0)
		return 0;
	p = &inp[9];
	while (*p && (*p == ' '))
		p++;

	while (*p && ((*p >= '0' && *p <= '9') || *p == '.'))
		*str++ = *p++;
	*str = '\0';
	return 1;
}

/**********************************************/
static int has_sessionID(char *inp, char *sessionID)
{
	int		l;
	char	*p;
	l = strlen(inp);

	if (l < 9)		/* "Session:x" (9) */
		return 0;
	if (strn_casecmp(inp, "session", 7) != 0)
		return 0;
	p = strchr(inp, ':');
	p++;
	while (*p && (*p == ' '))
		p++;
	if (p) {
		strcpy(sessionID, p);
		return 1;
	}
	else
		return 0;
}

/**********************************************/
static int has_client_port(char *inp, int *port)
{
	int		l;
	char	*p;
	l = strlen(inp);

	if (l < 23)		/* "Transport:<>client_port=n" (23) */
		return 0;
	if (strn_casecmp(inp, "transport", 9) != 0)
		return 0;
	
	p = inp;
	while (p) {
		p = strchr(p, '=');
		if (p - 11 < inp) {
			p++;
			continue;
		}
		if (strn_casecmp(&p[-11], "client_port=", 12) != 0) {
			p++;
			continue;
		}
		*port = atoi(&p[1]);
		*++p = '\0';
		return 1;
	}
	return 0;
}

/**********************************************/
static char *find_transport_header(char *inp)
{
	inp = strchr(inp, ';');
	if (inp != NULL)
		inp++;
	return inp;
}

/**********************************************/
static int has_ports(char *inp, int *client_port, int *server_port)
{
	int		l, got_server = 0, got_client = 0;
	char	*p;
	l = strlen(inp);

	if (l < 40)		/* "Transport:<>client_port=n-nserver_port=n-n" (40) */
		return 0;
	if (strn_casecmp(inp, "transport", 9) != 0)
		return 0;

	p = inp;
	while (p && !(got_client && got_server)) {
		p = strchr(p, '=');
		if (p - 11 < inp) {
		}
		else if (strn_casecmp(&p[-11], "client_port=", 12) == 0) {
			got_client = 1;
			*client_port = atoi(&p[1]);
		}
		else if (strn_casecmp(&p[-11], "server_port=", 12) == 0) {
			got_server = 1;
			*server_port = atoi(&p[1]);
		}
		p++;
	}
	if (got_client && got_server)
		return 1;
	else
		return 0;
}

/**********************************************/
void service_session(rtsp_session *s)
{
	int			i, num, canRead, numDelims = 0;
	int			HdrContentLength = 0;
	char		*pBuf, *p;
	char		temp[RTSP_SESSION_BUF_SIZE];
	track_info	*t;
	char		cmd[256], *w;

	/* see if we have any commands coming in */
	pBuf = &s->cinbuf[s->amtInClientInBuffer];
	canRead = sizeof(s->cinbuf) - s->amtInClientInBuffer - 1;
	if (canRead > 0) {
		if ((num = recv_tcp(s->client_skt, pBuf, canRead)) == SOCKET_ERROR) {
			switch (GetLastSocketError(s->client_skt)) {
				case EAGAIN:
						/* do nothing, no data to be read. */
					break;
				case EPIPE:			// connection broke
				case ENOTCONN:		// shut down
				case ECONNRESET:
					s->state = stClientShutdown;
					break;
				default:
					ErrorString1("problems reading from session socket (%d)", GetLastSocketError(s->client_skt));
					break;
			}
		}
		else if (num == 0) {
			// if readable and # of bytes is 0, then the client has shut down
			s->state = stClientShutdown;
		}
		else {
			pBuf[num] = '\0';
			s->amtInClientInBuffer += num;
		}
	}

	/* see what we have to do as a result of this or previous commands */
	switch (s->state) {
		case stIdle:
			break;

		case stRecvClientCommand:
			//
			// have we read a full command yet?
			if (s->amtInClientInBuffer == 0 || ! has_two_crlfs(s->cinbuf))
				break;
			strcpy(temp, s->cinbuf);
			pBuf = temp;
			if ( (p = str_sep(&pBuf, "\r\n")) != NULL ) {
				if (is_command(p, cmd, temp)) {
					if (s->server_address)
						free(s->server_address);
					s->server_address = malloc(strlen(temp) + 1);
					strcpy(s->server_address, temp);
					//
					// take port off address (if any)
					if ((w = strchr(s->server_address, ':')) != NULL)
						*w++ = '\0';
					//
					// make an async request for the IP number
#if USE_THREAD
					name_to_ip_num(s->server_address, &s->tempIP, true);
#else
					name_to_ip_num(s->server_address, &s->tempIP, false);
#endif
					s->state = stWaitingForIPAddress;
				}
			}
			else {
				ErrorStringS("Couldn't make sense of client command", temp);
				s->state = stError;
			}
			break;

		case stWaitingForIPAddress:
			if (s->tempIP != kPENDING_ADDRESS) {
				add_to_IP_cache(s->server_address, s->tempIP);
				s->state = stParseClientCommand;
			}
			if (s->tempIP == -1) {
				send_rtsp_error(s->client_skt, kServerNotFound);
				s->state = stBadServerName;
			}
			break;

		case stParseClientCommand:
			//
			// see what the command and server address is
			//
			// munge the data and snarf what we need
			for (pBuf = s->cinbuf; (p = str_sep(&pBuf, "\r\n")) != NULL; ) {
				//
				// Count the empty fields; three in a row is the end of the header
				if (*p == '\0') {
					if (++numDelims == 3)
						break;
					continue;
				}
				else
					numDelims = 0;

				//
				// see if we can snarf our data out of the headers
				if (is_command(p, cmd, temp)) {
					int		ip;
					//
					// get server address
					if (s->server_address)
						free(s->server_address);
					s->server_address = malloc(strlen(temp) + 1);
					strcpy(s->server_address, temp);
					//
					// get server port (if any)
					if ((w = strchr(s->server_address, ':')) != NULL) {
						*w++ = '\0';
						s->server_port = atoi(w);
					}
					//
					// check to see if command is pointing to the same server that
					// we're already connected to.
					name_to_ip_num(s->server_address, &ip, false);
					if ((ip != s->server_ip) && (s->server_skt != INVALID_SOCKET)) {
						close_socket(s->server_skt);
						s->server_skt = INVALID_SOCKET;
						s->server_ip = ip;
					}
					if (ip == -1) {
						s->state = stBadServerName;
						return;
					}
					s->server_ip = ip;
					if (gVerbose)
						printf("%s command for server %s:%d (ip %s)\n",
								cmd, s->server_address, s->server_port,
								ip_to_string(s->server_ip));
					s->transaction_type = cmd_to_transaction_type(cmd);

					if (s->transaction_type == ttSetup && has_trackID(p, &i)) {
						if ((num = track_id_to_idx(s, i)) == -1) {
							num = s->numTracks;
							s->trk[s->numTracks++].ID = i;
						}
						s->cur_trk = num;
					}
				}
				else if (s->transaction_type == ttSetup
						&& has_client_port(p, &(s->trk[s->cur_trk].ClientRTPPort))) {
					t = &(s->trk[s->cur_trk]);
					if (gDebug)
						printf("Client ports for track %d are %d-%d\n",
								s->cur_trk, t->ClientRTPPort, t->ClientRTPPort + 1);
					//
					// make rtp/rtcp port pair for proxy=>server
					if (make_udp_port_pair(gProxyIP, s->server_ip, &t->RTP_S2P, &t->RTCP_S2P) == -1) {						s->server_skt = INVALID_SOCKET;
						ErrorString1("Couldn't create udp port pair for proxy=>server", GetLastSocketError(s->server_skt));
						s->state = stError;
						break;
					}

					if (gDebug)
						printf("Created ports for server to proxy on track %d are %d-%d sockets %d-%d\n",
								s->cur_trk, t->RTP_S2P->port, t->RTCP_S2P->port,
								t->RTP_S2P->socket, t->RTCP_S2P->socket);
					//
					// reconstruct the client port string
					sprintf(temp, "%s%d-%d", p, t->RTP_S2P->port, t->RTCP_S2P->port);
					p = temp;
				}

				//
				// put the line in the outgoing buffer
				num = strlen(p);
				memcpy(&s->soutbuf[s->amtInServerOutBuffer], p, num);
				s->amtInServerOutBuffer += num;
				s->soutbuf[s->amtInServerOutBuffer++] = '\r';
				s->soutbuf[s->amtInServerOutBuffer++] = '\n';
			}

			s->soutbuf[s->amtInServerOutBuffer++] = '\r';
			s->soutbuf[s->amtInServerOutBuffer++] = '\n';

			s->amtInClientInBuffer -= s->amtInServerOutBuffer;
			if (s->amtInClientInBuffer > 0)
				memcpy(s->cinbuf, &s->cinbuf[s->amtInServerOutBuffer], s->amtInClientInBuffer);
			else if (s->amtInClientInBuffer < 0)
				s->amtInClientInBuffer = 0;
			s->state = stServerTransactionSend;

			if (gDebug)
				DebugStringS("%s", s->soutbuf);

			gNeedsService++;
			break;

		case stServerTransactionSend:
			//
			// check to see if we've got a connection to the server open
			if (s->server_skt == INVALID_SOCKET) {
				//
				// create a connection if we don't have one
				if ((s->server_skt = new_socket_tcp(false)) == INVALID_SOCKET) {
					ErrorString("Couldn't open a socket to connect to server.");
					s->state = stError;
					return;
				}
				set_socket_reuse_address(s->server_skt);
				make_socket_nonblocking(s->server_skt);
				s->server_skt_pending_connection = true;
#if DO_ASYNC
				if ((i = connect_to_address(s, conn_finished_proc, s->server_skt, s->server_ip, s->server_port)) == SOCKET_ERROR) {
#else
				if ((i = connect_to_address(s->server_skt, s->server_ip, s->server_port)) == SOCKET_ERROR) {
#endif
					num = GetLastSocketError(s->server_skt);
					switch (GetLastSocketError(s->server_skt)) {
						case EISCONN:		/* already connected */
							break;
						case EINPROGRESS:	/* connection can't be completed immediately */
						case EAGAIN:		/* connection can't be completed immediately */
						case EALREADY:		/* previous connection attempt hasn't been completed */
							break;
						default:
							close_socket(s->server_skt);
							s->server_skt = INVALID_SOCKET;
							ErrorString1("Couldn't connect to server %d", GetLastSocketError(s->server_skt));
							s->state = stCantConnectToServer;
							return;
					}
				}
				else
					s->server_skt_pending_connection = false;
			}

			//
			// check to see if we're connected (writable) and send the command
			if (s->amtInServerOutBuffer && (s->server_skt_pending_connection == false || isWritable(s->server_skt))) {
				s->server_skt_pending_connection = false;
				if ((num = send_tcp(s->server_skt, s->soutbuf, s->amtInServerOutBuffer)) == SOCKET_ERROR) {
					switch (GetLastSocketError(s->server_skt)) {
						case EPIPE:			// connection broke
						case ENOTCONN:		// shut down
						case ECONNRESET:
							s->state = stServerShutdown;
							break;
						case EAGAIN:	// was busy - try again
						case EINTR:		// got interrupted - try again
							break;
						default:
							ErrorString1("writing to server error (%d)", GetLastSocketError(s->server_skt));
							s->state = stError;
							return;
					}
				}
				else if (num == 0)
					s->state = stServerShutdown;
				else {
					s->amtInServerOutBuffer -= num;
					if (s->amtInServerOutBuffer == 0)
						s->state = stServerTransactionRecv;
				}
			}

			gNeedsService++;
			break;

		case stServerTransactionRecv:
			//
			// check to see if we've got a response from the server
			if (s->server_skt == INVALID_SOCKET) {
				s->state = stServerShutdown;
				break;
			}
			pBuf = &s->sinbuf[s->amtInServerInBuffer];
			canRead = sizeof(s->sinbuf) - s->amtInServerInBuffer - 1;
			if (canRead > 0) {
				if ((num = recv_tcp(s->server_skt, pBuf, canRead)) == SOCKET_ERROR) {
					switch (GetLastSocketError(s->server_skt)) {
						case EAGAIN:
								/* do nothing, no data to be read. */
							break;
						case EPIPE:			// connection broke
						case ENOTCONN:		// shut down
						case ECONNRESET:
							s->state = stServerShutdown;
							break;
						default:
							ErrorString1("problems reading from server (%d)", GetLastSocketError(s->server_skt));
							break;
					}
				}
				else {
					pBuf[num] = '\0';
					if (gDebug)
						printf("read %d bytes from server:%s\n", num, pBuf);
					s->amtInServerInBuffer += num;
				}
			}
			//
			// did we get a complete response yet?
			if (! has_two_crlfs(s->sinbuf))
				break;

			//
			// munge the data for the client, while snarfing what we need
			for (pBuf = s->sinbuf; (p = str_sep(&pBuf, "\r\n")) != NULL; ) {
				//
				// Count the empty fields; three in a row is end of the header
				if (*p == '\0') {
					if (++numDelims == 3)
						break;
					continue;
				}
				else
					numDelims = 0;

				// see if we can snarf any data out of the headers
				if (has_content_length(p, &HdrContentLength)) {
				}
				else if (has_sessionID(p, temp)) {
					if (!s->sessionID) {
						s->sessionID = malloc(strlen(temp) + 1);
						strcpy(s->sessionID, temp);
					}
					else if (str_casecmp(s->sessionID, temp) != 0)
							ErrorString("Bad session ID in response from server");
				}
				else if (s->transaction_type == ttSetup && has_ports(p, &i, &num)) {
					t = &(s->trk[s->cur_trk]);
					t->ServerRTPPort = num;
					
					if (gDebug)
						printf("Server ports for track %d are %d-%d \n",
								s->cur_trk, t->ServerRTPPort, t->ServerRTPPort + 1);
					//
					// make rtp/rtcp port pair here proxy=>client
					if (make_udp_port_pair(gProxyIP, s->client_ip, &t->RTP_P2C, &t->RTCP_P2C) == -1) {						s->server_skt = INVALID_SOCKET;
						ErrorString1("Couldn't create udp port pair for proxy=>client", GetLastSocketError(s->server_skt));
						s->state = stError;
						break;
					}
					//
					// set up transfer param blocks
					t->RTP_S2C_tpb.status = &s->die;
					t->RTP_S2C_tpb.send_from = t->RTP_P2C;
					t->RTP_S2C_tpb.send_to_ip = s->client_ip;
					t->RTP_S2C_tpb.send_to_port = t->ClientRTPPort;
					upon_receipt_from(t->RTP_S2P, s->server_ip, transfer_data, &(t->RTP_S2C_tpb));

					t->RTCP_S2C_tpb.status = &s->die;
					t->RTCP_S2C_tpb.send_from = t->RTCP_P2C;
					t->RTCP_S2C_tpb.send_to_ip = s->client_ip;
					t->RTCP_S2C_tpb.send_to_port = t->ClientRTPPort + 1;
					upon_receipt_from(t->RTCP_S2P, s->server_ip, transfer_data, &(t->RTCP_S2C_tpb));

					t->RTCP_C2S_tpb.status = &s->die;
					t->RTCP_C2S_tpb.send_from = t->RTCP_S2P;
					t->RTCP_C2S_tpb.send_to_ip = s->server_ip;
					t->RTCP_C2S_tpb.send_to_port = t->ServerRTPPort + 1;
					upon_receipt_from(t->RTCP_P2C, s->client_ip, transfer_data, &(t->RTCP_C2S_tpb));

					if (gDebug)
						printf("Created ports for proxy to client on track %d are %d-%d sockets %d-%d\n",
								s->cur_trk,
								t->RTP_P2C->port, t->RTCP_P2C->port,
								t->RTP_P2C->socket, t->RTCP_P2C->socket);
					//
					// reconstruct the client;server string
					w = find_transport_header(p);
					if (w != NULL)
						*w = '\0';
					sprintf(temp, "%sclient_port=%d-%d;server_port=%d-%d", p,
						t->ClientRTPPort, t->ClientRTPPort+1,
						t->RTP_P2C->port, t->RTCP_P2C->port);
					p = temp;
				}

				//
				// put the line in the outgoing buffer
				num = strlen(p);
				memcpy(&s->coutbuf[s->amtInClientOutBuffer], p, num);
				s->amtInClientOutBuffer += num;
				s->coutbuf[s->amtInClientOutBuffer++] = '\r';
				s->coutbuf[s->amtInClientOutBuffer++] = '\n';
			}

			s->coutbuf[s->amtInClientOutBuffer++] = '\r';
			s->coutbuf[s->amtInClientOutBuffer++] = '\n';

			//
			// munge and add the content if there is any
			if (HdrContentLength) {
				if (s->transaction_type == ttDescribe) {
					for (; (p = str_sep(&pBuf, "\r\n")) != NULL; ) {
						if (*p != '\0') {
							if (has_IN_IP(p, temp)) {
								//
								// reconstruct the IN IP string, but
								// make sure our replacement string is the
								// same length as the original string
								// so as not to change the Content-Length string
								switch (strlen(p) - 12) {
									case 4:
										sprintf(temp, "c=IN IP4 0.0.0.0");
										break;
									case 5:
										sprintf(temp, "c=IN IP4 0.0.0.00");
										break;
									case 6:
										sprintf(temp, "c=IN IP4 0.0.0.000");
										break;
									case 7:
										sprintf(temp, "c=IN IP4 0.0.00.000");
										break;
									case 8:
										sprintf(temp, "c=IN IP4 0.0.000.000");
										break;
									case 9:
										sprintf(temp, "c=IN IP4 0.00.000.000");
										break;
									case 10:
										sprintf(temp, "c=IN IP4 0.000.000.000");
										break;
									case 11:
										sprintf(temp, "c=IN IP4 00.000.000.000");
										break;
									case 12:
										sprintf(temp, "c=IN IP4 000.000.000.000");
										break;
								}
								p = temp;
							}
							
							//
							// put the line in the outgoing buffer
							num = strlen(p);
							memcpy(&s->coutbuf[s->amtInClientOutBuffer], p, num);
							s->amtInClientOutBuffer += num;
							s->coutbuf[s->amtInClientOutBuffer++] = '\r';
							s->coutbuf[s->amtInClientOutBuffer++] = '\n';
						}
					}
				}
				else {
					memcpy(&s->coutbuf[s->amtInClientOutBuffer], pBuf, HdrContentLength);
					s->amtInClientOutBuffer += HdrContentLength;
				}
			}

			s->state = stSendClientResponse;
			s->amtInServerInBuffer = 0;

			gNeedsService++;
			break;

		case stSendClientResponse:
			//
			// check to see that we're still connected (writable) and send the response
			if (s->amtInClientOutBuffer && isWritable(s->client_skt)) {
				if ((num = send_tcp(s->client_skt, s->coutbuf, s->amtInClientOutBuffer)) == SOCKET_ERROR) {
					switch (GetLastSocketError(s->client_skt)) {
						case EPIPE:			// connection broke
						case ENOTCONN:		// shut down
						case ECONNRESET:
							s->state = stClientShutdown;
							break;
						case EAGAIN:	// was busy - try again
						case EINTR:		// got interrupted - try again
							break;
						default:
							ErrorString1("writing to client error (%d)", GetLastSocketError(s->client_skt));
							s->state = stError;
							return;
					}
				}
				else {
					s->amtInClientOutBuffer -= num;
					if (s->amtInClientOutBuffer == 0)
						if (s->transaction_type == ttTeardown)
							s->state = stClientShutdown;
						else
							s->state = stRecvClientCommand;
				}
			}

			gNeedsService++;
			break;
			

		case stClientShutdown:
			if (gDebug) DebugString1("Client shutdown (ip %s)", ip_to_string(s->client_ip));
			s->die = true;
			gNeedsService++;
			break;

		case stBadServerName:
			send_rtsp_error(s->client_skt, kServerNotFound);
			s->state = stServerShutdown;
			break;

		case stCantConnectToServer:
			send_rtsp_error(s->client_skt, kServerNotFound);
			s->state = stServerShutdown;
			break;
			
		case stServerShutdown:
			if (gDebug) DebugString1("Server shutdown (ip %s)", ip_to_string(s->server_ip));
			s->die = true;
			gNeedsService++;
			break;

		case stError:
			send_rtsp_error(s->client_skt, kUnknownError);
			if (gDebug) DebugString("error condition.");
			s->die = true;
			gNeedsService++;
			break;
	}
}

/**********************************************/
#define MAX_CONFIG_LINE_LEN	512
void read_config() {
	int			fd, eof, num;
	char		buf, line[MAX_CONFIG_LINE_LEN], temp[MAX_CONFIG_LINE_LEN];
	int			line_pos;
	regmatch_t	pmatch[3];
	regex_t		regexpAllow, regexpComment, regexpUsers;
	regex_t		regexpListen, regexpPortRange;

	if ((fd = open(gConfigFilePath, O_RDONLY)) == -1) {
		switch (errno) {
			case EACCES:
				ErrorStringS("Config file %s inaccessible", gConfigFilePath);
				break;
			default:
				ErrorString1("Problems opening config file (%d)", errno);
				break;
		}
		return;
	}
	
	if (gVerbose)
		printf("Opened config file %s\n", gConfigFilePath);

	regcomp(&regexpAllow, "^[	 ]*allow[ 	]+([0-9\\.]+)/([0-9]+).*$",
			REG_EXTENDED | REG_ICASE);
	regcomp(&regexpUsers, "^[	 ]*users[ 	]+([0-9]+).*$",
			REG_EXTENDED | REG_ICASE);
	regcomp(&regexpListen, "^[	 ]*listen[ 	]+([0-9]+).*$",
			REG_EXTENDED | REG_ICASE);
	regcomp(&regexpPortRange, "^[	 ]*port-range[ 	]+([0-9]+)-([0-9]+).*$",
			REG_EXTENDED | REG_ICASE);
	regcomp(&regexpComment, "^[	 ]*#.*$", REG_EXTENDED | REG_ICASE);

	eof = false;
	while (!eof) {
		//
		// read a line from the configuration file
		line_pos = 0;
		while (1) {
			num = read(fd, &buf, 1);
			if (num <= 0) {
				eof = true;
				break;
			}
			if (buf == '\n' || buf == '\r')
				break;
			if (line_pos < MAX_CONFIG_LINE_LEN)
				line[line_pos++] = buf;
		}
		line[line_pos] = '\0';
		if (gDebug) DebugStringS("Read config line: %s", line);

		//
		// if the line has anything, try to parse it
		if (line_pos > 0) {
			if (regexec(&regexpAllow, line, 3, pmatch, 0) == 0) {
				int		ip, range;

				num = pmatch[1].rm_eo - pmatch[1].rm_so;
				memcpy(temp, line + pmatch[1].rm_so, num);
				temp[num] = '\0';
				range = atoi(line + pmatch[2].rm_so);
				name_to_ip_num(temp, &ip, false);
				if (gVerbose)
					printf("Allow connections from %s/%d\n", ip_to_string(ip), range);
				add_allow_subnet(ip, range);
			}
			else if (regexec(&regexpUsers, line, 3, pmatch, 0) == 0) {
				gUserLimit = atoi(line + pmatch[1].rm_so);
				if (gVerbose)
					printf("Limit number of users to %d\n", gUserLimit);
			}
			else if (regexec(&regexpListen, line, 3, pmatch, 0) == 0) {
				int port = atoi(line + pmatch[1].rm_so);
				add_rtsp_port_listener(port);
			}
			else if (regexec(&regexpPortRange, line, 3, pmatch, 0) == 0) {
				int minPort, maxPort;
				minPort = atoi(line + pmatch[1].rm_so);
				maxPort = atoi(line + pmatch[2].rm_so);
				set_udp_port_min_and_max(minPort, maxPort);
				if (gVerbose)
					printf("Use UDP ports %d through %d\n", minPort, maxPort);
			}
			else if (regexec(&regexpComment, line, 0, NULL, 0) == 0) {
				// comment - do nothing
			}
			else {
				ErrorStringS("invalid configuration line [%s]", line);
			}
		}
	}

	regfree(&regexpAllow);
	regfree(&regexpComment);
	regfree(&regexpUsers);
	regfree(&regexpListen);
	regfree(&regexpPortRange);

	close(fd);
}

/**********************************************/
void add_allow_subnet(int ip, int range)
{
	subnet_allow *allow;

	allow = (subnet_allow*)malloc(sizeof(subnet_allow));
	allow->ip = ip;
	allow->range = range;
	allow->next = gAllowedNets;
	gAllowedNets = allow;
}

/**********************************************/
static void bitfill(int *x, int bits)
{
	int i;
	*x = 0;
	for (i=31; i>31-bits; i--)
		*x |= (1 << i);
}

/**********************************************/
bool allow_ip(int ip)
{
	int				mask;
	subnet_allow	*cur;

	cur = gAllowedNets;
	if (cur == NULL)
		return true;

	while (cur) {
		bitfill(&mask, cur->range);
		if ((cur->ip & mask) == (ip & mask))
			return true;
		cur = cur->next;
	}

	return false;
}

/**********************************************/
void send_rtsp_error(int skt, int refusal)
{
	char *refusal_string;

	switch (refusal) {
		case kServerNotFound:
 			refusal_string = "RTSP/1.0 462 Destination unreachable\r\n";
			break;
		case kUnknownError:
 			refusal_string = "RTSP/1.0 500 Unknown proxy error\r\n";
			break;
		case kPermissionDenied:
 			refusal_string = "RTSP/1.0 403 Proxy denied\r\n";
			break;
		case kTooManyUsers:
 			refusal_string = "RTSP/1.0 503 Too many proxy users\r\n";
			break;
		default:
 			refusal_string = "RTSP/1.0 500 Unknown proxy error\r\n";
			break;
	}

	send_tcp(skt, refusal_string, strlen(refusal_string));
}

/**********************************************/
/**********************************************/
