/*  -*- c -*-  */
/* -------------------------------------------------------------------- *
**  copyright (c) 1995 ipvr stuttgart and thomas harrer
** -------------------------------------------------------------------- *
**
**  libhelp
**
**  a comprehensive hypertext help system for OSF/Motif(tm) applications. 
**  based on libhtmlw from NCSA Mosaic(tm) version 2.4
**
**  written by thomas harrer
**  e-mail: Thomas.Harrer@rus.uni-stuttgart.de
**  
** -------------------------------------------------------------------- *
*h  $Id: util.c,v 1.10 1995/06/28 12:59:30 thomas Exp $
** -------------------------------------------------------------------- *
**
*h  module:		util.c
**
**  contents:		utilities for extracting components in urls and
**			path names and for classifying filenames.
**
**  interface:		procedures
**			* get_file
**			* get_anchor
**			* convert_newlines_to_spaces
**			* get_file_and_strip_path
**			* get_path
**			* file_type
**			
** -------------------------------------------------------------------- *
**  license and copying issues:
**
**  this software is free; you can redistribute it and/or modify it 
**  under terms similar to the gnu general public license (version 1 
**  or any later version published by the free software foundation). 
**  see the file Licence for more details.
**
**  this program is distributed in the hope that it will be useful,
**  but without any warranty; without even the implied warranty of
**  merchantability or fitness for a particular purpose.  
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*g  include files
** -------------------------------------------------------------------- */
#include <sys/stat.h>
#include <ctype.h>
#include <dirent.h>

#include "helpp.h"
#include "util.h"

/* -------------------------------------------------------------------- *
*g  module identification
** -------------------------------------------------------------------- */
#ifdef RCSID
static char rcsid [] =
    "$Id: util.c,v 1.10 1995/06/28 12:59:30 thomas Exp $";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
*g  local defs.
** -------------------------------------------------------------------- */
#define LOCAL "file:"
#define LOCAL_LEN 5

/* -------------------------------------------------------------------- *
*p  procedure-name:	get_file
**
**  purpose:		extracts file (plus opt. path) from url.
**			strips of prefix and anchor.
** -------------------------------------------------------------------- *
**  args:		filename and opt. anchor in a '\0' terminated
**			string.  format: "[file:]<file>[#<anchor>]".
**  return type:	char* (allocated). caller is responsible for
**			freeing the returned memory (if != NULL).
** -------------------------------------------------------------------- */
char*
get_file (/* i  */ char* url)
{
    char* filename = NULL;	/* filename to be returned.		*/
    char* ptr;			/* pointer for string manipulation.	*/

    execute ("get_file");

    /* we work arround URL's containing file: prefix.  */
    if (url) {

	if (0 == c_strncmp (url, LOCAL, LOCAL_LEN)) {
	    checked_strdup (filename, url + LOCAL_LEN);
	} else  {
	    checked_strdup (filename, url); /* scope: returned. */
	}

	ptr = filename;

	/* for absolute specified files we ignore the anchor.  */
	/* we neet this to access filesnames which contain '#' chars.  */
	if ((url[0] != '.') && (url[0] != '/')) {

	    /* else we strip off the optional anchor  */
	    while (*ptr != '\0') {
		if (*ptr == '#') {
		    *ptr = '\0';
		    break;
		}
		ptr++;
	    }
	}
    
	return filename;
    }
    return NULL;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	get_anchor
**
**  purpose:		returns the anchor from the given url or an
**			empty string if there was no anchor.
** -------------------------------------------------------------------- *
**  args:		filename + opt anchor.
**  returns:		the anchor or "" (\0 terminated).
**			the application must not free the returned ptr.
** -------------------------------------------------------------------- */
char*
get_anchor (/* i  */ char* url)
{
    char* ptr = url;
    
    execute ("get_anchor");

    while (*ptr != '\0') {
	if (*ptr == '#') {
	    return (ptr + 1);
	}
	else ptr++;
    }
    return ptr;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	convert_newlines_to_spaces
**
**  purpose:		removes leading whitespace and converts all
**			newlines with spaces.
**  and:		shows we can fluently deal with pointers :-)
** -------------------------------------------------------------------- *
**  args:		string str
**  return type:	char*
**  precondition:	\0 terminated string must be passed.
**  postcondition:	see purpose. side effect is, that non blank part
**			of str is copied to the beginning of string.
**  error handling.:	assumes \0 at the end of string.
** -------------------------------------------------------------------- */
char*
convert_newlines_to_spaces (/* io */ char*	str)
{
    if (str) {

	char* ptr = str;
    
	execute ("convert_newlines_to_spaces");
    
	/* we try to move to the first nonspace position.  */
	while ((*ptr != '\0') && (isspace (*ptr))) {
	    ptr++;
	}

	/* if we moved the position, now we move the string.  */
	if (ptr != str) {
	    /* is this destructive ??  */
	    c_memmove (str, ptr, (c_strlen (ptr) + 1));
	}
	
	/* in the result, we convert newlines to spaces.  */
	ptr = str;
	while (*ptr) {
	    if (*ptr == '\n') *ptr = ' ';
	    ptr++;
	}
    
	return str;
    }

    /* if we've got nothing, we return nothing.  */
    return NULL;    
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	get_file_and_strip_path
**
**  purpose:		strips the path component from a filename.
** -------------------------------------------------------------------- *
**  return-value:	returns a newly allocated string filename.
**			or null if ref does not contain a filename 
**			or ref is a directory.
**			the caller is responsible to free the returned 
**			string.
** -------------------------------------------------------------------- */
char*
get_file_and_strip_path (/* i  */ char* url)
{
    execute ("get_file_and_strip_path");
    
    if (url) {

	char* file = get_file (url); /* scope return  */

	if (file) {

	    /* here we need the filetype:  */
	    int type = file_type (file);

	    if (type & file_dir)  {

		checked_free (file);
		return NULL;
	    
	    } else {

		char* last = NULL;
		char* ptr = file;

		/* we extract the last element of the path.  */
		while (*ptr != '\0') {
		    if (*ptr == '/') {
			last = ptr;
		    }
		    ptr++;
		}

		/* now we move the last component to the 1st position. */
		if ((last) && (*last))  { 
		    last++;
		    c_memmove (file, last, (c_strlen (last) + 1));
		}
		return file;
	    }
	}
    }

    return NULL;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	get_path
**
**  purpose:		returns the path component of filename
** -------------------------------------------------------------------- *
**  precondition:	file must be accessible (for reading).
**
**  return value:	path component of file in a newly allocated 
**			string or NULL if name of file
** -------------------------------------------------------------------- */
char*
get_path (/* i  */ char* filename)
{
    execute ("get_path");

    if (filename) {

	struct stat 	st;
	char* 		fn = NULL;
	
	/* we work arround URL's containing file: prefix.  */
	if (0 == c_strncmp (filename, LOCAL, LOCAL_LEN)) {
	    fn = filename + LOCAL_LEN;
	} else  {
	    fn = filename;
	}
	if (stat (fn, &st) >= 0) {

	    int 	len = c_strlen (fn);
	    char* 	path;	/* scope: returned. */
	
	    checked_malloc (path, len + 2, char);
	    c_strcpy (path, fn);

	    /* we apply some changes to the path.  */
	    if (S_ISDIR (st.st_mode)) {
		/* for directories we add "/" if not already at the end.  */
		if ((len > 0) && (path[len - 1] != '/'))
		    c_strcat (path, "/");
	    } else {
	    
		/* for all others: we take the string before the last '/'. */
		char* pos = NULL;
		char* ptr = path;
    
		while (*ptr) {	/* searches path!  */
		    if (*ptr == '/') pos = ptr;
		    ptr++;
		}

		if (pos)  {	/* changes path! */
		    *(pos + 1) = '\0';
		} else {
		    *ptr = '/';
		    *(ptr + 1) = '\0';
		}
	    }
	    return path;
	} else  {
	    return NULL;
	}

    } else  {
	return NULL;
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	file_type
**
**  purpose:		returns the type of the file `filename'.
**			
**			possible values are:
**			
**			* file_html
**			* file_dir
**			* file_plain
**			* file_gif
**			* file_executable
**			* file_no_access
**			
**			possible more than one attributes belong to a file
**			the caller can test if attribute is true by
**			using the & (and) operator. e.g. 
**			
**			xxx = file_type ("?");
**			if (xxx & file_dir) printf ("? is a dir\n");
** -------------------------------------------------------------------- */
int
file_type (/* i  */ char* filename)
{

/* some local definitions. */
#define GIF_SUF ".gif"
#define GIF_SUF_LEN 4
#define HTML_SUF ".html"
#define HTML_SUF_LEN 5
#define HTMLGZ_SUF ".html.gz"
#define HTMLGZ_SUF_LEN 8

    execute ("file_type");

    if (filename) {

	int 		ftype = 0;	/* the return value. */
	struct stat 	st;		/* tmp. stat buffer. */

	/* we check only if file is accessible  */
	if (stat (filename, &st) < 0) return file_no_access;
	if (0 != access (filename, R_OK)) return file_no_access;

	if (S_ISREG (st.st_mode)) {

	    int len = c_strlen (filename);
	    int htmlpos = (len > HTML_SUF_LEN ? (len - HTML_SUF_LEN) : 0);
	    int gzpos = (len > HTMLGZ_SUF_LEN ? (len - HTMLGZ_SUF_LEN) : 0);
	    int gifpos = (len > GIF_SUF_LEN ? (len - GIF_SUF_LEN) : 0);
	
	    /* html file ?  */
	    if (0 == c_strcmp (filename + htmlpos, HTML_SUF)) {
		ftype |= file_html;
	    } else if (0 == c_strcmp (filename + gzpos, HTMLGZ_SUF)) {
		ftype |= file_html;
	    } else if (0 == c_strcmp (filename + gifpos, GIF_SUF)) {
		/* gif file ?  */
		ftype = file_gif;
	    } else {
		ftype |= file_plain;
	    }
	    /* executable ?  */
	    if (0 == access (filename, X_OK)) ftype |= file_executable;

	} else if (S_ISDIR (st.st_mode)) {
	    ftype = file_dir;
	}

	return ftype;
    } else {
	return file_no_access;
    }
}

/* -------------------------------------------------------------------- *
*l  emacs:
**  local variables:
**  mode:		c
**  outline-regexp:	"\*[HGPLT]"
**  comment-column:	32
**  eval:		(outline-minor-mode t)
**  end:
** -------------------------------------------------------------------- */
