/*  -*- c -*-  */
/* -------------------------------------------------------------------- *
**  copyright (c) 1995 ipvr stuttgart and thomas harrer
** -------------------------------------------------------------------- *
**
**  libhelp
**
**  a comprehensive hypertext help system for OSF/Motif(tm) applications. 
**  based on libhtmlw from NCSA Mosaic(tm) version 2.4
**
**  written by thomas harrer
**  e-mail: Thomas.Harrer@rus.uni-stuttgart.de
**  
** -------------------------------------------------------------------- *
*h  $Id: buffer.c,v 1.16 1995/06/28 12:59:30 thomas Exp $
** -------------------------------------------------------------------- *
**
*h  module:		buffer.c
**
**  contents:		object output buffer
**
**  interface:	        opaque type buffer_t;
**			functions
**			* bf_new
**			* bf_free
**			* bf_strcpy
**			* bf_strcat
**			* bf_sprintf
**			* bf_the_buffer
**			
**			* bf_memory_usage
**			* bf_buffer_usage
**
** -------------------------------------------------------------------- *
**  license and copying issues:
**
**  this software is free; you can redistribute it and/or modify it 
**  under terms similar to the gnu general public license (version 1 
**  or any later version published by the free software foundation). 
**  see the file Licence for more details.
**
**  this program is distributed in the hope that it will be useful,
**  but without any warranty; without even the implied warranty of
**  merchantability or fitness for a particular purpose.  
** -------------------------------------------------------------------- */
#include <stdio.h>
#include <stdarg.h>
#include "helpp.h"
#include "buffer.h"

/* -------------------------------------------------------------------- *
*g  module identification
** -------------------------------------------------------------------- */
#ifdef RCSID
static char rcsid [] =
    "$Id: buffer.c,v 1.16 1995/06/28 12:59:30 thomas Exp $";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
*g  definitions
** -------------------------------------------------------------------- */
#define PAGE_SIZE 4096

/* -------------------------------------------------------------------- *
*g  private buffer structure
** -------------------------------------------------------------------- */
struct _buffer_s {

    char* buffer;		/* the buffer to be enlarged		*/
    int	  size;			/* the allocation size			*/
    int	  pos;			/* the position for the next write.	*/
    int   refs;			/* number of copies.			*/

};

/* -------------------------------------------------------------------- *
*p  private prototypes
** -------------------------------------------------------------------- */
static void enlarge_buffer (buffer_t*, int);


/* -------------------------------------------------------------------- *
*g  global data:	statistics
** -------------------------------------------------------------------- */
static int mem_usage = 0;
static int mem_filled = 0;
static int num_buffers = 0;

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_new
**
**  purpose:		returns a newly allocated and initialized 
**			buffer structure.
** -------------------------------------------------------------------- */
buffer_t*
bf_new (void)
{
    buffer_t*	ptr;
    
    /* we allocate one instance of a buffer.  */
    checked_malloc (ptr, 1, buffer_t);

    ptr->buffer = NULL;
    ptr->size = 0;
    ptr->pos = 0;
    ptr->refs = 1;

    num_buffers++;
    
    return ptr;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_clone
**
**  purpose:		return the same buffer with just incremented 
**			reference count. it is like a hard link for 
**			files.
**  note:		cloned buffers should not be accessed for writing
**			by different tasks. the result can be somewhat
**			ugly.
** -------------------------------------------------------------------- */
buffer_t*
bf_clone (/* io */ buffer_t* buf)
{
    if (buf) {
	if (buf->refs >= 1) {
	    buf->refs++;
	} else {
	    fatal_error (error_inconsistency);
	}
    }
    return buf;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_free
**
**  purpose:		drestroys the buffer
** -------------------------------------------------------------------- */
int
bf_free (/* i  */ buffer_t*	buf)
{
    if (buf) {
	
	/*
	 *  if this buffer has some more clone, we don't free it.
	 *  we just remove the reference to the clone.
	 */
	if (buf->refs > 1) {
	    
	    buf->refs--;

	} else {

	    /* if this is the last clone, we can free the memory.  */
	    if (buf->buffer)  {
		mem_usage -= buf->size;
		mem_filled -= buf->pos;
		checked_free (buf->buffer);
	    }
	    checked_free (buf);

	    num_buffers--;
	}
    }

    return no_error;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_the_buffer
**
**  purpose:		returns the text pointer of the buffer, or NULL
**			if there is no text.
** -------------------------------------------------------------------- */
char*
bf_the_buffer (/* i  */ buffer_t* buf)
{
    char* ret_buf = NULL;
    
    if (buf != NULL) {
	ret_buf = buf->buffer;
    }
    return ret_buf;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_strcat
**
**  purpose:		adds text to the buffer.
** -------------------------------------------------------------------- */
int
bf_strcat (/* io */ buffer_t*	buf,
	   /* i  */ char*	text)
{
    if (buf) {
	if (text) {

	    /* data.  */
	    int len = c_strlen (text);
	    /* actions.  */
	    enlarge_buffer (buf, len);
	    c_strcpy (buf->buffer + buf->pos, text);
	    buf->pos += len;
	    mem_filled += len;
	    return len;

	} else return no_error;
    } else return BF_sorry;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_strcpy
**
**  purpose:		starts text in an already existent buffer.
** -------------------------------------------------------------------- */
int
bf_strcpy (/* i  */ buffer_t*	buf,
	   /* i  */ char*	text)
{
    if (buf) {
	mem_filled -= buf->pos;
	buf->pos = 0;
	return bf_strcat (buf, text);
    } else {
	return BF_sorry;
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_sprint
**
**  purpose:		adds text to the buffer (but with arguments like
**			sprintf).
** -------------------------------------------------------------------- *
**  caveat:		the resulting string must not be larger than
**			PAGE_SIZE bytes (4k default).
**			i found no way to make vsprintf save (unless 
**			there is no vsnprint or vasprintf :(). i could't 
**			find these on much systems.
** -------------------------------------------------------------------- */
int
bf_sprint (/* io */ buffer_t* 	buf, 
	   /* i  */ char* 	format_template,
	   /* i  */ char* 	string)
{
    /* local prototype  */
    if (format_template && string) {
    
    	char* 	buffer;
	int 	size = PAGE_SIZE + c_strlen (format_template) 
	    + c_strlen (string);
	int 	len = 0;

	/* first we format all into a new buffer.  */
	checked_malloc (buffer, size, char);
	
	(void) sprintf (buffer, format_template, string);
	len = c_strlen (buffer);

	if (len > size - 1) {
	    fatal_error (error_out_of_memory);
	}

	enlarge_buffer (buf, len);
	c_strcpy (buf->buffer + buf->pos, buffer);
	mem_filled += len;
	buf->pos += len;

	/* and we don't forget to free the internal buffer.  */
	checked_free (buffer);
    }

    return no_error;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_dprint
**
**  purpose:		adds text to the buffer (but with arguments like
**			sprintf with one decimal arg.).
** -------------------------------------------------------------------- *
**  caveat:		the resulting string must not be larger than
**			PAGE_SIZE bytes (4k default).
**			i found no way to make vsprintf save (unless 
**			there is no vsnprint or vasprintf :(). i could't 
**			find these on much systems.
** -------------------------------------------------------------------- */
int
bf_dprint (/* io */ buffer_t* 	buf, 
	   /* i  */ char* 	format_template,
	   /* i  */ int		value)
{
    /* local prototype  */
    if (format_template) {
    
	char* 	buffer;
	int 	size = PAGE_SIZE  + c_strlen (format_template);
	int 	len = 0;

	/* first we format all into a new buffer.  */
	checked_malloc (buffer, size, char);
	
	(void) sprintf (buffer, format_template, value);
	len = c_strlen (buffer);

	if (len > size - 1) {
	    fatal_error (error_out_of_memory);
	}

	enlarge_buffer (buf, len);
	c_strcpy (buf->buffer + buf->pos, buffer);
	mem_filled += len;
	buf->pos += len;

	/* and we don't forget to free the internal buffer.  */
	checked_free (buffer);
    }

    return no_error;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bf_fread
**
**  purpose:		adds the contents of file into buffer
** -------------------------------------------------------------------- */
int
bf_fread (/* i  */ buffer_t*	buf,
	  /* i  */ FILE*	stream)
{
    int 	is_binary = 0;

    if (buf) {

	while (1) {
	    
	    char* 	ptr = buf->buffer + buf->pos;
	    int 	try_size = buf->size - buf->pos;
	    int 	len;

	    len = fread (ptr, sizeof (char), (size_t) try_size, stream);
	    mem_filled += len;
	    buf->pos += len;
	    if (len < try_size) break;

	    /* if there is a '\0' character we don't display it.  */
	    if (NULL != memchr (ptr, 0, (size_t) len))  {
		is_binary = 1; break;
	    }
	    enlarge_buffer (buf, PAGE_SIZE);
	}
	enlarge_buffer (buf, 1);
	buf->buffer[buf->pos] = '\0';
    } else  {
	return BF_sorry;
    }

    if (is_binary) return BF_is_binary;
    return no_error;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	enlarge_buffer
**
**  purpose:		enlarges the buffer so that len bytes can 
**			be stored additionally.
** -------------------------------------------------------------------- */
static void
enlarge_buffer (/* io */ buffer_t* buf,
		/* i  */ int	   len)
{
    if (buf) {

	int new_len;
	int new_size;
	
	/* if we have no buffer, we allocate it.  */
	if (!(buf->buffer)) {
	    buf->size = PAGE_SIZE;
	    checked_malloc (buf->buffer, buf->size, char);
	    buf->pos = 0;
	    mem_usage += buf->size;
	}
	
	/* linear growing scheme.  */
	new_len = buf->pos + len;
	new_size = ((new_len / PAGE_SIZE) + 1) * PAGE_SIZE;
	
	if (buf->size < new_size) {
	    checked_realloc (buf->buffer, new_size, char);
	    mem_usage += (new_size - buf->size);
	    buf->size = new_size;
	}
    }
}

/* -------------------------------------------------------------------- *
*g  -------------------- small information functions ------------------
** -------------------------------------------------------------------- */
int bf_memory_usage (void) 
{
    return mem_usage;
}
int bf_buffer_usage (void)
{
    return num_buffers;
}
int bf_memory_filled (void)
{
    return mem_filled;
}
int bf_page_size (void)
{
    return PAGE_SIZE;
}

/* -------------------------------------------------------------------- *
*g  module test:	simple example program
**			does not depends on other modules.
** -------------------------------------------------------------------- */
#ifdef MODULE_TEST
int main (int argc, char* argv[])
{
    buffer_t*	buf = bf_new ();
    bf_strcpy (buf, "thomas");
    bf_sprint (buf, "%s harrer ", "version");
    bf_dprint (buf, "%d\n", 10);
    bf_fread (buf, stdin);
    printf (bf_the_buffer (buf));
    bf_free (buf);
}
#endif /* MODULE_TEST  */

/* -------------------------------------------------------------------- *
*l  emacs:
**  local variables:
**  mode:		c
**  outline-regexp:	"\*[HGPLT]"
**  comment-column:	32
**  eval:		(outline-minor-mode t)
**  end:
** -------------------------------------------------------------------- */
