// -*- mode: cpp; mode: fold -*-
// Description								/*{{{*/
// $Id: tabdialog.cc,v 1.3 1998/08/27 04:10:04 jgg Exp $
/* ######################################################################

   Tabbed Dialog - A dialog that has a number of tabs on one side to 
                   represent pages.

   The technique used to layout the dialog and achive the visual effect
   is pretty simple. Each tab exists as it's own widget with one side 
   missing. By having the tab overlap the page by one pixel we can 
   control the position of the blank space quite easially using ToFront
   and ToBack. Each page exists as a separate widget that is delay 
   realized. When the page is made active the old page is unrealized
   and the new one is realized in it's pace.
   
   ##################################################################### */
									/*}}}*/
// Include Files							/*{{{*/
#ifdef __GNUG__
#pragma implementation "deity/tabdialog.h"
#endif  
#include <deity/tabdialog.h>
#include <iostream.h>
									/*}}}*/

// TabDialog::TabDialog - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
TabDialog::TabDialog(Widget *Parent) : BasicWidget(Parent),
           CurPage(0), PageList(0), iLocation(Top),
           iEFlags(FullWidth)
{
   BorderWidth(0);
}
									/*}}}*/
// TabDialog::TabDialog - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
TabDialog::~TabDialog()
{
   while (PageList != 0)
   {
      Page *Tmp = PageList;
      PageList = Tmp->NextPage;
      delete Tmp;
   }
}
									/*}}}*/
// TabDialog::Resize - Resize member					/*{{{*/
// ---------------------------------------------------------------------
/* We make the child pane track our size as well as controlling the size
   of the tabs */
void TabDialog::Resize(Rect NewSize)
{
   bool ReLayout = false;
   if (NewSize.w != NewSize.w || NewSize.h != NewSize.h)
      ReLayout = true;
   
   BasicWidget::Resize(NewSize);
   
   LayoutTabs();
   if (CurPage != 0)
   {
      CurPage->PageBase->Resize(PageLoc);
      CurPage->PageBase->Damage();
   }   
}
									/*}}}*/
// TabDialog::Realize - Realize the widget				/*{{{*/
// ---------------------------------------------------------------------
/* Because all of the pages are not realized with the parent we must realize
   the current active page here. */
void TabDialog::Realize()
{
   LayoutTabs();
   BasicWidget::Realize();
   CurPage->PageBase->RealizeFamily();
}
									/*}}}*/
// TabDialog::LayoutTabs - Position the tabs				/*{{{*/
// ---------------------------------------------------------------------
/* The meaning of the two layout flags is a bit odd, FullWidth only 
   has meaning for horizontal layout and will cause the tabs to occupy
   the whole width of the dialog. Same size causes the X size of the
   tabs to be the same for both horiz and vertical layout. */
void TabDialog::LayoutTabs()
{
   Point P = CanvasSize();
   PageLoc = Rect(0,0,P.x,P.y);

   long Overlap = 0;
   if (GraphicGC::GC != 0)
      Overlap = 1;
   
   /* Extent all the tabs and determine the largest size and
      total size in both directions */
   Rect MaxSz;
   for (Page *I = PageList; I != 0; I = I->NextPage)
   {
      Point Sz = I->IdealSize();
      MaxSz.x += Sz.x;
      if (Sz.x > MaxSz.w)
	 MaxSz.w = Sz.x;
      MaxSz.y += Sz.y;
      if (Sz.y > MaxSz.h)
	 MaxSz.h = Sz.y;
   }
   
   // Vertical layout
   if (iLocation == Left || iLocation == Right)
   {
      // Make room
      if (iLocation == Right)
      {
	 PageLoc.w -= MaxSz.w;
      }
      else
      {
	 PageLoc.x += MaxSz.w - Overlap;
	 PageLoc.w -= MaxSz.w - Overlap;
      }
      
      // Reposition
      long CurY = 0;
      for (Page *I = PageList; I != 0; I = I->NextPage)
      {
	 Point P = I->IdealSize();
	 Rect Cur = Rect(0,0,P.x,P.y);
	 
	 // Make all tabs the same size if desired
	 if ((iEFlags & SameSize) == SameSize)
	    Cur.w = MaxSz.w;
	 
	 // Position the tab
	 Cur.y = CurY;
	 if (iLocation == Right)
	    Cur.x = PageLoc.w + PageLoc.x;
	 else
	    Cur.x = MaxSz.w - Cur.w;
	 I->Resize(Cur);
	 
	 CurY += Cur.h;
      }

      // Add an extra bit so the page overlaps with the tabs
      if (iLocation == Right)
	 PageLoc.w += Overlap;
   }   
   
   // Horizontal layout
   if (iLocation == Top || iLocation == Bottom)
   {
      // Make room
      if (iLocation == Bottom)
      {
	 PageLoc.h -= MaxSz.h;
      }
      else
      {
	 PageLoc.y += MaxSz.h - Overlap;
	 PageLoc.h -= MaxSz.h - Overlap;
      }

      // Compute a scaling factor for 
      float Scale = 1;
      if ((iEFlags & FullWidth) == FullWidth)
	 Scale = PageLoc.w/float(MaxSz.x);
      
      // Reposition
      long CurX = 0;
      long RealX = 0;
      for (Page *I = PageList; I != 0; I = I->NextPage)
      {
	 Point P = I->IdealSize();
	 Rect Cur = Rect(0,0,P.x,P.y);
	 
	 // Make all tabs the same width if desired
	 RealX += Cur.w;
	 if ((iEFlags & SameSize) == SameSize)
	    Cur.w = MaxSz.w;
	 else
	    Cur.w = int(RealX*Scale) - CurX;

	 // There is some rounding error in the scaling..
	 if (I->NextPage == 0)
	    Cur.w = PageLoc.w - CurX;
	 
	 // Position the tab
	 Cur.x = CurX;
	 if (iLocation == Top)
	    Cur.y = 0;
	 else
	    Cur.y = PageLoc.y + PageLoc.h;
	 I->Resize(Cur);
	 CurX += Cur.w;
      }

      // Add an extra bit so the page overlaps with the tabs
      if (iLocation == Bottom)
	 PageLoc.h += Overlap;
   }

   // Resize the base page widget.
   if (CurPage != 0)
      CurPage->PageBase->Resize(PageLoc);
}
									/*}}}*/
// TabDialog::GotoPage - Jump to the given page				/*{{{*/
// ---------------------------------------------------------------------
/* */
void TabDialog::GotoPage(Page *Target)
{
   if (Target == 0)
      return;
   
   if (CurPage != 0)
      CurPage->PageBase->UnRealize();

   CurPage = Target;
   CurPage->PageBase->Resize(PageLoc);
   CurPage->PageBase->RealizeFamily();
   CurPage->PageBase->ToFront();
   CurPage->ToFront();
}
									/*}}}*/

// TabDialog::Page::Page - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* This automatically links the widget into the parents list and 
   if it is the first widget causes it to be made active. */
TabDialog::Page::Page(TabDialog *Owner) : BasicWidget(Owner), 
                 NextPage(0)
{
   LastWidget = PageBase = new BasicWidget(Owner);
   if (GraphicGC::GC != 0)
      PageBase->BorderWidth(1);
   if (TextGC::GC != 0)
      PageBase->BorderWidth(0);
   
   PageBase->Flag(DelayRealize);

   // Link us into the parents list, at the end.
   Page **Last = &Owner->PageList;
   for (Page *I = Owner->PageList; I != 0; I = I->NextPage)
      Last = &I->NextPage;
   *Last = this;

   // Become the active page if we are the first inserted page.
   ToBack();
   if (Owner->PageList == this)
      Owner->GotoPage(this);
}
									/*}}}*/
// TabDialog::Page::~Page - Destructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
TabDialog::Page::~Page()
{
   delete PageBase;
}
									/*}}}*/

// TabPage::TabPage - Constructor					/*{{{*/
// ---------------------------------------------------------------------
/* */
TabPage::TabPage(string Test,TabDialog *Owner) : TabDialog::Page(Owner), 
           iString(Test)
{
   if (TextGC::GC != 0)
      BorderWidth(0);
   Pos.x = -1;
}
									/*}}}*/
// TabPage::IdealSize - Compute the widgets size			/*{{{*/
// ---------------------------------------------------------------------
/* */
Point TabPage::IdealSize()
{
   GenGC::GC->SetFont(iFont);
   Rect Sz = GenGC::GC->ExtentText(iString);
   return Point(2*(iMargins.x + BorderX) + Sz.w,
		2*(iMargins.y + BorderY) + Sz.h);
}
									/*}}}*/
// TabPage::Render - Draw the widget					/*{{{*/
// ---------------------------------------------------------------------
/* The side which the pane is on will have a blank space drawn for a 
   border. */
void TabPage::Render(CombinedGC &GC)
{
   TabDialog *Owner = (TabDialog *)Parent;
   
   if (GC.IsGraphic() == true)
   {
      AbsRect Fill = AbsRect(0,0,Pos.w,Pos.h);
      TabDialog::TabLocation Loc = Owner->Location();
      
      // Draw the upper left side
      GC->SetColor(iBorderUl);
      if (Loc != TabDialog::Bottom)
      {
	 GC.gGC->Line(Point(0,0),Point(Pos.w - 1,0));
	 Fill.y1++;
      }
      if (Loc != TabDialog::Right)
      {
	 GC.gGC->Line(Point(0,0),Point(0,Pos.h - 1));
	 Fill.x1++;
      }
      
      // Draw the lower right side
      GC->SetColor(iBorderLr);
      if (Loc != TabDialog::Top)
      {
	 GC.gGC->Line(Point(Pos.w - 1,Pos.h - 1),Point(0,Pos.h - 1));
	 Fill.y2--;
      }
      if (Loc != TabDialog::Left)
      {
	 GC.gGC->Line(Point(Pos.w - 1,Pos.h - 1),Point(Pos.w - 1,0));
	 Fill.x2--;
      }
   }
   
   GC->SetColor(iColor);
   GC->Background(iBackground);
   GC->DrawString(Rect(BorderX,0,
		       Pos.w - 2*BorderX,
		       Pos.h - BorderY),Point(Pos.w/2,Pos.h/2 + BorderX),
		  iString,GenGC::XCenter | GenGC::YCenter);
}
									/*}}}*/
// TabPage::Mouse - Mouse click handler					/*{{{*/
// ---------------------------------------------------------------------
/* */
void TabPage::Mouse(const MouseEvent &Event)
{
   if (Event.IsClick() == true)
      Select();
}
									/*}}}*/
