/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"
#include <dirent.h>


#define	PROC_SENSORS_DIR	"/proc/sys/dev/sensors"

typedef struct
	{
	gchar		*name;
	gchar		*temp_file;
	gfloat		factor;			/* Scale sensor tempx reading	*/
	gfloat		offset;			/* Add to sensor tempx reading	*/

	GtkWidget	*entry;
	GtkWidget	*factor_spin_button;
	GtkWidget	*offset_spin_button;
	}
	Sensor;

Sensor	*sensors			= NULL;

GList	*temp_file_glist	= NULL;
gint	n_temp_files;


gchar *
chip_sensor(gchar *sensor_file)
	{
	gchar	*s;

	s = sensor_file + strlen(PROC_SENSORS_DIR);
	if (*s == '/')
		return s + 1;
	return "";
	}

  /* Associate a name (mb, cpu, cpu0, cpu1, ...) with chip/temperature
  |  file (eg w83782d-i2c-0-2d/temp1).
  */
static Sensor *
link_temperature_name(gchar *name, gchar *temp)
	{
	GList	*list;
	Sensor	*sensor;
	gchar	*s;

	for (list = temp_file_glist; list; list = list->next)
		{
		sensor = (Sensor *) list->data;
		if (sensor->temp_file)
			{
			s = chip_sensor(sensor->temp_file);
			if (*s != '\0' && strcmp(s, temp) == 0)
				{
				if (sensor->name)
					g_free(sensor->name);
				sensor->name = g_strdup(name);
				return sensor;
				}
			}
		}
	return NULL;
	}

gint
read_temperature(gchar *name)
	{
	FILE	*f;
	GList	*list;
	Sensor	*sensor;
	gfloat	temp	= 0;

	list = temp_file_glist;
	for (list = temp_file_glist; list; list = list->next)
		{
		sensor = (Sensor *) list->data;
		if (sensor->name && strcmp(sensor->name, name) == 0)
			{
			if ((f = fopen(sensor->temp_file, "r")) != NULL)
				{
				fscanf(f, "%*f %*f %f", &temp);
				fclose(f);
				temp = temp * sensor->factor + sensor->offset;
				if (UC.sensor_temp_fahrenheit)
					temp = 1.8 * temp + 32.0;
				}
			break;
			}
		}
	return (int) temp;
	}

static void
scan_for_temp_files(gchar *chip_dir)
	{
	DIR				*dir;
	struct dirent	*dentry;
	Sensor			*sensor;
	gchar			temp_file[256];

	GK.sensor_temp_files = 0;
	if ((dir = opendir(chip_dir)) == NULL)
		return;
	while ((dentry = readdir(dir)) != NULL)
		{
		if (dentry->d_name[0] != '.' && dentry->d_ino > 0)
			{
			if (strncmp(dentry->d_name, "temp", 4) == 0)
				{
				sprintf(temp_file, "%s/%s", chip_dir, dentry->d_name);
				sensor = g_new0(Sensor, 1);
				sensor->temp_file = g_strdup(temp_file);
				sensor->name = g_strdup("");			/* Simplify free code */
				sensor->factor = 1.0;
				sensor->offset = 0;
				temp_file_glist = g_list_append(temp_file_glist, sensor);
				++GK.sensor_temp_files;
				}
			}
		}
	closedir(dir);
	}

void
register_sensors()
	{
	DIR				*dir;
	struct dirent	*dentry;
	gchar			chip_dir[256];

	if ((dir = opendir(PROC_SENSORS_DIR)) == NULL)
		return;
	while ((dentry = readdir(dir)) != NULL)
		{
		if (dentry->d_name[0] != '.' && dentry->d_ino > 0)
			{
			sprintf(chip_dir, "%s/%s", PROC_SENSORS_DIR, dentry->d_name);
			if (isdir(chip_dir))
				scan_for_temp_files(chip_dir);
			}
		}
	closedir(dir);
	}


void
write_sensor_config(FILE *f)
	{
	GList	*list;
	Sensor	*s;
	gfloat	factor;

	for (list = temp_file_glist; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (s->name && *(s->name) != '\0')
			{
			if ((factor = s->factor) < 0.1)
				factor = 1.0;
			fprintf(f, "sensor %s %s %f %f\n", s->name,
					chip_sensor(s->temp_file), factor, s->offset);
			}
		}
	}

void
load_sensor_config(gchar *arg)
	{
	gchar	*name, *chip, *factor, *offset;
	Sensor	*s;
	gchar	buf[96];

	strcpy(buf, arg);
	name = strtok(buf, " \t\n");
	if (name == NULL)
		return;
	chip = strtok(NULL, " \t\n");

	if (   chip == NULL
		|| (s = link_temperature_name(name, chip)) == NULL
	   )
		return;

	offset = NULL;
	factor = strtok(NULL, " \t\n");
	if (factor)
		offset = strtok(NULL, " \t\n");
	s->factor = factor ? atof(factor) : 1.0;
	s->offset = offset ? atof(offset) : 0;
	}


/* ------------------------------------------------------------------- */
static GtkWidget	*fahrenheit_button;


  /* Apply sensor tab widget values to the UC and to temp_file_glist entries.
  |  apply_stat_config() will handle applying what is updated here to the
  |  cpu and proc panels.
  */
void
apply_sensors_config()
	{
	GList	*list;
	Sensor	*s;
	gchar	*name;
	gfloat	factor, offset;

	if (!GK.sensor_temp_files)
		return;
	for (list = temp_file_glist; list; list = list->next)
		{
		s = (Sensor *) list->data;
		name = gtk_entry_get_text(GTK_ENTRY(s->entry));
		factor = gtk_spin_button_get_value_as_float(
						GTK_SPIN_BUTTON(s->factor_spin_button));
		offset = gtk_spin_button_get_value_as_float(
						GTK_SPIN_BUTTON(s->offset_spin_button));
		if (strcmp(s->name, name) != 0)
			{
			if (s->name)
				g_free(s->name);
			s->name = g_strdup(name);
			}
		if (s->factor != factor)
			s->factor = factor;
		if (s->offset != offset)
			s->offset = offset;
		}
	UC.sensor_temp_fahrenheit = GTK_TOGGLE_BUTTON(fahrenheit_button)->active;
	}

void
create_sensors_tab(GtkWidget *vbox)
	{
	GtkWidget		*table;
	GtkWidget		*spinbutton;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;
	GtkWidget		*separator;
	GtkWidget		*label;
	GtkWidget		*entry;
	GList			*list;
	Sensor			*s;
	gchar			buf[256];
	gchar			*string;
	gint			i;

	table = gtk_table_new (GK.sensor_temp_files + 2, 4, FALSE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 3);

	label = gtk_label_new("Sensor Label");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

	label = gtk_label_new(PROC_SENSORS_DIR);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);
	label = gtk_label_new("Temperature Files");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 1, 2);

	label = gtk_label_new("Temperature Correction");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 4, 0, 1);
	label = gtk_label_new("Factor");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 1, 2);
	label = gtk_label_new("Offset");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 3, 4, 1, 2);

	for (i = 2, list = temp_file_glist; list; list = list->next, ++i)
		{
		s = (Sensor *) list->data;

		entry = gtk_entry_new_with_max_length(8);
		s->entry = entry;
		if (s->name)
			gtk_entry_set_text(GTK_ENTRY(entry), s->name);
		gtk_widget_set_usize(entry, 36, 0);
		gtk_table_attach_defaults(GTK_TABLE(table), entry, 0, 1, i, i + 1);

		label = gtk_label_new(chip_sensor(s->temp_file));
		gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, i, i + 1);

		adj = (GtkAdjustment *) gtk_adjustment_new(s->factor,
					0.5, 2.0, 0.01 /*step*/, 0.1 /*page*/, 0.0);
		spinbutton = gtk_spin_button_new (adj, 0.5, 2 /*digits*/);
		s->factor_spin_button = spinbutton;
		gtk_widget_set_usize(spinbutton, 36, 0);
		spin = GTK_SPIN_BUTTON(spinbutton);
		gtk_spin_button_set_numeric(spin, TRUE);
		gtk_table_attach_defaults(GTK_TABLE(table), spinbutton, 2, 3, i, i+1);

		adj = (GtkAdjustment *) gtk_adjustment_new(s->offset,
					-50.0, 50.0, 1.0 /*step*/, 5.0 /*page*/, 0.0);
		spinbutton = gtk_spin_button_new (adj, 0.5, 0 /*digits*/);
		s->offset_spin_button = spinbutton;
		gtk_widget_set_usize(spinbutton, 36, 0);
		spin = GTK_SPIN_BUTTON(spinbutton);
		gtk_spin_button_set_numeric(spin, TRUE);
		gtk_table_attach_defaults(GTK_TABLE(table), spinbutton, 3, 4, i, i+1);
		}

	if (smp_cpus == 0)
		string = "\"cpu\" or \"mb\"";
	else if (smp_cpus == 1)
		string = "\"cpu0\" or \"mb\"";
	else if (smp_cpus == 2)
		string = "\"cpu0\", \"cpu1\", or \"mb\"";
	else
		string = "\"cpu0\", \"cpu1\"..., or \"mb\"";

	sprintf(buf,
			"Sensor Labels can be %s for a motherboard\n"
			"temp which will show on the Proc panel.  See the README file.\n"
			"Temperature Correction Offset units are centigrade.",
			string);

	label = gtk_label_new(buf);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);

	fahrenheit_button = gtk_check_button_new_with_label("Display fahrenheit");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(fahrenheit_button),
			UC.sensor_temp_fahrenheit);
	gtk_box_pack_start(GTK_BOX(vbox), fahrenheit_button, TRUE, TRUE, 0);
	}

