/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"
#include <pwd.h>

#include	"pixmaps/frame_horizontal.xpm"
#include	"pixmaps/frame_side.xpm"
#include	"pixmaps/bg_spacer.xpm"
#include	"pixmaps/bg_bordered.xpm"
#include	"pixmaps/bg_chart.xpm"
#include	"pixmaps/chart_grid.xpm"

  /* These data images are used only for the default theme
  */
#include	"pixmaps/data_in.xpm"
#include	"pixmaps/data_out.xpm"
#include	"pixmaps/data_in_etch.xpm"
#include	"pixmaps/data_out_etch.xpm"

#include	"pixmaps/net_leds.xpm"
#include	"pixmaps/decals.xpm"
#include	"pixmaps/ppp_button.xpm"

#include	"pixmaps/krell_panel.xpm"
#include	"pixmaps/krell_meter.xpm"

#include	"pixmaps/krell_meter_mail.xpm"
#include	"pixmaps/decal_mail.xpm"

#ifdef KRELL_METER_FS
#include	"pixmaps/krell_meter_fs.xpm"
#endif

  /* I do a hoop jump to get these installed without
  |  compromising theme image heirarchy behavior.
  */
#include	"pixmaps/bg_panel.xpm"
#include	"pixmaps/bg_meter_fs.xpm"
#include	"pixmaps/bg_timer.xpm"

/* fonts */
/* -adobe-courier-medium-r-normal-*-*-100-*-*-m-*-iso8859-1 */
/* -adobe-helvetica-medium-r-normal-*-*-80-*-*-p-*-iso8859-1 */


gchar *
homedir()
	{
	static gchar	*homedir;
	struct passwd	*pw;

	if (   homedir == NULL
		&& (homedir = getenv("HOME")) == NULL
	   )
		{
		if ((pw = getpwuid(getuid())))
			homedir = pw->pw_dir;
		}
	return homedir;
	}

gint
isdir(char *s)
	{
	struct stat st;

	if (stat(s, &st) == 0 && S_ISDIR(st.st_mode))
		return 1;
	return 0;
	}

static gchar	*image_type[] =
	{
	".png", ".jpg", ".xpm", ".gif"
	};

gchar *
theme_file_exists(char *base)
	{
	gint			i;
	static gchar	pathname[80];
	struct stat		st;

	if (*(GK.theme_path) == '\0')
		return NULL;

	for (i = 0; i < sizeof(image_type) / sizeof(char *); ++i)
		{
		sprintf(pathname, "%s/%s%s", GK.theme_path, base, image_type[i]);
		if (   stat(pathname, &st) == 0
			&& (S_ISREG(st.st_mode) || S_ISLNK(st.st_mode))
		   )
			return pathname;
		}
	return NULL;
	}


void
set_border(GdkImlibBorder *border, char *string)
	{
	gchar			*s;
	static gchar	*delim = " ,\t\n";

	if (string == NULL || border == NULL)
		{
		printf("set_border() NULL pointer.\n");
		return;
		}
	if (GK.debug)
		printf("set_border(%s)\n", string);
	if ((s = strtok(string, delim)) != NULL)
		{
		border->left = atoi(s);
		if ((s = strtok(NULL, delim)) != NULL)
			{
			border->right = atoi(s);
			if ((s = strtok(NULL, delim)) != NULL)
				{
				border->top = atoi(s);
				if ((s = strtok(NULL, delim)) != NULL)
					border->bottom = atoi(s);
				}
			}
/*		gdk_imlib_set_image_border(im, border); */
		}
	}

  /* This routine needs work.  Must clean it up with tables when I have time.
  */
void
set_style(gchar *sty)
	{
	Style	**style, *st, *st_default;
	gchar	*name, *val, *class, *type;

	type = NULL;
	name = NULL;
	style = NULL;
	class = strtok(sty, ". \t\n");	/* chart, meter, or uptime	*/
	if (class)
		type = strtok(NULL, ". =:\t\n");	/* cpu, disk, net; mem, swap */
	if (class == NULL || type == NULL)	/* or may be name			*/
		return;

	if (strcmp(class, "chart") == 0)
		style = &GK.chart_style[0];
	else if (strcmp(class, "meter") == 0)
		style = &GK.meter_style[0];

	if (style == NULL)
		{
		printf("NULL style for class <%s> dude!\n", class);
		return;
		}
	st_default = *style;

	if (strcmp(type, "cpu") == 0)
		style += CPU_IMAGE;
	else if (strcmp(type, "disk") == 0)
		style += DISK_IMAGE;
	else if (strcmp(type, "net") == 0)
		style += NET_IMAGE;
	else if (strcmp(type, "proc") == 0)
		style += PROC_IMAGE;
	else if (strcmp(type, "inet") == 0)
		style += INET_IMAGE;
	else if (strcmp(type, "mem") == 0)
		style += MEM_IMAGE;
	else if (strcmp(type, "swap") == 0)
		style += SWAP_IMAGE;
	else if (strcmp(type, "fs") == 0)
		style += FS_IMAGE;
	else if (strcmp(type, "apm") == 0)
		style += APM_IMAGE;
	else if (strcmp(type, "mail") == 0)
		style += MAIL_IMAGE;
	else
		{
		name = type;
		type = "xxx";
		}

	if (name == NULL)
		name = strtok(NULL, " =:\t\n");

	if (*style == NULL)
		{
		if (GK.debug)
			printf("Allocating style\n");
		*style = (Style *) g_new0(Style, 1);
		**style = *st_default;		/* Initialize	*/
		}
	val = NULL;
	if (name)
		val = strtok(NULL, " =:\t\n");
	if (name == NULL || val == NULL)
		{
		printf("Can't get style name(%s) or value(%s) for %s.\n",
			name == NULL ? "NULL" : name, val == NULL ? "NULL": val, class);
		return;
		}
	if (GK.debug)
		printf("Setting style for %s.%s.%s='%s'\n", class, type,
					name, val);
	if (strcmp(name, "label") == 0)
		{
		if (strcmp(val, "center") == 0)
			(*style)->label_position = LABEL_CENTER;
		else if (isdigit(*val))
			(*style)->label_position = atoi(val);
		else
			(*style)->label_position = LABEL_NONE;
		if ((*style)->label_position > LABEL_MAX)
			(*style)->label_position = LABEL_MAX;
		}
	if (strcmp(name, "krell_enable") == 0)
		{
		if (strcmp(val, "false") == 0)
			(*style)->krell_enable = FALSE;
		else
			(*style)->krell_enable = TRUE;
		}
	if (strcmp(name, "krell_yoff") == 0)
		(*style)->krell_yoff = atoi(val);
	if (strcmp(name, "krell_expand") == 0)
		{
		if (strcmp(val, "left") == 0)
			(*style)->krell_expand = LEFT;
		else if (strcmp(val, "right") == 0)
			(*style)->krell_expand = RIGHT;
		else
			(*style)->krell_expand = NONE;
		}
	if (strcmp(name, "krell_x_hot") == 0)
		(*style)->krell_x_hot = atoi(val);
	if (strcmp(name, "krell_ema_period") == 0)
		(*style)->krell_ema_period = atoi(val);
	if (strcmp(name, "krell_depth") == 0)
		(*style)->krell_depth = atoi(val);

	st = *style;
	if (strcmp(name, "panel_border") == 0)
		set_border(&st->border_panel, val);
	if (strcmp(name, "chart_border") == 0)
		set_border(&st->border_chart, val);
	}


static struct	_images
	{
	gchar			*name;
	gchar			**xpm;
	GdkImlibImage	**im;
	}
	images[]	=
	{
	{ "frame_horizontal", frame_horizontal_xpm, &GK.frame_horizontal_image },
	{ "frame_top",		NULL,				&GK.frame_top_image },
	{ "frame_bottom",	NULL,				&GK.frame_bottom_image },
	{ "frame_side", 	frame_side_xpm, 	&GK.frame_side_image },
	{ "frame_left",  	NULL,				&GK.frame_left_image },
	{ "frame_right", 	NULL, 				&GK.frame_right_image },

	{ "bg_spacer",		bg_spacer_xpm,		&GK.bg_spacer_image },
	{ "bg_bordered",	bg_bordered_xpm,	&GK.bg_bordered_image },

	{ "chart_grid", 	chart_grid_xpm,		&GK.chart_grid_image[0] },
	{ "chart_grid_cpu",	NULL,			&GK.chart_grid_image[CPU_IMAGE] },
	{ "chart_grid_disk", NULL,			&GK.chart_grid_image[DISK_IMAGE] },
	{ "chart_grid_net",	NULL,			&GK.chart_grid_image[NET_IMAGE] },
	{ "chart_grid_proc",NULL,			&GK.chart_grid_image[PROC_IMAGE] },
	{ "chart_grid_inet",NULL,			&GK.chart_grid_image[INET_IMAGE] },

	{ "bg_chart",	 	bg_chart_xpm,		&GK.bg_chart_image[0] },
	{ "bg_chart_cpu", 	NULL,				&GK.bg_chart_image[CPU_IMAGE] },
	{ "bg_chart_disk", 	NULL,				&GK.bg_chart_image[DISK_IMAGE] },
	{ "bg_chart_net", 	NULL,				&GK.bg_chart_image[NET_IMAGE] },
	{ "bg_chart_proc", 	NULL,				&GK.bg_chart_image[PROC_IMAGE] },
	{ "bg_chart_inet", 	NULL,				&GK.bg_chart_image[INET_IMAGE] },

	/* bg_ images with no default.  If there is no theme image present for
	|  them, they will be initialized to the basic bg_spacer or bg_border.
	|  In theory, that is.  I do a hoop jump to get around this.
	*/
	{ "bg_panel",		NULL,			&GK.bg_panel_image[0] },
	{ "bg_panel_cpu",	NULL,			&GK.bg_panel_image[CPU_IMAGE] },
	{ "bg_panel_disk",	NULL,			&GK.bg_panel_image[DISK_IMAGE] },
	{ "bg_panel_net",	NULL,			&GK.bg_panel_image[NET_IMAGE] },
	{ "bg_panel_proc",	NULL,			&GK.bg_panel_image[PROC_IMAGE] },
	{ "bg_panel_inet",	NULL,			&GK.bg_panel_image[INET_IMAGE] },

	{ "bg_meter",		NULL,			&GK.bg_meter_image[0] },
	{ "bg_meter_mem",	NULL,			&GK.bg_meter_image[MEM_IMAGE] },
	{ "bg_meter_swap",	NULL,			&GK.bg_meter_image[SWAP_IMAGE] },
	{ "bg_meter_fs",	NULL,			&GK.bg_meter_image[FS_IMAGE] },
	{ "bg_meter_apm",	NULL,			&GK.bg_meter_image[APM_IMAGE] },
	{ "bg_meter_mail",	NULL,			&GK.bg_meter_image[MAIL_IMAGE] },

	{ "bg_timer",		NULL,			&GK.bg_timer_image },

	{ "ppp_button",		ppp_button_xpm,	&GK.ppp_button_image },
	{ "net_leds",		net_leds_xpm,	&GK.led_image },
	{ "decals",			decals_xpm,		&GK.decal_image },
	{ "decal_mail",		decal_mail_xpm,	&GK.decal_mail_image },

	{ "data_in",	 	NULL,			&GK.data_in_image },
	{ "data_in_etch", 	NULL,			&GK.data_in_etch_image },
	{ "data_out",	 	NULL,			&GK.data_out_image },
	{ "data_out_etch", 	NULL,			&GK.data_out_etch_image },

	{ "krell_panel",	krell_panel_xpm,	&GK.krell_panel_image[0] },
	{ "krell_panel_cpu",  NULL,			&GK.krell_panel_image[CPU_IMAGE] },
	{ "krell_panel_disk", NULL,			&GK.krell_panel_image[DISK_IMAGE] },
	{ "krell_panel_net",  NULL,			&GK.krell_panel_image[NET_IMAGE] },
	{ "krell_panel_proc", NULL,			&GK.krell_panel_image[PROC_IMAGE] },
	{ "krell_panel_inet", NULL,			&GK.krell_panel_image[INET_IMAGE] },

	{ "krell_meter",	krell_meter_xpm,	&GK.krell_meter_image[0] },
	{ "krell_meter_mem",  NULL,			&GK.krell_meter_image[MEM_IMAGE] },
	{ "krell_meter_swap", NULL,			&GK.krell_meter_image[SWAP_IMAGE] },
	{ "krell_meter_fs",	  NULL,			&GK.krell_meter_image[FS_IMAGE] },
	{ "krell_meter_apm",  NULL,			&GK.krell_meter_image[APM_IMAGE] },

	/* Mail monitor is not a pure meter, but is set up like one and has
	|  its own krell for mail notification animation.
	*/
	{ "krell_meter_mail", krell_meter_mail_xpm,
								&GK.krell_meter_image[MAIL_IMAGE] }

	};


static void
check_images()
	{
	gint	i;

	for (i = 0; i < N_CHART_IMAGES; ++i)
		{
		if (GK.chart_grid_image[i] == NULL)
			printf("chart_grid_image[%d] is NULL\n", i);
		if (GK.bg_chart_image[i] == NULL)
			printf("bg_chart_image[%d] is NULL\n", i);
		if (GK.bg_panel_image[i] == NULL)
			printf("bg_panel_image[%d] is NULL\n", i);
		if (GK.krell_panel_image[i] == NULL)
			printf("krell_panel_image[%d] is NULL\n", i);
		}
	for (i = 0; i < N_METER_IMAGES; ++i)
		{
		if (GK.bg_meter_image[i] == NULL)
			printf("bg_meter_image[%d] is NULL\n", i);
		if (GK.krell_meter_image[i] == NULL)
			printf("krell_meter_image[%d] is NULL\n", i);
		}
	if (GK.frame_horizontal_image == NULL)
			printf("frame_horizontal_image is NULL\n");
	if (GK.led_image == NULL)
			printf("net_leds_image is NULL\n");
	if (GK.decal_image == NULL)
			printf("decal_image is NULL\n");
	if (GK.decal_mail_image == NULL)
			printf("decal_mail_image is NULL\n");
	}


  /* I need a back door so my default theme can have a little more stuff
  |  than just the base level background images.  But I need to make it
  |  as if no sublevels have been specified.
  */
static gint		new_bg_spacer, new_bg_timer, new_bg_bordered;

void
jump_through_hoop(char *name)
	{
	gchar	*s;

	if ((s = strrchr(name, '.')) != NULL)
		*s = '\0';
	if (strcmp(name, "bg_spacer") == 0)
		new_bg_spacer = TRUE;
	if (strcmp(name, "bg_bordered") == 0)
		new_bg_bordered = TRUE;
	if (strcmp(name, "bg_timer") == 0)
		new_bg_timer = TRUE;
	}

void
setup_images()
	{
	gint	i, w, h;

	if (GK.trace)
		printf("setup_images()\n");

	/* If no new top level theme images are being used, it is safe to
	|  stick in my sub level images so I can get a custom look.
	*/
	if (! new_bg_spacer)  /* Use my custom bg_panel */
		{
		if (GK.bg_panel_image[FS_IMAGE] == NULL)
			GK.bg_panel_image[0] =
				gdk_imlib_create_image_from_xpm_data(bg_panel_xpm);
		if (   (! new_bg_timer && GK.ppp_border_mode == 0)
			&& (! new_bg_bordered || GK.ppp_border_mode > 0)
			&& GK.bg_timer_image == NULL
		   )
			GK.bg_timer_image =
				gdk_imlib_create_image_from_xpm_data(bg_timer_xpm);
		}
	if (! new_bg_bordered)
		{
		if (GK.bg_meter_image[FS_IMAGE] == NULL)
			GK.bg_meter_image[FS_IMAGE] =
				gdk_imlib_create_image_from_xpm_data(bg_meter_fs_xpm);
#ifdef KRELL_METER_FS
/* If I want to have a custom fs krell - not yet */
		if (GK.krell_meter_image[FS_IMAGE] == NULL)
			GK.krell_meter_image[FS_IMAGE] =
				gdk_imlib_create_image_from_xpm_data(krell_meter_fs_xpm);
#endif
		}

	if (GK.frame_left_image == NULL)
		GK.frame_left_image = gdk_imlib_clone_image(GK.frame_side_image);
	if (GK.frame_right_image == NULL)
		GK.frame_right_image = gdk_imlib_clone_image(GK.frame_side_image);
	gdk_imlib_set_image_border(GK.frame_left_image, &GK.frame_side_border);
	gdk_imlib_set_image_border(GK.frame_right_image, &GK.frame_side_border);

	/* Since frame_horizontal is split, I handle frame top/bottom when used.*/

	if (GK.bg_panel_image[0] == NULL)
		GK.bg_panel_image[0] = gdk_imlib_clone_image(GK.bg_spacer_image);
	if (GK.bg_meter_image[0] == NULL)
		GK.bg_meter_image[0] = gdk_imlib_clone_image(GK.bg_bordered_image);
	/* If bg_timer_image is NULL, it stays NULL */

	if (GK.trace)
		printf("  borders/styles...\n");
	gdk_imlib_set_image_border(GK.bg_panel_image[0],
			&GK.chart_style[0]->border_panel);
	gdk_imlib_set_image_border(GK.bg_chart_image[0],
			&GK.chart_style[0]->border_chart);
	for (i = 1; i < N_CHART_IMAGES; ++i)
		{
		if (GK.chart_style[i] == NULL)
			GK.chart_style[i] = GK.chart_style[0];

		if (GK.chart_grid_image[i] == NULL)
			GK.chart_grid_image[i] = GK.chart_grid_image[0];

		if (GK.bg_chart_image[i] == NULL)
			GK.bg_chart_image[i] = GK.bg_chart_image[0];
		else
			gdk_imlib_set_image_border(GK.bg_chart_image[i],
					&GK.chart_style[i]->border_chart);

		if (GK.bg_panel_image[i] == NULL)
			GK.bg_panel_image[i] = GK.bg_panel_image[0];
		else
			gdk_imlib_set_image_border(GK.bg_panel_image[i],
					&GK.chart_style[i]->border_panel);

		if (GK.krell_panel_image[i] == NULL)
			GK.krell_panel_image[i] = GK.krell_panel_image[0];
		}

	gdk_imlib_set_image_border(GK.bg_meter_image[0],
					&GK.meter_style[0]->border_panel);
	for (i = 1; i < N_METER_IMAGES; ++i)
		{
		if (GK.meter_style[i] == NULL)
			GK.meter_style[i] = GK.meter_style[0];

		if (GK.bg_meter_image[i] == NULL)
			GK.bg_meter_image[i] = GK.bg_meter_image[0];
		else
			gdk_imlib_set_image_border(GK.bg_meter_image[i],
						&GK.meter_style[i]->border_panel);

		if (GK.krell_meter_image[i] == NULL)
			GK.krell_meter_image[i] = GK.krell_meter_image[0];
		}

	gdk_imlib_set_image_border(GK.bg_bordered_image, &GK.bordered_bg_border);
	if (GK.bg_timer_image)  /* Is legal for this to be NULL */
		gdk_imlib_set_image_border(GK.bg_timer_image, &GK.timer_border);
	if (GK.frame_top_image)
		gdk_imlib_set_image_border(GK.frame_top_image, &GK.frame_top_border);
	if (GK.frame_bottom_image)
		gdk_imlib_set_image_border(GK.frame_bottom_image,
								&GK.frame_bottom_border);

	w = GK.led_image->rgb_width;
	if (GK.allow_scaling && GK.chart_width_ref != UC.chart_width)
		w = w * UC.chart_width / GK.chart_width_ref;
	gdk_imlib_render(GK.led_image, w, GK.led_image->rgb_height);
	GK.led_pixmap = gdk_imlib_copy_image(GK.led_image);
	GK.led_mask = gdk_imlib_copy_mask(GK.led_image);

	GK.led_width = ((GdkWindowPrivate *) GK.led_pixmap)->width;
	GK.led_height = ((GdkWindowPrivate *) GK.led_pixmap)->height / N_LEDS;
	if (GK.debug)
		printf("net_leds w=%d h=%d\n", GK.led_width, GK.led_height);

	gdk_imlib_render(GK.decal_image,
					GK.decal_image->rgb_width, GK.decal_image->rgb_height);
	GK.decal_pixmap = gdk_imlib_copy_image(GK.decal_image);
	GK.decal_mask = gdk_imlib_copy_mask(GK.decal_image);
	GK.decal_width = ((GdkWindowPrivate *) GK.decal_pixmap)->width;
	GK.decal_height =
			((GdkWindowPrivate *) GK.decal_pixmap)->height / N_DECALS;

	gdk_imlib_render(GK.decal_mail_image,
			GK.decal_mail_image->rgb_width, GK.decal_mail_image->rgb_height);
	GK.decal_mail_pixmap = gdk_imlib_copy_image(GK.decal_mail_image);
	GK.decal_mail_mask = gdk_imlib_copy_mask(GK.decal_mail_image);
	GK.decal_mail_width = ((GdkWindowPrivate *) GK.decal_mail_pixmap)->width;
	GK.decal_mail_height =
	((GdkWindowPrivate *) GK.decal_mail_pixmap)->height / GK.decal_mail_frames;

	/* If default theme, use my data in/out imagees, else use in/out colors
	|  unless new theme has its own in/out images.
	*/
	if (*(GK.theme_path) == '\0')
		{
		GK.data_in_image = gdk_imlib_create_image_from_xpm_data(data_in_xpm);
		GK.data_out_image = gdk_imlib_create_image_from_xpm_data(data_out_xpm);
		GK.data_in_etch_image =
				gdk_imlib_create_image_from_xpm_data(data_in_etch_xpm);
		GK.data_out_etch_image =
				gdk_imlib_create_image_from_xpm_data(data_out_etch_xpm);
		}
	for (i = 0, h = 0; i < N_CHART_MONITORS; ++i)
		if (h < UC.chart_height[i])
			h = UC.chart_height[i];
	w = UC.chart_width;
	GK.max_chart_height = h;
	if (GK.data_in_image)
		{
		gdk_imlib_render(GK.data_in_image, w, h);
		GK.data_in_pixmap = gdk_imlib_copy_image(GK.data_in_image);
		}
	if (GK.data_out_image)
		{
		gdk_imlib_render(GK.data_out_image, w, h);
		GK.data_out_pixmap = gdk_imlib_copy_image(GK.data_out_image);
		}
	if (GK.data_in_etch_image)
		{
		if ((h = GK.data_in_etch_image->rgb_height) > 2)
			h = 2;
		gdk_imlib_render(GK.data_in_etch_image, w, h);
		GK.data_in_etch_pixmap = gdk_imlib_copy_image(GK.data_in_etch_image);
		}
	if (GK.data_out_etch_image)
		{
		if ((h = GK.data_out_etch_image->rgb_height) > 2)
			h = 2;
		gdk_imlib_render(GK.data_out_etch_image, w, h);
		GK.data_out_etch_pixmap = gdk_imlib_copy_image(GK.data_out_etch_image);
		}
	check_images();
	if (GK.trace)
		printf("  <-\n");
	}

void
load_images()
	{
	gchar	*fname;
	gint	i;

	if (GK.trace)
		printf("load_images()\n");
	for (i = 0; i < sizeof(images) / sizeof(struct _images); ++i)
		{
		if ((fname = theme_file_exists(images[i].name)) == NULL)
			{
			if (images[i].xpm != NULL)
				*(images[i].im) =
						gdk_imlib_create_image_from_xpm_data(images[i].xpm);
			}
		else
			{
			if (GK.debug)
				printf("Using theme image '%s'\n", fname);
			*(images[i].im) = gdk_imlib_load_image(fname);
			jump_through_hoop(images[i].name);
			}
		if (*(images[i].im) == NULL && images[i].xpm != NULL)
			printf("Cannot load image %s\n", images[i].name);
		}

	setup_images();

	if (GK.trace)
		printf("  <-\n");
    }


static gchar
            *timer_border,
			*frame_top_border,
			*frame_bottom_border,
			*frame_side_border,
			*bordered_bg_border;

#define OLD_BORDERS

#ifdef OLD_BORDERS
/* These will go away after conversion to new style settings. */
static gchar *meter_border,
            *chart_border,
            *panel_border;
#endif

struct	_config
	{
	gchar	*option;
	gint	*value;
	gchar	**arg;
	gint	minimum;
	}
	config []	=
	{
	{"frame_width",			&GK.frame_width,			NULL,			2  },
	{"chart_width_min",		&GK.chart_width_min,		NULL,			30 },
	{"chart_width_max",		&GK.chart_width_max,		NULL,			45 },
	{"chart_width_ref",		&GK.chart_width_ref,		NULL,			30 },
	{"chart_height_min",	&GK.chart_height_min,		NULL,			15 },
	{"chart_height_max",	&GK.chart_height_max,		NULL,			20 },
	{"allow_scaling",		&GK.allow_scaling,			NULL,			0 },

	{"rx_led_x",			&GK.rx_led_x,				NULL,			-16 },
	{"rx_led_y",			&GK.rx_led_y,				NULL,			0   },
	{"tx_led_x",			&GK.tx_led_x,				NULL,			-16 },
	{"tx_led_y",			&GK.tx_led_y,				NULL,			0   },

	{"decal_mail_frames",	&GK.decal_mail_frames,	NULL,			0  },
	{"decal_mail_delay",	&GK.decal_mail_delay,	NULL,			1  },

	{"ppp_border_mode",		&GK.ppp_border_mode,			NULL,		0 },

	{"large_font",			NULL,		&GK.large_font_string,		-100 },
	{"label_font",			NULL,		&GK.label_font_string,		-100 },
	{"alt_font",			NULL,		&GK.alt_font_string,		-100 },
	{"label_draw_effect",	NULL,		&GK.label_effect_string,	-16 },
	{"meter_draw_effect",	NULL,		&GK.meter_effect_string,	-16 },
	{"time_draw_effect",	NULL,		&GK.time_effect_string,	-16 },
	{"alt1_draw_effect",	NULL,		&GK.alt1_effect_string,		-16 },
	{"alt2_draw_effect",	NULL,		&GK.alt2_effect_string,		-16 },

	{"label_color",			NULL,		&GK.label_color_string,		-100 },
	{"meter_color",			NULL,		&GK.meter_color_string,		-100 },
	{"time_color",			NULL,		&GK.time_color_string,	-100 },
	{"alt1_color",			NULL,		&GK.alt1_color_string,		-100 },
	{"alt2_color",			NULL,		&GK.alt2_color_string,		-100 },

	{"chart_in_color",		NULL,		&GK.chart_in_color,		-100 },
	{"chart_in_color_grid",	NULL,		&GK.chart_in_color_grid,	-100 },
	{"chart_out_color",		NULL,		&GK.chart_out_color,		-100 },
	{"chart_out_color_grid",NULL,		&GK.chart_out_color_grid,	-100 },

	{"chart_grid_mode",		&GK.chart_grid_mode,		NULL,			0  },
    {"timer_border",		NULL,		&timer_border,				-100 },
    {"bordered_border",		NULL,		&bordered_bg_border,		-100 },
    {"frame_top_border",	NULL,		&frame_top_border,			-100 },
    {"frame_bottom_border",	NULL,		&frame_bottom_border,		-100 },
    {"frame_side_border",	NULL,		&frame_side_border,			-100 },


#ifdef OLD_BORDERS
    {"meter_border",        NULL,       &meter_border,              -100 },
    {"label_border",        NULL,       &panel_border,              -100 },
    {"panel_border",        NULL,       &panel_border,              -100 },
    {"chart_border",        NULL,       &chart_border,              -100 },

#endif

	{"style",				NULL,		NULL,						-100 }

	};


#ifdef OLD_BORDERS
void
fix_border(GdkImlibBorder *b, char *values)
	{
	set_border(b, values);
	}

void
cleanup_old_borders()
	{
	/* XXX Fixme, these 1st 2 are not old. */
    if (bordered_bg_border)
        fix_border(&GK.bordered_bg_border, bordered_bg_border);
    if (timer_border)
        fix_border(&GK.timer_border, timer_border);
    if (frame_top_border)
        fix_border(&GK.frame_top_border, frame_top_border);
    if (frame_bottom_border)
        fix_border(&GK.frame_bottom_border, frame_bottom_border);
    if (frame_side_border)
        fix_border(&GK.frame_side_border, frame_side_border);

    if (meter_border)
        fix_border(&GK.meter_style[0]->border_panel, meter_border);
    if (panel_border)
        fix_border(&GK.chart_style[0]->border_panel, panel_border);
    if (chart_border)
        fix_border(&GK.chart_style[0]->border_chart, chart_border);
	}
#endif

  /* This routine need to go away.  Use parse_config_file() instead.
  */
void
parse_gkrellmrc(FILE *f)
	{
	gchar	*s, *arg;
	gint	i, val, foundit;
	gchar	buf[120], pbuf[120];

	if (GK.trace)
		printf("parse_gkrellmrc()\n");
	while (fgets(buf, sizeof(buf), f))
		{
		s = strtok(buf, " \t:=\n");
		if (s == NULL || *s == '#' || *s == '\n' || *s == '\0')
			continue;

		if (strcmp(s, "style") == 0)
			{
			arg = strtok(NULL, "\n#");
			set_style(arg);
			continue;
			}
		arg = strtok(NULL, " \t:=\n");

		/* Must allow '#' char for arguments - color setting.
		*/
		if (arg == NULL || *arg == '\n' || *arg == '\0')
			continue;
		if (isdigit(*arg) || *arg == '-')
			val = atoi(arg);
		else
			val = 0;
		if (strcmp(arg, "off") == 0 || strcmp(arg, "false") == 0)
			val = 0;
		if (strcmp(arg, "on") == 0 || strcmp(arg, "true") == 0)
			val = 1;

		foundit = FALSE;
		for (i = 0; i < sizeof(config) / sizeof(struct _config); ++i)
			{
			if (strcmp(config[i].option, s) == 0)
				{
				if (config[i].value)
					*(config[i].value) = val;
				if (config[i].arg)
					{
					/* String args can have quotes around them.
					*/
					if (*arg == '"')
						{
						strcpy(pbuf, arg + 1);
						arg = strtok(NULL, "#\n");	/* Get rest of line */
						if (arg)
							{
							strcat(pbuf, " ");
							strcat(pbuf, arg);
							}
						arg = strchr(pbuf, (int) '"');
						if (arg)
							*arg = '\0';
						*(config[i].arg) = g_strdup(pbuf);
						}
					else
						*(config[i].arg) = g_strdup(arg);
					}
				if (config[i].minimum > -100 && val < config[i].minimum)
					*(config[i].value) = config[i].minimum;
				foundit = TRUE;
				break;
				}
			}
		if (!foundit && GK.debug)
			printf("Ugh! cannot grok %s in gkrellmrc.\n", s);
		}
#ifdef OLD_BORDERS
	cleanup_old_borders();
#endif
	if (GK.trace)
		printf("  <-\n");

	}

void
parse_config_file(FILE *f, struct _config *cf, gint size)
	{
	struct _config	*pc;
	gchar			*s, *arg;
	gint			i, val, foundit;
	gchar			buf[120], pbuf[120], xbuf[120];

	if (GK.trace)
		printf("parse_config_file()\n");
	while (fgets(buf, sizeof(buf), f))
		{
		s = strtok(buf, " \t:=\n");
		if (s == NULL || *s == '#' || *s == '\n' || *s == '\0')
			continue;

		/* Special case diversions. XXX Put a function call in the table!!
		*/
		if (strcmp(s, "style") == 0)
			{
			arg = strtok(NULL, "\n#");
			set_style(arg);
			continue;
			}
		if (strcmp(s, "net_resolution") == 0)
			{
			arg = strtok(NULL, "\n#");
			load_net_config(arg);
			continue;
			}
		if (strcmp(s, "sensor") == 0)
			{
			arg = strtok(NULL, "\n#");
			load_sensor_config(arg);
			continue;
			}
		if (strcmp(s, "inet") == 0)
			{
			arg = strtok(NULL, "\n#");
			load_inet_config(arg);
			continue;
			}
		if (strcmp(s, "fs") == 0)
			{
			arg = strtok(NULL, "\n#");
			load_fs_config(arg);
			continue;
			}
		if (strcmp(s, "mail") ==0)
		        {
			arg = strtok(NULL, "\n#");
			load_mail_config(arg);
			continue;
			}
		arg = strtok(NULL, " \t:=\n");

		/* Must allow '#' char for arguments - color setting.
		*/
		if (arg == NULL || *arg == '\n' || *arg == '\0')
			continue;
		if (isdigit(*arg) || *arg == '-')
			val = atoi(arg);
		else
			val = 0;
		if (strcmp(arg, "off") == 0 || strcmp(arg, "false") == 0)
			val = 0;
		if (strcmp(arg, "on") == 0 || strcmp(arg, "true") == 0)
			val = 1;

		foundit = FALSE;
		for (i = 0; i < size; ++i)
			{
			pc = cf + i;
			if (strcmp(pc->option, s) == 0)
				{
				if (pc->value)
					*(pc->value) = val;
				if (pc->arg)
					{
					/* Backwards compatibility check for old non quoted
					|  ppp commands.  Take me out soon.
					*/
					if (strncmp(s, "ppp", 3) == 0 && *arg != '"')
						{
						sprintf(xbuf, "\"%s", arg);
						arg = xbuf;
						}

					/* String args can have quotes around them.
					*/
					if (*arg == '"')
						{
						strcpy(pbuf, arg + 1);
						arg = strtok(NULL, "#\n");	/* Get rest of line */
						if (arg)
							{
							strcat(pbuf, " ");
							strcat(pbuf, arg);
							}
						arg = strchr(pbuf, (int) '"');
						if (arg)
							*arg = '\0';
						*(pc->arg) = g_strdup(pbuf);
						}
					else
						*(pc->arg) = g_strdup(arg);
					}
				if (pc->minimum > -100 && val < pc->minimum)
					*(pc->value) = pc->minimum;
				foundit = TRUE;
				if (GK.debug)
					printf("parse_config %s %s\n", s, arg);
				break;
				}
			}
		if (GK.debug && !foundit)
			printf("Ugh! parse_config cannot grok %s.\n", s);
		}
	if (GK.trace)
		printf("  <-\n");
	}

void
user_config_defaults()
	{
	gint	i;

	for (i = 0; i < N_MONITORS; ++i)
		UC.enable[i] = TRUE;

	UC.enable_hostname = TRUE;
	UC.hostname_short = FALSE;
	UC.enable_clock = TRUE;
	UC.enable_mailcheck = FALSE;
	UC.enable_net_timer = TRUE;
	UC.net_timer_is_ppp = TRUE;

	UC.clock_24hr_format = FALSE;
	UC.clock_12hr_seconds = FALSE;

	UC.chart_width = 60;
	for (i = 0; i < N_CHART_MONITORS; ++i)
		UC.chart_height[i] = 36;

	UC.super_mute_mailcheck = FALSE;
	UC.smp_mode = 0;

	UC.proc_load_bar_graph = FALSE;
	UC.proc_processes_enable = TRUE;
	UC.load_resolution = 100;
	UC.proc_clip_processes = TRUE;

	UC.disk_resolution = 50;
	UC.enable_auto_eject = FALSE;

	/* Default net resolution is set in net.c because the default will
	|  be function of bytes or packets units.
	*/

	UC.ppp_on_command = g_strdup("pon");
	UC.ppp_off_command = g_strdup("poff");

	UC.ema_period = 4;
	UC.update_HZ = 10;
/*	UC.bump_HZ = 1; */
	}

void
config_defaults()
	{
	GK.frame_width = 4;
	GK.chart_width_min = 50;
	GK.chart_width_max = 200;
	GK.chart_height_min = 15;
	GK.chart_height_max = 150;
	GK.chart_width_ref = 60;
	GK.allow_scaling = TRUE;

	GK.rx_led_x = 2;
	GK.rx_led_y = 6;
	GK.tx_led_x = -2;
	GK.tx_led_y = 6;

	GK.decal_mail_frames = 18;
	GK.decal_mail_delay = 1;

	GK.ppp_border_mode = 0;

/*  A nice oblique variation */
/*	"-adobe-courier-medium-o-normal-*-*-100-*-*-m-*-iso8859-1" */

	GK.large_font_string =
			"-adobe-courier-medium-r-normal-*-*-120-*-*-m-*-iso8859-1";
	GK.label_font_string =
			"-adobe-courier-medium-r-normal-*-*-100-*-*-m-*-iso8859-1";
	GK.alt_font_string =
			"-adobe-helvetica-medium-r-normal-*-*-80-*-*-p-*-iso8859-1";

	GK.label_effect_string = "shadow";
	GK.meter_effect_string = "shadow";
	GK.time_effect_string = "shadow";
	GK.alt1_effect_string = "shadow";
	GK.alt2_effect_string = "shadow";

	GK.label_color_string = "#ede6ef";
	GK.meter_color_string = "#f8b8a0";
	GK.time_color_string = "#ffeac4";
	GK.alt1_color_string = "#99e38e";
	GK.alt2_color_string = "#efd097";
	
	GK.chart_in_color = "#10d3d3";
	GK.chart_in_color_grid = "#00a3a3";
	GK.chart_out_color = "#f4ac4a";
	GK.chart_out_color_grid = "#b47c20";

	/* Setup the default styles.  Substyles may be set in gkrellmrc.  If
	|  they are not, then they will be set to point to the default after
	|  parsing the gkrellmrc file.
	*/
	GK.chart_style[0] = (Style *) g_new0(Style, 1);
	GK.chart_style[0]->label_position = LABEL_CENTER;
	GK.chart_style[0]->krell_enable = TRUE;
	GK.chart_style[0]->krell_yoff   = 0;
	GK.chart_style[0]->krell_expand = NONE;
	GK.chart_style[0]->krell_x_hot  = 3;
	GK.chart_style[0]->krell_ema_period = 4;
	GK.chart_style[0]->krell_depth  = 4;
	GK.chart_style[0]->border_panel.left   = 0;
	GK.chart_style[0]->border_panel.right  = 0;
	GK.chart_style[0]->border_panel.top    = 3;
	GK.chart_style[0]->border_panel.bottom = 1;
	GK.chart_style[0]->border_chart.left   = 0;
	GK.chart_style[0]->border_chart.right  = 0;
	GK.chart_style[0]->border_chart.top    = 0;
	GK.chart_style[0]->border_chart.bottom = 0;

	GK.meter_style[0] = (Style *) g_new0(Style, 1);
	*GK.meter_style[0] = *GK.chart_style[0];	/* same as chart except for: */
	GK.meter_style[0]->krell_x_hot  = -1;		/* Half of pixmap width */
	GK.meter_style[0]->krell_yoff   = 1;
	GK.meter_style[0]->krell_depth  = 1;
	GK.meter_style[0]->krell_ema_period  = 1;
	GK.meter_style[0]->border_panel.left = 3;
	GK.meter_style[0]->border_panel.right = 3;
	GK.meter_style[0]->border_panel.top = 3;
	GK.meter_style[0]->border_panel.bottom = 3;

	GK.meter_style[FS_IMAGE] = (Style *) g_new0(Style, 1);
	*GK.meter_style[FS_IMAGE] = *GK.meter_style[0];
	GK.meter_style[FS_IMAGE]->label_position = 0;	/* Left justify */
	GK.meter_style[FS_IMAGE]->border_panel.top = 3;
	GK.meter_style[FS_IMAGE]->border_panel.bottom = 2;

	GK.meter_style[APM_IMAGE] = (Style *) g_new0(Style, 1);
	*GK.meter_style[APM_IMAGE] = *GK.meter_style[0];
	GK.meter_style[APM_IMAGE]->border_panel.top = 2;
	GK.meter_style[APM_IMAGE]->border_panel.bottom = 2;

	GK.meter_style[MAIL_IMAGE] = (Style *) g_new0(Style, 1);
	*GK.meter_style[MAIL_IMAGE] = *GK.meter_style[0];
	GK.meter_style[MAIL_IMAGE]->label_position = 70;
	GK.meter_style[MAIL_IMAGE]->krell_yoff   = 0;
	GK.meter_style[MAIL_IMAGE]->krell_depth  = 15;

	GK.bordered_bg_border = GK.meter_style[0]->border_panel;
	GK.timer_border  = GK.meter_style[0]->border_panel;
	}

void
gkrellmrc_config()
	{
	FILE	*f;
	gchar	gkrellmdir[80], path[80], theme_path[80], *s;

	if (GK.trace)
		printf("gkrellmrc_config()\n");
	register_sensors();
	config_defaults();
	user_config_defaults();

	sprintf(gkrellmdir, "%s/%s", homedir(), GKRELLM_DIR);
	if (!isdir(gkrellmdir))
		{
		if (mkdir(gkrellmdir, 0755) < 0)
			printf("Cannot create gkrellm directory %s\n", gkrellmdir);
		}

	if (*(GK.theme_path) == '\0')	/* Could have been command line set */
		{
		sprintf(path, "%s/%s", homedir(), GKRELLM_THEME);
		if ((f = fopen(path, "r")) != NULL)
			{
			fgets(path, sizeof(path), f);
			fclose(f);
			s = strtok(path, " \n\t");
			if (s && *s != '#' && *s != '\0')
				{
				if (*s == '/')
					sprintf(theme_path, "%s", s);
				else
					sprintf(theme_path, "%s/%s", gkrellmdir, s);
				GK.theme_path = g_strdup(theme_path);

				/* A theme is requested, so try config file from the theme_path
				*/
				sprintf(path, "%s/%s", GK.theme_path, GKRELLMRC);
				if ((f = fopen(path, "r")) != NULL)
					{
					parse_gkrellmrc(f);
					fclose(f);
					}
				}
			}
		}
	else
		{
		sprintf(path, "%s/%s", GK.theme_path, GKRELLMRC);
		if ((f = fopen(path, "r")) != NULL)
			{
			parse_gkrellmrc(f);
			fclose(f);
			}
		}
	if (GK.debug)
		{
		if (GK.theme_path == NULL)
			printf("gkrellm using default skin.\n");
		else
			printf("gkrellm using theme directory <%s>\n", GK.theme_path);
		}
	if (GK.trace)
		printf("  <-\n");
	}


/* --------------------------------------------------------------*/

struct	_config	user_config[] =
	{
	{"enable_proc",		&UC.enable[MON_PROC],	NULL,				0  },
	{"enable_disk",		&UC.enable[MON_DISK],	NULL,				0  },

	{"enable_inet",		&UC.enable[MON_INET],	NULL,				0  },
	{"enable_net_timer", &UC.enable_net_timer,	NULL,				0  },
	{"net_timer_is_ppp", &UC.net_timer_is_ppp,	NULL,				0  },

	{"enable_hostname",	&UC.enable_hostname,	NULL,				0  },
	{"enable_clock",	&UC.enable_clock,		NULL,				0  },
	{"clock_24hr",		&UC.clock_24hr_format,	NULL,				0  },
	{"clock_12hr_seconds",	&UC.clock_12hr_seconds,	NULL,			0  },
	{"enable_mailcheck",    &UC.enable_mailcheck,   NULL,			0  },
	{"hostname_short",	&UC.hostname_short,		NULL,				0  },
	{"sensor_temp_units", &UC.sensor_temp_fahrenheit, NULL,			0  },

	{"enable_extra_info", &UC.enable_extra_info, NULL,				0  },
	{"save_position",	&UC.save_position,		NULL,				0  },
	{"on_top",			&UC.on_top,				NULL,				0  },
	{"fixed_scale",		&UC.fixed_scale,		NULL,				0  },

	{"chart_width",		&UC.chart_width,			NULL,			0  },
	{"height_cpu",		&UC.chart_height[MON_CPU],	NULL,			0  },
	{"height_proc",		&UC.chart_height[MON_PROC],	NULL,			0  },
	{"height_disk",		&UC.chart_height[MON_DISK],	NULL,			0  },
	{"height_net",		&UC.chart_height[MON_NET],	NULL,			0  },
	{"height_inet",		&UC.chart_height[MON_INET],	NULL,			0  },

	{"enable_auto_eject", &UC.enable_auto_eject,	NULL,			0  },

	{"super_mute_mailcheck", &UC.super_mute_mailcheck, NULL,		0  },
	{"smp_chart_mode",	&UC.smp_mode,			NULL,				0  },
	{"ema_period",		&UC.ema_period,			NULL,				0  },
	{"update_HZ",		&UC.update_HZ,			NULL,				0  },
/*	{"bump_HZ",			&UC.bump_HZ,			NULL,				0  }, */

	{"ppp_on_command",		NULL,				&UC.ppp_on_command,	0 },
	{"ppp_off_command",		NULL,				&UC.ppp_off_command, 0 },

	{"disk_resolution",	&UC.disk_resolution,	NULL,				0  },
	{"load_bar_graph",	&UC.proc_load_bar_graph, NULL,				0  },
	{"clip_processes",	&UC.proc_clip_processes, NULL,				0  },

	};

gint
read_user_config()
	{
	FILE	*f;
	gchar	config_file[128];

	if (GK.trace)
		printf("read_user_config()\n");
	sprintf(config_file, "%s/%s", homedir(), GKRELLM_USER_CONFIG);
	if ((f = fopen(config_file, "r")) == NULL)
		{
		if (GK.debug)
			printf("Cannot open config file %s for reading.\n", config_file);
		return FALSE;
		}
	parse_config_file(f, &user_config[0],
				sizeof(user_config) / sizeof (struct _config));
	fclose(f);

	if (UC.chart_width < GK.chart_width_min)
		UC.chart_width = GK.chart_width_min;
	if (UC.chart_width > GK.chart_width_max)
		UC.chart_width = GK.chart_width_max;

	if (GK.trace)
		printf("  <-\n");

	return TRUE;
	}

void
save_user_config()
	{
	FILE	*f;
	gint	i;
	gchar	config_file[128];

	sprintf(config_file, "%s/%s", homedir(), GKRELLM_USER_CONFIG);
	if ((f = fopen(config_file, "w")) == NULL)
		{
		printf("Cannot open config file %s for writing.\n", config_file);
		return;
		}
	fprintf(f, "### GKrellM user config.  Auto written, do not edit ###\n");
	fprintf(f, "### Version %d.%d.%d ###\n",
			VERSION_MAJOR, VERSION_MINOR, VERSION_REV);
	for (i = 0; i < sizeof(user_config) / sizeof(struct _config); ++i)
		{
		if (user_config[i].value)
			fprintf(f, "%s %d\n", user_config[i].option,
							*(user_config[i].value));
		else if (user_config[i].arg)	/* Put quotes around strings */
			fprintf(f, "%s \"%s\"\n",user_config[i].option,
						*(user_config[i].arg));
		}
	/* Net.c keeps track of its own grid resolutions.
	*/
	write_net_config(f);
	write_sensor_config(f);
	write_inet_config(f);
	write_fs_config(f);
	write_mail_config(f);
	fclose(f);
	}
