
/*  
   1998 Roberto Alameda <roberto@myokay.net>
  You may modify and distribute this file under the terms of the GNU
  General Public License, version 2, or any later version, at your
  convenience. See the file COPYING for details. 

*/

#include <config.h>

#if HAVE_UNISTD_H
# include <sys/types.h>
# include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <limits.h>
#include <errno.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>


#include "gfont.h"
#include "t1enc.h"


extern GdkVisual *visual;
extern GdkGC *bwGC, *defGC, *redGC;

extern GtkWidget *mainwindow;
extern GtkWidget *samplet, *stbar, *prbar;
extern GtkStyle  *stylebold;
extern GdkPixmap *emptypixmap;

extern bool kerning, antialiasing;
extern bool enable_antialiasing;

extern int xresolution, yresolution;


// Window Info structure 
// Contains all data for the fonttable to work
struct WInfo {
  FontData *fd;
  double fontsize;
  GdkPixmap *pixmap;
  GtkWidget *drawing_area, *info_label, *option_menu;
  GtkItemFactory *item_factory;
  int pixmapwidth, pixmapheight;
  guint antialiasing:1;  // Whether the fonts in this window are smoothed
  guint char_codes:1;    // Whether the character codes are shown
};


// TTInfo is used in the properties window
struct TTInfo {
  GtkWidget *property, *value;
};


// A linked list of all open fonttable windows
static GSList *winfolist = NULL;


// ************** Function prototypes (internally used in this module)
static Callback destroy_fonttable, newencoding, showallglyphs;
static GdkImage* gdk_image_t1bitmap(char* data, int width, int height);
static GdkImage* gdk_image_t1pixmap(char* data, int width, int height);
static BBox getfontbbox(int fontID);
static char* readline(char* line, int size, FILE* fp);
static GtkWidget* showglyph(GtkWindowType type, GLYPH* glyph, int fontID, 
			    int width, int height);
static GtkWidget* aashowglyph(GtkWindowType type, GLYPH* glyph, int fontID, 
			      int width, int height);
static gint drawarea_event(GtkWidget *widget, GdkEvent *event, gpointer);
static void fill_pixmap(WInfo *winfo);
static WInfo* drawfont_inframe(GtkWidget *frame, FontData *fd, double size);

static guint32 getblocklength(FILE *fp);
static bool t1_downloadpfb(FILE *ifp, FILE *ofp);

static void make_entries(GtkWidget *frame, struct TTInfo *props, int num);
static GtkWidget* general_info(FontData *fd);








// ********************* Begin of program functions

const char* GetT1error(int err)
{
  static GString* mes = NULL;
  if (mes == NULL) mes = g_string_new(NULL);

  switch(err) {
    // Type 1 Font File Scan Errors
#ifdef T1ERR_SCAN_FONT_FORMAT
  case T1ERR_SCAN_FONT_FORMAT:
    g_string_assign(mes, "Multiple master fonts cannot be handled");
    break;
#endif
  case T1ERR_SCAN_FILE_OPEN_ERR:
  case T1ERR_FILE_OPEN_ERR: 
    g_string_assign(mes, "Cannot open font file: ");
    g_string_append(mes, g_strerror(errno));
    break;
  case T1ERR_SCAN_OUT_OF_MEMORY:
    g_string_assign(mes, "Font uses too much memory");
    break;
  case T1ERR_SCAN_ERROR:
  case T1ERR_SCAN_FILE_EOF:
  case T1ERR_PARSE_ERROR:
    g_string_assign(mes, "Syntactical error in font file");
    break;
    
    //Path Generation Errors
#ifdef T1ERR_TYPE1_ABORT
  case T1ERR_TYPE1_ABORT:
#endif
  case T1ERR_PATH_ERROR:
    g_string_assign(mes, "Error in the font");
    break;
    
    // T1lib Errors
  case T1ERR_INVALID_FONTID:
    g_string_assign(mes, "Invalid font ID");
    break;
  case T1ERR_INVALID_PARAMETER:
    g_string_assign(mes, "Invalid parameter");
    break;
  case T1ERR_OP_NOT_PERMITTED:
    g_string_assign(mes, "Operation not permited");
    break;
  case T1ERR_ALLOC_MEM:
    g_string_assign(mes, "Out of memory");
    break;
  case T1ERR_UNSPECIFIED:
    g_string_assign(mes, "Unspecified error");
    break;
    
  default:
    g_string_sprintf(mes, "Invalid T1lib error code: %d", err);
    break;
  }
  return mes->str;
}



static GdkImage* gdk_image_t1bitmap(char* data, int width, int height)
{
  static GdkImage* dummy = NULL;
  Visual *xvisual;
  GdkImage *image;
  GdkImagePrivate *priv;

  if (!dummy) dummy = gdk_image_new_bitmap(visual, NULL, 1, 1);
  priv = g_new(GdkImagePrivate, 1);
  image = (GdkImage *)priv;
  priv->xdisplay = gdk_display;
  priv->image_put = ((GdkImagePrivate*)dummy)->image_put;  // Only way to set
  image->type   = GDK_IMAGE_NORMAL;
  image->visual = visual;
  image->width  = width;
  image->height = height;
  image->depth  = 1;
  xvisual = ((GdkVisualPrivate*)visual)->xvisual;
  priv->ximage = XCreateImage(priv->xdisplay, xvisual, 1, XYBitmap,
			      0, 0, width, height, T1_GetBitmapPad(), 0);
  priv->ximage->data = data;
  priv->ximage->bitmap_bit_order = LSBFirst;
  priv->ximage->byte_order = LSBFirst;
  image->byte_order = GDK_LSB_FIRST;
  image->mem = priv->ximage->data;
  image->bpl = priv->ximage->bytes_per_line;
  image->bpp = 1;
  return(image);
}



static GdkImage* gdk_image_t1pixmap(char* data, int width, int height)
{
  static GdkImage* dummy = NULL;
  Visual *xvisual;
  GdkImage *image;
  GdkImagePrivate *priv;

  if (!dummy) dummy = gdk_image_new_bitmap(visual, NULL, 1, 1);
  priv  = g_new(GdkImagePrivate, 1);
  image = (GdkImage *)priv;
  priv->xdisplay  = gdk_display;
  priv->image_put = ((GdkImagePrivate*)dummy)->image_put;  // Only way to set
  image->type   = GDK_IMAGE_NORMAL;
  image->visual = visual;
  image->width  = width;
  image->height = height;
  image->depth  = visual->depth;
  xvisual = ((GdkVisualPrivate*)visual)->xvisual;
  priv->ximage = XCreateImage(priv->xdisplay, xvisual, visual->depth, ZPixmap,
			      0, data, width, height, T1_GetBitmapPad(), 0);
  priv->ximage->bitmap_bit_order = LSBFirst;  // Not MSB!
  priv->ximage->byte_order = LSBFirst;
  image->byte_order = GDK_LSB_FIRST;
  image->mem = priv->ximage->data;
  image->bpl = priv->ximage->bytes_per_line;
  image->bpp = visual->depth/8;
  return(image);
}





static GtkWidget* showglyph(GtkWindowType type, GLYPH* glyph, int fontID, 
			    int width, int height)
{
  GtkWidget* window = gtk_window_new(type);
  gtk_container_border_width(GTK_CONTAINER(window), 10);
  gtk_window_set_title(GTK_WINDOW(window), T1_GetFontName(fontID));

  GdkImage* image = gdk_image_t1bitmap(glyph->bits, width, height);
  GtkWidget* gtkimage = gtk_image_new(image, NULL);
  gtk_container_add(GTK_CONTAINER(window), gtkimage);
  glyph->bits=NULL;    /* Since XDestroyImage() frees this also! */

  gtk_signal_connect(GTK_OBJECT(window), "destroy", 
		     (GtkSignalFunc)delete_image, (gpointer)image);
  int hid = gtk_signal_connect_object(GTK_OBJECT(window), "delete_event", 
				      GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_object_set_data(GTK_OBJECT(window), "delete_handler", 
		      GINT_TO_POINTER(hid));
  gtk_object_set_data(GTK_OBJECT(window), "image", (gpointer)image);
  return window;
}




static GtkWidget* aashowglyph(GtkWindowType type, GLYPH* glyph, int fontID, 
			      int width, int height)
{
  GtkWidget* window = gtk_window_new(type);
  gtk_container_border_width(GTK_CONTAINER(window), 10);
  gtk_window_set_title(GTK_WINDOW(window), T1_GetFontName(fontID));

  GdkImage* image = gdk_image_t1pixmap(glyph->bits, width, height);
  GtkWidget* gtkimage = gtk_image_new(image, NULL);
  gtk_container_add(GTK_CONTAINER(window), gtkimage);
  glyph->bits=NULL;    /* Since XDestroyImage() frees this also! */

  gtk_signal_connect(GTK_OBJECT(window), "destroy", 
		     (GtkSignalFunc)delete_image, (gpointer)image);
  int hid = gtk_signal_connect_object(GTK_OBJECT(window), "delete_event", 
				      GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_object_set_data(GTK_OBJECT(window), "delete_handler", 
		      GINT_TO_POINTER(hid));
  gtk_object_set_data(GTK_OBJECT(window), "image", (gpointer)image);
  return window;
}


static BBox getfontbbox(int fontID)
{
  BBox bbox = T1_GetFontBBox(fontID);
  // Check for null box (yes, some fonts have it...)
  if (bbox.urx==0 && bbox.ury==0 && bbox.llx==0 && bbox.lly==0) {
    bbox.urx = bbox.ury = INT_MIN;
    bbox.llx = bbox.lly = INT_MAX;
    // Calculate font BBox by overlapping all character BBoxes
    for (int i=0; i<256; i++) {
      BBox bboxi = T1_GetCharBBox(fontID, (char)i);
      if (bboxi.urx==0 && bboxi.ury==0 && bboxi.llx==0 && bboxi.lly==0) continue;
      if (bboxi.urx > bbox.urx) bbox.urx = bboxi.urx;
      if (bboxi.ury > bbox.ury) bbox.ury = bboxi.ury;
      if (bboxi.llx < bbox.llx) bbox.llx = bboxi.llx;
      if (bboxi.lly < bbox.lly) bbox.lly = bboxi.lly;
    } 
  }
  return bbox;
}



void t1_drawsample(FontData* fd, const char *msg, double size)
{
  int height, width;
  int fontID = fd->t1data.fontID;

  gdk_window_clear(samplet->window);  // Clear old font sample

  GLYPH* glyph = T1_SetString(fontID, (char*)msg, 0, 0, 0, size, NULL);
  if (glyph==NULL) {
    gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL);  
    errormsg("Cannot generate string: %s", GetT1error(T1_errno));
    return;
  }
  height = glyph->metrics.ascent - glyph->metrics.descent;
  width = glyph->metrics.rightSideBearing - glyph->metrics.leftSideBearing;
  if (width == 0 || height == 0) {
    gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL);  
    return;
  }
  GdkImage* image = gdk_image_t1bitmap(glyph->bits, width, height);
  GdkPixmap* pixmap = gdk_pixmap_new(mainwindow->window, width, height, -1);
  gdk_draw_image(pixmap, defGC, image, 0, 0, 0, 0, -1, -1);
  gdk_image_destroy(image);
  glyph->bits = NULL;    /* Since XDestroyImage() frees this also! */

  gtk_pixmap_set(GTK_PIXMAP(samplet), pixmap, NULL);  // Deletes old pixmap too
}



// Behaves like fgets, but treates a shitty DOS \r as a \n
static char* readline(char* line, int size, FILE* fp)
{
  int n = 0;

  while (n<size) {
    int v = fgetc(fp);
    if (v == EOF) return NULL;
    if (v == '\r' || v == '\n') {
      line[n] = '\0';
      break;
    }
    line[n++] = v;
  }
  return line;
}


bool is_t1font (gchar *pathname, FontData *fd)
{
  int ret = FALSE;
  gchar line[128]; // characters per line
  static char name[128];  // characters in font name

  char *filename = strrchr(pathname, '/') + 1;
  /* Look if it has a .pf[ab] extension */
  int len = strlen(pathname);
  if (len<4) return FALSE;
  if (!(pathname[len-4] == '.' &&
      (pathname[len-3] == 'p' || pathname[len-3] == 'P') &&
      (pathname[len-2] == 'f' || pathname[len-2] == 'F') &&
      (pathname[len-1] == 'a' || pathname[len-1] == 'A' ||
       pathname[len-1] == 'b' || pathname[len-1] == 'B'))) return FALSE;
  /* Now get the font name */
    FILE* fontfile = fopen(pathname, "r");
    if (fontfile == NULL) {
      add_error(fd, "Cannot open file: %s", g_strerror(errno));
      return FALSE;
    }
    /* first line should have the magic message */
    if (readline(line, sizeof(line), fontfile) == NULL) {
      fclose(fontfile);
      return FALSE;
    }    
    // Try as ASCII font (pfa)
    if (!(strstr(line, "%!PS-AdobeFont-1") || strstr(line, "%!FontType1"))) {
      // Try as binary font (pfb)
      rewind(fontfile);
      for (int i=0, rc=1; i<6 && rc!=EOF; i++) rc = fgetc(fontfile);  // Skip 6 bytes
      if (readline(line, sizeof(line), fontfile) == NULL) {
	fclose(fontfile);
	return FALSE;
      }    
      if (!(strstr(line, "%!PS-AdobeFont-1") || strstr(line, "%!FontType1"))) {
	add_error(fd, "Font has no valid ID string");
	fclose(fontfile);
	return FALSE;
      }
    }
    // Look for FontName
    while(readline(line, sizeof(line), fontfile)) {
      if (sscanf(line, " /FontName /%s", name) == 1) {
	ret = TRUE;
	break;
      }
    }
    fclose(fontfile);
    if (ret == FALSE) {
      add_error(fd, "Name of the font not found in file");
      return FALSE;
    }
    if ((fd->t1data.fontID=T1_AddFont(filename))<0) {
      add_error(fd, "Cannot add font to t1lib");
      return FALSE;
    }    
    fd->fontName = g_strdup(name);
    fd->num_glyphs = -1;  // Not computed
    fd->t1data.encoding = NULL;  // Default encoding
    fd->fontType = T1FONT;
    return TRUE;
}




void t1_showstring(FontData* fd, const char* string, double size)
{
  unsigned int height, width;
  int modflag = 0;
  GtkWidget* window;
  GLYPH* glyph;
  
  T1_errno=0;
  int fontID = fd->t1data.fontID;
  if (kerning) modflag |= T1_KERNING;
  if (antialiasing) 
    glyph = T1_AASetString(fontID, (char*)string, 0, 0, modflag, size, NULL);
  else 
    glyph = T1_SetString(fontID, (char*)string, 0, 0, modflag, size, NULL);
  if (glyph==NULL) {
    errormsg("Cannot generate string: %s", GetT1error(T1_errno));
    return;
  }
  height = glyph->metrics.ascent - glyph->metrics.descent;
  width = glyph->metrics.rightSideBearing - glyph->metrics.leftSideBearing;
  if (width == 0 || height == 0) {
    errormsg("Image has zero width or height");
    return;
  }

  if (antialiasing) 
    window = aashowglyph(GTK_WINDOW_TOPLEVEL, glyph, fontID, width, height);
  else 
    window = showglyph(GTK_WINDOW_TOPLEVEL, glyph, fontID, width, height);
  gtk_widget_set_name(window, "string window");

  // The delete and destroy handlers of the window are already set

  GtkWidget *popupmenu = make_imagepopupmenu(window);
  gint event_mask = gtk_widget_get_events(window);
  event_mask |= GDK_BUTTON_PRESS_MASK; 
  gtk_widget_set_events(window, event_mask);
  gtk_signal_connect(GTK_OBJECT(window), "button_press_event",
		     GTK_SIGNAL_FUNC(image_clicked), GTK_OBJECT(popupmenu));
  gtk_widget_show_all(window);
}



void t1_showchar(FontData* fd, guint character, double size)
{
  unsigned int height, width;
  GtkWidget* window;
  GLYPH* glyph;

  int fontID = fd->t1data.fontID;
  T1_errno=0;
  if (antialiasing) 
    glyph = T1_AASetChar(fontID, character, size, NULL);
  else 
    glyph = T1_SetChar(fontID, character, size, NULL);
  if (glyph==NULL) {
    errormsg("Cannot generate character: %s", GetT1error(T1_errno));
    return;
  }
  height = glyph->metrics.ascent - glyph->metrics.descent;
  width = glyph->metrics.rightSideBearing - glyph->metrics.leftSideBearing;
  if (width == 0 || height == 0) {
    errormsg("Image has zero width or height");
    return;
  }
  if (antialiasing) 
    window = aashowglyph(GTK_WINDOW_TOPLEVEL, glyph, fontID, width, height);
  else 
    window = showglyph(GTK_WINDOW_TOPLEVEL, glyph, fontID, width, height);
  gtk_widget_set_name(window, "char window");

  // The delete and destroy handlers of the window are already set

  GtkWidget *popupmenu = make_imagepopupmenu(window);
  gint event_mask = gtk_widget_get_events(window);
  event_mask |= GDK_BUTTON_PRESS_MASK; 
  gtk_widget_set_events(window, event_mask);
  gtk_signal_connect(GTK_OBJECT(window), "button_press_event",
		     GTK_SIGNAL_FUNC(image_clicked), GTK_OBJECT(popupmenu));
  gtk_widget_show_all(window);
}
  



static gint drawarea_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gint x, y;
  WInfo *winfo;
  GdkModifierType mask;
  gint state;
  gint width_w, height_w;  // Of the drawing_area window
  gint width, height;      // Of a cell
  gint col, row;  // In drawing_area
  gint chr;       // Character code (0-255)
  gint chrsize, fontID;
  
  static GString *info_string = NULL; // Character names shown below drawing_area
  static GtkWidget *popup = NULL;     // Enlarged character window
  static guint buttonclicked = 0;     // Button causing popup to activate

  if (info_string == NULL) info_string = g_string_new(NULL);

  switch(event->type) {

  case GDK_LEAVE_NOTIFY:
    winfo = (WInfo*)data;
    // Delete info_label if no mouse key pressed
    state = event->crossing.state;
    if (state&GDK_BUTTON1_MASK || state&GDK_BUTTON2_MASK || 
	state&GDK_BUTTON3_MASK)
      return FALSE;  
    g_string_truncate(info_string, 0);
    gdk_window_clear(winfo->info_label->window);  // Delete old info string
    return FALSE;

  case GDK_MOTION_NOTIFY:
    {
    if (event->motion.is_hint)
      gdk_window_get_pointer(event->motion.window, &x, &y, &mask);
    // Do not go further if mouse moved with a mouse key pressed
    if (mask&GDK_BUTTON1_MASK || mask&GDK_BUTTON2_MASK || 
	mask&GDK_BUTTON3_MASK)
      return FALSE;  
    
    winfo = (WInfo*)data;
    width_w = winfo->pixmapwidth; height_w = winfo->pixmapheight;
    fontID = winfo->fd->t1data.fontID;
    width = width_w/16;  height = height_w/16;
    col = x/width; row = y/height;
    if (col<0 || col>=16 || row<0 || row>=16) {
      g_string_truncate(info_string, 0);
      gdk_window_clear(winfo->info_label->window);  // Delete old info string
      return FALSE; // Out of the grid
    }  
    
    chr = col + row*16;
    char *ps_name = T1_GetCharName(fontID, chr);
    if (strcmp(ps_name, info_string->str) == 0) return FALSE; // info_string not changed
    
    g_string_assign(info_string, ps_name);
    gdk_window_clear(winfo->info_label->window);  // Delete old info string
    gdk_draw_string(winfo->info_label->window, 
		    winfo->info_label->style->font, 
		    winfo->info_label->style->fg_gc[GTK_STATE_NORMAL], 
		    2, 2 + winfo->info_label->style->font->ascent, 
		    info_string->str);
    return FALSE; 
    }
    
  case GDK_BUTTON_PRESS:
    {
    if (buttonclicked != 0) return FALSE;  // Do not allow a second parallel click
    winfo = (WInfo*)data;
    width_w = winfo->pixmapwidth;  height_w = winfo->pixmapheight;
    width = width_w/16;  height = height_w/16;
    col = ((int)event->button.x)/width; row = ((int)event->button.y)/height;
    if (col<0 || col>=16 || row<0 || row>=16) return FALSE;
    chr = col + row*16;
    fontID = winfo->fd->t1data.fontID;
    switch (event->button.button) {
    case 1: chrsize = 100; break;
    case 2: chrsize = 200; break;
    case 3: chrsize = 300; break;
    default: // sorry, scrolling wheels don't do anything here 
      return FALSE;
    }
    //printf("Clicked on %s\n", T1_GetCharName(fontID, chr));
    GLYPH *glyph  = T1_SetChar(fontID, chr, chrsize, 0);
    if ((glyph==NULL) ||
	((height=glyph->metrics.ascent-glyph->metrics.descent)==0) ||
	((width=glyph->metrics.rightSideBearing-
	  glyph->metrics.leftSideBearing)==0)) {
      return FALSE;
    }
    buttonclicked = event->button.button;  // Record which button was clicked
    popup = showglyph(GTK_WINDOW_POPUP, glyph, fontID, width, height);
    int pos_x = (int)event->button.x_root;
    int pos_y = (int)event->button.y_root;
    if (pos_x+width > gdk_screen_width()) pos_x -= width;
    if (pos_y+height > gdk_screen_height()) pos_y -= height;
    gtk_widget_set_uposition(popup, pos_x, pos_y);
    gtk_widget_show_all(popup);
    gtk_grab_add(widget);
    return FALSE;
    }

  case GDK_BUTTON_RELEASE:
    if (event->button.button != buttonclicked) return FALSE;
    winfo = (WInfo*)data;
    buttonclicked = 0;
    if (popup) {
      gtk_widget_destroy(popup);
      gtk_widget_destroyed(popup, &popup);  // Sets popup to NULL (2nd argument)
      gtk_grab_remove(widget);
      g_string_truncate(info_string, 0);
      gdk_window_clear(winfo->info_label->window);  // Delete old info string
    }
    return FALSE;

  default: break;
  }
  return FALSE;
}





// Actually draw the contents of the fonttable in the pixmap
static void fill_pixmap(WInfo *winfo)
{
  GdkPixmap *pixmap = winfo->pixmap;
  int width_w = winfo->pixmapwidth;
  int height_w = winfo->pixmapheight;
  int fontID = winfo->fd->t1data.fontID;
  double size = winfo->fontsize;
  int width = width_w/16;  // Dimensions of a cell
  int height = height_w/16;

  GdkFont *numfont = NULL;  // Font for the character codes
  static GString *auxstr = g_string_new(NULL);

  GtkWidget *top = gtk_widget_get_toplevel(winfo->drawing_area);
  if (GTK_WIDGET_VISIBLE(top)) set_window_busy(top, TRUE);
  if (winfo->char_codes) {
    int fsize = height/4;  // Select the font size in pixels relative to the cell size
    g_string_sprintf(auxstr, "-adobe-helvetica-medium-r-narrow-*-%d-*-*-*-p-*-iso8859-1[48_57]", fsize);
    numfont = gdk_font_load(auxstr->str);
    g_string_sprintf(auxstr, "-adobe-helvetica-medium-r-*-*-%d-*-*-*-p-*-iso8859-1[48_57]", fsize);
    if (!numfont) numfont = gdk_font_load(auxstr->str);
    g_string_sprintf(auxstr, "-*-courier-medium-r-*-*-%d-*-*-*-m-*-iso8859-1", fsize);
    if (!numfont) numfont = gdk_font_load(auxstr->str);
  }

  double xscale = xresolution/72.0;
  double yscale = yresolution/72.0;
  BBox fontbbox = getfontbbox(fontID);  // In charspace units
  int maxdescent = (int)(fontbbox.lly*size*yscale/1000.0);  // In pixels
  int maxleft = (int)(fontbbox.llx*size*xscale/1000.0);

  // Make pixmap white
  gdk_draw_rectangle(pixmap, mainwindow->style->white_gc, TRUE, 0, 0, -1, -1);  
  for (int i=1; i<=16; i++) {  // Draw grid
    gdk_draw_line(pixmap, defGC, 0, height*i, width_w, height*i);
    gdk_draw_line(pixmap, defGC, width*i, 0, width*i, height_w);
  }

  for (int j=0, total=16*16; j<16; j++)  // Put glyphs; i=column, j=row
    for (int i=0; i<16; i++) {
      GLYPH* glyph;
      GdkImage *image;
      int chr = 16*j+i;

      gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), double(chr)/total);
      while (gtk_events_pending()) gtk_main_iteration();
      
      // If char is not defined, show a special (stippled) effect
      char *ps_name = T1_GetCharName(fontID, chr);
      if (strcmp(ps_name, ".notdef")==0) {
	gdk_draw_rectangle(pixmap, bwGC, TRUE, i*width, j*height, 
			   width, height);
	continue;
      }
      if (numfont) {
	// Show the char code at the lower right corner
	g_string_sprintf(auxstr, "%d", chr);
	gint w = gdk_string_width(numfont, auxstr->str);
	gdk_draw_string(pixmap, numfont, defGC, (i+1)*width - w - 2, 
			(j+1)*height - numfont->descent, auxstr->str);
      }
      // Show the glyph
      if (winfo->antialiasing) 
	glyph = T1_AASetChar(fontID, chr, size, 0);
      else 
	glyph = T1_SetChar(fontID, chr, size, 0);
      int height_c = glyph->metrics.ascent - glyph->metrics.descent;
      int width_c = glyph->metrics.rightSideBearing - glyph->metrics.leftSideBearing;
      // if there is no such glyph, show a red stippled effect
      if (glyph==NULL || (height_c==0 && width_c==0)) {
	gdk_draw_rectangle(pixmap, redGC, TRUE, i*width, j*height, 
			   width, height);
	continue;
      }
      if (winfo->antialiasing) 
	image = gdk_image_t1pixmap(glyph->bits, width_c, height_c);
      else 
	image = gdk_image_t1bitmap(glyph->bits, width_c, height_c);
      gdk_draw_image(pixmap, bwGC, image, 0, 0, 
		     i*width + glyph->metrics.leftSideBearing - 
		     maxleft + width/12, 
		     (j+1)*height - glyph->metrics.ascent + 
		     maxdescent - height/12,
		     width_c, height_c);
      gdk_image_destroy(image);
      glyph->bits=NULL;    /* Since XDestroyImage() frees this also! */
    }
  gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 0.0);
  if (numfont) gdk_font_unref(numfont);
  if (GTK_WIDGET_VISIBLE(top)) set_window_busy(top, FALSE);
}




// Change option menu in window of winfo according to the font encoding
// If no proper encoding is found in the option menu, do nth
static void sync_optionmenu(WInfo *winfo)
{
  GtkWidget *menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(winfo->option_menu));
  int pos = 0;
  
  // Go thru all menu items looking for one with the proper encoding
  for (GList *itl=GTK_MENU_SHELL(menu)->children; itl; itl=itl->next, pos++) {
    GtkWidget *item = GTK_WIDGET(itl->data);
    GtkWidget *w = GTK_BIN(item)->child;
    if (!w) continue;
    int custompage = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(item), 
							 "custompage"));

    if (custompage == 0) { // Defined encodings
      char **encoding = (char**)gtk_object_get_data(GTK_OBJECT(item), 
						    "encoding");
      if (winfo->fd->t1data.encoding == encoding) {
	gtk_option_menu_set_history(GTK_OPTION_MENU(winfo->option_menu), pos);
	break;
      }
    }
    else  { // Custom encodings
      if (winfo->fd->t1data.encoding &&
	  strcmp(winfo->fd->t1data.encoding[256], GTK_LABEL(w)->label)==0) {
	gtk_option_menu_set_history(GTK_OPTION_MENU(winfo->option_menu), pos);
	break;
      }
    }
  }
}


// Called for each fonttable window to repaint when encoding changed
static void repaintfont(gpointer data1, gpointer data2)
{
  WInfo *winfo = (WInfo*)data1;     // The font table
  FontData *fd = (FontData*)data2;  // The font that changed encoding
  // If window is fonttable of font fd, repaint it
  if (winfo->fd == fd) {
    sync_optionmenu(winfo);
    fill_pixmap(winfo);
    gdk_draw_pixmap(winfo->drawing_area->window, bwGC, winfo->pixmap, 0, 0, 
		    0, 0, winfo->pixmapwidth, winfo->pixmapheight);
  }
}


static void newencoding(GtkWidget *item, gpointer data)
{
  WInfo *winfo = (WInfo*)data;
  FontData *fd = winfo->fd;
  int fontID = fd->t1data.fontID;
  char **newencoding;

  // If encoding is 'Custom x', free all components of encoding vector
  if (fd->t1data.encoding && 
      strncmp(fd->t1data.encoding[256], "Custom", 6) == 0) {
    for (int i=0; i<=256; i++) g_free(fd->t1data.encoding[i]);
    g_free(fd->t1data.encoding);
    fd->t1data.encoding = NULL;
  }

  int custompage = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(item), 
						       "custompage"));
  if (custompage == 0) { // Defined encodings
    newencoding = (char**)gtk_object_get_data(GTK_OBJECT(item), "encoding");
  }
  else {
    // Show all glyphs with a custom encoding
    custompage--;
    int num_glyphs = -1;
    char **names = T1_GetAllCharNames(fontID);
    while (names[++num_glyphs]);  // Calculate size of ptr array (last element is NULL)
    int maxnum = MIN(256, num_glyphs - custompage*256);  // Number of glyphs on this page
  // Encode the font with the set of glyphs
    char buf[32];
    int code = -1;
    newencoding = g_new(char*, 257);
    while (++code < maxnum) // encode all defined glyphs
      newencoding[code] = g_strdup(names[custompage*256+code]);
    while (code < 256)  // encode the undefined glyphs as .notdef
      newencoding[code++] = g_strdup(".notdef");
    // put the name of the encoding (the string 'Custom xx' causes the delete event to free the memory of the chars)
    sprintf(buf, "Custom %d", custompage+1);
    newencoding[256] = g_strdup(buf);  
  }
  
  T1_DeleteAllSizes(fontID);
  int rc = T1_ReencodeFont(fontID, newencoding);
  if (rc < 0) {
    errormsg("Cannot reencode font: %s", GetT1error(T1_errno));
    return;
  }
  fd->t1data.encoding = newencoding;
  
  g_slist_foreach(winfolist, repaintfont, fd); 
  return;
}



// Creates a scrolled window in the given frame with a drawing area 
// and makes the character map of the font in it in the specified size
static WInfo* drawfont_inframe(GtkWidget *frame, FontData *fd, double size)
{
  int fontID = fd->t1data.fontID;
  T1_errno=0;

  BBox fontbbox = getfontbbox(fontID);  // In charspace units
  int sizex = fontbbox.urx - fontbbox.llx;
  int sizey = fontbbox.ury - fontbbox.lly;
  if (sizex<=0 || sizey<=0) {
    errormsg("Invalid bounding box in font");
    return NULL;
  }
  // 1 charspace unit = 1/1000 bp, 1 bp = 1/72 inch (at 72 dpi resolution = 1 pixel)
  double xscale = xresolution/72.0;
  double yscale = yresolution/72.0;

  // Get the size in pixels of the cell surrounding the glyph:
  // it should be 1.2 times the size of the glyph
  int width = (int)((sizex*size*xscale*1.2)/1000.0);  
  int height = (int)((sizey*size*yscale*1.2)/1000.0);

  int width_w = width*16 + 1;   // 16 cells in a row or column
  int height_w = height*16 + 1;

  WInfo *winfo = g_new(WInfo, 1);
  GtkWidget* drawing_area = gtk_drawing_area_new();
  GdkPixmap* pixmap = gdk_pixmap_new(mainwindow->window, width_w,
				     height_w, -1);
  winfolist = g_slist_append(winfolist, (gpointer)winfo);

  winfo->fd = fd;
  winfo->fontsize = size;
  winfo->pixmap = pixmap;
  winfo->drawing_area = drawing_area;
  winfo->pixmapwidth = width_w;
  winfo->pixmapheight = height_w;
  winfo->antialiasing = antialiasing;
  winfo->char_codes = FALSE;

  // Scrolling window with drawing area
  GtkWidget* scwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwindow), 
				 GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(frame), scwindow);

  gtk_widget_set_events(drawing_area, GDK_EXPOSURE_MASK |
			GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK |
			GDK_POINTER_MOTION_MASK | 
			GDK_POINTER_MOTION_HINT_MASK |
			GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwindow), 
					drawing_area);

  // Set the size of the area so the scwindow knows
  gtk_drawing_area_size(GTK_DRAWING_AREA(drawing_area), width_w, height_w);
  /* And the size of the scwindow to request a view of the whole pixmap;
     if the dimensions of the screen are exceeded, it will be limited in
     t1_fonttable()
   */
  gtk_widget_set_usize(GTK_BIN(scwindow)->child, width_w, height_w);

  // Do the job
  fill_pixmap(winfo);
  fd->refcount++;  // FontData* is now used by this window

  // Connections
  gtk_signal_connect(GTK_OBJECT(drawing_area), "destroy", 
		     (GtkSignalFunc)destroy_fonttable, (gpointer)winfo);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "expose_event", 
		     (GtkSignalFunc)expose_event, (gpointer)pixmap);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "motion_notify_event", 
		     (GtkSignalFunc)drawarea_event, (gpointer)winfo);
  gtk_signal_connect(GTK_OBJECT(winfo->drawing_area), "leave_notify_event", 
		     (GtkSignalFunc)drawarea_event, (gpointer)winfo);

  if (size<100) {  // Expansion window only for small sizes
    gtk_signal_connect(GTK_OBJECT(drawing_area), "button_press_event", 
		       (GtkSignalFunc)drawarea_event, (gpointer)winfo);
    gtk_signal_connect(GTK_OBJECT(drawing_area), "button_release_event", 
		       (GtkSignalFunc)drawarea_event, (gpointer)winfo);
  }
  return winfo;
}




// Called when closing a fonttable window (when frame is destroyed)
static void destroy_fonttable(GtkWidget *window, gpointer data)
{
  WInfo *winfo = (WInfo*)data;
  winfolist = g_slist_remove(winfolist, data);
  winfo->fd->refcount--;  // FontData* no longer used by this window
  // If the font is no longer being displayed in the font list, try to delete it
  if (!winfo->fd->visible) destroyfont(winfo->fd);

  gdk_pixmap_unref(winfo->pixmap);
  g_free(winfo);
}



static void saveasGIF(gpointer callback_data,
		      guint    callback_action,
		      GtkWidget *widget) 
{
  GtkWidget *window = (GtkWidget*)callback_data;
  WInfo *winfo = (WInfo*)gtk_object_get_data(GTK_OBJECT(window), "winfo");
  /* Get image from pixmap and put the reference to it 
     in the 'window' object */
  GdkImage *image = gdk_image_get(winfo->pixmap, 0, 0, 
				  winfo->pixmapwidth, winfo->pixmapheight);
  gtk_object_set_data(GTK_OBJECT(window), "image", (gpointer)image); 
  /* The 'image' and 'delete_handler' in window are needed by save_asgif */
  save_asgif(window);
  gdk_image_destroy(image); 
}



/* 
   Called to toggle the representation of the map in the 
   font table window
*/
static void switch_options(gpointer callback_data,
			   guint    callback_action,
			   GtkWidget *widget) 
{
  GtkWidget *item;
  GtkWidget *window = (GtkWidget*)callback_data;
  WInfo *winfo = (WInfo*)gtk_object_get_data(GTK_OBJECT(window), "winfo");

  item = gtk_item_factory_get_item(winfo->item_factory, 
				   "/View/Antialiasing");
  winfo->antialiasing = GTK_CHECK_MENU_ITEM(item)->active;
  item = gtk_item_factory_get_item(winfo->item_factory, 
				   "/View/View Codes");
  winfo->char_codes = GTK_CHECK_MENU_ITEM(item)->active;

  fill_pixmap(winfo); 
  gdk_draw_pixmap(winfo->drawing_area->window, bwGC, winfo->pixmap, 
		  0, 0, 0, 0, winfo->pixmapwidth, winfo->pixmapheight); 
}


static GtkItemFactoryEntry ftmenu_items[] = { 
  // path accelerator callback callback_action item_type
{ "/_File",      NULL,           NULL, 0, "<Branch>" },
{ "/File/_Save as GIF",  "<control>S",  (GtkItemFactoryCallback)saveasGIF,  0, NULL },
{ "/File/_Close",  "<control>C",  (GtkItemFactoryCallback)gtk_widget_destroy,  0, NULL },
{ "/_View",      NULL,           NULL, 0, "<Branch>" },
{ "/View/View _Codes",  "<control>O",  (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"},
{ "/View/_Antialiasing",  "<control>A",  (GtkItemFactoryCallback)switch_options,  0, "<CheckItem>"}
};




void t1_fonttable(FontData* fd, double size)
{
  int fontID = fd->t1data.fontID;
  int num_glyphs = -1;
        
  if (T1_LoadFont(fontID) == -1) {
    errormsg("Cannot load fontid %d: %s", fontID, GetT1error(T1_errno));
    return;
  }
  char **names = T1_GetAllCharNames(fontID);
  while (names[++num_glyphs]);  // Calculate size of names array (last element is NULL)

  // Create window for the image
  GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name(window, "fonttable window");
  gtk_window_set_title(GTK_WINDOW(window), T1_GetFontName(fontID));
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
  int hid = gtk_signal_connect_object(GTK_OBJECT(window), "delete_event", 
				      GTK_SIGNAL_FUNC(gtk_false), NULL);
  gtk_object_set_data(GTK_OBJECT(window), "delete_handler", 
		      GINT_TO_POINTER(hid));

  // Vertical container
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(window), vbox);
  
  // Menubar
  GtkAccelGroup *accel_group = gtk_accel_group_new();
  GtkItemFactory *item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, 
						      "<main>", accel_group);
  gint nmenu_items = sizeof(ftmenu_items) / sizeof (ftmenu_items[0]);
  gtk_item_factory_create_items(item_factory, nmenu_items, 
				ftmenu_items, window);
  GtkWidget *menubar = gtk_item_factory_get_widget(item_factory, "<main>");
  gtk_accel_group_attach(accel_group, GTK_OBJECT(window));
  gtk_box_pack_start(GTK_BOX(vbox), menubar, FALSE, FALSE, 0);
  
  // Set antialias menu item to current state of check button
  GtkWidget *aaliasmenu = gtk_item_factory_get_item(item_factory, 
						    "/View/Antialiasing");
  GTK_CHECK_MENU_ITEM(aaliasmenu)->active = antialiasing;

  // Set showing of character codes to FALSE
  GtkWidget *codesmenu = gtk_item_factory_get_item(item_factory, 
						   "/View/View Codes");
  GTK_CHECK_MENU_ITEM(codesmenu)->active = FALSE;

  // Upper horizontal box containing encoding selector
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 10);

  // Font map
  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);
  WInfo *winfo = drawfont_inframe(frame, fd, size);
  if (winfo == NULL) {
    gtk_widget_destroy(window);
    return;
  }
  winfo->char_codes = GTK_CHECK_MENU_ITEM(codesmenu)->active;
  winfo->item_factory = item_factory;
  gtk_object_set_data(GTK_OBJECT(window), "winfo", winfo);    

  GtkWidget *label = gtk_label_new("Encoding:");
  gtk_widget_set_style(label, stylebold);
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 10);

  // Encodings option menu
  GtkWidget *optionmenu = gtk_option_menu_new();
  GtkWidget *menu = gtk_menu_new();

  winfo->option_menu = optionmenu;
  GtkWidget *item = gtk_menu_item_new_with_label("Default");
  gtk_widget_show(item);	// This call is needed;
				// otherwise the menu is badly displayed
  gtk_object_set_data(GTK_OBJECT(item), "encoding", NULL);
  gtk_signal_connect(GTK_OBJECT(item), "activate",
		     GTK_SIGNAL_FUNC(newencoding), (gpointer)winfo);
  gtk_menu_append(GTK_MENU(menu), item);
  
  item = gtk_menu_item_new_with_label("AdobeStandard");
  gtk_widget_show(item);
  gtk_object_set_data(GTK_OBJECT(item), "encoding",
		      (gpointer)StandardEncoding);
  gtk_signal_connect(GTK_OBJECT(item), "activate", 
		     GTK_SIGNAL_FUNC(newencoding), (gpointer)winfo);
  gtk_menu_append(GTK_MENU(menu), item);

  item = gtk_menu_item_new_with_label("ISO Latin1");
  gtk_widget_show(item);
  gtk_object_set_data(GTK_OBJECT(item), "encoding",
		      (gpointer)ISOLatin1Encoding);
  gtk_signal_connect(GTK_OBJECT(item), "activate", 
		     GTK_SIGNAL_FUNC(newencoding), (gpointer)winfo);
  gtk_menu_append(GTK_MENU(menu), item);

  item = gtk_menu_item_new_with_label("Expert");
  gtk_widget_show(item);
  gtk_object_set_data(GTK_OBJECT(item), "encoding",
		      (gpointer)ExpertEncoding);
  gtk_signal_connect(GTK_OBJECT(item), "activate", 
		     GTK_SIGNAL_FUNC(newencoding), (gpointer)winfo);
  gtk_menu_append(GTK_MENU(menu), item);

  item = gtk_menu_item_new_with_label("TeXBase1");
  gtk_widget_show(item);
  gtk_object_set_data(GTK_OBJECT(item), "encoding",
		      (gpointer)TeXBase1Encoding);
  gtk_signal_connect(GTK_OBJECT(item), "activate", 
		     GTK_SIGNAL_FUNC(newencoding), (gpointer)winfo);
  gtk_menu_append(GTK_MENU(menu), item);

  item = gtk_menu_item_new_with_label("Cork");
  gtk_widget_show(item);
  gtk_object_set_data(GTK_OBJECT(item), "encoding", (gpointer)CorkEncoding);
  gtk_signal_connect(GTK_OBJECT(item), "activate", 
		     GTK_SIGNAL_FUNC(newencoding), (gpointer)winfo);
  gtk_menu_append(GTK_MENU(menu), item);

  // menu separator (empty item)
  item = gtk_menu_item_new();
  gtk_widget_show(item);
  gtk_menu_append(GTK_MENU(menu), item);

  // Make menu entries for the custom encodings
  // that permits to view all glyphs in the font
  for (int page=0; page<=(num_glyphs-1)/256; page++) {
    char buf[32];
    sprintf(buf, "Custom %d", page+1);
    item = gtk_menu_item_new_with_label(buf);
    gtk_widget_show(item);
    gtk_object_set_data(GTK_OBJECT(item), "custompage", 
			GINT_TO_POINTER(page+1));
    gtk_signal_connect(GTK_OBJECT(item), "activate", 
		       GTK_SIGNAL_FUNC(newencoding), (gpointer)winfo);
    gtk_menu_append(GTK_MENU(menu), item);
  }
  
  gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenu), menu);
  gtk_box_pack_start(GTK_BOX(hbox), optionmenu, FALSE, FALSE, 0);  
  sync_optionmenu(winfo);


  // Info label
  GtkWidget *evb = gtk_event_box_new();
  gtk_widget_set_name(evb, "info label");
  gtk_box_pack_end(GTK_BOX(vbox), evb, FALSE, FALSE, 0);
  label = gtk_label_new(NULL);
  gtk_widget_set_name(label, "info label");
  gtk_misc_set_padding(GTK_MISC(label), 0, 2);
  gtk_container_add(GTK_CONTAINER(evb), label);
  winfo->info_label = evb;

  /* Compute the size requested by the widgets in the window */
  GtkRequisition req;
  gtk_widget_show_all(vbox);
  gtk_widget_size_request(window, &req);

  /* Set the size so that the complete fontmap is visible on screen, 
     but only up to the screen size */
  gtk_widget_set_usize(window, 
		       MIN(req.width,  gdk_screen_width()-30),
  		       MIN(req.height, gdk_screen_height()-40));
  gtk_widget_show(window);
}




// ******* Font download functions ******

// Read a 4 byte length from PFB file
static guint32 getblocklength(FILE *fp)
{
  guint32 len;

  len = (guint32)(getc(fp) & 0xff);
  len |= (guint32)(getc(fp) & 0xff)<<8;
  len |= (guint32)(getc(fp) & 0xff)<<16;
  len |= (guint32)(getc(fp) & 0xff)<<24;
  return len;
}



// Read, interpret and download the PFB font from ifp to ofp
// Return success as boolean
// Based on the t1utils by Eddie Kohler (eddietwo@lcs.mit.edu)
//    Copyright (c) 1992 by I. Lee Hetherington, all rights reserved
static bool t1_downloadpfb(FILE *ifp, FILE *ofp)
{
  static const char *hexchar = "0123456789ABCDEF";
  int v, hexcol;
  guint32 len;

  while ((v=fgetc(ifp))!=EOF) {
    if (v != 128) {  // 128 is the marker byte
      errormsg("Invalid Type 1 font");
      return FALSE;
    }
    
    switch (fgetc(ifp)) {  // Block type
    case 1:  // ASCII block
      fputc('\n', ofp);
      for (len=getblocklength(ifp); len > 0; len--) {
	v = fgetc(ifp);
	if (v == '\r') v = '\n';
	fputc(v, ofp);
      }
      break;
    case 2:  // Binary block
      for (len=getblocklength(ifp), hexcol=0; len > 0; len--) {
	v = fgetc(ifp);
	fputc(hexchar[(v>>4)&0xf], ofp);  // Translate to hex
	fputc(hexchar[v&0xf], ofp);
	hexcol += 2;
	if (hexcol>64) {  // Truncate line
	  fputc('\n', ofp);
	  hexcol = 0;
	}
      }
      break;
    case 3:  // Done; end block
      return TRUE;
    default: // Bad block
      errormsg("Invalid Type 1 font");
      return FALSE;
    }  // switch
  } // while
  // EOF was read
  errormsg("Invalid Type 1 font");
  return FALSE;
}



// Download the Type 1 font onto the given file (for printing)
// Return success as boolean
bool t1_downloadfont(FILE *ofp, FontData* fd)
{
  FILE *ifp;      // Input (font file)
  int rc = FALSE; // Unsuccessfull

  ifp = fopen(fd->fontFile, "r");
  if (!ifp) {
    errormsg("Cannot open file: %s", g_strerror(errno));
    return FALSE;
  }

  int len = strlen(fd->fontFile);  
  if (fd->fontFile[len-1] == 'a' || fd->fontFile[len-1] == 'A') {
    // ASCII Font, PFA
    // Just copy it as is onto output file
    int v;
    while ((v=fgetc(ifp))!=EOF) fputc(v, ofp);
    rc = TRUE;
  }
  else {  // Binary Font, PFB
    rc = t1_downloadpfb(ifp, ofp);
  }
  fclose(ifp);
  return rc;
}



// ********** Font properties functions ***********

static void make_entries(GtkWidget *frame, struct TTInfo *props, int num)
{
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_container_border_width(GTK_CONTAINER(vbox), 5);

  GtkWidget *table = gtk_table_new(num, 2, FALSE); // A table with num rows, 2 columns
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);

  for (int i=0; i<num; i++) {
    GtkWidget *label = props[i].property;
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_widget_set_style(label, stylebold);
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, i, i+1, 
		     GTK_FILL, GTK_EXPAND, 0, 0);
    label = props[i].value;
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
    gtk_table_attach(GTK_TABLE(table), label, 1, 2, i, i+1, 
		     (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), GTK_EXPAND, 0, 0);
  }
}



static GtkWidget* general_info(FontData *fd)
{
  struct TTInfo props[20];
  char buf[25], *str;
  int nprops = 0;

  int fontID = fd->t1data.fontID;
  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_container_border_width(GTK_CONTAINER(frame), 3);

  props[nprops].property = gtk_label_new("File name:");
  props[nprops++].value = gtk_label_new(g_basename(fd->fontFile));

  props[nprops].property = gtk_label_new("Font name:");
  props[nprops++].value = gtk_label_new(T1_GetFontName(fontID));

  props[nprops].property = gtk_label_new("Font full name:");
  props[nprops++].value = gtk_label_new(T1_GetFullName(fontID));

  props[nprops].property = gtk_label_new("Font family:");
  props[nprops++].value = gtk_label_new(T1_GetFamilyName(fontID));

  props[nprops].property = gtk_label_new("Font version:");
  props[nprops++].value = gtk_label_new(T1_GetVersion(fontID));

  props[nprops].property = gtk_label_new("Copyright:");
  props[nprops++].value = gtk_label_new(T1_GetNotice(fontID));

  props[nprops].property = gtk_label_new("Weight:");
  props[nprops++].value = gtk_label_new(T1_GetWeight(fontID));

  props[nprops].property = gtk_label_new("Italic angle:");
  sprintf(buf, "%.1f", T1_GetItalicAngle(fontID));
  props[nprops++].value = gtk_label_new(buf);

  props[nprops].property = gtk_label_new("Monospaced:");
  sprintf(buf, "%s", T1_GetIsFixedPitch(fontID)?"yes":"no");
  props[nprops++].value = gtk_label_new(buf);

  make_entries(frame, props, nprops);
  gtk_widget_show_all(frame);
  return frame;
}




void t1_showproperties(FontData *fd, GtkWidget *window)
{
  GtkWidget *frame, *label;

  GtkWidget *notebook = gtk_notebook_new();
  gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);
  gtk_container_add(GTK_CONTAINER(window), notebook);

  // General info
  frame = general_info(fd);
  label = gtk_label_new("General");
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  gtk_widget_show(notebook);
  gtk_widget_show(window);
}

