
/*
  This will be a font displaying program some day 
  
   1998 Roberto Alameda <roberto@myokay.net>
  You may modify and distribute this file under the terms of the GNU
  General Public License, version 2, or any later version, at your
  convenience. See the file COPYING for details. 

*/


#include <config.h>

#if HAVE_UNISTD_H
# include <unistd.h>
# include <sys/types.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <locale.h>
#include <limits.h>
#include <errno.h>
#include <sys/stat.h>

#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "gfont.h"

#include "mini-ofolder.xpm"
#include "font.xpm"
#include "emptypixmap.xpm"
#include "stipple.xbm"



// Global variables
GdkVisual *visual;
GdkColormap *colormap;
GdkGC *bwGC, *defGC, *redGC;
GdkColorContext* mCC;
GtkStyle *stylebold;
GdkColor white, black;
GdkPixmap* emptypixmap;

GtkWidget* mainwindow;  // The main window widget
GtkWidget* fontclistw;    // Font list widget
GtkWidget* dirl;        // Directory name label
GtkWidget* stbar;       // Status bar below
GtkWidget* prbar;       // Progress bar below
GtkWidget* samplet;     // The sample text (a pixmap)
InputDialog *fontsizeD, *characterD, *stringD;

bool antialiasing = FALSE;
bool kerning = FALSE;
bool enable_antialiasing = TRUE;

int xresolution, yresolution;  // Resolution of the display in dpi

GdkColor graypalette[5];  // Colors for antialiasing: white - light - medium - dark - black

TT_Engine ttengine;  // For the FreeType library

GSList *fontlist;  // All fonts in the present directory



// ************** Function prototypes (internally used in this module)
gint delete_event(GtkWidget *widget, GdkEvent *event, gpointer data);
Callback newfontdir, leave, showchar, showstring, fonttable, toggle_button;
static void gfont_init(int *argc, char*** argv);
static void select_fontlist(GtkWidget *widget, gint row, gint column, GdkEventButton *event);
static GtkWidget* makefontlist(void);
static void destroyrow(gpointer data);
static gint compareEntries(gconstpointer first, gconstpointer second);
static bool fillfontlist(GtkWidget* clist, char* directory);
static void okfontdir(gpointer diag);
static void load_directory(GtkCombo *combo);





// ************** Menubar
static GtkItemFactoryEntry menu_items[] = {  
{ "/_File",          NULL,          NULL,                               0, "<Branch>" },  
{ "/File/Open",      "<control>O",  (GtkItemFactoryCallback)newfontdir,       0, NULL },  
{ "/File/Print",     "<control>P",  (GtkItemFactoryCallback)printfont,        0, NULL }, 
{ "/File/Properties","<control>R",  (GtkItemFactoryCallback)show_properties,  0, NULL }, 
{ "/File/Save Names","<control>S",  (GtkItemFactoryCallback)save_names,  0, NULL }, 
{ "/File/sep1",      NULL,          NULL,                            0, "<Separator>" },  
{ "/File/Quit",      "<control>Q",  (GtkItemFactoryCallback)leave,            0, NULL },  
{ "/_View",          NULL,          NULL,                               0, "<Branch>" },  
{ "/View/Errors",    NULL,          (GtkItemFactoryCallback)show_message_window, 0, NULL },
{ "/_Help",          NULL,          NULL,                           0, "<LastBranch>" },  
{ "/Help/About",     NULL,          (GtkItemFactoryCallback)makeabout,        0, NULL }
};



// ********************* Begin of program functions

#if 0
void info()
{
  Display *dp = GDK_DISPLAY();
  int screen = gdk_screen;

  printf("X%dR%d revision %d\n", ProtocolVersion(dp), XlibSpecificationRelease,
         ProtocolRevision(dp));
  printf("%s, vendor release %d\n", ServerVendor(dp), VendorRelease(dp));

  printf("Display: %s, screen %d, visual id %#lx, colormap id %#lx\n", DisplayString(dp), screen,
         XVisualIDFromVisual(GDK_VISUAL_XVISUAL(visual)), GDK_COLORMAP_XCOLORMAP(colormap));
  printf("Colormap cells: %d\n", colormap->size);
  printf("Depth of visual: %d planes\n", visual->depth);

  printf("Width:%d Height:%d\n", DisplayWidth(dp,screen), DisplayHeight(dp,screen));
  printf("WidthMM:%d HeightMM:%d\n", DisplayWidthMM(dp,screen), DisplayHeightMM(dp,screen));
  printf("Default root window id of display: %#lx\n", DefaultRootWindow(dp));

  printf("X %s support locale %s\n", (XSupportsLocale()==True)?"does":"does not",
         setlocale(LC_ALL, NULL));
  printf("Locale modifiers:%s\n", XSetLocaleModifiers(NULL));
}



void dumpGC(GdkGC* gc)
{
  XGCValues values;
  GC xgc = GDK_GC_XGC(gc);
  
  XGetGCValues(GDK_DISPLAY(), xgc, GCForeground | GCBackground, &values);
  printf("Foreground: %ld, Background:%ld\n", values.foreground, values.background);
}
#endif




static void gfont_init(int *argc, char*** argv)
{
  // Init GTK and X11 stuff
  gtk_init(argc, argv);
  gtk_set_locale();
  gtk_rc_init();
  gtk_rc_parse(".gfontviewrc");

  visual = gdk_visual_get_system();
  colormap = gdk_colormap_get_system();
  mCC = gdk_color_context_new(visual, colormap);
  if (visual->depth<8 || visual->depth==15) enable_antialiasing = FALSE;

  gdk_color_black(colormap, &black);
  gdk_color_white(colormap, &white);
 
  // Calculate screen resolution in dots per inch
  xresolution = (int)(0.5 + gdk_screen_width() / (gdk_screen_width_mm()/25.4));
  yresolution = (int)(0.5 + gdk_screen_height() / (gdk_screen_height_mm()/25.4));

  // Main Window has to be created now
  mainwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name(mainwindow, "main window");
  gtk_widget_ensure_style(mainwindow); 

  // Get selected basis font and make a bold version in stylebold
  // Have to access a private component for that...
  GdkFontPrivate *priv = (GdkFontPrivate *)mainwindow->style->font;
  const char *fnames = (const char*)priv->names->data;

  // It can be a fontset; get all components of it (separated by comma)
  gchar **fonts = g_strsplit(fnames, ",", -1);
  for (gint i=0; fonts[i]; i++) {
    // For all fonts in the fontset, find a bold version
    gchar **namecomps = g_strsplit(fonts[i], "-", 14);
    g_free(namecomps[3]);
    namecomps[3] = g_strdup("bold");
    gchar *boldfont = g_strjoinv("-", namecomps);
    g_strfreev(namecomps);
    
    g_free(fonts[i]);
    fonts[i] = g_strdup(boldfont);
  }
  // Make a complete fontset
  gchar *boldfonts = g_strjoinv(",", fonts);
  g_strfreev(fonts);

  // Attach the bold font or fontset to the stylebold style
  GdkFont *bold;
  stylebold = gtk_style_copy(mainwindow->style);
  switch (stylebold->font->type) {
  case GDK_FONT_FONT: 
    bold = gdk_font_load(boldfonts);
    break;
  case GDK_FONT_FONTSET:
    bold = gdk_fontset_load(boldfonts);
    break;
  }
  if (bold) {
    gdk_font_unref(stylebold->font);
    stylebold->font = bold;
  }

  // Fill graypalette[] for antialiasing
  if (enable_antialiasing) {
    char *list[] = {"white", "gray75", "gray50", "gray25", "black"};
    for (guint i=0; i<sizeof(list)/sizeof(char*); i++) {
      int failed = 1;
      if (gdk_color_parse(list[i], &graypalette[i]))
	graypalette[i].pixel = 
	  gdk_color_context_get_pixel(mCC, graypalette[i].red, 
				      graypalette[i].green, 
				      graypalette[i].blue, &failed);
      if (failed) graypalette[i].pixel = black.pixel;
    }
  }

  // Init T1lib
  T1_SetLogLevel(T1LOG_STATISTIC);
  T1_SetBitmapPad(16);
  T1_SetDeviceResolutions(xresolution, yresolution);
  if (T1_InitLib(NO_LOGFILE | IGNORE_FONTDATABASE | IGNORE_CONFIGFILE)==NULL){
    fprintf(stderr, "Initialization of T1-library failed\n");
    exit(1); 
  }
  if (enable_antialiasing) {
    if (T1_AASetBitsPerPixel(visual->depth) < 0) {
      fprintf(stderr, "Cannot set T1-library antialiasing depth to %d: %s\n", 
	      visual->depth, GetT1error(T1_errno));
      exit(1);
    }
    T1_AASetLevel(T1_AA_LOW);
    T1_AASetGrayValues(graypalette[0].pixel, graypalette[1].pixel, 
		       graypalette[2].pixel, graypalette[3].pixel, 
		       graypalette[4].pixel); 
  }
  
  // Init FreeType
  int error = TT_Init_FreeType(&ttengine);
  if (error) {
    fprintf(stderr, "Could not create TT engine instance\n");
    exit(1);
  }
  error = TT_Init_Kerning_Extension(ttengine);
  if (error) {
    fprintf(stderr, "Could not init the kerning extension\n");
    exit(1);
  }
  error = TT_Init_Post_Extension(ttengine);
  if (error) {
    fprintf(stderr, "Could not init the PostScript extension\n");
    exit(1);
  }
  if (enable_antialiasing) {
    // Gray palette: the values are indexes into graypalette[]
    guchar palette[5] = {0, 1, 2, 3, 4};  // white - light - medium - dark - black
    TT_Set_Raster_Gray_Palette(ttengine, palette);
  }
}




InputDialog::InputDialog(const char* labeltext, const char* entrytext)
{
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(hbox), 0);

  label = gtk_label_new(labeltext);
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_widget_set_style(label, stylebold);

  entry = gtk_entry_new();
  gtk_widget_set_usize(entry, 50, -1);
  gtk_entry_set_text(GTK_ENTRY(entry), entrytext);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 0);
}


void delete_image(GtkWidget*, gpointer data)
{
  GdkImage *image = (GdkImage*)data;
  //printf("Destroying image %p...\n", image);
  gdk_image_destroy(image);
}


void delete_pixmap(GtkWidget*, gpointer data)  // destroy callback
{
  GdkPixmap *pixmap = (GdkPixmap*)data;
  //printf("Destroying pixmap %p...\n", pixmap);
  gdk_pixmap_unref(pixmap);
}



gint expose_event(GtkWidget *widget, GdkEventExpose *event, gpointer data)
{
  GdkPixmap *pixmap = (GdkPixmap*)data;
  //printf("In the expose event, pixmap:%p\n", pixmap);
  gdk_draw_pixmap(widget->window, bwGC, pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);
  return FALSE;
}


/*
  Called when mouse key is released and the combo box was displayed
 */
static void dirl_release(GtkWidget *widget, GdkEvent *event, GtkCombo *combo) 
{
  /* If the popwin still has the grab, the click was not in the 
     list area, so ignore */
  if (!GTK_WIDGET_HAS_GRAB(combo->popwin)) {
    while (gtk_events_pending()) gtk_main_iteration(); 
    XSync(GDK_DISPLAY(), False);  // Get the combox box out of screen
    load_directory(combo);
  }
}


/* 
   Called when user pressed ENTER in font directory entry combo
   or after user pressed OK in directory selection dialog
   It causes the list to be refreshed from the new directory
*/
static void load_directory(GtkCombo *combo)
{
  bool add_to_list = TRUE;  // Whether to add the dir to the combo list
  char *dir = gtk_entry_get_text(GTK_ENTRY(combo->entry));
  if (*dir == '\0') {  // if empty string, do nothing
    return;
  }
  gdk_window_clear(samplet->window);  // Clear old font sample
  gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL); 
  if (fillfontlist(fontclistw, dir) == FALSE) return;
  // Only goes further if directory was correctly read

  // Go thru all list items of the list; if dir is already there, do not
  // add it again
  GList *list = GTK_LIST(GTK_COMBO(dirl)->list)->children; 
  while (list && list->data) {  // list->data is a list item
    gchar *ltext;
    GtkWidget *label = GTK_BIN(list->data)->child;
    if (!label || !GTK_IS_LABEL(label)) continue;
    gtk_label_get(GTK_LABEL(label), &ltext); 
    if (strcmp(dir, ltext) == 0) {
      add_to_list = FALSE;
      break;
    }
    list = list->next;
  }
  
  if (add_to_list) {
    /* Adding a new element to the list corrupts the string returned
       by gtk_entry_get_text; it cannot be accessed any more */
    GtkWidget *item = gtk_list_item_new_with_label(dir);
    gtk_widget_show(item); 
    // Add new item to beginning of list
    GList *iteml = g_list_alloc(); 
    iteml->data = item;
    gtk_list_prepend_items(GTK_LIST(GTK_COMBO(dirl)->list), iteml);
  }
}



void showchar(GtkWidget *button, gpointer list)
{
  char *invl;

  GList *selection = GTK_CLIST(list)->selection;
  if (selection == NULL) {
    errormsg("No row selected");
    return;
  }
  double size = atof(fontsizeD->getText());
  if (size<1.0 || size>1000.0) {
    errormsg("Invalid size");
    return;
  }
  const char *chr = characterD->getText();
  long rc = strtol(chr, &invl, 0);
  if (*chr=='\0' || *invl!='\0' || rc==LONG_MIN || rc==LONG_MAX || 
      rc<0 || rc>0xFFFF) {
    errormsg("Invalid character");
    return;
  }

  set_window_busy(mainwindow, TRUE);

  int row = GPOINTER_TO_INT(selection->data);
  FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);
  switch (fd->fontType) {
  case T1FONT:
    t1_showchar(fd, (guint)rc, size);
    break;
  case TTFONT: 
    tt_showchar(fd, (guint)rc, size);
    break;
  }
  set_window_busy(mainwindow, FALSE);
}



void fonttable(GtkWidget *button, gpointer list)
{
  GList *selection = GTK_CLIST(list)->selection;
  if (selection == NULL) {
    errormsg("No row selected");
    return;
  }
  double size = atof(fontsizeD->getText());
  if (size<1.0 || size>1000.0) {
    errormsg("Invalid size");
    return;
  }
  int row = GPOINTER_TO_INT(selection->data);
  FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);

  gtk_statusbar_push(GTK_STATUSBAR(stbar), 1, "Making font table...");
  set_window_busy(mainwindow, TRUE);

  switch (fd->fontType) {
  case T1FONT:
    t1_fonttable(fd, size);
    break;
  case TTFONT: 
    tt_fonttable(fd, size);
    break;
  }
  gtk_statusbar_pop(GTK_STATUSBAR(stbar), 1);
  set_window_busy(mainwindow, FALSE);
}




void showstring(GtkWidget *button, gpointer list)
{
  GList *selection = GTK_CLIST(list)->selection;
  if (selection == NULL) {
    errormsg("No row selected");
    return;
  }
  const char* string = stringD->getText();
  if (string==NULL || *string=='\0') string = "Test";
  double size = atof(fontsizeD->getText());
  if (size<1.0 || size>1000.0) {
    errormsg("Invalid size");
    return;
  }

  set_window_busy(mainwindow, TRUE);

  int row = GPOINTER_TO_INT(selection->data);
  FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);
  switch(fd->fontType) {
  case T1FONT:
    t1_showstring(fd, string, size);
    break;
  case TTFONT: 
    tt_showstring(fd, string, size);
    break;
  }
  set_window_busy(mainwindow, FALSE);
}



/* 
   Gets the list of all directories with fonts under 'dirname'
   and appends it to 'list', giving the new list as result
 */
static GList* get_dirtree(GList *list, const char* dirname)
{  
  struct dirent *entry;
  DIR *dir;
  GString *file = g_string_new(NULL);
  bool dir_hasfonts = FALSE;
  
  dir = opendir(dirname);
  if (dir == NULL) return list;

  while ((entry=readdir(dir))) {
    struct stat sbuf;
    char *name = entry->d_name;
    int len = strlen(name);
    g_string_sprintf(file, "%s/%s", dirname, name);
    if (stat(file->str, &sbuf) != 0) continue;
  
    if (S_ISDIR(sbuf.st_mode) && 
	strcmp(name, ".")!=0 && 
	strcmp(name, "..")!=0) list = get_dirtree(list, file->str);
    if (dir_hasfonts==FALSE && S_ISREG(sbuf.st_mode)) {
      // Type 1
      if (len>4 && name[len-4] == '.' &&
	  (name[len-3] == 'p' || name[len-3] == 'P') &&
	  (name[len-2] == 'f' || name[len-2] == 'F') &&
	  (name[len-1] == 'a' || name[len-1] == 'A' ||
	   name[len-1] == 'b' || name[len-1] == 'B'))
	dir_hasfonts = TRUE;    
      // True Type
      if (len>4 && name[len-4] == '.' &&
	(name[len-3] == 't' || name[len-3] == 'T') &&
	(name[len-2] == 't' || name[len-2] == 'T') &&
	(name[len-1] == 'f' || name[len-1] == 'F' ||
	 name[len-1] == 'c' || name[len-1] == 'C'))
	dir_hasfonts = TRUE;
    }
  }    
  if (dir_hasfonts) list = g_list_append(list, g_strdup(dirname));
  closedir(dir);    
  g_string_free(file, TRUE);
  return list;
}




static GList* get_some_cool_dirs(void)
{
  GList *list = NULL;
  const char *dlist[] = {"/usr/share/ghostscript/fonts", "/usr/X11R6/lib/X11/fonts", "/usr/share/texmf/fonts", "/usr/local/share/texmf/fonts", "/usr/local/fonts"};

  for (guint i=0; i<sizeof(dlist)/sizeof(char*); i++) 
    list = get_dirtree(list, dlist[i]);
  return list;
}



void toggle_button(GtkWidget*, gpointer data)
{
  bool *variable =(bool*)data;
  *variable = !*variable;  // Invert state of variable
}




/* Called when user clicked on a list item, i.e., a font */
static void select_fontlist(GtkWidget *widget, gint row, gint column, 
			    GdkEventButton *event)
{
  //printf("GtkCList Selection: row %d column %d button %d\n", row, column, event?event->button:0);
  FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(widget), row);
  // fillfontlist will generate 1 select event before the data is there
  if (fd == NULL) return;  // So check for it
  switch(fd->fontType) {
  case T1FONT: 
    set_window_busy(mainwindow, TRUE); 
    t1_drawsample(fd, "the quick brown fox jumps over the lazy dog", 18.0);
    set_window_busy(mainwindow, FALSE);
    break;
  case TTFONT:
    tt_drawsample(fd, "the quick brown fox jumps over the lazy dog", 18.0);
    break;
  }
  /* 
     The next 2 lines are a workaround for a clist bug present in GTK+ 1.2.2:
     if a modal window (an error message) is displayed within the 
     select_row event, the pointer is grabbed and the X-Server 
     does not respond to mouse events
   */
  gtk_grab_remove(widget);
  if (gdk_pointer_is_grabbed()) gdk_pointer_ungrab(GDK_CURRENT_TIME);
}




static GtkWidget* makefontlist(void)
{
  char* titles[] = {"File name", "Glyphs", "Font name", "Kern pairs"};

  GtkWidget* scwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwindow), 
				 GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);

  fontclistw = gtk_clist_new_with_titles(sizeof(titles)/sizeof(char*), titles);
  gtk_widget_ensure_style(fontclistw);
  gtk_clist_column_titles_passive(GTK_CLIST(fontclistw));
  gtk_clist_set_column_width(GTK_CLIST(fontclistw), 0, 100);
  gtk_clist_set_column_width(GTK_CLIST(fontclistw), 1, 50);
  gtk_clist_set_column_width(GTK_CLIST(fontclistw), 2, 180);
  gtk_clist_set_selection_mode(GTK_CLIST(fontclistw), GTK_SELECTION_BROWSE);
  gtk_clist_set_shadow_type(GTK_CLIST(fontclistw), GTK_SHADOW_ETCHED_OUT);
  gtk_container_border_width(GTK_CONTAINER(fontclistw), 0);
  gtk_signal_connect(GTK_OBJECT(fontclistw), "select_row", 
		     (GtkSignalFunc)select_fontlist, NULL);

  gtk_widget_set_usize(fontclistw, -1, 10*(mainwindow->style->font->ascent+
					 mainwindow->style->font->descent));
  gtk_container_add(GTK_CONTAINER(scwindow), fontclistw);
  return scwindow;
}



// Destroy row in clist fontlist; called when displaying a new directory
// to free all structures related to fonts in the old directory
static void destroyrow(gpointer data)
{
  FontData *fd = (FontData*)data;
  fd->visible = FALSE; // No longer being displayed in list
  destroyfont(fd);
}


// Try to destroy a font if unused: free all memory related to its FontData
void destroyfont(FontData *fd)
{
  if (fd->refcount > 0) return;  // Still in use

  switch(fd->fontType) {
  case T1FONT: 
    T1_DeleteFont(fd->t1data.fontID);
    // If encoding is 'Custom x', free all components of encoding vector
    if (fd->t1data.encoding && 
	strncmp(fd->t1data.encoding[256], "Custom", 6) == 0) {
      for (int i=0; i<=256; i++) g_free(fd->t1data.encoding[i]);
      g_free(fd->t1data.encoding);
    }
    break;
  case TTFONT:
    TT_Close_Face(fd->ttdata.face);  
    break;
  }
  g_free(fd->fontName);
  g_free(fd->fontFile);
  g_free(fd);
}




static gint compareEntries(gconstpointer first, gconstpointer second)
{
  FontData* fd1 = (FontData*)first;
  FontData* fd2 = (FontData*)second;

  return(strcmp(fd1->fontName, fd2->fontName));
}



/*
  As the name says, fills the list widget of fonts: it scans the directory,
  looks for fonts in it, creates the data structure FontData associated to
  each font, and then creates the rows of the widget. Each row has a FontData
  struct associated; when the list is destroyed, the function destroyrow will
  be called for each font.
  The FontData structs are stored in the linked list fontlist

 */
static bool fillfontlist(GtkWidget* clist, char* directory)
{
  struct dirent *entry;
  struct stat buf;
  int nfiles=0, readfiles=0;

  g_slist_free(fontlist);
  fontlist = NULL;

  GString* fullname = g_string_new("");
  gtk_clist_clear(GTK_CLIST(clist));
  if (T1_AddToFileSearchPath(T1_PFAB_PATH | T1_AFM_PATH, 
			     T1_PREPEND_PATH, directory)==-1) {
    errormsg("Cannot add directory %s to T1 path: %s", directory, 
	     GetT1error(T1_errno));
    g_string_free(fullname, TRUE);
    return FALSE;
  }
  g_string_sprintf(fullname, "%s/afm", directory);
  if (T1_AddToFileSearchPath(T1_AFM_PATH, T1_PREPEND_PATH, fullname->str)==-1) {
    errormsg("Cannot add directory %s to T1 afm path: %s", fullname->str, 
	     GetT1error(T1_errno));
    g_string_free(fullname, TRUE);
    return FALSE;
  }
  // Calculate number of entries in directory (variable nfiles)
  DIR* dir = opendir(directory);
  if (dir == NULL) {
    errormsg("Cannot open %s: %s", directory, g_strerror(errno));
    return FALSE;
  }
  
  set_window_busy(mainwindow, TRUE);
  gtk_statusbar_push(GTK_STATUSBAR(stbar), 1, "Reading files...");
  while (gtk_events_pending()) gtk_main_iteration();
  while((entry=readdir(dir))) nfiles++;
  rewinddir(dir);

  // Read all files in directory and create a sorted linked list with the valid ones
  while((entry=readdir(dir))) {
    int fontID;
    char *fontname;

    gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 
			    double(++readfiles)/nfiles);
    while (gtk_events_pending()) gtk_main_iteration();

    g_string_sprintf(fullname, "%s/%s", directory, entry->d_name);
    if (stat(fullname->str, &buf)==0 && S_ISREG(buf.st_mode)) {
      FontData *fontdesc = NULL;
      if (fontdesc == NULL) {
	fontdesc = g_new(FontData, 1);
	fontdesc->refcount = 0;
	fontdesc->visible = TRUE;
	fontdesc->fontName = NULL;
      }
      fontdesc->fontFile = fullname->str;
      // File is either T1
      if (is_t1font(fullname->str, fontdesc)) {
	fontlist = g_slist_insert_sorted(fontlist, fontdesc, compareEntries);
	fontdesc->fontFile = g_strdup(fullname->str);  
	fontdesc = NULL;
	continue;
      }
      // Or TT
      if (is_ttfont(fullname->str, fontdesc)) {
	fontlist = g_slist_insert_sorted(fontlist, fontdesc, compareEntries);
	fontdesc->fontFile = g_strdup(fullname->str);  
	fontdesc = NULL;
	continue;
      }
      // Or TT font collection
      if (is_ttfontcollection(fullname->str, fontdesc)) {
	int num = fontdesc->ttdata.properties.num_Faces;  // Number of fonts in collection
	TT_Close_Face(fontdesc->ttdata.face);
	for (int i=0; i<num; i++) {  // Make a list entry for each font in collection
	  FontData *fd = g_new(FontData, 1);
	  fd->fontFile = g_strdup(fullname->str);
	  int error = TT_Open_Collection(ttengine, fullname->str, i, 
					 &fd->ttdata.face);
	  if (error) {
	    add_error(fd, "Cannot open font number %d in collection", i);
	    g_free(fd->fontFile);
	    g_free(fd);
	    continue;
	  }
	  fill_ttfontdata(fd);
	  fontlist = g_slist_insert_sorted(fontlist, fd, compareEntries);
	}
	continue;
      }
      // Or unknown; so forget it
    }
  }
  g_string_free(fullname, TRUE);
  closedir(dir);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 0.0);
  gtk_statusbar_pop(GTK_STATUSBAR(stbar), 1);
  while (gtk_events_pending()) gtk_main_iteration();

  // Put the linked list in the list widget
  gtk_clist_freeze(GTK_CLIST(clist));
  gtk_statusbar_push(GTK_STATUSBAR(stbar), 1, "Making entries...");
  while (gtk_events_pending()) gtk_main_iteration();
  int row = 0;
  for (GSList* p=fontlist; p; p=p->next, row++) {
    gchar *rowentries[4];
    char buf[24], buf2[24];
    FontData* fd = (FontData*)p->data;
    rowentries[0] = g_basename(fd->fontFile);
    rowentries[2] = fd->fontName;
    if (fd->num_glyphs >= 0)
      sprintf(buf, "%u", fd->num_glyphs);
    else 
      strcpy(buf, "---");
    rowentries[1] = buf;
    if (fd->fontType == TTFONT) 
      sprintf(buf2, "%u", fd->ttdata.kernpairs);
    else
      strcpy(buf2, "---");
    rowentries[3] = buf2;

    gtk_clist_append(GTK_CLIST(clist), rowentries);

    // Each row has a FontData*
    gtk_clist_set_row_data_full(GTK_CLIST(clist), row, p->data, destroyrow);  
  }
  gtk_statusbar_pop(GTK_STATUSBAR(stbar), 1);
  gtk_clist_thaw(GTK_CLIST(clist));
  if (row > 0) gtk_clist_select_row(GTK_CLIST(clist), 0, -1); // Select first row

  set_window_busy(mainwindow, FALSE);
  return TRUE;
}



// The user selected to display a new directory
static void okfontdir(gpointer diag)
{
  GtkWidget* filesel = GTK_WIDGET(diag);
  char *newdir = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dirl)->entry), newdir);
  gtk_widget_destroy(filesel);

  load_directory(GTK_COMBO(dirl));
}



// Selection window to choose a new directory
void newfontdir(GtkWidget*, gpointer)
{
  set_window_busy(mainwindow, TRUE);

  GtkWidget* filesel = 
    gtk_file_selection_new("Font Viewer directory selection");
  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));
  gtk_widget_set_sensitive(GTK_FILE_SELECTION(filesel)->file_list, FALSE);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    GTK_OBJECT(filesel));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(okfontdir), 
			    GTK_OBJECT(filesel));
  char *actualdir = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dirl)->entry));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), actualdir);
  gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(mainwindow));
  gtk_widget_show(filesel);
  set_window_busy(mainwindow, FALSE);
}




gint delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  gtk_statusbar_push(GTK_STATUSBAR(stbar), 1, "Exiting...");
  while (gtk_events_pending()) gtk_main_iteration();
  return FALSE; // Send a destroy event
}


// Exit the program
void leave(GtkWidget*, gpointer data)
{
  T1_CloseLib();
  TT_Done_FreeType(ttengine);
  gtk_exit(0);
}




int main(int argc, char** argv)
{
  char *firstdirectory = DEFAULT_FONTDIR;

  gfont_init(&argc, &argv);

  // Check for filename or directory as rest argument
  if (argc > 1) {
    struct stat buf;
    char *givenarg = argv[1];        
    if (stat(givenarg, &buf) != 0) {
      fprintf(stderr, "Cannot find %s: %s\n", givenarg, g_strerror(errno));
      exit(1);
    }
    if (S_ISDIR(buf.st_mode)) {  // If a directory, display it
      firstdirectory = givenarg;
    }
    if (S_ISREG(buf.st_mode)) {  // If a file, display its directory
      firstdirectory = g_dirname(givenarg);
    }
  }


  //******** Main window

  gtk_container_border_width(GTK_CONTAINER(mainwindow), 0);
  gtk_window_set_title(GTK_WINDOW(mainwindow), "Font Viewer");
  gtk_window_set_policy(GTK_WINDOW(mainwindow), FALSE, TRUE, FALSE); /* shrink, grow, auto_shrink */
  gtk_widget_realize(mainwindow);	 
  gtk_signal_connect(GTK_OBJECT(mainwindow), "destroy", 
		     GTK_SIGNAL_FUNC(leave), NULL);
  gtk_signal_connect(GTK_OBJECT(mainwindow), "delete_event", 
		     GTK_SIGNAL_FUNC(delete_event), NULL);

  // Global vbox
  GtkWidget *vboxglobal = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(mainwindow), vboxglobal);

  // Menubar
  GtkAccelGroup *accel_group = gtk_accel_group_new();
  GtkItemFactory *item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, 
						      "<main>", accel_group);
  gint nmenu_items = sizeof(menu_items) / sizeof (menu_items[0]);
  gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, NULL);
  GtkWidget *menubar = gtk_item_factory_get_widget(item_factory, "<main>");
  gtk_accel_group_attach(accel_group, GTK_OBJECT(mainwindow));
  gtk_box_pack_start(GTK_BOX(vboxglobal), menubar, FALSE, FALSE, 0);

  // Main vertical container
  GtkWidget* vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(vbox), 10);
  gtk_box_pack_start(GTK_BOX(vboxglobal), vbox, TRUE, TRUE, 0);


  //****** Current directory list combo
  GtkWidget* hboxl = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hboxl, FALSE, FALSE, 5);

  GtkWidget* dirb = gtk_button_new();
  GtkWidget* icondir = makeicon(mainwindow, miniofolder);
  gtk_container_add(GTK_CONTAINER(dirb), icondir);
  gtk_signal_connect(GTK_OBJECT(dirb), "clicked", 
		     GTK_SIGNAL_FUNC(newfontdir), NULL);
  gtk_box_pack_start(GTK_BOX(hboxl), dirb, FALSE, FALSE, 0);

  dirl = gtk_combo_new();
  gtk_combo_disable_activate(GTK_COMBO(dirl)); // Do not popup list when ENTER
  gtk_combo_set_use_arrows_always(GTK_COMBO(dirl), TRUE);
  gtk_box_pack_start(GTK_BOX(hboxl), dirl, TRUE, TRUE, 0);
  // Reload file list when user presses ENTER
  gtk_signal_connect_object(GTK_OBJECT(GTK_COMBO(dirl)->entry), "activate", 
			    (GtkSignalFunc)load_directory, GTK_OBJECT(dirl));
  // Also when user releases mouse button in combo box list
  gtk_signal_connect_after(GTK_OBJECT(GTK_COMBO(dirl)->list), 
			   "button_release_event", 
			   (GtkSignalFunc)dirl_release, dirl);


  // Put some default dirs into the list
  GList *ilist = get_some_cool_dirs();
  if (ilist) {
    gtk_combo_set_popdown_strings(GTK_COMBO(dirl), ilist);
    g_list_free(ilist);
  }
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dirl)->entry), firstdirectory); 

  GtkWidget* flist = makefontlist();
  gtk_box_pack_start(GTK_BOX(vbox), flist, TRUE, TRUE, 0);

  //******** Sample text
  // Use a viewport to clip pixmap
  GtkWidget* viewport = gtk_viewport_new(NULL, NULL);
  gtk_widget_set_usize(viewport, -1, 35);
  gtk_viewport_set_shadow_type(GTK_VIEWPORT(viewport), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(vbox), viewport, FALSE, FALSE, 0);
  // This prevents the size requests of the viewport from propagating up to the toplevel
  gtk_container_set_resize_mode(GTK_CONTAINER(viewport->parent), 
				GTK_RESIZE_QUEUE);

  GdkBitmap* mask;
  GtkStyle* style = gtk_widget_get_style(mainwindow);
  emptypixmap = gdk_pixmap_create_from_xpm_d(mainwindow->window, &mask,
					     &style->bg[GTK_STATE_NORMAL], 
					     emptypixmap_d);
  samplet = gtk_pixmap_new(emptypixmap, mask);
  gtk_container_add(GTK_CONTAINER(viewport), samplet);
  
     
  //******** Input entries
  fontsizeD = new InputDialog("Font Size:", "24");
  gtk_box_pack_start(GTK_BOX(vbox), fontsizeD->getWidget(), FALSE, FALSE, 0);

  GtkWidget* hboxe = gtk_hbox_new(TRUE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxe), 0);
  gtk_box_pack_start(GTK_BOX(vbox), hboxe, FALSE, FALSE, 0);

  characterD = new InputDialog("Character:", "65");
  gtk_box_pack_start(GTK_BOX(hboxe), characterD->getWidget(), TRUE, TRUE, 0);

  stringD = new InputDialog("Test String:", "Test");
  stringD->setEntryWidth(100);
  gtk_box_pack_start(GTK_BOX(hboxe), stringD->getWidget(), TRUE, TRUE, 0);


  //********** Toggle buttons
  GtkWidget* hboxt = gtk_hbox_new(TRUE, 0);
  gtk_container_border_width(GTK_CONTAINER(hboxt), 3);
  gtk_box_pack_start(GTK_BOX(vbox), hboxt, FALSE, FALSE, 0);

  GtkWidget* aabutton = gtk_check_button_new_with_label("Antialias");
  gtk_box_pack_start(GTK_BOX(hboxt), aabutton, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(aabutton), "toggled", 
		     GTK_SIGNAL_FUNC(toggle_button), &antialiasing);
  if (!enable_antialiasing) gtk_widget_set_sensitive(aabutton, FALSE);

  GtkWidget* kernbutton = gtk_check_button_new_with_label("Kerning");
  gtk_box_pack_start(GTK_BOX(hboxt), kernbutton, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(kernbutton), "toggled", 
		     GTK_SIGNAL_FUNC(toggle_button), &kerning);


  //********** Separator
  GtkWidget* sep1 = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), sep1, FALSE, FALSE, 0);


  //******** Action buttons
  GtkWidget* hboxa = gtk_hbox_new(TRUE, 0);
  gtk_container_border_width(GTK_CONTAINER(hboxa), 0);
  gtk_box_pack_start(GTK_BOX(vbox), hboxa, FALSE, FALSE, 0);
  GtkWidget* charb = gtk_button_new_with_label("Make character");
  gtk_signal_connect(GTK_OBJECT(charb), "clicked", 
		     GTK_SIGNAL_FUNC(showchar), fontclistw);
  gtk_box_pack_start(GTK_BOX(hboxa), charb, TRUE, TRUE, 0);
  GtkWidget* stringb = gtk_button_new_with_label("Make string");
  gtk_signal_connect(GTK_OBJECT(stringb), "clicked", 
		     GTK_SIGNAL_FUNC(showstring), fontclistw);
  gtk_box_pack_start(GTK_BOX(hboxa), stringb, TRUE, TRUE, 0);
  GtkWidget* tableb = gtk_button_new_with_label("Make font table");
  gtk_signal_connect(GTK_OBJECT(tableb), "clicked", 
		     GTK_SIGNAL_FUNC(fonttable), fontclistw);
  gtk_box_pack_start(GTK_BOX(hboxa), tableb, TRUE, TRUE, 0);

  //********** Message and progress area  -- Last in window
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_end(GTK_BOX(vbox), hboxm, FALSE, FALSE, 0);
  stbar = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(hboxm), stbar, TRUE, TRUE, 0);
  prbar = gtk_progress_bar_new();
  gtk_box_pack_end(GTK_BOX(hboxm), prbar, FALSE, FALSE, 0);
  gtk_widget_set_usize(prbar, 100, -1);

  
  //********** Final steps
  gtk_widget_show_all(mainwindow);
 
  // define GCs 
  
  // bwGC: black & white GC, with stippled fill pattern
  GdkGCValues gcval;
  int gcmask = GDK_GC_FUNCTION | GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | 
    GDK_GC_FILL | GDK_GC_STIPPLE;

  gcval.function = GDK_COPY;
  gcval.foreground.pixel = black.pixel;
  gcval.background.pixel = white.pixel;
  gcval.fill = GDK_STIPPLED;
  gcval.stipple = gdk_bitmap_create_from_data(mainwindow->window, 
					      stipple_bits, 
					      stipple_width, stipple_height);
  bwGC = gdk_gc_new_with_values(mainwindow->window, &gcval,
				(GdkGCValuesMask)gcmask);
   
  // redGC: red & white GC, with stippled fill pattern
  GdkColor red;
  gdk_color_parse("red", &red);
  gdk_colormap_alloc_color(colormap, &red, FALSE, TRUE);
  redGC = gdk_gc_new(mainwindow->window); 
  gdk_gc_copy(redGC, bwGC);
  gdk_gc_set_foreground(redGC, &red);
   
  // defGC: fg is black, bg is the default window bg (possibly gray)
  defGC = gdk_gc_new(mainwindow->window); 
  gdk_gc_set_foreground(defGC, &black);
  gdk_gc_set_background(defGC, &style->bg[GTK_STATE_NORMAL]);
  // Set the bg colour of black_gc in the default style to white
  gdk_gc_set_background(style->black_gc, &white);

  GtkWidget *windowicon = makeicon(mainwindow, font_xpm);
  gdk_window_set_icon(mainwindow->window, NULL, 
		      GTK_PIXMAP(windowicon)->pixmap, NULL);

  make_message_window();  // The async messages
  while (gtk_events_pending()) gtk_main_iteration();
  load_directory(GTK_COMBO(dirl));

  // Geometry hints
  GdkGeometry geometry;
  GdkWindowHints geometry_mask; 

  geometry_mask = GDK_HINT_MIN_SIZE;
  geometry.min_width = 350;
  geometry.min_height = 300;

  gtk_window_set_geometry_hints(GTK_WINDOW(mainwindow), NULL, 
				&geometry, geometry_mask); 

  gtk_main();
  return 1;  // Should not reach
}
