/*
 * Copyright (c) 1998 Ethan Fischer <allanon@u.washington.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "../../configure.h"

#ifdef NEWLOOK

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <X11/Xproto.h>
#include <X11/Xatom.h>

#include "../../include/afterstep.h"
#include "../../include/misc.h"
#include "../../include/parse.h"
#include "../../include/screen.h"
#include "../../include/stepgfx.h"

/*
 * if you add a member to this list, or to the MyStyle structure, 
 * remember to update new_style(), delete_style(), merge_styles(), and 
 * ParseMyStyle(). also fix_styles(), if the default from new_style() 
 * isn't good enough for Scr.MSDefault
 */
struct config style_config[] =
{
  {"~MyStyle", set_func_arg, (char **) F_DONE},
  {"Inherit", set_func_arg, (char **) F_INHERIT},
  {"Font", set_func_arg, (char **) F_FONT},
  {"ForeColor", set_func_arg, (char **) F_FORECOLOR},
  {"BackColor", set_func_arg, (char **) F_BACKCOLOR},
  {"TextStyle", set_func_arg, (char **) F_TEXTSTYLE},
#ifndef NO_TEXTURE
  {"MaxColors", set_func_arg, (char **) F_MAXCOLORS},
  {"BackGradient", set_func_arg, (char **) F_BACKGRADIENT},
  {"BackPixmap", set_func_arg, (char **) F_BACKPIXMAP},
#endif
  {"", NULL, NULL}
};

int style_func;
char *style_arg;

void
make_styles (void)
{
/* 
 * the user may not have set the default style, so set it here
 * all we need are reasonable defaults, and most defaults have been set 
 *  by new_style() already
 * we need FONT, FORECOLOR, and BACKCOLOR, at a minimum
 */
  if (Scr.MSDefault == NULL)
    if ((Scr.MSDefault = find_style ("default")) == NULL)
      Scr.MSDefault = new_style_with_name ("default");
  if ((Scr.MSDefault->set_flags & F_FORECOLOR) == 0)
    Scr.MSDefault->user_flags |= F_FORECOLOR;
  if ((Scr.MSDefault->set_flags & F_BACKCOLOR) == 0)
    {
      Scr.MSDefault->relief.fore = GetHilite (Scr.MSDefault->colors.back);
      Scr.MSDefault->relief.back = GetShadow (Scr.MSDefault->colors.back);
      Scr.MSDefault->user_flags |= F_BACKCOLOR;
    }
  if ((Scr.MSDefault->set_flags & F_FONT) == 0)
    {
      if (load_font (NULL, &Scr.MSDefault->font) == False)
	{
	  afterstep_err ("couldn't load default font", NULL, NULL, NULL);
	  exit (1);
	}
      Scr.MSDefault->user_flags |= F_FONT;
    }
  Scr.MSDefault->set_flags = Scr.MSDefault->inherit_flags | Scr.MSDefault->user_flags;

/* make sure the globals are defined */
  if (Scr.MSFWindow == NULL)
    Scr.MSFWindow = new_style_with_name ("FWindow");
  if (Scr.MSUWindow == NULL)
    Scr.MSUWindow = new_style_with_name ("UWindow");
  if (Scr.MSSWindow == NULL)
    Scr.MSSWindow = new_style_with_name ("SWindow");
  if (Scr.MSMenuTitle == NULL)
    Scr.MSMenuTitle = new_style_with_name ("MenuTitle");
  if (Scr.MSMenuItem == NULL)
    Scr.MSMenuItem = new_style_with_name ("MenuItem");
  if (Scr.MSMenuHilite == NULL)
    Scr.MSMenuHilite = new_style_with_name ("MenuHilite");
  if (Scr.MSMenuStipple == NULL)
    Scr.MSMenuStipple = new_style_with_name ("MenuStipple");
}

void
fix_styles (void)
{
  MyStyle *style;

/* fix up the styles, using the default style */
  for (style = Scr.first_style; style != NULL; style = style->next)
    merge_styles (Scr.MSDefault, style, False, False);
}


void
draw_styled_text (Window w, MyStyle * style, char *text, int x, int y)
{
  int len = strlen (text);
  change_style (style);
  switch (style->text_style)
    {
#ifdef I18N
#undef FONTSET
#define FONTSET (*style).font.fontset
#endif
/* 3d look #1 */
    case 1:
      XDrawString (dpy, w, Scr.ReliefGC, x - 2, y - 2, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x - 1, y - 2, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x + 0, y - 2, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x + 1, y - 2, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x + 2, y - 2, text, len);

      XDrawString (dpy, w, Scr.ReliefGC, x - 2, y - 1, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x - 2, y + 0, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x - 2, y + 1, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x - 2, y + 2, text, len);

      XDrawString (dpy, w, Scr.ShadowGC, x + 2, y + 0, text, len);
      XDrawString (dpy, w, Scr.ShadowGC, x + 2, y + 1, text, len);
      XDrawString (dpy, w, Scr.ShadowGC, x + 2, y + 2, text, len);
      XDrawString (dpy, w, Scr.ShadowGC, x + 1, y + 2, text, len);
      XDrawString (dpy, w, Scr.ShadowGC, x + 0, y + 2, text, len);

      XDrawString (dpy, w, Scr.ForeGC, x + 0, y + 0, text, len);
      break;
/* 3d look #2 */
    case 2:
      XDrawString (dpy, w, Scr.ShadowGC, x - 1, y - 1, text, len);
      XDrawString (dpy, w, Scr.ReliefGC, x + 0, y + 0, text, len);
      XDrawString (dpy, w, Scr.ForeGC, x + 1, y + 1, text, len);
      break;
/* normal text */
    default:
    case 0:
      XDrawString (dpy, w, Scr.ForeGC, x + 0, y + 0, text, len);
      break;
    }
}

Pixmap
make_styled_pixmap (MyStyle * style, int width, int height, Pixmap cache)
{
  Pixmap pixmap = None;
#ifndef NO_TEXTURE
  GC gc;
  XGCValues gcv;
  unsigned long gcm;

  pixmap = XCreatePixmap (dpy, Scr.Root, width, height, Scr.d_depth);
  gcv.foreground = style->colors.fore;
  gcv.background = style->colors.back;
  gcv.fill_style = FillSolid;
  gcv.graphics_exposures = False;
  gcm = GCForeground | GCBackground | GCFillStyle | GCGraphicsExposures;
  if (style->texture_type >= 128)
    {
      gcv.fill_style = FillTiled;
      gcv.tile = style->pixmap;
      gcm |= GCTile;
    }
  gc = XCreateGC (dpy, pixmap, gcm, &gcv);
  switch (style->texture_type)
    {
    case 0:
    case 128:
      XFillRectangle (dpy, pixmap, gc, 0, 0, width, height);
      break;
    case 1:
      if (!DrawDegradeRelief (dpy, pixmap, 0, 0, width, height,
			      style->gradient.from,
			      style->gradient.to, 0,
			      style->max_colors))
	{
	  XFreePixmap (dpy, pixmap);
	  pixmap = None;
	}
      break;
    case 2:
    case 3:
      /* use the cache if we can */
      if ((cache != None) && (width < Scr.MyDisplayWidth))
	{
	  XCopyArea (dpy, cache, pixmap, gc, 0, 0, width, height, 0, 0);
	}
      else if (!DrawHGradient (dpy, pixmap, 0, 0, width, height,
			       style->gradient.from,
			       style->gradient.to, 0,
			       style->max_colors,
			       style->texture_type - 2))
	{
	  XFreePixmap (dpy, pixmap);
	  pixmap = None;
	}
      break;
    case 4:
    case 5:
      if (!DrawVGradient (dpy, pixmap, 0, 0, width, height,
			  style->gradient.from,
			  style->gradient.to, 0,
			  style->max_colors,
			  style->texture_type - 4))
	{
	  XFreePixmap (dpy, pixmap);
	  pixmap = None;
	}
      break;
    case 129:
    default:
      XFreePixmap (dpy, pixmap);
      pixmap = None;
      break;
    }
  XFreeGC (dpy, gc);
#endif /* NO_TEXTURE */
  return pixmap;
}

/* set a window's background for XClearArea */
void
change_styled_background (Window w, MyStyle * style)
{
#ifndef NO_TEXTURE
  if (style->set_flags & F_BACKPIXMAP)
    XSetWindowBackgroundPixmap (dpy, w, style->pixmap);
  else
#endif /* NO_TEXTURE */
    XSetWindowBackground (dpy, w, style->colors.back);
}

/*
 * set the standard global drawing GCs
 * avoids resetting the GCs when possible
 */
void
change_style (MyStyle * style)
{
  static MyStyle *old_style = NULL;

/* set the GCs if the style is different */
  if (old_style != style)
    {
      old_style = style;
      change_styled_gc (style, Scr.ForeGC, Scr.BackGC, Scr.ReliefGC, Scr.ShadowGC);
    }
}

void
change_styled_gc (MyStyle * style, GC foreGC, GC backGC, GC reliefGC, GC shadowGC)
{
  XGCValues gcv;
  unsigned long gcm;

/* set the drawing GC */
  gcv.foreground = style->colors.fore;
  gcv.background = style->colors.back;
  gcv.font = style->font.font->fid;
  gcv.fill_style = FillSolid;
  gcm = GCForeground | GCBackground | GCFont | GCFillStyle;
  XChangeGC (dpy, foreGC, gcm, &gcv);

#ifndef NO_TEXTURE
  if ((style->texture_type != 0) && (style->pixmap != None))
    {
      gcv.fill_style = FillTiled;
      gcv.tile = style->pixmap;
      gcm |= GCTile;
    }
#endif /* NO_TEXTURE */
  XChangeGC (dpy, backGC, gcm, &gcv);

/* set the relief GC */
  gcv.foreground = style->relief.fore;
  gcv.background = style->relief.back;
  gcv.fill_style = FillSolid;
  gcm = GCForeground | GCBackground | GCFont | GCFillStyle;
  XChangeGC (dpy, reliefGC, gcm, &gcv);

/* set the shadow GC */
  gcv.foreground = style->relief.back;
  gcv.background = style->relief.fore;
  gcv.fill_style = FillSolid;
  gcm = GCForeground | GCBackground | GCFont | GCFillStyle;
  XChangeGC (dpy, shadowGC, gcm, &gcv);
}

MyStyle *
new_style (void)
{
  MyStyle *style = (MyStyle *) safemalloc (sizeof (MyStyle));

  style->next = Scr.first_style;
  Scr.first_style = style;

  style->user_flags = 0;
  style->inherit_flags = 0;
  style->set_flags = 0;
  style->name = NULL;
  style->text_style = 0;
  style->font.name = NULL;
  style->font.font = NULL;
#ifdef I18N
  style->font.fontset = NULL;
#endif
  style->colors.fore = GetColor ("white");
  style->colors.back = GetColor ("black");
  style->relief.fore = style->colors.back;
  style->relief.back = style->colors.fore;
  style->texture_type = 0;
#ifndef NO_TEXTURE
  {
    int i;
    for (i = 0; i < 3; i++)
      {
	style->gradient.from[i] = style->colors.fore;
	style->gradient.to[i] = style->colors.fore;
      }
  }
  style->pixmap = None;
  if (Scr.d_depth > 16)
    style->max_colors = 128;
  else if (Scr.d_depth > 8)
    style->max_colors = 32;
  else
    style->max_colors = 10;
#endif
  return style;
}

MyStyle *
new_style_with_name (char *name)
{
  MyStyle *style = new_style ();

  if (name != NULL)
    {
      style->name = (char *) safemalloc (strlen (name) + 1);
      strcpy (style->name, name);
    }
  return style;
}

/*
 * deletes memory alloc'd for style (including members)
 */

void
delete_style (MyStyle * style)
{
  MyStyle *s1;
  MyStyle *s2;

  /* remove ourself from the list */
  for (s1 = s2 = Scr.first_style; s1 != NULL; s2 = s1, s1 = s1->next)
    if (s1 == style)
      break;
  if (s1 != NULL)
    {
      if (Scr.first_style == s1)
	Scr.first_style = s1->next;
      else
	s2->next = s1->next;
    }
  /* delete members */
  if (style->name != NULL);
  free (style->name);
  if (style->user_flags & F_FONT)
    {
      if (style->font.name != NULL);
      free (style->font.name);
      XFreeFont (dpy, style->font.font);
    }
#ifndef NO_TEXTURE
  if (style->user_flags & F_BACKPIXMAP)
    XFreePixmap (dpy, style->pixmap);
#endif

  /* free our own mem */
  free (style);
}

/* find a style by name */
MyStyle *
find_style (char *name)
{
  MyStyle *style;
  for (style = Scr.first_style; style != NULL; style = style->next)
    if (!mystrcasecmp (style->name, name))
      break;
  return style;
}

/*
 * merges two styles, with the result in child
 * if override == False, will not override fields already set
 * if copy == True, copies instead of inheriting; this is important, because 
 *   inherited members are deleted when their real parent is deleted; don't 
 *   inherit if the parent style could go away before the child
 */
void
merge_styles (MyStyle * parent, MyStyle * child, Bool override, Bool copy)
{
  if (parent->set_flags & F_FONT)
    {
      if ((override == True) && (child->user_flags & F_FONT))
	unload_font (&child->font);
      if ((override == True) || !(child->set_flags & F_FONT))
	{
	  if (copy == False)
	    {
	      child->font = parent->font;
	      child->user_flags &= ~F_FONT;
	      child->inherit_flags |= F_FONT;
	    }
	  else
	    {
	      if (load_font (parent->font.name, &child->font) == True)
		{
		  child->user_flags |= F_FONT;
		  child->inherit_flags &= ~F_FONT;
		}
	    }
	}
    }
  if (parent->set_flags & F_TEXTSTYLE)
    {
      if ((override == True) || !(child->set_flags & F_TEXTSTYLE))
	{
	  child->text_style = parent->text_style;

	  if (copy == False)
	    {
	      child->user_flags &= ~F_TEXTSTYLE;
	      child->inherit_flags |= F_TEXTSTYLE;
	    }
	  else
	    {
	      child->user_flags |= F_TEXTSTYLE;
	      child->inherit_flags &= ~F_TEXTSTYLE;
	    }
	}
    }
  if (parent->set_flags & F_FORECOLOR)
    {
      if ((override == True) || !(child->set_flags & F_FORECOLOR))
	{
	  if (override == True)
	    child->texture_type = parent->texture_type;
	  child->colors.fore = parent->colors.fore;
	  if (copy == False)
	    {
	      child->user_flags &= ~F_FORECOLOR;
	      child->inherit_flags |= F_FORECOLOR;
	    }
	  else
	    {
	      child->user_flags |= F_FORECOLOR;
	      child->inherit_flags &= ~F_FORECOLOR;
	    }
	}
    }
  if (parent->set_flags & F_BACKCOLOR)
    {
      if ((override == True) || !(child->set_flags & F_BACKCOLOR))
	{
	  child->colors.back = parent->colors.back;
	  child->relief = parent->relief;
	  if (copy == False)
	    {
	      child->user_flags &= ~F_BACKCOLOR;
	      child->inherit_flags |= F_BACKCOLOR;
	    }
	  else
	    {
	      child->user_flags |= F_BACKCOLOR;
	      child->inherit_flags &= ~F_BACKCOLOR;
	    }
	}
    }
#ifndef NO_TEXTURE
  if (parent->set_flags & F_MAXCOLORS)
    {
      if ((override == True) || !(child->set_flags & F_MAXCOLORS))
	{
	  child->max_colors = parent->max_colors;
	  if (copy == False)
	    {
	      child->user_flags &= ~F_MAXCOLORS;
	      child->inherit_flags |= F_MAXCOLORS;
	    }
	  else
	    {
	      child->user_flags |= F_MAXCOLORS;
	      child->inherit_flags &= ~F_MAXCOLORS;
	    }
	}
    }
  if (parent->set_flags & F_BACKGRADIENT)
    {
      if ((override == True) || !(child->set_flags & F_BACKGRADIENT))
	{
	  if (override == True)
	    child->texture_type = parent->texture_type;
	  child->gradient = parent->gradient;
	  if (copy == False)
	    {
	      child->user_flags &= ~F_BACKGRADIENT;
	      child->inherit_flags |= F_BACKGRADIENT;
	    }
	  else
	    {
	      child->user_flags |= F_BACKGRADIENT;
	      child->inherit_flags &= ~F_BACKGRADIENT;
	    }
	}
    }
  if (parent->set_flags & F_BACKPIXMAP)
    {
      if ((override == True) && (child->user_flags & F_BACKPIXMAP))
	XFreePixmap (dpy, child->pixmap);
      if ((override == True) || !(child->set_flags & F_BACKPIXMAP))
	{
	  if (override == True)
	    child->texture_type = parent->texture_type;
	  if (copy == False)
	    {
	      child->pixmap = parent->pixmap;
	      child->user_flags &= ~F_BACKPIXMAP;
	      child->inherit_flags |= F_BACKPIXMAP;
	    }
	  else
	    {
	      Window root;
	      int junk, width, height, depth;
	      GC gc = DefaultGC (dpy, Scr.screen);
	      XGetGeometry (dpy, parent->pixmap, &root, &junk, &junk,
			    &width, &height, &junk, &depth);
	      child->pixmap = XCreatePixmap (dpy, root, width, height, depth);
	      XCopyArea (dpy, parent->pixmap, child->pixmap, gc,
			 0, 0, width, height, 0, 0);
	      child->user_flags |= F_BACKPIXMAP;
	      child->inherit_flags &= ~F_BACKPIXMAP;
	    }
	}
    }
#endif /* NO_TEXTURE */
  child->set_flags = child->user_flags | child->inherit_flags;
}

void
ParseMyStyle (char *tline, FILE * fd, char **junk, int *junk2)
{
  MyStyle *style;
  char *newline = safemalloc (MAXLINELENGTH + 1);
  int i;
  Bool done = False;
  char *name = stripcpy2 (tline, 0, True);

  if (name == NULL)
    {
      afterstep_err ("bad style name: '%s'", tline, NULL, NULL);
      return;
    }
/* if this style was already defined, find it */
  if ((style = find_style (name)) != NULL)
    {
      free (name);
      name = style->name;
    }
  else
    {
      style = new_style ();
      style->name = name;
    }

  while (done == False)
    {
      char *p, *tempstring;
      p = fgets (newline, MAXLINELENGTH, fd);
      if (p == NULL)
	break;
      while (isspace (*p))
	p++;
      /* remove comments from the line */
      tempstring = p;
      while (*tempstring != '\0' && (tempstring = strchr (tempstring, '#')) != NULL)
	{
	  i = 1;
	  /* make sure it's not color */
	  while (isxdigit (*(tempstring + i)))
	    i++;
	  /* the minimum color definition is '#000' or similar */
	  if (i < 4)
	    *tempstring = '\0';
	  else
	    tempstring += i;
	}
      if (*p != '\0')
	{
	  style_func = F_ERROR;
	  style_arg = NULL;
	  match_string (style_config, p, "bad style body:", fd);
	  if (style_arg == NULL)
	    {
	      afterstep_err ("bad style arg: %s", p, NULL, NULL);
	    }
	  else
	    {
	      style->inherit_flags &= ~style_func;
	      switch (style_func)
		{
		case F_INHERIT:
		  {
		    MyStyle *parent = find_style (style_arg);
		    if (parent != NULL)
		      merge_styles (parent, style, True, False);
		  }
		  break;
		case F_FONT:
		  if (style->user_flags & style_func)
		    unload_font (&style->font);
		  style->user_flags &= ~style_func;
		  if (load_font (style_arg, &style->font) == True)
		    style->user_flags |= style_func;
		  break;
		case F_TEXTSTYLE:
		  style->text_style = strtol (style_arg, NULL, 10);
		  style->user_flags |= style_func;
		  break;
		case F_FORECOLOR:
		  style->colors.fore = GetColor (style_arg);
		  style->user_flags |= style_func;
		  break;
		case F_BACKCOLOR:
		  style->texture_type = 0;
		  style->colors.back = GetColor (style_arg);
		  style->relief.fore = GetHilite (style->colors.back);
		  style->relief.back = GetShadow (style->colors.back);
		  style->user_flags |= style_func;
		  break;
#ifndef NO_TEXTURE
		case F_MAXCOLORS:
		  style->max_colors = strtol (style_arg, NULL, 10);
		  style->user_flags |= style_func;
		  break;
		case F_BACKGRADIENT:
		  {
		    char *ptr;
		    int type = strtol (style_arg, &ptr, 10);
		    if (ParseColor (ptr, style->gradient.from, style->gradient.to) == True)
		      {
			style->texture_type = type;
			style->user_flags |= style_func;
		      }
		    else
		      afterstep_err ("bad gradient: %s", style_arg, NULL, NULL);
		  }
		  break;
		case F_BACKPIXMAP:
		  {
		    char *ptr;
		    int type = strtol (style_arg, &ptr, 10);
		    char *tmp = stripcpy (ptr);
		    int colors = -1;
		    if (style->set_flags & F_MAXCOLORS)
		      colors = style->max_colors;
		    if (style->user_flags & style_func)
		      XFreePixmap (dpy, style->pixmap);
		    style->user_flags &= ~style_func;
		    if ((style->pixmap = GetXPMTile (tmp, colors)) != None)
		      {
			style->texture_type = type;
			style->user_flags |= style_func;
		      }
		    else
		      afterstep_err ("unable to load pixmap: '%s'", ptr, NULL, NULL);
		    free (tmp);
		  }
		  break;
#endif
		case F_DONE:
		  done = True;
		  break;

		case F_ERROR:
		default:
		  afterstep_err ("unknown style command: %s", p, NULL, NULL);
		  break;
		}
	    }
	  style->set_flags = style->inherit_flags | style->user_flags;
	  free (style_arg);
	}
    }

  free (newline);
}

void
set_style (char *text, FILE * fd, char **style, int *junk2)
{
  char *name = stripcpy2 (text, 0, True);
  if (name != NULL)
    {
      *(MyStyle **) style = find_style (name);
      if (*style == NULL)
	afterstep_err ("unknown style name: '%s'", name, NULL, NULL);
      free (name);
    }
}

void
set_func_arg (char *text, FILE * fd, char **value, int *junk)
{
  style_arg = stripcpy (text);
  style_func = (unsigned long) value;
}

#endif /* NEWLOOK */
