/*
 * Copyright (C) 1998 Ethan Fischer <allanon@crystaltokyo.com>
 * Based on code by Guylhem Aznar <guylhem@oeil.qc.ca>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>

#include "../configure.h"
#include "../include/aftersteplib.h"

#define NEW(a) ((a*)safemalloc(sizeof(a)))

#ifdef I18N
#ifdef __STDC__
#define XTextWidth(x,y,z)	XmbTextEscapement(x ## set,y,z)
#else
#define XTextWidth(x,y,z)	XmbTextEscapement(x/**/set,y,z)
#endif
#define XDrawString(t,u,v,w,x,y,z)	XmbDrawString(t,u,FONTSET,v,w,x,y,z)
#endif

Bool balloon_show = 0;

static Balloon *balloon_first = NULL;
static Balloon *balloon_current = NULL;
static GC balloon_GC = NULL;
static MyFont balloon_font =
{NULL, NULL};
static ColorPair balloon_colors =
{0, 0};
static int balloon_yoffset = 0;
static int balloon_border_width = 0;
static Pixel balloon_border_color = 0;
static Window balloon_window = None;
static int balloon_delay = 0;

Bool
balloon_parse (char *tline, FILE * fd)
{
  Bool handled = False;
  while (isspace (*tline))
    tline++;

  /* check - is this one of our keywords? */
  if (mystrncasecmp (tline, "Balloon", 7))
    return False;

  if (mystrncasecmp (tline, "Balloons", 8) == 0)
    {
      handled = True;
      balloon_show = True;
    }
  else if (mystrncasecmp (tline, "BalloonBack", 11) == 0)
    {
      int len;
      handled = True;
      for (tline += 11; isspace (*tline); tline++);
      for (len = 0; tline[len] != '\0' && !isspace (tline[len]); len++);
      tline[len] = '\0';
      balloon_colors.back = GetColor (tline);
    }
  else if (mystrncasecmp (tline, "BalloonFore", 11) == 0)
    {
      int len;
      handled = True;
      for (tline += 11; isspace (*tline); tline++);
      for (len = 0; tline[len] != '\0' && !isspace (tline[len]); len++);
      tline[len] = '\0';
      balloon_colors.fore = GetColor (tline);
    }
  else if (mystrncasecmp (tline, "BalloonFont", 11) == 0)
    {
      int len;
      handled = True;
      for (tline += 11; isspace (*tline); tline++);
      for (len = 0; tline[len] != '\0' && !isspace (tline[len]); len++);
      tline[len] = '\0';
      load_font (tline, &balloon_font);
    }
  else if (mystrncasecmp (tline, "BalloonBorderColor", 18) == 0)
    {
      int len;
      handled = True;
      for (tline += 18; isspace (*tline); tline++);
      for (len = 0; tline[len] != '\0' && !isspace (tline[len]); len++);
      tline[len] = '\0';
      balloon_border_color = GetColor (tline);
    }
  else if (mystrncasecmp (tline, "BalloonBorderWidth", 18) == 0)
    {
      handled = True;
      balloon_border_width = strtol (&tline[18], NULL, 10);
      if (balloon_border_width < 0)
	balloon_border_width = 0;
    }
  else if (mystrncasecmp (tline, "BalloonYOffset", 14) == 0)
    {
      handled = True;
      balloon_yoffset = strtol (&tline[14], NULL, 10);
    }
  else if (mystrncasecmp (tline, "BalloonDelay", 12) == 0)
    {
      handled = True;
      balloon_delay = strtol (&tline[12], NULL, 10);
      if (balloon_delay < 0)
	balloon_delay = 0;
    }
  return handled;
}

/* call this after parsing config files to finish initialization */
void
balloon_setup (void)
{
  XSizeHints hints;
  XSetWindowAttributes attributes;
  XGCValues gcv;

  if (balloon_font.font == NULL)
    {
      balloon_font.font = XLoadQueryFont (dpy, "fixed");
      if (balloon_font.font == NULL)
	{
	  fprintf (stderr, "%s: No fonts available.\n", MyName);
	  exit (1);
	}
    }
  balloon_font.height = balloon_font.font->ascent + balloon_font.font->descent;
  balloon_font.y = balloon_font.font->ascent;

  if (balloon_colors.fore == balloon_colors.back)
    {
      balloon_colors.fore = GetColor ("white");
      balloon_colors.back = GetColor ("black");
    }

  attributes.override_redirect = True;
  attributes.border_pixel = balloon_border_color;
  balloon_window = XCreateWindow (dpy, RootWindow (dpy, screen), 0, 0, 64, 64,
				  balloon_border_width, CopyFromParent,
				  InputOutput, CopyFromParent,
				  CWOverrideRedirect | CWBorderPixel,
				  &attributes);
  XSelectInput (dpy, balloon_window, ExposureMask);
  hints.flags = USPosition;
  XSetWMNormalHints (dpy, balloon_window, &hints);
  XSetWindowBackground (dpy, balloon_window, balloon_colors.back);

  /* make the drawing gc */
  gcv.graphics_exposures = False;
  gcv.foreground = balloon_colors.fore;
  gcv.background = balloon_colors.back;
  gcv.font = balloon_font.font->fid;
  balloon_GC = XCreateGC (dpy, RootWindow (dpy, screen), GCGraphicsExposures | GCForeground | GCBackground | GCFont, &gcv);
}

/*
 * parent must be selecting EnterNotify and LeaveNotify events
 */
Balloon *
balloon_new (Window parent)
{
  Balloon *balloon;

  balloon = NEW (Balloon);

  (*balloon).next = balloon_first;
  balloon_first = balloon;

  (*balloon).parent = parent;
  (*balloon).text = NULL;

  return balloon;
}

Balloon *
balloon_new_with_text (Window parent, char *text)
{
  Balloon *balloon;

  balloon = balloon_new (parent);
  if (text != NULL)
    (*balloon).text = mystrdup (text);

  return balloon;
}

Balloon *
balloon_find (Window parent)
{
  Balloon *balloon;

  for (balloon = balloon_first; balloon != NULL; balloon = (*balloon).next)
    if ((*balloon).parent == parent)
      break;
  return balloon;
}

/* balloon_delete() checks for NULL so that:

 *   balloon_delete(balloon_find(win));
 *
 * will always work
 */
void
balloon_delete (Balloon * balloon)
{
  if (balloon == NULL)
    return;

  while (timer_remove_by_data (balloon));

  if (balloon_first == balloon)
    balloon_first = (*balloon).next;
  else if (balloon_first != NULL)
    {
      Balloon *ptr;
      for (ptr = balloon_first; (*ptr).next != NULL; ptr = (*ptr).next)
	if ((*ptr).next == balloon)
	  break;
      if ((*ptr).next == balloon)
	(*ptr).next = (*balloon).next;
    }

  if ((*balloon).text != NULL)
    free ((*balloon).text);

  free (balloon);
}

void
balloon_set_text (Balloon * balloon, char *text)
{
  if ((*balloon).text != NULL)
    free ((*balloon).text);
  (*balloon).text = (text != NULL) ? mystrdup (text) : NULL;
}

static void
balloon_map (Balloon * balloon)
{
  if ((*balloon).text != NULL)
    {
      XWindowChanges wc;
      Window root;
      int x, y, width, height, border, junk;
      XGetGeometry (dpy, (*balloon).parent, &root, &x, &y, &width, &height, &border, &junk);
      XTranslateCoordinates (dpy, (*balloon).parent, RootWindow (dpy, screen), 0, 0, &wc.x, &wc.y, &root);
      wc.width = 4 + XTextWidth (balloon_font.font, (*balloon).text, strlen ((*balloon).text));
      wc.height = 4 + balloon_font.height;
      wc.x += (width - wc.width) / 2 - balloon_border_width;
      if (balloon_yoffset >= 0)
	wc.y += balloon_yoffset + height + 2 * border;
      else
	wc.y += balloon_yoffset - wc.height - 2 * balloon_border_width;
      /* clip to screen */
      if (wc.x < 2)
	wc.x = 2;
      if (wc.x > DisplayWidth (dpy, screen) - wc.width - 2)
	wc.x = DisplayWidth (dpy, screen) - wc.width - 2;
      XConfigureWindow (dpy, balloon_window, CWX | CWY | CWWidth | CWHeight, &wc);
      XMapRaised (dpy, balloon_window);
    }
}

static void
balloon_timer_handler (void *data)
{
  balloon_map ((Balloon *) data);
}

Bool
balloon_handle_event (XEvent * event)
{
  Balloon *balloon;

  if (!balloon_show)
    return False;

  /* is this our event? */
  if ((*event).type != EnterNotify && (*event).type != LeaveNotify && (*event).type != Expose)
    return False;
  if ((*event).type == EnterNotify || (*event).type == LeaveNotify)
    {
      balloon = balloon_find (event->xcrossing.window);
      if (balloon == NULL)
	return False;
      balloon_current = balloon;
    }
  else if (event->xexpose.window != balloon_window)
    return False;

  balloon = balloon_current;

  /* handle the event */
  switch ((*event).type)
    {
    case EnterNotify:
      if (event->xcrossing.detail != NotifyInferior)
	{
	  if (balloon_delay == 0)
	    balloon_map (balloon);
	  else
	    timer_new (balloon_delay, &balloon_timer_handler, (void *) balloon);
	}
      break;
    case LeaveNotify:
      if (event->xcrossing.detail != NotifyInferior)
	{
	  if (!timer_remove_by_data (balloon))
	    XUnmapWindow (dpy, balloon_window);
	}
      break;
    case Expose:
#undef FONTSET
#define FONTSET balloon_font.fontset
      XDrawString (dpy, balloon_window, balloon_GC, 2, 1 + balloon_font.y, (*balloon).text, strlen ((*balloon).text));
      break;
    default:
      break;
    }

  return True;
}
