/* 
   sitecopy, for managing remote web sites. WebDAV client routines.
   Copyright (C) 1998-99, Joe Orton <joe@orton.demon.co.uk>
                                                                     
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   $Id: httpdav.c,v 1.1.1.1 1999/11/21 19:47:13 davek Exp $
*/

/* This file is a collection of routines to implement a basic WebDAV
 * client, including an HTTP/1.1 client. 
 * Transparently supports basic and digest authentication.
 */

/* HTTP method implementation:
 *   Call, in this order:
 *     1.  http_request_init()  - set up the request
 *     2.  http_request()       - make the request
 *     3.  http_request_end()   - clean up the request
 */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#ifdef __EMX__
#include <sys/select.h>
#endif

#include <netinet/in.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif 
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif /* HAVE_STDLIB_H */
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */

#ifndef HAVE_SNPRINTF
#include <snprintf.h>
#endif

#include <basename.h>
#include <dirname.h>
#include <strsplit.h>

#include "frontend.h"
#include "protocol.h"
#include "httpdav.h"
#include "common.h"
#include "socket.h"
#include "base64.h"
#include "md5.h"

/* Connection information... */
int http_sock;
/*bool http_persistent;  Whether persistent connections work */
/* For persistent connections, this is set to false when the socket
 * is closed by the server, so the next request has to open it again.
 * For non-persistent connections, this is not used */
bool http_connected;

/* Time in seconds to wait for the server to give us a 100 Continue
 * after submitting a PUT with an "Expect: 100-continue" header.
 */
#define HTTP_EXPECT_TIMEOUT 15
/* 100-continue only used if size > HTTP_EXPECT_MINSIZ */
#define HTTP_EXPECT_MINSIZE 512

/* Whether the current server respects the Expect: 100-continue header */
int http_expect_works; /* == 0 if it we don't know, 1 if it does,
			    -1 if it doesn't support 100-continue */

bool http_webdav_server = false;
bool http_init_checks = true;

const char *http_quotes = "\"'";
const char *http_whitespace = " \r\n\t";

#ifdef HAVE_LIBEXPAT
/* WebDAV Fetch mode */

#include <xmlparse.h>

typedef enum {
    dav_xml_multistatus = 0,
    dav_xml_response,
    dav_xml_responsedescription,
    dav_xml_href,
    dav_xml_propstat,
    dav_xml_prop,
    dav_xml_status,
    dav_xml_getlastmodified,
    dav_xml_getcontentlength,
    dav_xml_resourcetype,
    dav_xml_collection,
    dav_xml_unknown,
    dav_xml_root
} dav_xml_tag_t;

/* We get the tag_t from the names array below */
const char *dav_xml_tagnames[] = {
    "DAV:multistatus",
    "DAV:response",
    "DAV:responsedescription",
    "DAV:href",
    "DAV:propstat",
    "DAV:prop",
    "DAV:status",
    "DAV:getlastmodified",
    "DAV:getcontentlength",
    "DAV:resourcetype",
    "DAV:collection",
    NULL, /* end-of-list marker */
    "@<root>@" /* filler, so we can index this by dav_xml_tag_t */
};

typedef struct dav_xml_ns_s dav_xml_ns;

/* Linked list of namespace scopes */
struct dav_xml_ns_s {
    char *name;
    char *value;
    dav_xml_ns *next;
};

struct dav_xml_state {
    dav_xml_tag_t tag;
    char *tag_name; /* The full tag name */
    char *default_ns; /* The current default namespace */
    dav_xml_ns *nspaces; /* List of other namespace scopes */
    struct dav_xml_state *parent; /* The parent in the tree */
};

/* We pass around a dav_xmldoc as the userdata using expat.  This
 * maintains the current state of the parse and various other bits and
 * bobs. Within the parse, we store the current branch of the tree,
 * i.e., the current element and all its parents, but nothing other
 * than that. 
 *
 * The files list is filled as we go (by dav_fetch_gotresource), but
 * always kept in sorted order, since no ordering of resources in the
 * PROPFIND response is given. 
 */
struct dav_xmldoc {
    /* Points to the root of the document */
    struct dav_xml_state *root;
    /* Points to the current element in the document */
    struct dav_xml_state *current;
    /* Whether we want to collect CDATA at the moment or not */
    bool want_cdata;
    /* The cdata we've collected so far - grows as we collect
     * more. */
    char *cdata;
    /* How big the buffer is atm */
    size_t cdata_buflen;
    /* How much cdata we've collected so far */
    size_t cdata_len; 
    /* Is it valid? */
    bool valid;
    /* Temporary store of file info */
    struct proto_file_t *file;
    /* The complete fetch list */
    struct proto_file_t *files;
    /* The collection we did a PROPFIND on */
    const char *fetch_root;
};

/* The initial size of the cdata buffer, and the minimum size by which
 * it grows each time we overflow it. For the PROPFIND requests
 * we do for sitecopy, the only cdata segments we collect are very small.
 */
#define CDATABUFSIZ 128
/* If the cdata buffer size is larger than this when we've finished
 * with its contents, then we reallocate it. Otherwise, we just
 * zero it out. Reallocation -> a free() and a malloc(). No realloc -> 
 * just a memset().
 */
#define CDATASHRINK 128

/* Prototypes */
void dav_xml_startelm( void *userdata, const char *tag, const char **atts );
void dav_xml_endelm( void *userdata, const char *tag );
void dav_xml_cdata( void *userdata, const char *cdata, int len );
bool dav_xml_parsetag( struct dav_xml_state *state,
		       const char *tag, const char **atts );

int dav_fetch_getdepth( const char *href );
bool dav_fetch_parse_href( struct dav_xmldoc *doc );
void dav_fetch_gotresource( struct dav_xmldoc *doc );

void dav_xml_parsebody( void *userdata, const char *buffer, const size_t len );
void http_get_content_charset( const char *name, const char *value );

char *http_content_charset;

#endif /* HAVE_LIBEXPAT */

/* Handy macro to free things. */
#define DOFREE(x) if( x!=NULL ) free( x )

#define EOL "\r\n"
#define HTTP_PORT 80

const char *http_useragent = PACKAGE "/" VERSION;

/* We need to remember the remote host and port even after connecting
 * the socket e.g. so we can form the Destination: header */
char *http_remotehost;
struct in_addr http_remoteaddr;
int http_remoteport;

/* The transfer encoding types */
enum http_te_t {
    /* TODO: Support other types */
    http_te_none,
    http_te_chunked,
    http_te_unknown
};

typedef enum {
    http_auth_basic,
    http_auth_digest,
    http_auth_none
} http_auth_t;

/* HTTP Authentication.
 *   http_auth holds the type of authentication last used.
 *   http_auth_cookie holds the last cookie used for basic auth.
 *   http_username and http_password hold the username and password
 *     last used.
 *   http_auth_realm, _opaquea, _nonce are all used by digest auth.
 */
http_auth_t http_auth;
char *http_username, *http_password;
char *http_auth_cookie;
char *http_auth_realm, *http_auth_opaque, *http_auth_nonce;

/* Adds the client authentication headers to the request */
void http_auth_request( http_req_t *req, char *headers );

/* Parses the authentication details sent by the server */
void http_auth_challenge( char *challenge );

int http_mkdir_works;

/* Sets up the body size for the given request */
int http_req_bodysize( http_req_t *req );

/* The callback for GET requests (to allow signalling progress to the FE) */
void http_get_callback( void *user, const char *buffer, const size_t len );

/* Do a dummy MKDIR with PUT */
int http_mkdir_with_put( const char *realdir );

/* Holds the error message to be passed up */
char http_error[BUFSIZ];

/* Put the fixed headers into the request */
void http_req_fixedheaders( http_req_t *req );

/* Concatenates the remote server name with :port if port!=80 on to 
 * the end of str. */
void http_strcat_remoteserver( char *str );

/* Header parser for OPTIONS requests. */
void http_options_parsehdr( const char *name, const char *value );

int http_parse_status( http_req_t *req, char *status_line );

/* Sends the body down the wire */
int http_req_sendbody( http_req_t *req );

/* Encodes the absPath sectionf of a URI using %<hex><hex> encoding */
char *uri_abspath_encode( const char *abs_path );

/* Decodes a URI */
char *uri_decode( const char *uri );

/* Opens the connection to the remote server.
 * Returns:
 *  PROTO_OK       on success
 *  PROTO_CONNECT  if the socket couldn't be connected
 */
int http_open( void );

/* Closes the connection.
 * Always returns PROTO_OK
 */
int http_close( void );

/* This doesn't really connect to the server.
 * Returns
 *  PROTO_LOOKUP if the hostname of the server could not be resolved
 *  PROTO_LOCALHOST if the hostname of the local machine could not be
 *    found
 *  PROTO_CONNECT if the socket could not be connected
 *  PROTO_OK if the connection was made successfully.
 */
int http_init( const char *remote_root,
	       const char *hostname, const int port,
	       const char *username, const char *password ) {
    int ret;
    /* Look up the remote host name */
    if( http_remotehost!=NULL ) free( http_remotehost );
    http_remotehost = strdup( hostname );
    http_remoteport = port;
    fe_connection( fe_namelookup );
    if( host_lookup( http_remotehost, &http_remoteaddr ) )
	return PROTO_LOOKUP;
    http_connected = false;
    http_expect_works = 0; /* we don't know yet */
    http_mkdir_works = 0; /* we don't know yet */
    /* Initialize the authentication variables */
    DOFREE( http_auth_nonce );
    DOFREE( http_auth_realm );
    DOFREE( http_auth_opaque );
    http_auth = http_auth_none;
    http_username = strdup( username );
    http_password = strdup( password );
    ret = http_open();
    /* Drop out if that failed, or they don't want the OPTIONS */
    if( (!http_init_checks) || (ret != PROTO_OK) )
	return ret;
    /* Capability discovery... we don't care whether this
     * actually works or not, we just want http_webdav_server
     * set appropriately. */
    (void) http_options( remote_root );
    return PROTO_OK;
}

int http_finish( void ) {
    free( http_username );
    free( http_password );
    if( http_connected ) http_close( );
    return PROTO_OK;
}

/* This parses the response status line, and plonks the numeric
 * status value in status.
 * Returns: PROTO_OK on success, PROTO_ERROR otherwise */
int http_parse_status( http_req_t *req, char *status_line ) {
    char *part;
    DEBUG( DEBUG_HTTP, "HTTP response line: %s", status_line );
    /* Save the line for error codes later */
    memset( http_error, 0, BUFSIZ );
    strncpy( http_error, status_line, BUFSIZ );
    /* Strip off the CRLF for the status line */
    if( (part = strchr( http_error, '\r' )) != NULL )
	*part = '\0';
    /* Parse the status code */
    part = strtok( status_line, " " ); /* "HTTP/1.1" */
    if( part==NULL ) return PROTO_ERROR;
    part = strtok( NULL, " " ); /* xxx */
    if( part==NULL ) return PROTO_ERROR;
    req->status = atoi( part );
    req->class = req->status / 100;
    DEBUG( DEBUG_HTTP, "HTTP reponse code: %d\n", req->status );
    return PROTO_OK;
}

/* Add the authentication headers to the request. 
 * FIXME: Check whether this leaks like a sieve.
 */
void http_auth_request( http_req_t *req, char *headers ) {
    char tmp[BUFSIZ];
    char *a1, *md5_a1, *a2, *x, *y, *z, *temp, *auth_response,
	*v, *w, *ei; /* yarg, that enough? */

    switch( http_auth ) {
    case http_auth_basic:
	strcat( headers, "Authorization: Basic " );
	strcat( headers, http_auth_cookie );
	strcat( headers, EOL );
	break;
    case http_auth_digest:
/* RFC2069 Says:
--- begins ---
     response-digest     =
          <"> < KD ( H(A1), unquoted nonce-value ":" H(A2) > <">

     A1             = unquoted username-value ":" unquoted realm-value
                                                ":" password
     password       = < user's password >
     A2             = Method ":" digest-uri-value
--- ends ---

Which means, in English...
   A1 = username + ":" + realm + ":" password
   A2 = method + ":" + uri
   x = nonce + ":" + MD5( A2 )
   y = MD5( A1 ) + ":" + x
   z = MD5( y )
   ( and response-digest = "\"" + z + "\"", but we only need to 
   calculate z, which is put into http_auth_response );
*/
	a1 = malloc( strlen( http_username ) +
		     strlen( http_auth_realm ) +
		     strlen( http_password ) + 3 );
	strcpy( a1, http_username );
	strcat( a1, ":" );
	strcat( a1, http_auth_realm );
	strcat( a1, ":" );
	strcat( a1, http_password );
	DEBUG( DEBUG_HTTP, "A1: %s\n", a1 );
	a2 = malloc( strlen( req->method ) +
		     strlen( req->uri ) + 2 );
	strcpy( a2, req->method );
	strcat( a2, ":" );
	strcat( a2, req->uri );
	DEBUG( DEBUG_HTTP, "A2: %s\n", a2 );
	temp = md5_hex( a2, strlen( a2 ) );
	free( a2 );
	x = malloc( strlen( http_auth_nonce ) +
		    strlen( temp ) + 2 );
	strcpy( x, http_auth_nonce );
	strcat( x, ":" );
	strcat( x, temp );
	free( temp );
	DEBUG( DEBUG_HTTP, "x: %s\n", x );
	md5_a1 = md5_hex( a1, strlen( a1 ) );
	free( a1 );
	y = malloc( strlen( md5_a1 ) +
		    strlen( x ) + 2 );
	strcpy( y, md5_a1 ); 
	strcat( y, ":" );
	strcat( y, x );
	/* take a DEEP breath */
	auth_response = md5_hex( y, strlen( y ) );
	DEBUG( DEBUG_HTTP, "Response digest: %s\n", temp );
	free( x );
	free( y );
	snprintf( tmp, BUFSIZ, "Authorization: Digest "
		  "username=\"%s\", realm=\"%s\", "
		  "nonce=\"%s\", uri=\"%s\", response=\"%s\"",
		  http_username,
		  http_auth_realm,
		  http_auth_nonce,
		  req->uri,
		  auth_response );

	free( auth_response );

/* Now for the entity digest. RFC2069 says:
--- begins ---
entity-digest = <"> KD (H(A1), unquoted nonce-value ":" Method ":"
                           date ":" entity-info ":" H(entity-body)) <">
       ; format is <"> *LHEX <">

date = = rfc1123-date            ; see section 3.3.1 of [2]
entity-info = H(
          digest-uri-value ":"
          media-type ":"         ; Content-type, see section 3.7 of [2]
          *DIGIT ":"             ; Content length, see 10.12 of [2]
          content-coding ":"     ; Content-encoding, see 3.5 of [2]
          last-modified ":"      ; last modified date, see 10.25 of [2]
          expires                ; expiration date; see 10.19 of [2]
          )

last-modified   = rfc1123-date  ; see section 3.3.1 of [2]
expires         = rfc1123-date
--- ends ---

Since we do not pass last-modified, content-encoding, expires, or
content-encoding headers, we can skip those fields.

So what we do is this:

   ei = "::" + body_size + ":::"
   y = MD5( body )
   date = date from date header (in format specified by RFC1123)
   z = nonce + ":" + method + ":" + date + ":" + ei + ":" + y
   md5_a1 = MD5( A1 )    (from above)
   v = md5_a1 + ":" + z
   w = MD5( v )   where w is the unquoted entity-digest
*/
	if( req->body!=http_body_none ) {
	    temp = malloc( BUFSIZ );
	    snprintf( temp, BUFSIZ, "%d", req->body_size );
	    ei = malloc( strlen(temp) + 5 + 1 );
	    strcpy( ei, "::" );
	    strcat( ei, temp ); free( temp );
	    strcat( ei, ":::" );
	    DEBUG( DEBUG_HTTP, "Entity-info: %s\n", ei );
	    if( req->body == http_body_file ) {
		y = md5_hex_stream( req->body_file );
		rewind( req->body_file ); /* leave it at the beginning */
	    } else {
		y = md5_hex( req->body_buffer, req->body_size );
	    }
	    z = malloc( strlen( http_auth_nonce ) + strlen( req->method ) +
			strlen( req->date ) + strlen( ei ) + strlen( y ) + 5 );
	    strcpy( z, http_auth_nonce );
	    strcat( z, ":" );
	    strcat( z, req->method );
	    strcat( z, ":" );
	    strcat( z, req->date );
	    strcat( z, ":" );
	    strcat( z, ei ); free( ei );
	    strcat( z, ":" );
	    strcat( z, y ); free( y );
	    DEBUG( DEBUG_HTTP, "Z: %s\n", z );
	    v = malloc( strlen( md5_a1 ) + strlen( z ) + 2 );
	    strcpy( v, md5_a1 );
	    strcat( v, ":" );
	    strcat( v, z ); free( z );
	    DEBUG( DEBUG_HTTP, "V: %s\n", v );
	    w = md5_hex( v, strlen( v ) );
	    free( v );
	    DEBUG( DEBUG_HTTP, "W: %s\n", w );
	    /* Now lob it on the end */
	    strcat( tmp, ", digest=\"" );
	    strcat( tmp, w ); free( w );
	    strcat( tmp, "\"" );
	}
	free( md5_a1 );

	if( http_auth_opaque!=NULL ) {
	    strcat( tmp, ", opaque=\"" );
	    strcat( tmp, http_auth_opaque );
	    strcat( tmp, "\"" );
	}
	strcat( tmp, EOL );
	strcat( headers, tmp );
	break;
    default:
	break;
    }
}

/* This is called when a WWW-Authenticate header is received in the response.
 * It is used to perform any necessary computation required for presenting
 * authorisation information on the next (retry-with-auth) request.
 * i.e., to work out http_auth_cookie.
 * Basic auth is simple - we just take the string
 *   username:password
 * and Base64 it.
 * Digest auth is complex - read RFC2069.
 */
void http_auth_challenge( char *challenge ) {
    /* Parse the authentication challenge */
    char *temp, *key, *val, **auth_parms;
    int parm;
    size_t len;
    /* Here, we split the entire challenge into the challenge part and the
     * other parameters - everything before, and after the first space char,
     * respectively. 
     */
    len = strcspn( challenge, " " );
    DEBUG( DEBUG_HTTP, "Challenge: [%s]\n", challenge );
    if( len == 0 ) {
	DEBUG( DEBUG_HTTP, "Invalid challenge header: [%s]\n", challenge );
	return;
    }
    /* Now process it */
    if( strncasecmp( challenge, "basic", len ) == 0 ) {
	if( http_auth==http_auth_digest ) {
	    /* We're being spoofed, don't hand out the password.
	     * FIXME: Get this to the user, somehow. */
	    DEBUG( 0, 
"WARNING: Spoof attack probable. The server has switched to using basic\n"
"authentication from digest authenticaion, which may be an attempt to\n"
"discover your password. Basic auth will NOT be used.\n" );
	    return;
	}
	DEBUG( DEBUG_HTTP, "Using basic authentication.\n" );
	http_auth = http_auth_basic;
	temp = malloc( strlen(http_username) + strlen(http_password) + 2 );
	strcpy( temp, http_username );
	strcat( temp, ":" );
	strcat( temp, http_password );
	DEBUG( DEBUG_HTTP, "Uname/password pair: [%s]\n", temp );
	http_auth_cookie = base64( temp );
	DEBUG( DEBUG_HTTP, "Basic auth cookie: %s\n", http_auth_cookie );
	http_auth = http_auth_basic;
	free( temp );
    } else if( strncasecmp( challenge, "digest", len ) == 0 ) {
	/* Digest authentication */
	DEBUG( DEBUG_HTTP, "Using digest authentication.\n" );
	/* Get the parameter pairs. */
	auth_parms = strpairs( challenge+len+1, ',', '=',
			       http_quotes, http_whitespace );
	/* Chuck out the old nonce + realm */
	DOFREE( http_auth_nonce );
	DOFREE( http_auth_realm );
	DOFREE( http_auth_opaque );
	for( parm = 0; auth_parms[2*parm] != NULL; auth_parms+=2 ) {
	    key = auth_parms[2*parm];
	    val = auth_parms[2*parm+1];
	    DEBUG( DEBUG_HTTP, "Got: [%s] - [%s]\n", key, val );
	    /* Value is ALWAYS a quoted string - use strstrip to
	     * strip them off. */
	    if( strcasecmp( key, "nonce" ) == 0 ) {
		http_auth_nonce = strstrip(val, '\"');
	    } else if( strcasecmp( key, "realm" ) == 0 ) {
		http_auth_realm = strstrip(val, '\"');
	    } else if( strcasecmp( key, "opaque" ) == 0 ) {
		http_auth_opaque = strstrip(val, '\"');
	    }
	}
	DEBUG( DEBUG_HTTP, "Nonce: %s, realm: %s\n",
	       http_auth_nonce, http_auth_realm );    
	if( (http_auth_nonce != NULL) && (http_auth_realm != NULL) ) {
	    http_auth = http_auth_digest;
	} else {
	    http_auth = http_auth_none;
	    DEBUG( DEBUG_HTTP, "Could not use digest authentication.\n" );
	}
	strpairs_free( auth_parms );
    } else {
	http_auth = http_auth_none;
	DEBUG( DEBUG_HTTP, "Unrecognised authentication method.\n" );
    }
}

/* Sends the body down the socket.
 * Returns PROTO_OK on success, PROTO_ERROR otherwise */
int http_req_sendbody( http_req_t *req ) {
    int ret;
    switch( req->body ) {
    case http_body_file:
	ret = transfer( fileno(req->body_file), http_sock, req->body_size );
	rewind( req->body_file ); /* since we may have to send it again */
	break;
    case http_body_buffer:
	DEBUG( DEBUG_HTTP, "Sending body:\n%s\n", req->body_buffer );
	ret = send_string( http_sock, req->body_buffer );
	break;
    default:
	DEBUG( DEBUG_HTTP, "Argh in http_req_sendbody!" );
	ret = -1;
    }
    if( ret == -1 ) { 
	/* transfer failed */
	return PROTO_ERROR;
    } else {
	return PROTO_OK;
    }
}

/* Deal with the body size */
int http_req_bodysize( http_req_t *req ) {
    struct stat bodyst;
    /* Do extra stuff if we have a body */
    switch( req->body ) {
    case http_body_file:
	/* Get file length */
	if( fstat( fileno(req->body_file), &bodyst ) < 0 ) {
	    /* Stat failed */
	    DEBUG( DEBUG_HTTP, "Stat failed: %s\n", strerror( errno ) );
	    return PROTO_ERROR;
	}
	req->body_size = bodyst.st_size;
	break;
    case http_body_buffer:
	req->body_size = strlen( req->body_buffer );
	break;
    default:
	/* No body, so no size. */
	return PROTO_OK;
    }
    if( req->body != http_body_none ) {
	char tmp[BUFSIZ];
	/* Add the body length header */
	snprintf( tmp, BUFSIZ, "Content-Length: %d" EOL, req->body_size );
	strcat( req->headers, tmp );
    }
    return PROTO_OK;
}

void http_strcat_remoteserver( char *str ) {
    strcat( str, http_remotehost );
    /* Only add the port if it isn't 80 */
    if( http_remoteport != HTTP_PORT ) {
	static char buffer[128];
	snprintf( buffer, 128, ":%d", http_remoteport );
	strcat( str, buffer );
    }
}

/* Lob the User-Agent, connection and host headers in to the request
 * headers */
void http_req_fixedheaders( http_req_t *req ) {
    strcat( req->headers, "User-Agent: " );
    strcat( req->headers, http_useragent );
    strcat( req->headers, EOL 
	    "Connection: Keep-Alive" EOL 
	    "Host: " );
    http_strcat_remoteserver( req->headers );
    strcat( req->headers, EOL );
}

/* Decodes a URI */
char *uri_decode( const char *uri ) {
    const char *pnt;
    char *ret, *retpos, buf[5] = { "0x00\0" };
    retpos = ret = malloc( strlen( uri ) + 1 );
    for( pnt = uri; *pnt != '\0'; pnt++ ) {
	if( *pnt == '%' ) {
	    if( !isxdigit(pnt[1]) || !isxdigit(pnt[2]) ) {
		/* Invalid URI */
		return NULL;
	    }
	    buf[2] = *++pnt; buf[3] = *++pnt; /* bit faster than memcpy */
	    *retpos++ = strtol( buf, NULL, 16 );
	} else {
	    *retpos++ = *pnt;
	}
    }
    *retpos = '\0';
    return ret;
}

/* RFC2396 spake:
 * "Data must be escaped if it does not have a representation 
 * using an unreserved character".
 * ...where...
 *  unreserved  = alphanum | mark
 *  mark = "-" | "_" | "." | "!" | "~" | "*" | "'" | "(" | ")"
 * 
 * We need also to skip reserved characters
 * reserved    = ";" | "/" | "?" | ":" | "@" | "&" |
 *               "=" | "+" | "$" | ","
 */

/* Lookup table:
 * 1 marks an RESERVED character. 2 marks a UNRESERVED character.
 * 0 marks everything else. 
 */

#define RE 1
#define UN 2 
const short uri_chars[128] = {
/* 0 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 16 */  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 32 */  0, UN, 0, 0, RE, 0, RE, UN, UN, UN, UN, RE, RE, UN, UN, RE,
/* 48 */ UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, RE, RE, 0, RE, 0, RE,
/* 64 */ RE, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,
/* 80 */ UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, 0, 0, 0, 0, UN,
/* 96 */ 0, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,
/* 112 */ UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, 0, 0, 0, UN, 0 
};
#undef RE
#undef UN

/* Encodes the abspath segment of a URI.
 * TODO: Make this parse a complete URI */
char *uri_abspath_encode( const char *abs_path ) {
    const char *pnt;
    char *ret, *retpos;
    /* Rather than mess about growing the buffer, allocate as much as
     * the URI could possibly need, i.e. every character gets %XX
     * escaped. Hence 3 times input size. 
     */
    retpos = ret = malloc( strlen( abs_path ) * 3 + 1 );
    for( pnt = abs_path; *pnt != '\0'; pnt++ ) {
	/* Escape it:
	 *  - if it isn't 7-bit
	 *  - if it is a reserved character (but ignore '/')
	 *  - otherwise, if it is not an unreserved character
	 * (note, there are many characters that are neither reserved
	 * nor unreserved)
	 */
	if( *pnt<0 || (uri_chars[(int) *pnt] < 2 && *pnt!='/' )) {
	    /* Escape it - %<hex><hex> */
	    sprintf( retpos, "%%%02x", (unsigned char) *pnt );
	    retpos += 3;
	} else {
	    /* It's cool */
	    *retpos++ = *pnt;
	}
    }
    *retpos = '\0';
    return ret;
}

#ifdef URITEST

void fe_transfer_progress( size_t progress, size_t total ) {}

int main( int argc, char *argv[] ) {
    char *tmp;
    if( argc!=2 ) {
	printf( "doh. usage:\nuritest a_uri_abspath_segment\n"
		"e.g. uritest \"/this/is/a silly<filename>/but/hey\"\n" );
	exit(-1);
    }
    printf( "Input URI: %s\n", argv[1] );
    tmp = uri_abspath_encode( argv[1] );
    printf( "Encoded: %s\n", tmp );
    printf( "Decoded: %s\n", uri_decode( tmp ) );
    return 0;
}

#endif /* URITEST */

/* Initializes the request with given method and URI.
 * URI must be abs_path - i.e., NO scheme+hostname. It will BREAK 
 * otherwise. */
void http_request_init( http_req_t *req, 
			const char *method, const char *uri ) {
    /* Clear it out */
    memset( req, 0, sizeof( http_req_t ) );

    /* Add in the fixed headers */
    http_req_fixedheaders( req );

    /* Set the standard stuff */
    req->method = method;
    req->uri = uri_abspath_encode( uri );
    
    req->body_callback = NULL;
    req->body = http_body_none;
    
}

void http_request_end( http_req_t *req ) {
    if( req->uri != NULL ) {
	free( req->uri );
    }
}

/* The HTTP/1.x request/response mechanism 
 *
 * Returns:
 *   PROTO_OK if the request was made (not related to status code)
 *   PROTO_ERROR if the request could not be made
 * The STATUS CODE is placed in req->status. The error string is
 * placed in http_error.
 * 
 * TODO: This should be chopped up into smaller chunks, and get rid of
 * the horrid goto's.  
 */
int http_request( http_req_t *req ) {
    char tmp[BUFSIZ], buffer[BUFSIZ], *part, *name, *value;
    int ret, length, attempt, con_attempt;
    bool using_expect, /* whether we have sent a Expect: 100 header */
	close_connection,
	dead_connection,
	wants_body; /* whether the caller wants the response body
		     * callbacks */
    enum http_te_t encoding;

#ifdef USE_BROKEN_PROPFIND
    bool is_propfind = (strcmp( req->method, "PROPFIND" ) == 0);
#endif

#define HTTP_FATAL_ERROR(a) {				\
	DEBUG( DEBUG_HTTP, a );					\
	ret = PROTO_ERROR;					\
	close_connection = true;				\
	goto http_request_finish_proc;				\
    }

    /* Initialization... */
    DEBUG( DEBUG_HTTP, "Request started...\n" );
    strcpy( http_error, "Unknown error." );
    ret = PROTO_OK;

    if( http_req_bodysize( req ) != PROTO_OK )
	return PROTO_ERROR;

    /* I shall try this only twice...
     * First time, with default authentication stuff (either, what we
     * did last time, or none at all), then with up-to-the-minute
     * what-the-server-requested authentication stuff. */

    attempt = con_attempt = 1;

    do {
	char request[REQSIZ]; 

	/* The Request-Line */
	snprintf( request, REQSIZ, 
		  "%s %s HTTP/1.1" EOL, req->method, req->uri );
	/* The caller-supplied headers */
	strcat( request, req->headers );

	/* For the Date: header, and digest auth */
	req->date = rfc1123_date( time(NULL) );

	if( http_auth!=http_auth_none ) {
	    /* Add the authorization headers in */
	    http_auth_request( req, request );
	}

	/* Now handle the body. */
	using_expect = false;
	if( req->body!=http_body_none ) {
	    /* Add the date header */
	    strcat( request, "Date: " );
	    strcat( request, req->date );
	    strcat( request, EOL );
	    if( (http_expect_works > -1) &&
		(req->body_size > HTTP_EXPECT_MINSIZE) 
#ifdef USE_BROKEN_PROPFIND
		/* ... definitely NOT if we're doing PROPFIND */
		&& (!is_propfind)
#endif /* USE_BROKEN_PROPFIND */
		) {
		/* Add Expect: 100-continue. */
		strcat( request, "Expect: 100-continue" EOL );
		using_expect = true;
	    }
	}

	free( req->date ); /* that's enough of that */

	/* Final CRLF */
	strcat( request, EOL );
	
	/* Now send the request */

	/* Open the connection if necessary */
	if( !http_connected ) {
	    if( (ret = http_open()) != PROTO_OK )
		return ret;
	}

	dead_connection = false;

#ifdef USE_BROKEN_PROPFIND
	if( !is_propfind ) {
#endif
	/* Send the headers */
	DEBUG( DEBUG_HTTP, "Sending request headers:\n%s", request );
	if( send_string( http_sock, request ) < 0 ) {
	    dead_connection = true;
	    HTTP_FATAL_ERROR( "Could not send request!\n" );
	}

	DEBUG( DEBUG_HTTP, "Request sent.\n" );
	
#ifdef USE_BROKEN_PROPFIND
	}
#endif /* USE_BROKEN_PROPFIND */

	/* Now, if we are doing a Expect: 100, hang around for a short
	 * amount of time, to see if the server actually cares about the 
	 * Expect and sends us a 100 Continue response if the request
	 * is valid, else an error code if it's not. This saves sending
	 * big files to the server when they will be rejected.
	 */
	
	if( using_expect ) {
	    DEBUG( DEBUG_HTTP, "Waiting for response...\n" );
	    ret = sock_block( http_sock, HTTP_EXPECT_TIMEOUT );
	    switch( ret ) {
	    case -1: /* error */
		HTTP_FATAL_ERROR( "Wait (select) failed.\n" );
		break;
	    case 1: /* we got us a response! */
		DEBUG( DEBUG_HTTP, "Wait got data.\n" );
		http_expect_works = 1; /* it works - use it again */
		break;
	    case 0: 
		/* Timed out - i.e. Expect: ignored. There is a danger
		 * here that the server DOES respect the Expect: header,
		 * but was going SO slowly that it didn't get time to
		 * respond within HTTP_EXPECT_TIMEOUT.
		 * TODO: while sending the body, check to see if the
		 * server has sent anything back - if it HAS, then
		 * stop sending - this is a spec compliance SHOULD */
		DEBUG( DEBUG_HTTP, "Wait timed out.\n" );
		http_expect_works = -1; /* don't try that again */
		/* and give them the body */
		if( http_req_sendbody( req ) != PROTO_OK )
		    HTTP_FATAL_ERROR( "Could not send body.\n" );
		break;
	    }
	} else if( req->body != http_body_none ) {
#ifdef USE_BROKEN_PROPFIND
	    if( is_propfind ) {
		char *pfbuf;
		DEBUG( DEBUG_HTTP, "Broken PROPFIND handler...\n" );
		pfbuf = malloc( strlen(request) + req->body_size + 1 );
		memset( pfbuf, 0, strlen(request) + req->body_size + 1 );
		strcpy( pfbuf, request );
		strcat( pfbuf, req->body_buffer );
		DEBUG( DEBUG_HTTP, "Sending PROPFIND request...\n" );
		if( send_string( http_sock, pfbuf ) < 0 ) {
		    HTTP_FATAL_ERROR( "Could not send request.\n" );
		}
		DEBUG( DEBUG_HTTP, "Sending newline.\n" );
		if( send_line( http_sock, "" ) < 0 ) {
		    HTTP_FATAL_ERROR( "Could not send newline.\n" );
		}
		free( pfbuf );
	    } else {
		
#endif /* USE_BROKEN_PROPFIND */
	    /* Just chuck the file down the socket */
	    DEBUG( DEBUG_HTTP, "Sending body...\n" );
	    if( http_req_sendbody( req ) == PROTO_ERROR )
		HTTP_FATAL_ERROR( "Could not send body.\n" );
	    /* FIXME: This. Should it be here? */
	    DEBUG( DEBUG_HTTP, "Sending newline.\n" );
	    if( send_line( http_sock, "" ) < 0 ) {
		HTTP_FATAL_ERROR( "Could not send newline.\n" );
	    }
	    DEBUG( DEBUG_HTTP, "Body sent.\n" );
#ifdef USE_BROKEN_PROPFIND
	    }
#endif /* USE_BROKEN_PROPFIND */
	}
	
	/* Now, we have either:
	 *   - Sent the header and body, or
	 *   - Sent the header incl. Expect: line, and got some response.
	 * In any case, we get the status line of the response.
	 */
	
	/* HTTP/1.1 says we MUST be able to accept any number of
	 * 100 (Continue) responses prior to the normal response.
	 * So loop while we get them.
	 */
	
	do {
	    if( read_line( http_sock, tmp, BUFSIZ ) < 0 ) {
		dead_connection = true;
		HTTP_FATAL_ERROR( "Could not read status line.\n" );
	    }
	    
	    DEBUG( DEBUG_HTTP, "[Status Line] < %s", tmp );
	    
	    /* Got the status line - parse it */
	    if( http_parse_status( req, tmp ) == PROTO_ERROR )
		HTTP_FATAL_ERROR( "Could not parse status line.\n" );

	    if( req->class == 1 ) {
		DEBUG( DEBUG_HTTP, "Got 1xx-class.\n" );
		/* Skip any headers, we don't need them */
		do {
		    if( read_line( http_sock, tmp, BUFSIZ ) < 0 )
			HTTP_FATAL_ERROR( "Could not read header.\n" );
		    DEBUG( DEBUG_HTTP, "[Ignored header] < %s", tmp );
		} while( strcmp( tmp, EOL ) != 0 );
	
		if( using_expect && (req->status == 100) ) {
		    /* We are using Expect: 100, and we got a 100-continue 
		     * return code... send the request body */
		    DEBUG( DEBUG_HTTP, "Got continue... sending body now.\n" );
		    if( http_req_sendbody( req ) != PROTO_OK )
			HTTP_FATAL_ERROR( "Could not send body.\n" );
		    DEBUG( DEBUG_HTTP, "Body sent.\n" );
		}
	    }
	} while( req->class == 1 );
	
	/* We've got the real status line... now get the headers */
	
	length = -1;
	encoding = http_te_unknown;
	close_connection = false;
	
	/* Now read the rest of the header... up to the next blank line */
	while( read_line( http_sock, buffer, BUFSIZ ) > 0 ) {
	    char extra[BUFSIZ], *pnt;
	    DEBUG( DEBUG_HTTP, "[Header:%d] < %s", strlen(buffer), buffer );
	    if( strcmp( buffer, EOL ) == 0 ) {
		DEBUG( DEBUG_HTTP, "CRLF: End of headers.\n" );
		break;
	    }
	    while(true) {
		/* Collect any extra lines into buffer */
		ret = sock_recv( http_sock, extra, 1, MSG_PEEK);
		if( ret <= 0 ) {
		    HTTP_FATAL_ERROR( "Couldn't peek at next line.\n" );
		}
		if( extra[0] != ' ' && extra[0] != '\t' ) {
		    /* No more headers */
		    break;
		}
		ret = read_line( http_sock, extra, BUFSIZ );
		if( ret == -2 ) {
		    /* No newline within BUFSIZ bytes. This is a 
		     * loooong header. */
		    DEBUG( DEBUG_HTTP, 
			   "Header line longer than buffer, skipped.\n" );
		    break;
		} else if( ret <= 0 ) { 
		    HTTP_FATAL_ERROR( "Couldn't read next header line.\n" );
		} else {
		    DEBUG( DEBUG_HTTP, "[Cont:%d] < %s", strlen(extra), extra);
		}
		/* Append a space to the end of the last header, in
		 * place of the CRLF. */
		pnt = strchr( buffer, '\r' );
		pnt[0] = ' '; pnt[1] = '\0';
		/* Skip leading whitespace off next line */
		for( pnt = extra; *pnt!='\0' && 
			 ( *pnt == ' ' || *pnt =='\t' ); pnt++ ) /*oneliner*/;
		DEBUG( DEBUG_HTTP, "[Continued] < %s", pnt );
		if( strlen(buffer) + strlen(pnt) >= BUFSIZ ) {
		    DEBUG( DEBUG_HTTP, "Exceeded header buffer space.\n" );
		    /* Note, we don't break out of the loop here, cos
		     * we need to collect all the continued lines */
		} else {
		    strcat( buffer, pnt );
		}
	    }
	    /* Now parse the header line. This is all a bit noddy. */
	    name = strtok( buffer, ":" );
	    if( name != NULL ) {
		value = strtok( NULL, ":" );
		if( value != NULL ) {
		    /* Strip leading whitespace, and trailing CRLF */
		    while( *value!='\0' && *value==' ' ) value++;
		    if( (part = strchr( value, '\r' )) != NULL ) 
			*part = '\0';
		    if( (part = strchr( value, '\n' )) != NULL ) 
			*part = '\0';
		} else {
		    /* No value given */
		    value = "";
		}

		if( strcasecmp( name, "Content-Length" ) == 0 ) {
		    /* TODO: 2068 says we MUST notify the user if this
		     * is not a real number. */
		    length = atoi( value );
		} else if( strcasecmp( name, "Transfer-Encoding" ) == 0 ) {
		    if( strcasecmp( value, "chunked" ) == 0 ) {
			encoding = http_te_chunked;
		    } else {
			encoding = http_te_unknown;
		    }
		} else if( strcasecmp( name, "Connection" ) == 0 ) {
		    if( strcasecmp( value, "close" ) == 0 ) {
			close_connection = true;
		    }
		} else if( strcasecmp( name, "WWW-Authenticate" ) == 0 ) {
		    /* Parse the authentication challenge */
		    http_auth_challenge( value );
		} else if( req->hdrs_callback != NULL ) {
		    (*req->hdrs_callback)( name, value );
		}
	    }
	}

	/* Body length calculation, bit icky.
	 * Here, we set:
	 * length==-1 if we DO NOT know the exact body length
	 * length>=0 if we DO know the body length.
	 *
	 * RFC2068, section 4.3: 
	 * NO body is returned if the method is HEAD, or the resp status
	 * is 204 or 304
	 */
	if( (strcmp( req->method, "HEAD" ) == 0 ) ||
	    req->status==204 ||
	    req->status==304 ) {
	    length = 0;
	} else {
	    /* RFC2068, section 4.4: if we have a transfer encoding
	     * and a content-length, then ignore the content-length. */
	    if( ( length>-1 ) && (encoding!=http_te_unknown ) ) {
		length = -1;
	    }
	}

	/* The caller only wants the body if this request
	 * was 2xx class. */
	/* FIXME: Let the caller decide when they want it*/
	wants_body = (req->class == 2);

	/* Now, read the body */

	if( length == 0 ) {
	    /* Doing a HEAD, [23]04, or got Content-Length: 0.
	     * Nowt to do. */
	    DEBUG( DEBUG_HTTP, "No response body to fetch.\n" );
	} else if( encoding==http_te_chunked ) {
	    /* The response body is encoded using 'chunks'.
	     * It goes:  `SIZE CRLF CHUNK CRLF SIZE CRLF CHUNK CRLF ...'
	     * ended by a `SIZE CHUNK 0 CRLF', a 0-byte chunk */
	    int chunk_size; /* the size of the next chunk */
	    char *chunk; /* the data of the next chunk */

	    DEBUG( DEBUG_HTTP, "Reading chunked body.\n" );

	    if( req->resp_body_size )
		*req->resp_body_size = 0;

	    do {
		/* Pity the HTTP spec doesn't specify a maximum
		 * chunk size */
		if( read_line( http_sock, buffer, BUFSIZ ) < 0 )
		    HTTP_FATAL_ERROR( "Could not read chunk size.\n" );
		DEBUG( DEBUG_HTTP, "[Chunk Size] < %s", buffer );
		sscanf( buffer, "%x", &chunk_size );
		DEBUG( DEBUG_HTTP, "Got chunk size: %d\n", chunk_size );
		if( req->resp_body_size ) /* best we can do */
		    *req->resp_body_size += chunk_size;
		if( chunk_size > 0 ) {
		    chunk = malloc( chunk_size + 1 );
		    if( read_data( http_sock, chunk, chunk_size ) < 0 ) {
			/* It broke */
			free( chunk );
			HTTP_FATAL_ERROR( "Could not read chunk data.\n" );
		    }
		    *(chunk+chunk_size) = '\0';
		    DEBUG( DEBUG_SOCKET, "<<< Chunk:\n%s\n", chunk );
		    /* Do they want to have a look? */
		    if( req->body_callback && wants_body )
			(*req->body_callback)( req->body_callback_userdata,
					       chunk, chunk_size );
		    /* Now chuck the chunk, we don't want it */
		    free( chunk ); 
		    /* Read till CRLF - skip trailing headers */
		    do {
			if( read_line( http_sock, buffer, BUFSIZ ) < 0 ) {
			    /* It broke */
			    HTTP_FATAL_ERROR( "Could not read trailer.\n" );
			}
			DEBUG( DEBUG_HTTP, "Read trailer line: %s", buffer );
		    } while( strcmp( buffer, EOL ) != 0 );
		} else {
		    /* Finished getting body */
		    if( req->body_callback && wants_body )
			(*req->body_callback)( req->body_callback_userdata,
					       NULL, 0 );
		}
	    } while( chunk_size > 0 ); /* loop until 0-length chunk */
	    /* Read in that last line */
	    DEBUG( DEBUG_HTTP, "Reading trailing newline.\n" );
	    if( read_line( http_sock, buffer, BUFSIZ ) < 0 ) {
		/* It broke, kill it */
		HTTP_FATAL_ERROR( "Could not reading trailing CRLF.\n" );
	    }
	    DEBUG( DEBUG_HTTP, "[Trail] < %s", buffer );

	} else if( length > 0 ) {
	    char block[BUFSIZ+1]; /* +1 so we can null-term it */
	    int block_size;
	    /* No encoding, but we've got a Content-Length. */
	    if( req->resp_body_size )
		*req->resp_body_size = length;
	    while( length > 0 ) {
		DEBUG( DEBUG_HTTP,
		      "Reading block of size: %d\n", min(BUFSIZ,length) );
		block_size = sock_read( http_sock, block, min(BUFSIZ,length) );
		if( block_size < 1 )
		    HTTP_FATAL_ERROR( "Block read failed.\n" );
		block[block_size] = '\0';
		/* Let them inspect the block */
		DEBUG( DEBUG_HTTP, "[Body block: Size %d] <<<\n%s\n", 
		       block_size, block );
		length -= block_size;
		DEBUG( DEBUG_HTTP, "Body length remaining: %d\n", length );
		if( req->body_callback && wants_body )
		    (*req->body_callback)( req->body_callback_userdata,
					   block, block_size );
	    }
	    if( req->body_callback && wants_body )
		(*req->body_callback)( req->body_callback_userdata,
				       block, 0 );
	    DEBUG( DEBUG_HTTP, "Read body successfully.\n" );
	} else {
	    /* Unknown body length - read till socket is closed.
	     * TODO: Merge this into the above case. */
	    char block[BUFSIZ+1]; /* +1 so we can null-term it */
	    int block_size;
	    DEBUG( DEBUG_HTTP, "Reading body till server closes socket.\n" );
	    /* We have to close the connection after we're done here, 
	     * since that is the method by which the message is 
	     * delimited. */
	    close_connection = true;
	    do {
		block_size = sock_read( http_sock, block, BUFSIZ );
		if( block_size < 0 )
		    HTTP_FATAL_ERROR( "Block read failed.\n" );
		if( block_size > 0 ) {
		    block[block_size] = '\0';
		    /* Let them inspect the block */
		    DEBUG( DEBUG_HTTP, "[Body block: Size %d] <<<\n%s\n", 
			   block_size, block );
		    if( req->body_callback && wants_body )
			(*req->body_callback)( req->body_callback_userdata,
					       block, block_size );
		} else {
		    DEBUG( DEBUG_HTTP, "Socket closed remotely.\n" );
		}
	    } while( block_size > 0 );
	    if( req->body_callback && wants_body )
		(*req->body_callback)( req->body_callback_userdata,
				       block, 0 );
	    DEBUG( DEBUG_HTTP, "Read body successfully.\n" );
	}
	
	/* If they got this far, it's worked */
	ret = PROTO_OK;
    
http_request_finish_proc:
	/* Now, do we close the connection? */
	if( close_connection ) {
	    DEBUG( DEBUG_HTTP, "Forced connection close.\n" );
	    http_close( );
	}
    
	if( !dead_connection && (http_auth == http_auth_none) )
	    /* We can't authenticate... give up.
	     * http_auth will have been set by http_auth_challenge
	     * by know if it found out a way in which we can 
	     * authenticate with the server. */
	    break;
	
	/* Now, do that all *again* if it didn't work.
	 * Otherwise, give up */

    } while( (dead_connection && (++con_attempt<4)) || ((++attempt<3) && (req->status==401)) );

    DEBUG( DEBUG_HTTP, "Req ends, status %d class %dxx, status line:\n%s\n", 
	   req->status, req->class, http_error );

    return ret;
}

/* Simple HTTP put. 
 * local is the local filename. Remote is the destination URI (URI?)
 * Make it proper.
 * Returns:
 *   PROTO_FILE if no local file
 *   PROTO_ERROR if something general goes wrong
 *   PROTO_OK if it all works fine
 */
int http_put( const char *local, const char *remote, const bool ascii ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "PUT", remote );

    req.body_file = fopen( local, "r" );
    if( req.body_file == NULL ) {
	strcpy( http_error, "Could not open file." );
	ret = PROTO_FILE;
    } else {
	req.body = http_body_file;
	
	ret = http_request( &req );
	fclose( req.body_file );
	
	if( ret == PROTO_OK && req.class != 2 )
	    ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

int http_get_fd;
bool http_get_working;
size_t http_get_total, http_get_progress;

void http_get_callback( void *user, const char *buffer, const size_t len ) {
    if( !http_get_working ) return;
    DEBUG( DEBUG_HTTP, "Got progress: %d out of %d\n", len, http_get_total );
    if( send_data( http_get_fd, buffer, len ) < 0 ) {
	http_get_working = false;
    } else {
	http_get_progress += len;
    }
    fe_transfer_progress( http_get_progress, http_get_total ); 
}

int http_get( const char *local, const char *remote, const int remotesize,
	      const bool ascii ) {
    http_req_t req;
    int ret;

#if defined (__EMX__) || defined (__CYGWIN__)
    /* We have to set O_BINARY, thus need open(). Otherwise it should be
       equivalent to creat(). */
    http_get_fd = open( local, O_WRONLY|O_TRUNC|O_CREAT|O_BINARY, 0644 );
#else
    http_get_fd = creat( local, 0644 );
#endif
    if( http_get_fd < 0 ) {
	snprintf( http_error, BUFSIZ, "Could not open local file: %s", 
		  strerror( errno ) );
	return PROTO_ERROR;
    }

    http_request_init( &req, "GET", remote );
    req.body_callback = http_get_callback;

    http_get_working = true;
    http_get_progress = 0;
    http_get_total = remotesize;

    DEBUG( DEBUG_HTTP, "Total remote size: %d\n", remotesize );

    ret = http_request( &req );
    
    if( close( http_get_fd ) < 0 ) {
	snprintf( http_error, BUFSIZ, "Error closing local file: %s",
		  strerror( errno ) );
	ret = PROTO_ERROR;
    } else if( ret == PROTO_OK && req.class != 2 ) {
	ret = PROTO_ERROR;
    }
    http_request_end( &req );
    return ret;
}

/* Perform the file operations */
int dav_move( const char *from, const char *to ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "MOVE", from );

    strcat( req.headers, "Destination: http://" );
    http_strcat_remoteserver( req.headers );
    strcat( req.headers, to );
    strcat( req.headers, EOL );

    if( ! http_webdav_server ) {
	/* For non-WebDAV servers */
	strcat( req.headers, "New-URI: " );
	strcat( req.headers, to );
	strcat( req.headers, EOL );
    }

    ret = http_request( &req );
    
    if( ret == PROTO_OK && req.class != 2 ) {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

/* Deletes the specified resource on the server */
int http_delete( const char *filename ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "DELETE", filename );
    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;

    http_request_end( &req );
    return ret;
}

/* Deletes the specified resource on the server.
 * I wish we could do Depth: 0 with DELETE. */
int dav_rmdir( const char *filename ) {
    http_req_t req;
    int ret;
    char *dirname;
    
    if( strlen( filename ) < 1 ) {
	return PROTO_ERROR;
    }

    if( *(filename+strlen(filename)) != '/' ) {
	dirname = malloc( strlen( filename ) + 2 );
	strcpy( dirname, filename );
	strcat( dirname, "/" );
    } else {
	dirname = strdup( filename );
    }

    http_request_init( &req, "DELETE", dirname );
    
    ret = http_request( &req );

    /* Check for 204 here -- if we get a 207, something has gone 
     * wrong */
    if( ret == PROTO_OK && req.status != 204 )
	ret = PROTO_ERROR;

    free( dirname );
    http_request_end( &req );
    return ret;

}

int http_mkdir_with_put( const char *realdir ) {
    char filename[BUFSIZ];
    http_req_t req;
    int ret;

    strcpy( filename, realdir );
    strcat( filename, "SitecopyTempDirCreator.txt" );

    http_request_init( &req, "PUT", filename );
    /* Can't send Content-Type, would break digest auth entity
     * digest code
    strcat( req.headers, "Content-type: text/plain" EOL );
     */
    
    ret = http_request( &req );

    if( ret == PROTO_OK && req.class == 2 ) {
	http_request_end( &req );
	http_request_init( &req, "DELETE", filename );

	ret = http_request( &req );
	if( ret != PROTO_OK || req.class != 2 )
	    /* NEED an FE warning function here */
            DEBUG( DEBUG_HTTP,
		   "Can\'t delete the temporary file http://%s:%d%s",
		   http_remotehost, http_remoteport, filename );
	ret = PROTO_OK;
    } else {
	ret = PROTO_ERROR;
    }

    http_request_end( &req );
    return ret;
}

int dav_mkcol( const char *dirname ) {
    http_req_t req;
    int ret;
    char *realdir;

    if( strlen( dirname ) < 1 ) {
	strcpy( http_error, "Invalid directory name." );
	return PROTO_ERROR;
    }
    if( *(dirname+strlen(dirname)-1) == '/' ) {
	realdir = strdup( dirname );
    } else {
	/* +2 since one for \0, one for / */
	realdir = malloc( strlen(dirname) + 2 );
	strcpy( realdir, dirname );
	strcat( realdir, "/" );
    }
    
    if( http_mkdir_works == 1 ) {
	/* Use MKDIR, since we know it works */

	http_request_init( &req, "MKDIR", realdir );
	ret = http_request( &req );

	if( ret == PROTO_OK && req.class != 2 )
	    ret = PROTO_ERROR;

    } else {
	/* Try MKCOL first */
	http_request_init( &req, "MKCOL", realdir );
	
	ret = http_request( &req );
	
	if( ret == PROTO_OK && req.class == 2 ) {
	    /* MKCOL works -> MKDIR doesn't */
	    http_mkdir_works = -1;
	} else if( (http_mkdir_works > -1) && !http_webdav_server ) { 
	    /* MKCOL failed, we're not on a DAV server, lets try MKDIR */
	    http_request_end( &req );
	    http_request_init( &req, "MKDIR", realdir );
	    ret = http_request( &req );
	    if( ret == PROTO_OK && req.class == 2 ) {
		/* MKDIR does work */
		http_mkdir_works = 1;
		ret = PROTO_OK;
	    } else { 
		/* MKDIR doesn't work */
		if( req.status == 501 ) {
		    /* MKDIR definitely isn't implemented */
		    http_mkdir_works = -1;
		}
		/* Try a dummy PUT/DELETE */
		return http_mkdir_with_put( realdir );
	    }
	} else {
	    ret = PROTO_ERROR;
	}
    }

    free( realdir );
    http_request_end( &req );
    return ret;
}

int http_open( void ) {
    DEBUG( DEBUG_SOCKET, "Connecting to %s:%d...\n", http_remotehost, http_remoteport );
    fe_connection( fe_connecting );
    http_sock = socket_connect( http_remoteaddr, http_remoteport );
    if( http_sock < 0 ) {
	DEBUG( DEBUG_SOCKET, "Could not connect: %s\n", strerror( errno ) );
	return PROTO_CONNECT;
    }
    DEBUG( DEBUG_SOCKET, "Connected.\n" );
    fe_connection( fe_connected );
    http_connected = true;
    return PROTO_OK;
}

int http_close( void ) {
    DEBUG( DEBUG_SOCKET, "Closing socket.\n" );
    socket_close( http_sock );
    http_connected = false;
    DEBUG( DEBUG_SOCKET, "Socket closed.\n" );
    return PROTO_OK;
}

int dav_mkref( const char *resref, const char *target ) {
    http_req_t req;
    int ret;

    http_request_init( &req, "MKREF", resref );
    strcat( req.headers, "Ref-Target: <" );
    strcat( req.headers, target );
    strcat( req.headers, ">" EOL );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}

int dav_chref( const char *resref, const char *target ) {
    int ret;

    /* Delete it */
    ret = dav_rmref( resref );
    if( ret != PROTO_OK ) 
	return ret;
    /* Then create it again */
    ret = dav_mkref( resref, target );

    return ret;
}

int dav_rmref( const char *resref ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "DELETE", resref );
    strcat( req.headers, "No-Passthrough: 1" EOL );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}

#ifdef HAVE_LIBEXPAT

/* TODO: Go read the XML specs again and use the correct terminology
 * for everything.  
 */

XML_Parser dav_xml_parser;

/* Parses the tag attributes, and handles XML namespaces. 
 * With a little bit of luck.
 * Return true on success. */
bool dav_xml_parsetag( struct dav_xml_state *state,
		       const char *tag, const char **atts ) {
    dav_xml_ns *ns;
    int attn;
    char *pnt;
    const char *tag_prefix, *tag_suffix;
    struct dav_xml_state *xmlt;

    DEBUG( DEBUG_XML, "Parsing tag of name: [%s]\n", tag );
    /* Parse the atts for namespace declarations */
    for( attn = 0; atts[attn]!=NULL; attn+=2 ) {
	DEBUG( DEBUG_XML, "Got attribute: [%s] = [%s]\n", atts[attn], atts[attn+1] );
	if( strcasecmp( atts[attn], "xmlns" ) == 0 ) {
	    /* New default namespace */
	    state->default_ns = strdup( atts[attn+1] );
	    DEBUG( DEBUG_XML, "New default namespace: %s\n", 
		   state->default_ns );
	} else if( strncasecmp( atts[attn], "xmlns:", 6 ) == 0 ) {
	    /* New namespace scope */
	    ns = malloc( sizeof( dav_xml_ns ) );
	    ns->next = state->nspaces;
	    state->nspaces = ns;
	    ns->name = strdup( atts[attn]+6 ); /* skip the xmlns= */
	    ns->value = strdup( atts[attn+1] );
	    DEBUG( DEBUG_XML, "New namespace scope: %s -> %s\n",
		   ns->name, ns->value );
	}
    }
    /* Now check the tag name for a namespace scope */
    pnt = strchr( tag, ':' );
    tag_prefix = NULL;
    tag_suffix = NULL;
    if( pnt == NULL ) {
	/* No scope - have we got a default? */
	DEBUG( DEBUG_XML, "No scope found, searching for default.\n" );
	for( xmlt = state; xmlt!=NULL; xmlt=xmlt->parent ) {
	    if( xmlt->default_ns != NULL ) {
		tag_prefix = xmlt->default_ns;
		break;
	    }
	}
	if( tag_prefix != NULL ) {
	    DEBUG( DEBUG_XML, "Found default namespace [%s]\n", tag_prefix );
	} else {
	    DEBUG( DEBUG_XML, "No default namespace, using empty.\n" );
	    tag_prefix = "";
	}
	tag_suffix = tag;
    } else {
	DEBUG( DEBUG_XML, "Got namespace scope. Trying to resolve..." );
	/* Have a scope - resolve it */
	for( xmlt = state; tag_prefix==NULL && xmlt!=NULL; xmlt=xmlt->parent ) {
	    for( ns = xmlt->nspaces; ns!=NULL; ns=ns->next ) {
		/* Just compare against the bit before the :
		 * pnt points to the colon. */
		if( strncasecmp( ns->name, tag, pnt-tag ) == 0 ) {
		    /* Scope matched! Hoorah */
		    tag_prefix = ns->value;
		    /* end the search */
		    break;
		}
	    }
	}
	if( tag_prefix != NULL ) {
	    DEBUG( DEBUG_XML, "Resolved scope to [%s]\n", tag_prefix );
	    /* The suffix is everything after the ':' */
	    tag_suffix = pnt+1;
	    if( *tag_suffix == '\0' ) {
		DEBUG( DEBUG_XML, "No element name after ':'. Failed.\n" );
		return false;
	    }
	} else {
	    DEBUG( DEBUG_XML, "Smeg. We lost it somewhere.\n" );
	    return false;
	}
    }
    /* here, we have tag_suffix and tag_prefix */
    DEBUG( DEBUG_XML, "prefix: [%s], suffix: [%s]\n", tag_prefix, tag_suffix );
    pnt = state->tag_name = malloc( strlen(tag_prefix) + strlen(tag_suffix) +
				    1 );
    strcpy( pnt, tag_prefix );
    strcat( pnt, tag_suffix );
    DEBUG( DEBUG_XML, "You gave me: %s, and I gave you this: %s\n", tag, pnt );
    return true;
}

/* This is the XML_StartElementHandler... called with the start of a
 * new element. */
void dav_xml_startelm( void *userdata, const char *tag, const char **atts ) {
    struct dav_xmldoc *doc = (struct dav_xmldoc *)userdata;
    struct dav_xml_state *s;
    int n;

    if( !doc->valid ) {
	/* We've stopped parsing */
	DEBUG( DEBUG_XML, "Parse died. Ignoring start of element: %s\n", tag );
	return;
    }
    /* Set the new state */
    s = malloc( sizeof(struct dav_xml_state) );
    memset( s, 0, sizeof(struct dav_xml_state) );
    s->parent = doc->current;
    doc->current = s;
    if( dav_xml_parsetag( s, tag, atts ) == false ) {
	/* nooo, it bombed */
	doc->valid = false;
	return;
    }
    /* Map the name to a tag */
    DEBUG( DEBUG_XML, "Mapping tag name... " );
    s->tag = dav_xml_unknown;
    for( n = 0; dav_xml_tagnames[n] != NULL; n++ ) {
	if( strcasecmp( dav_xml_tagnames[n], s->tag_name ) == 0 ) {
	    s->tag = n;
	    break;
	}
    }
    if( s->tag == dav_xml_unknown ) {
	DEBUG( DEBUG_XML, "unknown tag, ignoring.\n" );
	return;
    }
    DEBUG( DEBUG_XML, "mapped.\n" );

    /* Normally, we don't want to collect cdata */
    doc->want_cdata = false;

    /* expat is not a validating parser - check the new tag
     * is valid in the current context.
     */
    DEBUG( DEBUG_XML, "Checking context of tag (parent: %s)\n",
	   dav_xml_tagnames[s->parent->tag] );
    switch( s->parent->tag ) {
    case dav_xml_root:
	switch( s->tag ) {
	case dav_xml_multistatus:
	case dav_xml_response:
	    break;
	default:
	    doc->valid = false;
	    break;
	}
	break;
    case dav_xml_multistatus:
	switch( s->tag ) {
	case dav_xml_response:
	case dav_xml_responsedescription:
	    break;
	default:
	    doc->valid = false;
	    break;
	}
	break;
    case dav_xml_response:
	switch( s->tag ) {
	case dav_xml_href:
	case dav_xml_propstat:
	case dav_xml_responsedescription:
	case dav_xml_status:
	    break;
	default:
	    doc->valid = false;
	}
	break;
    default:
	break;
    }
    if( doc->valid == false ) {
	DEBUG( DEBUG_XML, "Parse error - not good context for this tag.\n" );
	return;
    }
    /* Now actually do something about it */
    switch( s->tag ) {
    case dav_xml_response:
	/* new file information */
	DEBUG( DEBUG_XML, "New file context.\n" );
	memset( doc->file, 0, sizeof(struct proto_file_t) );
	break;

    case dav_xml_href:
    case dav_xml_getcontentlength:
    case dav_xml_resourcetype:
    case dav_xml_getlastmodified:
	/* For these elements, collect the CDATA */
	DEBUG( DEBUG_XML, "Collecting CDATA...\n" );
	doc->want_cdata = true;
	break;
    case dav_xml_prop:
    case dav_xml_propstat:
    case dav_xml_collection:
    default:
	/* donothing */
	break;
    }
    if( doc->want_cdata ) {
	/* Ready the cdata buffer for new cdata */
	if( doc->cdata_buflen == -1 ) {
	    /* Create a buffer */
	    DEBUG( DEBUG_XML, "Allocating new cdata buffer.\n" );
	    doc->cdata = malloc( CDATABUFSIZ );
	    doc->cdata_buflen = CDATABUFSIZ;
	}
	/* Now zero-out the buffer. */
	memset( doc->cdata, 0, doc->cdata_buflen );
	/* And we've got nothing in it */
	doc->cdata_len = 0;
    }
}

/* Returns the depth of the file in the directory heirarchy,
 * i.e. 1 for /foo.html, 2 for /bar/norm.html, 4 for /a/b/c/d.html
 */
int dav_fetch_getdepth( const char *href ) {
    const char *pnt;
    int count = 0;
    for( pnt=href; *pnt != '\0'; pnt++ ) /* oneliner */
	if( *pnt == '/' ) count++;
    return count;
}

bool dav_fetch_parse_href( struct dav_xmldoc *doc ) {
    const char *tmp;
    char *dec;
    bool ret;
    size_t rootlen;
    DEBUG( DEBUG_HTTP, "Parsing href [%s]\n", doc->cdata );
    if( strncmp( doc->cdata, "http://", 7 ) == 0 ) {
	/* Absolute URI.
	 * Look for the path bit */
	DEBUG( DEBUG_HTTP, "Got absolute URI.\n" );
	tmp = strchr( doc->cdata+7, '/' );
	if( tmp == NULL ) {
	    DEBUG( DEBUG_HTTP, "No path segment found.\n" );
	    return false;
	}
    } else {
	tmp = doc->cdata;
    }
    DEBUG( DEBUG_HTTP, "Using abspath segment: %s\n", tmp );
    dec = uri_decode( tmp );
    DEBUG( DEBUG_HTTP, "Decoded is: [%s]. Root is [%s]\n", tmp,
	   doc->fetch_root );
    /* Now, dec points to the absPath section of the URI.
     * We check whether this resource is actually in the
     * collection we have done the PROPFIND against.
     */
    rootlen = strlen( doc->fetch_root );
    if( strncmp( dec, doc->fetch_root, rootlen ) != 0 ) {
	DEBUG( DEBUG_HTTP, "parse_href failed: root collection not matched." );
	ret = false;
    } else {
	/* We're in the right place */
	DEBUG( DEBUG_HTTP, "parse_href: Got [%s]\n", dec + rootlen );
	/* Filename must be the basename */
	doc->file->filename = strdup( base_name( dec + rootlen ) );
	/* And fill in the directory while we're here */
	doc->file->directory = dir_name( dec + rootlen );
	ret = true;
	DEBUG( DEBUG_HTTP, "parse_href: Filename [%s], Directory [%s]\n",
	       doc->file->filename, doc->file->directory );
    }
    free( dec );
    return ret;
}

void dav_fetch_gotresource( struct dav_xmldoc *doc ) {
    struct proto_file_t *current, *previous;

    DEBUG( DEBUG_HTTP, "Got resource:\n"
	   "filename = [%s] isdir = %s size = %d mtime = %s\n", 
	   doc->file->filename, doc->file->isdir?"true":"false", 
	   doc->file->size, rfc1123_date( doc->file->modtime ) );

    if( (strlen( doc->file->directory ) == 0) &&
	(strlen( doc->file->filename ) == 0 ) ) {
	DEBUG( DEBUG_HTTP, "Resource is root collection, ignoring.\n" );
	return;
    }
   
    if( doc->file->isdir ) {
	doc->file->directory[strlen(doc->file->directory)-1] = '\0';
    }
    DEBUG( DEBUG_HTTP, "Filename is really: %s\n", doc->file->filename );

    /* Depth in the hierarchy - i.e., how many directories deep the
     * resource is. */
    doc->file->depth = dav_fetch_getdepth( doc->file->directory );
    DEBUG( DEBUG_HTTP, "File is at depth: %d\n", doc->file->depth );

    /* Insert it into the list, keeping the list sorted by depth */
    for( current=doc->files, previous=NULL; current!=NULL; 
	 previous=current, current=current->next )
	/* one-liner */
	if( current->depth > doc->file->depth )
	    break;

    doc->file->next = current;

    if( previous == NULL ) {
	doc->files = doc->file;
    } else {
	previous->next = doc->file;
    }

    /* Create a new file, ready to be filled in */
    doc->file = malloc( sizeof( struct proto_file_t ) );
    memset( doc->file, 0, sizeof( struct proto_file_t ) );
}

/* End-of-element handler */
void dav_xml_endelm( void *userdata, const char *tag ) {
    struct dav_xmldoc *doc = (struct dav_xmldoc *)userdata;
    struct dav_xml_state *s;
    dav_xml_ns *this_ns, *next_ns;
    if( !doc->valid ) {
	/* We've stopped parsing */
	DEBUG( DEBUG_XML, "Parse died. Ignoring end of element: %s\n", tag );
	return;
    }
    s = doc->current;
    DEBUG( DEBUG_XML, "End of element %s.\n", tag);
    switch( s->tag ) {
    case dav_xml_href:
	doc->valid = dav_fetch_parse_href( doc );
	break;
    case dav_xml_getlastmodified:
	doc->file->modtime = rfc1123_parse( doc->cdata );
	if( doc->file->modtime == (time_t)-1 ) {
	    DEBUG( DEBUG_HTTP, "Date is not in RFC1123 format.\n" );
	    doc->valid = false;
	}
	break;
    case dav_xml_getcontentlength:
	doc->file->size = atoi( doc->cdata );
	break;
    case dav_xml_collection:
	doc->file->isdir = true;
	break;
    case dav_xml_response:
	dav_fetch_gotresource( doc );
	break;
    default:
	break;
    }
    /* Move the current pointer up the chain */
    doc->current = s->parent;
    DEBUG( DEBUG_XML, "Back in tag: %s\n", doc->current->tag_name );
    if( doc->want_cdata ) {
	/* Free the cdata buffer if it's grown to big for its boots. */
	if( doc->cdata_buflen > CDATASHRINK ) {
	    DEBUG( DEBUG_XML, "cdata buffer overgrown, freeing.\n" );
	    free( doc->cdata );
	    doc->cdata_buflen = -1;
	} 
	/* And we've stopped collecting it now, thanks */
	doc->want_cdata = false;
    }
    if( s->default_ns!=NULL ) free( s->default_ns );
    /* Free the namespaces */
    this_ns = s->nspaces;
    while( this_ns != NULL ) {
	next_ns = this_ns->next;
	free( this_ns );
	this_ns = next_ns;
    };
    free( s->tag_name );
    free( s );
    DEBUG( DEBUG_XML, "Cleanup okay.\n" );
}

/* CDATA handler. We keep the entire cdata for each element in
 * doc->cdata, and expand the buffer as necessary. */
void dav_xml_cdata( void *userdata, const char *cdata, int len ) {
    struct dav_xmldoc *doc = (struct dav_xmldoc *)userdata;
    size_t newlen;
    
    if( !doc->want_cdata ) return;
    /* First, if this is the beginning of the CDATA, skip all
     * leading whitespace, we don't want it. */
    if( doc->cdata_buflen < 0  ) {
	DEBUG( DEBUG_XML, "ALERT: Shouldn't be collecting.\n" );
	return;
    }
    DEBUG( DEBUG_XML, "Given %d bytes of cdata.\n", len );
    if( doc->cdata_len == 0 ) {
	size_t wslen = 0;
	/* Ignore any leading whitespace */
	while( wslen < len && 
	       ( cdata[wslen] == ' ' || cdata[wslen] == '\r' ||
		 cdata[wslen] == '\n' || cdata[wslen] == '\t' ) ) {
	    wslen++;
	}
	cdata += wslen;
	len -= wslen;
	DEBUG( DEBUG_XML, "Skipped %d bytes of leading whitespace.\n", 
	       wslen );
	if( len == 0 ) {
	    DEBUG( DEBUG_XML, "Zero bytes of content.\n" );
	    return;
	}
    }
    /* Work out whether we need to expand the cdata buffer to
     * include the new incoming cdata. We always keep one more
     * byte in the buffer than we need, for the null-terminator */
    for( newlen = doc->cdata_buflen; 
	 newlen < (doc->cdata_len + len + 1);
	 newlen += CDATABUFSIZ ) /* nullop */ ;
    if( newlen > doc->cdata_buflen ) { 
	size_t oldbuflen = doc->cdata_buflen;
	/* Reallocate bigger buffer */
	DEBUG( DEBUG_XML, "Growing CDATA buffer from %d to %d.\n",
	       oldbuflen, newlen );
	doc->cdata = realloc( doc->cdata, newlen );
	doc->cdata_buflen = newlen;
	/* Zero-out the new bit of buffer */
	memset( doc->cdata+oldbuflen, 0, newlen-oldbuflen );
    }
    /* Now simply copy the new cdata onto the end of the buffer */
    memcpy( doc->cdata+doc->cdata_len, cdata, len );
    doc->cdata_len += len;
    DEBUG( DEBUG_XML, "Collected %d bytes of cdata, buffer now:\n%s\n",
	   len, doc->cdata );
}

void http_get_content_charset( const char *name, const char *value ) {
    char **pairs;
    int n;
    if( strcasecmp( name, "Content-Type" ) == 0 ) {
	/* Get the charset so we can pass it on to expat */
	pairs = strpairs( value, ';', '=', http_quotes, http_whitespace );
	for( n = 0; pairs[n] != NULL; n+=2 ) {
	    if( (strcasecmp( pairs[n], "charset") == 0) &&
		pairs[n+1] != NULL ) {
		DOFREE( http_content_charset );
		/* Strip off the quotes */
		http_content_charset = strstrip( pairs[n+1], '\"' );
		DEBUG( DEBUG_HTTP, "Got content type charset: %s\n",
		       http_content_charset );
	    }
	}
	strpairs_free( pairs );
    }
    return;
}

void dav_xml_parsebody( void *userdata, 
			const char *buffer, const size_t len ) {
    struct dav_xmldoc *doc = userdata;
    int ret;
    /* duck out if it's broken */
    if( !doc->valid ) {
	DEBUG( DEBUG_XML, "Not parsing %d bytes!\n", len );
	return;
    }
    if( len == 0 ) {
	DEBUG( DEBUG_XML, "Got 0-length buffer, end of response.\n" );
	ret = XML_Parse( dav_xml_parser, "", 0, -1 );
    } else {
	DEBUG( DEBUG_XML, "Got %d length buffer.\n", len );
	ret = XML_Parse( dav_xml_parser, buffer, len, 0 );
    }
    DEBUG( DEBUG_XML, "XML_Parse returned %d\n", ret );
    if( ret == 0 ) {
	doc->valid = false;
    }
}

/* WebDAV fetch mode handler. */
int dav_fetch( const char *dirname, struct proto_file_t **files ) {
    http_req_t req;
    struct dav_xmldoc doc = {0};
    int ret;
    const char *propfind_body =
	"<?xml version=\"1.0\"?>" EOL /* should we use encoding=? */
	"<propfind xmlns=\"DAV:\">" EOL
	"  <prop>" EOL
	"    <getcontentlength/>" EOL
	"    <getlastmodified/>" EOL
	"    <resourcetype/>" EOL
	"  </prop>" EOL
	"</propfind>" EOL;
    const char *myheaders =
	"Content-Type: text/xml" EOL /* should we use charset=? */
	"Depth: infinity" EOL;
       
    dav_xml_parser = XML_ParserCreate( NULL );
    XML_SetElementHandler( dav_xml_parser, dav_xml_startelm, dav_xml_endelm );
    XML_SetCharacterDataHandler( dav_xml_parser, dav_xml_cdata );
    XML_SetUserData( dav_xml_parser, (void *) &doc );
    /* Create a dummy state to act as the root element in the 
     * tree. Just makes things a little bit easier since we can 
     * then always presume we have a ->parent element.
     */
    doc.root = malloc( sizeof( struct dav_xml_state ) );
    memset( doc.root, 0, sizeof( struct dav_xml_state ) );
    doc.root->tag = dav_xml_root;
    doc.root->tag_name = "@root@";
    /* And set it to the current element */
    doc.current = doc.root;
    /* Init the document */
    doc.files = NULL;
    doc.fetch_root = dirname;
    doc.valid = true; /* so far... */
    doc.file = malloc( sizeof( struct proto_file_t ) );
    memset( doc.file, 0, sizeof( struct proto_file_t ) );

    http_request_init( &req, "PROPFIND", dirname );
    req.body_callback = dav_xml_parsebody;
    req.body_callback_userdata = &doc;
    req.body = http_body_buffer;
    req.body_buffer = propfind_body;
    /* Add in the content type header */
    strcat( req.headers, myheaders );
    
    ret = http_request( &req );

    XML_ParserFree( dav_xml_parser );

    free( doc.root );

    if( ret == PROTO_OK && req.class == 2 && doc.valid ) {
	*files = doc.files;
	ret = PROTO_OK;
    } else {
	*files = NULL;
	ret = PROTO_ERROR;
    }
    http_request_end( &req );
    return ret;
}

#endif /* HAVE_LIBEXPAT */

int http_head( const char *directory ) {
    http_req_t req;
    int ret;
    
    http_request_init( &req, "HEAD", directory );

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}

void http_options_parsehdr( const char *name, const char *value ) {
    char **classes, **class;
    if( strcasecmp( name, "DAV" ) == 0 ) {
	DEBUG( DEBUG_HTTP, "Got OPTIONS header with value: %s\n", value );
	classes = strsplit( value, ',', http_quotes, http_whitespace );
	for( class = classes; *class!=NULL; class++ ) {
	    DEBUG( DEBUG_HTTP, "Got compliance class: [%s]\n", *class );
	    if( strncmp( *class, "1", 1 ) == 0 ) {
		DEBUG( DEBUG_HTTP, "Class 1 compliant server.\n" );
		http_webdav_server = true;
	    }
	}
	strsplit_free( classes );
    }
}

/* Performs an OPTIONS request.
 * Sets http_webdav_server appropriately.
 */
int http_options( const char *directory ) {
    http_req_t req;
    int ret;
    
    http_webdav_server = false;

    http_request_init( &req, "OPTIONS", directory );
    req.hdrs_callback = http_options_parsehdr;

    ret = http_request( &req );

    if( ret == PROTO_OK && req.class != 2 )
	ret = PROTO_ERROR;
    
    http_request_end( &req );
    return ret;
}
