/*
 * Copyright (c) 2000 Dave Jones <dave@powertweak.com>
 *
 * Can be freely distributed and used under the terms of the GNU GPL.
 *
 * DISCLAIMER:
 *	The authors cannot be held responsible for any loss incurred
 *	either directly, or indirectly as a result of running this
 *	utility. Like any other program, there may be bugs lurking
 *	someplace that haven't yet been noticed. As this program
 *	is tweaking low-level chipset registers, anything could happen.
 */

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <syslog.h>
#include <ctype.h>

#include <errno.h>
#include <sys/stat.h>

#include "version.h"
#include "pciutils.h"

#define FALSE 0
#define TRUE 1

/* Communication with libpci */
struct pci_access *pacc;

/* Prototypes */
void Optimise_CPU(void);
void Optimise_PCI(void);

#include "log.c"
#include "configfile.c"
#include "hardware/PCI/pci.c"
#include "proc/proc.c"

/**************************
    Main
 **************************/

int main(int argc, char **argv)
{
    int i;
    const char versionstring[]= "Powertweak-Linux v" POWERTWEAK_VERSION "\n(c) 2000 Dave Jones. [dave@powertweak.com]\n";
    const char TuningString1[]="[ 25%][......    ][PCI]\r";
    const char TuningString2[]="[ 75%][......    ][PROC]\r";
    const char TuningDone[]=   "[DONE][System tuned]    \n";
    const char options[] = "" GENERIC_OPTIONS ;
    const char help_msg[] = "\
Usage: Powertweak [<switches>]\n\
\n"
GENERIC_HELP
;

    if (argc == 2 && !strncasecmp(argv[1], "--version", 9)) {
		printf("v" POWERTWEAK_VERSION "\n");
		exit (EXIT_SUCCESS);
    }

    if (iopl(3)) {
		fprintf(stderr, "Powertweak can only be run by root.\n");
		exit(EXIT_FAILURE);

    } else {

		printf(versionstring);
		syslog(LOG_INFO, "%s", versionstring);

		if (open_log_file(&logfile)==TRUE) {
			fprintf(logfile, "%s", versionstring);
			close_log_file(&logfile);
		}

		/* Read in the config file. Abort program if it doesn't exist. */
		if (read_configfile() == FALSE) {
			exit(EXIT_FAILURE);
		}

		/* Set up the PCI access routines. */
		pacc = pci_alloc();
		pacc->error = die;

		/* Parse the command line arguments. */
		while ((i = getopt(argc, argv, options)) != -1)

			switch (i) {

				default:	if (parse_generic_option(i, pacc, optarg))
			    			break;
				bad:
							fprintf(stderr, help_msg, pacc->id_file_name);
							pci_cleanup(pacc);
							free (configfileptr);
							exit(EXIT_FAILURE);
			}

		if (optind < argc)
			goto bad;


// FIXME: A loop would be preferred here when we add more types of tuning. - DJ.

		/* PCI device tuning */
		printf (TuningString1);
		if (open_log_file(&logfile)==TRUE) {
	    		fprintf(logfile, "%s\n", TuningString1);
	    		close_log_file(&logfile);
		}
		pci_init(pacc);
		Optimise_PCI();
		pci_cleanup(pacc);


		/* /proc/sys tuning */
		printf (TuningString2);
		if (open_log_file(&logfile)==TRUE) {
	    		fprintf(logfile, "%s\n", TuningString2);
	    		close_log_file(&logfile);
		}
		Optimise_proc();


		/* Finished */
		if (open_log_file(&logfile)==TRUE) {
			fprintf(logfile, "%s\n", TuningDone);
		 	close_log_file(&logfile);
		}
		printf (TuningDone);

		free (configfileptr);

		exit(EXIT_SUCCESS);
	}
}
