/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>
#include <signal.h>
#include <errno.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

#ifndef HAVE_STRCASECMP
#ifdef __STDC__
int strcasecmp(const char *, const char *);
int strncasecmp(const char *, const char *, size_t);
#else /* not __STDC__ */
int strcasecmp();
int strncasecmp();
#endif /* not __STDC__ */
#endif /* not HAVE_STRCASECMP */

#ifndef HAVE_STRERROR
#ifdef __STDC__
char *strerror(int);
#else /* not __STDC__ */
char *strerror();
#endif /* not __STDC__ */
#endif /* HAVE_STRERROR */

/*
 * The maximum length of path name.
 */
#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX        MAXPATHLEN
#else /* not MAXPATHLEN */
#define PATH_MAX        1024
#endif /* not MAXPATHLEN */
#endif /* not PATH_MAX */

#include "fakelog.h"
#include "getopt.h"
#include "hostname.h"
#include "logpid.h"
#include "permission.h"
#include "readconf.h"
#include "wildcard.h"

#include "ndtpd.h"

/*
 * Unexported functions.
 */
static void output_version NDTPD_P((void));
static void output_help NDTPD_P((void));
static void output_try_help NDTPD_P((void));

/*
 * Program name.
 */
#define PROGRAM_NAME	"ndtpcontrol"

/*
 * Command line options.
 */
static const char *short_options = "c:hv";
static struct option long_options[] = {
    {"configuration-file", required_argument, NULL, 'c'},
    {"help",               no_argument,       NULL, 'h'},
    {"version",            no_argument,       NULL, 'v'},
    {NULL, 0, NULL, 0}
};

/*
 * Sub-commands.
 */
#define SUBCOM_STATUS		0
#define SUBCOM_RESTART		1
#define SUBCOM_TERMINATE	2
#define SUBCOM_KILL		3

typedef struct {
    const char *name;
    int id;
    int signal;
} Subcommand;

static const Subcommand subcommand_table[] = {
    {"status",    SUBCOM_STATUS,    -1},
    {"restart",   SUBCOM_RESTART,   SIGHUP},
    {"terminate", SUBCOM_TERMINATE, SIGTERM},
    {"kill",      SUBCOM_KILL,      SIGKILL},
    {NULL, 0}
};

/*
 * Unexported functions.
 */
#ifdef __STDC__
static const Subcommand *get_subcommand(const char *);
#else
static const Subcommand *get_subcommand();
#endif


int
main(argc, argv)
    int argc;
    char *argv[];
{
    const Subcommand *subcommand;
    pid_t pid;
    int ch;

    /*
     * Set program name and version.
     */
    invoked_name = argv[0];
    program_name = PROGRAM_NAME;
    program_version = VERSION;

    /*
     * Initialize global and static variables.
     */
    server_mode = SERVER_MODE_CONTROL;
    initialize_permission(&permissions);
    initialize_permission(&identifications);
    initialize_book_registry();

    if (PATH_MAX < strlen(DEFAULT_CONFIG_FILENAME)) {
	fprintf(stderr,
	    "%s: internal error, too long DEFAULT_CONFIG_FILENAME\n",
	    invoked_name);
	exit(1);
    }
    strcpy(configuration_filename, DEFAULT_CONFIG_FILENAME);

    /*
     * Set fakelog behavior.
     */
    set_fakelog_name(invoked_name);
    set_fakelog_mode(FAKELOG_TO_STDERR);
    set_fakelog_level(FAKELOG_ERR);

    /*
     * Parse command line options.
     */
    for (;;) {
	ch = getopt_long(argc, argv, short_options, long_options, NULL);
	if (ch == EOF)
	    break;
	switch (ch) {
	case 'c':
	    /*
	     * Option `-c file'.  Specify a configuration filename.
	     */
	    if (PATH_MAX < strlen(optarg)) {
		fprintf(stderr, "%s: too long configuration filename\n",
		    invoked_name);
		fflush(stderr);
		exit(1);
	    }
	    strcpy(configuration_filename, optarg);
	    break;

	case 'h':
	    /*
	     * Option `-h'.  Help.
	     */
	    output_help();
	    exit(0);

	case 'v':
	    /*
	     * Option `-v'.  Show the version number.
	     */
	    output_version();
	    exit(0);

	default:
	    output_try_help();
	    exit(1);
	}
    }

    /*
     * Check for the number of rest arguments.
     */
    if (argc - optind == 0) {
	fprintf(stderr, "%s: too few argument\n", invoked_name);
	output_try_help();
	exit(1);
    }
    if (1 < argc - optind) {
	fprintf(stderr, "%s: too many arguments\n", invoked_name);
	output_try_help();
	exit(1);
    }

    /*
     * Check for sub-command.
     */
    subcommand = get_subcommand(argv[optind]);
    if (subcommand == NULL) {
	output_try_help();
	exit(1);
    }

    /*
     * Read the configuration file.
     */
    if (read_configuration(configuration_filename, configuration_table) < 0) {
	fprintf(stderr, "%s: configuration failure\n", invoked_name);
	fflush(stderr);
	exit(1);
    }

    /*
     * Read PID from the pid-file.
     */
    if (probe_pid_file(pid_filename) < 0) {
	fprintf(stderr, "%s: PID file not found: %s\n", invoked_name,
	    pid_filename);
	fprintf(stderr, "%s: ndtpd not running\n", invoked_name);
	fflush(stderr);
	exit(1);
    }
    if (read_pid_file(pid_filename, &pid) < 0)
	exit(1);

    /*
     * When sub-command is `status', output a message and exit.
     */
    if (subcommand->id == SUBCOM_STATUS) {
	fprintf(stderr, "%s: PID file found: %s\n", invoked_name,
	    pid_filename);
	fprintf(stderr, "%s: ndtpd is running (PID = %u)\n", invoked_name,
	    (unsigned)pid);
	fflush(stderr);
	exit(0);
    }

    /*
     * Send a signal to the running `ndtpd'.
     */
    if (kill(pid, subcommand->signal) < 0) {
	fprintf(stderr, "%s: %s\n", invoked_name, strerror(errno));
	fflush(stderr);
	exit(1);
    }

    /*
     * When ndtpd is killed (send SIGKILL), ndtpd cannot removed
     * the pid file ndtpd.  The command try to remove it.
     */
    if (subcommand->id == SUBCOM_KILL) {
	if (remove_pid_file(pid_filename) < 0)
	    exit(1);
    }

    return 0;
}


static const Subcommand *
get_subcommand(subname)
    const char *subname;
{
    size_t sublen;
    const Subcommand *sub;
    const Subcommand *candidate;
    int matches;
    
    sublen = strlen(subname);
    matches = 0;
    for (sub = subcommand_table; sub->name != NULL; sub++) {
	if (strcasecmp(subname, sub->name) == 0)
	    return sub;
	else if (strncasecmp(subname, sub->name, sublen) == 0) {
	    candidate = sub;
	    matches++;
	}
    }

    if (matches == 0) {
	fprintf(stderr, "%s: unknown sub-command `%s'\n", invoked_name,
	    subname);
	fflush(stderr);
	return NULL;
    } else if (1 < matches) {
	fprintf(stderr, "%s: sub-command `%s' is ambigous\n", invoked_name,
	    subname);
	fflush(stderr);
	return NULL;
    }

    return candidate;
}


/*
 * Output the version number to standard out.
 */
static void
output_version()
{
    printf("%s (NDTPD) version %s\n", program_name, program_version);
    printf("Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara\n\n");
    printf("This is free software; you can redistribute it and/or modify\n");
    printf("it under the terms of the GNU General Public License as published by\n");
    printf("the Free Software Foundation; either version 2, or (at your option)\n");
    printf("any later version.\n\n");
    printf("This program is distributed in the hope that it will be useful,\n");
    printf("but WITHOUT ANY WARRANTY; without even the implied warranty\n");
    printf("of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    printf("GNU General Public License for more details.\n");
    fflush(stdout);
}


/*
 * Output the usage to standard out.
 */
static void
output_help()
{
    printf("Usage: %s [option...] sub-command\n", program_name);
    printf("Options:\n");
    printf("  -c FILE  --configuration-file FILE\n");
    printf("                             specify a configuration file\n");
    printf("                             (default: %s)\n",
	DEFAULT_CONFIG_FILENAME);
    printf("  -h  --help                 display this help, then exit\n");
    printf("  -v  --version              display version number, then exit\n");
    printf("\nSub-commands:\n");
    printf("  kill                       kill ndtpd (send SIGKILL)\n");
    printf("  restart                    restart ndtpd (send SIGHUP)\n");
    printf("  status                     output status of ndtpd\n");
    printf("  terminate                  terminate ndtpd (send SIGTERM)\n");
    printf("\nDefault value used in a configuration file:\n");
    printf("  work-path                  (default: %s)\n", 
	DEFAULT_WORK_PATH);
    printf("\nReport bugs to %s.\n", MAILING_ADDRESS);
    fflush(stdout);
}


/*
 * Output ``try ...'' message to standard error.
 */
static void
output_try_help()
{
    fprintf(stderr, "try `%s --help' for more information\n", invoked_name);
    fflush(stderr);
}


