/*
 * Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <sys/types.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef HAVE_STRCHR
#define strchr index
#define strrchr rindex
#endif /* HAVE_STRCHR */

/*
 * The maximum length of path name.
 */
#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX        MAXPATHLEN
#else /* not MAXPATHLEN */
#define PATH_MAX        1024
#endif /* not MAXPATHLEN */
#endif /* not PATH_MAX */

#include "fakelog.h"
#include "getopt.h"
#include "hostname.h"
#include "permission.h"
#include "readconf.h"
#include "wildcard.h"

#include "ndtpd.h"

/*
 * Unexported functions.
 */
static void output_version NDTPD_P((void));
static void output_help NDTPD_P((void));
static void output_try_help NDTPD_P((void));

/*
 * Program name.
 */
#define PROGRAM_NAME	"ndtpcheck"

/*
 * Command line options.
 */
static const char *short_options = "c:dhv";
static struct option long_options[] = {
    {"configuration-file", required_argument, NULL, 'c'},
    {"debug",              no_argument,       NULL, 'd'},
    {"help",               no_argument,       NULL, 'h'},
    {"verbose",            no_argument,       NULL, 'd'},
    {"version",            no_argument,       NULL, 'v'},
    {NULL, 0, NULL, 0}
};


int
main(argc, argv)
    int argc;
    char *argv[];
{
    int ch;

    /*
     * Set program name and version.
     */
    invoked_name = argv[0];
    program_name = PROGRAM_NAME;
    program_version = VERSION;

    /*
     * Initialize global and static variables.
     */
    server_mode = SERVER_MODE_CHECK;
    initialize_permission(&permissions);
    initialize_permission(&identifications);
    initialize_book_registry();

    if (PATH_MAX < strlen(DEFAULT_CONFIG_FILENAME)) {
	fprintf(stderr,
	    "%s: internal error, too long DEFAULT_CONFIG_FILENAME\n",
	    invoked_name);
	exit(1);
    }
    strcpy(configuration_filename, DEFAULT_CONFIG_FILENAME);

    /*
     * Set fakelog behavior.
     */
    set_fakelog_name(invoked_name);
    set_fakelog_mode(FAKELOG_TO_STDERR);
    set_fakelog_level(FAKELOG_ERR);

    /*
     * Parse command line options.
     */
    for (;;) {
	ch = getopt_long(argc, argv, short_options, long_options, NULL);
	if (ch == EOF)
	    break;
	switch (ch) {
	case 'c':
	    /*
	     * Option `-c file'.  Specify a configuration filename.
	     */
	    if (PATH_MAX < strlen(optarg)) {
		fprintf(stderr, "%s: too long configuration filename\n",
		    invoked_name);
		fflush(stderr);
		exit(1);
	    }
	    strcpy(configuration_filename, optarg);
	    break;

	case 'd':
	    /*
	     * Option `-d'.  Debug mode.
	     */
	    set_fakelog_level(FAKELOG_DEBUG);
	    break;

	case 'h':
	    /*
	     * Option `-h'.  Help.
	     */
	    output_help();
	    exit(0);

	case 'v':
	    /*
	     * Option `-v'.  Show the version number.
	     */
	    output_version();
	    exit(0);

	default:
	    output_try_help();
	    exit(1);
	}
    }

    /*
     * Check for the number of rest arguments.
     */
    if (0 < argc - optind) {
	fprintf(stderr, "%s: too many arguments\n", invoked_name);
	output_try_help();
	exit(1);
    }

    /*
     * Read the configuration file.
     */
    if (read_configuration(configuration_filename, configuration_table) < 0) {
	fprintf(stderr, "%s: configuration failure\n", invoked_name);
	fflush(stderr);
	exit(1);
    }

    return 0;
}


/*
 * Output the version number to standard out.
 */
static void
output_version()
{
    printf("%s (NDTPD) version %s\n", program_name, program_version);
    printf("Copyright (c) 1997, 1998, 1999  Motoyuki Kasahara\n\n");
    printf("This is free software; you can redistribute it and/or modify\n");
    printf("it under the terms of the GNU General Public License as published by\n");
    printf("the Free Software Foundation; either version 2, or (at your option)\n");
    printf("any later version.\n\n");
    printf("This program is distributed in the hope that it will be useful,\n");
    printf("but WITHOUT ANY WARRANTY; without even the implied warranty\n");
    printf("of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    printf("GNU General Public License for more details.\n");
    fflush(stdout);
}


/*
 * Output the usage to standard out.
 */
static void
output_help()
{
    printf("Usage: %s [option...]\n", program_name);
    printf("Options:\n");
    printf("  -c FILE  --configuration-file FILE\n");
    printf("                             specify a configuration file\n");
    printf("                             (default: %s)\n",
	DEFAULT_CONFIG_FILENAME);
    printf("  -d  --debug  --verbose     degug mode\n");
    printf("  -h  --help                 display this help, then exit\n");
    printf("  -v  --version              display version number, then exit\n");
    printf("\nDefault value used in a configuration file:\n");
    printf("  work-path                  (default: %s)\n", DEFAULT_WORK_PATH);
    printf("\nReport bugs to %s.\n", MAILING_ADDRESS);
    fflush(stdout);
}


/*
 * Output ``try ...'' message to standard error.
 */
static void
output_try_help()
{
    fprintf(stderr, "try `%s --help' for more information\n", invoked_name);
    fflush(stderr);
}


