/*
 * Copyright (c) 1997, 1998  Motoyuki Kasahara
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <syslog.h>

#if defined(STDC_HEADERS) || defined(HAVE_STRING_H)
#include <string.h>
#if !defined(STDC_HEADERS) && defined(HAVE_MEMORY_H)
#include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */
#else /* not STDC_HEADERS and not HAVE_STRING_H */
#include <strings.h>
#endif /* not STDC_HEADERS and not HAVE_STRING_H */

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_LIMITS_H
#include <limits.h>
#endif

#ifndef HAVE_STRCASECMP
#ifdef __STDC__
int strcasecmp(const char *, const char *);
int strncasecmp(const char *, const char *, size_t);
#else /* not __STDC__ */
int strcasecmp();
int strncasecmp();
#endif /* not __STDC__ */
#endif /* not HAVE_STRCASECMP */

/*
 * The maximum length of path name.
 */
#ifndef PATH_MAX
#ifdef MAXPATHLEN
#define PATH_MAX        MAXPATHLEN
#else /* not MAXPATHLEN */
#define PATH_MAX        1024
#endif /* not MAXPATHLEN */
#endif /* not PATH_MAX */

#include <eb/eb.h>
#include <eb/appendix.h>
#include <eb/language.h>

#include "ndtpd.h"
#include "permission.h"
#include "ticket.h"
#include "wildcard.h"

/*
 * Initialize the book list `book_registry'.
 */
void
initialize_book_registry()
{
    book_registry = NULL;
    book_registry_tail = NULL;
    book_count = 0;
    current_book = NULL;
}


/*
 * Clear the book list `book_registry'.
 * Free all nodes and allocated memories in `book_registry',
 * and then set it to NULL.
 */
void
clear_book_registry()
{
    Book *book = book_registry;
    Book *next;

    /*
     * Reset books list.  Free all malloced memories.
     */
    while (book != NULL) {
	clear_permission(&book->permissions);
	clear_ticket_stock(&book->ticket_stock);
	eb_clear(&book->book);
	eb_clear_appendix(&book->appendix);
	next = (Book *)book->next;
	if (book->path != NULL)
	    free(book->path);
	if (book->appendix_path != NULL)
	    free(book->appendix_path);
	free(book);
	book = next;
    }

    book_registry = NULL;
    book_registry_tail = NULL;
    book_count = 0;
    current_book = NULL;
}


/*
 * Count books in `book_registry'.
 */
int
count_book_registry()
{
    Book *book;
    int count;

    for (book = book_registry, count = 0;
	 book != NULL; book = (Book *)book->next, count++)
	;

    return count;
}


/*
 * Bind all books and appendixes in `book_registry'.
 */
int
activate_book_registry()
{
    Book *book;
    int avail = 0;
    char lock_filename[PATH_MAX + 1];

    /*
     * Unset the current book.
     */
    current_book = NULL;

    /*
     * If there is no book in the registry, do nothing.
     */
    if (book_registry == NULL)
	return;

    for (book = book_registry; book != NULL; book = (Book *)book->next) {
	/*
	 * Bind a book.
	 */
	if (eb_bind(&book->book, book->path) < 0) {
	    syslog(LOG_ERR, "the book is not available, %s: %s",
		eb_error_message(), book->path);
	    continue;
	}
	avail++;

	/*
	 * Initialize all subbooks when running as a standalone daemon.
	 */
	if (server_mode == SERVER_MODE_STANDALONE)
	    eb_initialize_all_subbooks(&book->book);

	/*
	 * Bind a ticket stock for the book.
	 */
	if (book->max_clients != 0) {
	    sprintf(lock_filename, "%s/%s.%s", work_path, &book->name,
		BOOK_LOCK_SUFFIX);
	    if (bind_ticket_stock(&book->ticket_stock, lock_filename,
		book->max_clients) < 0)
		return -1;
	}

	if (book->appendix_path != NULL) {
	    /*
	     * Bind an appendix.
	     */
	    if (eb_bind_appendix(&book->appendix, book->appendix_path) < 0) {
		syslog(LOG_ERR, "the appendix is not available, %s: %s",
		    eb_error_message(), book->appendix_path);
		continue;
	    }

	    /*
	     * Initialize all subbooks when running as a standalone daemon.
	     */
	    if (server_mode == SERVER_MODE_STANDALONE)
		eb_initialize_all_appendix_subbooks(&book->appendix);
	}
    }

    return avail;
}


/*
 * Add a new book to `book_registry'.
 * Upon return, `current_book' refers to the new book.
 * If succeeds, a node to the book is returned.
 * Otherwise NULL is returned.
 */
int
add_book()
{
    Book *book;

    book = (Book *)malloc(sizeof(Book));
    if (book == NULL) {
	syslog(LOG_ERR, "memory exhausted");
	return -1;
    }

    /*
     * Set initial values to the new book.
     */
    eb_initialize(&book->book);
    eb_initialize_appendix(&book->appendix);
    book->path = NULL;
    book->appendix_path = NULL;
    book->name[0] = '\0';
    book->title[0] = '\0';
    book->max_clients = DEFAULT_BOOK_MAX_CLIENTS;
    initialize_permission(&book->permissions);
    initialize_ticket_stock(&book->ticket_stock);
    book->next = NULL;
    if (book_registry == NULL)
	book_registry = book;
    else
	book_registry_tail->next = book;
    book_registry_tail = book;
    current_book = book;

    book_count++;

    return 0;
}


/*
 * Find the bound book with `name' in `book_regitry'.
 *
 * If found, the pointer to the book is returned.
 * Otherwise NULL is returned.
 */
Book *
find_book(name)
    const char *name;
{
    Book *book;

    /*
     * Find the bound book.
     */
    for (book = book_registry; book != NULL; book = (Book *)book->next) {
	if (eb_is_bound(&book->book) && strcasecmp(book->name, name) == 0)
	    return book;
    }

    return NULL;
}


/*
 * Search the current book for the subbook whose directory name is
 * `directory'.
 * If found, the subbook-code of the subbook is returned.
 * Otherwise, -1 is returned.
 */
EB_Subbook_Code
find_subbook(directory)
    const char *directory;
{
    EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];
    const char *directory2;
    int subcount;
    int i;

    /*
     * Find the subbook in the current book.
     */
    subcount = eb_subbook_list(&current_book->book, sublist);
    for (i = 0; i < subcount; i++) {
        directory2 = eb_subbook_directory2(&current_book->book, sublist[i]);
        if (directory2 == NULL)
	    continue;
	if (strcasecmp(directory, directory2) == 0)
	    return sublist[i];
    }

    return -1;
}


/*
 * Search the current appendix for the subbook whose directory name
 * is `directory'.
 * If found, the subbook-code of the subbook is returned.
 * Otherwise, -1 is returned.
 */
EB_Subbook_Code
find_appendix_subbook(directory)
    const char *directory;
{
    EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];
    const char *directory2;
    int subcount;
    int i;

    /*
     * Find the subbook in the current appendix.
     */
    subcount = eb_appendix_subbook_list(&current_book->appendix, sublist);
    for (i = 0; i < subcount; i++) {
        directory2 = eb_appendix_subbook_directory2(&current_book->appendix,
	    sublist[i]);
        if (directory2 == NULL)
	    continue;
        if (strcasecmp(directory, directory2) == 0)
	    return sublist[i];
    }

    return -1;
}


/*
 * For the current client,
 * check permission flags to the books in `book_registry'.
 */
void
check_book_permissions()
{
    Book *book;

    for (book = book_registry; book != NULL; book = (Book *)book->next) {
	book->permflag = test_permission(&book->permissions, client_hostname,
	    client_address, match_wildcard);
    }
}


/*
 * For the current client,
 * set permission flags to all books in `book_registry'.
 */
void
set_all_book_permissions()
{
    Book *book;

    for (book = book_registry; book != NULL; book = (Book *)book->next)
	book->permflag = 1;
}


